// dialogs.c
// LiVES (lives-exe)
// (c) G. Finch 2003
// Released under the GPL 2.0 or higher
// see file ../COPYING for licensing details

#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "main.h"
#include "interface.h"
#include "support.h"

// for ping-pong loops
extern gboolean dirchange_callback (GtkAccelGroup *, GObject *, guint, GdkModifierType, gpointer user_data); // defined in callbacks.c

// processing
static gint64 ratio=U_SECL/1000000; // how many U_SECs (ticks) in a microsecond
static gint64 event_start;
static gdouble audio_start;
static gboolean accelerators_swapped;
static gint frames_done;
static guint64 last_display_ticks=0;  // ticks when last display happened (fixed)

void on_warn_mask_toggled (GtkToggleButton *togglebutton, gpointer user_data)
{
  GtkWidget *tbutton;

  if (gtk_toggle_button_get_active(togglebutton)) prefs->warning_mask|=GPOINTER_TO_INT(user_data);
  else prefs->warning_mask^=GPOINTER_TO_INT(user_data);
  set_int_pref("lives_warning_mask",prefs->warning_mask);

  if ((tbutton=g_object_get_data(G_OBJECT(togglebutton),"auto"))!=NULL) {
    // this is for the cds window - disable autoreload if we are not gonna show this window
    if (gtk_toggle_button_get_active(togglebutton)) {
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tbutton),FALSE);
      gtk_widget_set_sensitive(tbutton,FALSE);
    }
    else {
      gtk_widget_set_sensitive(tbutton,TRUE);
    }
  }

}

static int progress_count;


#define PROG_LOOP_VAL 100

//Warning dialog
GtkWidget*
create_warn_dialog (gint warn_mask_number) {
  GtkWidget *dialog2;
  GtkWidget *dialog_vbox2;
  GtkWidget *dialog_action_area2;
  GtkWidget *warning_cancelbutton;
  GtkWidget *warning_okbutton;
  GtkWidget *checkbutton;
  GtkWidget *label;
  GtkWidget *hbox;

  dialog2 = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dialog2), _("LiVES: - Warning !"));
  if (palette->style&STYLE_1) {
    gtk_dialog_set_has_separator(GTK_DIALOG(dialog2),FALSE);
    gtk_widget_modify_bg(dialog2, GTK_STATE_NORMAL, &palette->normal_back);
  }

  gtk_window_set_modal (GTK_WINDOW (dialog2), TRUE);
  gtk_window_set_position (GTK_WINDOW (dialog2), GTK_WIN_POS_CENTER);
  gtk_window_set_default_size (GTK_WINDOW (dialog2), 350, 200);

  dialog_vbox2 = GTK_DIALOG (dialog2)->vbox;
  gtk_widget_show (dialog_vbox2);

  mainw->warning_label = gtk_label_new (_("warning"));
  gtk_widget_show (mainw->warning_label);
  gtk_box_pack_start (GTK_BOX (dialog_vbox2), mainw->warning_label, TRUE, TRUE, 0);
  gtk_label_set_justify (GTK_LABEL (mainw->warning_label), GTK_JUSTIFY_CENTER);
  gtk_label_set_line_wrap (GTK_LABEL (mainw->warning_label), FALSE);
  gtk_label_set_selectable (GTK_LABEL (mainw->warning_label), TRUE);

  if (palette->style&STYLE_1) {
    gtk_widget_modify_fg(mainw->warning_label, GTK_STATE_NORMAL, &palette->normal_fore);
  }

  if (warn_mask_number>0) {
    checkbutton = gtk_check_button_new ();
    label=gtk_label_new_with_mnemonic (_("Do _not show this warning any more\n(can be turned back on from Preferences/Warnings)"));
    gtk_label_set_mnemonic_widget (GTK_LABEL (label),checkbutton);
    if (palette->style&STYLE_1&&mainw!=NULL) {
      gtk_widget_modify_fg(label, GTK_STATE_NORMAL, &palette->normal_fore);
    }
    hbox = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (dialog_vbox2), hbox, FALSE, FALSE, 10);
    gtk_box_pack_start (GTK_BOX (hbox), checkbutton, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 10);
    gtk_widget_show_all (hbox);
    GTK_WIDGET_SET_FLAGS (checkbutton, GTK_CAN_DEFAULT|GTK_CAN_FOCUS);
    g_signal_connect (GTK_OBJECT (checkbutton), "toggled",
                      G_CALLBACK (on_warn_mask_toggled),
                      GINT_TO_POINTER(warn_mask_number));
  }

  dialog_action_area2 = GTK_DIALOG (dialog2)->action_area;
  gtk_widget_show (dialog_action_area2);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area2), GTK_BUTTONBOX_END);

  warning_cancelbutton = gtk_button_new_from_stock ("gtk-cancel");
  gtk_widget_show (warning_cancelbutton);
  gtk_dialog_add_action_widget (GTK_DIALOG (dialog2), warning_cancelbutton, GTK_RESPONSE_CANCEL);
  GTK_WIDGET_SET_FLAGS (warning_cancelbutton, GTK_CAN_DEFAULT);

  warning_okbutton = gtk_button_new_from_stock ("gtk-ok");
  gtk_widget_show (warning_okbutton);
  gtk_dialog_add_action_widget (GTK_DIALOG (dialog2), warning_okbutton, GTK_RESPONSE_OK);

  GTK_WIDGET_SET_FLAGS (warning_okbutton, GTK_CAN_DEFAULT);

  return dialog2;
}



gint
do_warning_dialog(const gchar *text) {
  return do_warning_dialog_with_check(text, 0);
}


gint do_warning_dialog_with_check (const gchar *text, gint warn_mask_number) {
  return do_warning_dialog_with_check_transient(text,warn_mask_number,GTK_WINDOW(mainw->LiVES));
}



gint
do_warning_dialog_with_check_transient(const gchar *text, gint warn_mask_number, GtkWindow *transient) {
  // show OK/CANCEL, returns FALSE if cancelled
  GtkWidget *warning;
  gint response=1;

  if (prefs->warning_mask&warn_mask_number) {
    return TRUE;
  }

  warning=create_warn_dialog(warn_mask_number);
  gtk_window_set_transient_for(GTK_WINDOW(warning),transient);
  gtk_label_set_text(GTK_LABEL(mainw->warning_label),text);

  response=gtk_dialog_run (GTK_DIALOG (warning));
  gtk_widget_destroy (warning);

  while (g_main_context_iteration(NULL,FALSE));
  return (response==GTK_RESPONSE_OK);
}



void 
do_error_dialog(const gchar *text) {
  // show error/info box
  do_error_dialog_with_check_transient(text,FALSE,0,GTK_WINDOW(mainw->LiVES));
}

void 
do_error_dialog_with_check(const gchar *text, gint warn_mask_number) {
  // show error/info box
  do_error_dialog_with_check_transient(text,FALSE,warn_mask_number,GTK_WINDOW(mainw->LiVES));
}


void 
do_blocking_error_dialog(const gchar *text) {
  // show error/info box - blocks until OK is pressed
  do_error_dialog_with_check_transient(text,TRUE,0,GTK_WINDOW(mainw->LiVES));
}


void 
do_error_dialog_with_check_transient(const gchar *text, gboolean is_blocking, gint warn_mask_number, GtkWindow *transient) {
  // show error/info box

  GtkWidget *err_box;
  if (prefs->warning_mask&warn_mask_number) return;
  err_box=create_dialog3(text,is_blocking,warn_mask_number);
  gtk_window_set_transient_for(GTK_WINDOW(err_box),transient);
  gtk_widget_show(err_box);
  gtk_window_present (GTK_WINDOW (err_box));

  if (is_blocking) {
    gtk_dialog_run(GTK_DIALOG (err_box));
    if (mainw!=NULL&&mainw->is_ready) {
      gtk_widget_queue_draw(GTK_WIDGET(transient));
    }
  }
}

void do_aud_during_play_error(void) {
  do_error_dialog_with_check_transient(_("Audio players cannot be switched during playback."),TRUE,0,GTK_WINDOW(prefsw->prefs_dialog));
}  

void 
do_memory_error_dialog (void) {
  do_error_dialog (_ ("\n\nLiVES was unable to perform this operation due to unsufficient memory.\nPlease try closing some other applications first.\n"));
}

void cancel_process(gboolean visible) {
  if (prefs->show_player_stats&&!visible&&mainw->fps_measure>0.) {
    // statistics
    gettimeofday(&tv, NULL);
    mainw->fps_measure/=(gdouble)(U_SECL*(tv.tv_sec-mainw->startsecs)+tv.tv_usec*ratio-mainw->origticks)/U_SEC;
  }
  if (visible) {
    if (mainw->preview_box!=NULL&&!mainw->preview) gtk_tooltips_set_tip (mainw->tooltips, mainw->p_playbutton,_ ("Play all"), NULL);
    if (accelerators_swapped) {
      if (!mainw->preview) gtk_tooltips_set_tip (mainw->tooltips, mainw->m_playbutton,_ ("Play all"), NULL);
      //gtk_widget_remove_accelerator (cfile->proc_ptr->preview_button, mainw->accel_group, GDK_p, 0);
      //gtk_widget_add_accelerator (mainw->playall, "activate", mainw->accel_group,GDK_p, 0, GTK_ACCEL_VISIBLE);
    }
    if (cfile->proc_ptr!=NULL) {
      gtk_widget_destroy(GTK_WIDGET(cfile->proc_ptr->processing));
      g_free(cfile->proc_ptr);
      cfile->proc_ptr=NULL;
    }
    mainw->is_processing=FALSE;
    if (!(cfile->menuentry==NULL)) {
      sensitize();
    }
  }
  else mainw->is_processing=TRUE;
  if (cfile->clip_type==CLIP_TYPE_DISK&&((mainw->cancelled!=CANCEL_NO_MORE_PREVIEW&&mainw->cancelled!=CANCEL_USER)||!cfile->opening)) {
    unlink(cfile->info_file);
  }
}

gboolean process_one (gboolean visible) {
  gint64 tc;
  gint first_frame,last_frame;
  gint oframeno=0;

#ifdef ENABLE_JACK
  gdouble audio_stretch;
  guint64 audio_ticks=0.;
  gboolean normal_time;
#endif


  if (!visible) {
    // INTERNAL PLAYER
    if (G_UNLIKELY(mainw->new_clip!=-1)) {
      do_quick_switch(mainw->new_clip);
      mainw->new_clip=-1;
    }
    
#ifdef ENABLE_JACK
    normal_time=TRUE;
    if (!mainw->foreign&&prefs->audio_player==AUD_PLAYER_JACK&&cfile->achans>0&&!mainw->is_rendering&&mainw->jackd->in_use) {
      mainw->currticks=jack_get_time(mainw->jackd);
      if (mainw->fixed_fps>-1.) normal_time=TRUE;
      else normal_time=FALSE;
    }
    if (normal_time) {
#endif
      gettimeofday(&tv, NULL);
      mainw->currticks=U_SECL*(tv.tv_sec-mainw->startsecs)+tv.tv_usec*ratio;
#ifdef ENABLE_JACK
      if (mainw->fixed_fps>-1.&&prefs->audio_player==AUD_PLAYER_JACK&&cfile->achans>0&&!mainw->is_rendering&&(audio_ticks>mainw->origticks)) {
	// fps is synched to external source, so we adjust the audio rate to fit
	audio_ticks=mainw->currticks;
	if ((audio_stretch=(mainw->currticks-mainw->origticks)/(audio_ticks-mainw->origticks))<2.) {
	  if (!cfile->play_paused) mainw->jackd->sample_in_rate=cfile->arate*cfile->pb_fps/cfile->fps*audio_stretch;
	  else mainw->jackd->sample_in_rate=cfile->arate*cfile->freeze_fps/cfile->fps*audio_stretch;
	}
      }
    }
#endif
    if (G_UNLIKELY(cfile->proc_ptr==NULL&&cfile->next_event!=NULL&&(tc=mainw->currticks-mainw->origticks)>=event_start)) {
      cfile->next_event=process_events (cfile->next_event,mainw->currticks-mainw->origticks);
      if (cfile->next_event==NULL&&mainw->preview) mainw->cancelled=CANCEL_EVENT_LIST_END;
      if (mainw->cancelled==CANCEL_NONE) return TRUE;
      cancel_process(visible);
      return FALSE;
    }
    else {
      oframeno=cfile->frameno;
      last_frame=(mainw->playing_sel&&!mainw->is_rendering)?cfile->end:mainw->play_end;
      first_frame=mainw->playing_sel?cfile->start:1;
      cfile->frameno+=cfile->play_paused?0:(gint)(((gint64)(mainw->currticks-mainw->startticks)+mainw->deltaticks)/mainw->period);
      if (mainw->noframedrop) cfile->frameno=cfile->frameno>oframeno?oframeno+1:cfile->frameno<oframeno?oframeno-1:cfile->frameno;
    }
    // see if we reached the start (backwards...)
    if (G_UNLIKELY(cfile->frameno<first_frame)) {
      // loop backwards...
      mainw->deltaticks=0;
      mainw->startticks=mainw->currticks-(cfile->frameno-oframeno)*mainw->period;
      if (mainw->ping_pong&&(cfile->pb_fps<0.||mainw->scratch!=SCRATCH_NONE)) {
	dirchange_callback (NULL,NULL,0,0,GINT_TO_POINTER(mainw->playing_sel));
	oframeno=(cfile->frameno=first_frame)-1;
      }
      else {
	oframeno=(cfile->frameno=last_frame)+1;
      }
      if (!mainw->foreign&&prefs->audio_player==AUD_PLAYER_JACK&&cfile->achans>0&&mainw->playing_sel&&!mainw->is_rendering) {
	// audio seek to start
#ifdef ENABLE_JACK
	if (prefs->audio_player==AUD_PLAYER_JACK&&mainw->jackd!=NULL&&!mainw->is_rendering) {
	  if (mainw->jackd->sample_in_rate<0) {
	    audio_seek_frame(mainw->jackd,cfile->end);
	  }
	  else {
	    audio_seek_frame(mainw->jackd,cfile->frameno);
	  }
	}
#endif
      }
    }
    
    // check if we are done
    if (G_UNLIKELY((cfile->frameno>last_frame)&&last_frame>0)) {
      if (mainw->whentostop==STOP_ON_VID_END) {
	mainw->cancelled=CANCEL_VID_END;
      }
      else {
	// loop forwards
	if (mainw->playing_sel) mainw->play_start=cfile->start;
	if (mainw->ping_pong&&(cfile->pb_fps>0.||mainw->scratch!=SCRATCH_NONE)) {
	  dirchange_callback (NULL,NULL,0,0,GINT_TO_POINTER(mainw->playing_sel));
	  oframeno=(cfile->frameno=last_frame)+1;
	}
	else {
	  oframeno=(cfile->frameno=first_frame)-1;
	}
	mainw->deltaticks=0;
	mainw->startticks=mainw->currticks-(cfile->frameno-oframeno)*mainw->period;
	if (!mainw->foreign&&prefs->audio_player==AUD_PLAYER_JACK&&cfile->achans>0&&mainw->playing_sel) {
	  if (!mainw->loop_cont&&mainw->whentostop==STOP_ON_AUD_END) {
	    mainw->cancelled=CANCEL_AUD_END;
	  }
	  else {
	    // audio seek to start
#ifdef ENABLE_JACK
	    if (prefs->audio_player==AUD_PLAYER_JACK&&mainw->jackd!=NULL&&!mainw->is_rendering) {
	      gint audio_frameno;
	      if (mainw->ping_pong&&(cfile->pb_fps<0.||mainw->scratch!=SCRATCH_NONE)&&(prefs->audio_opts&AUDIO_OPTS_FOLLOW_FPS)) {
		audio_frameno=cfile->frameno;
	      }
	      else audio_frameno=first_frame;
	      audio_seek_frame(mainw->jackd,audio_frameno);
	    }
#endif
	  }
	}
      }
    }
    
    mainw->scratch=SCRATCH_NONE;
    
    // play next frame
    if (G_LIKELY(mainw->cancelled==CANCEL_NONE)) {
      // calculate the audio 'frame'
      mainw->aframeno=(gint64)(mainw->currticks-mainw->firstticks)*cfile->fps/U_SEC+audio_start;
      if (G_UNLIKELY(mainw->loop_cont&&(mainw->aframeno>(mainw->audio_end?mainw->audio_end:cfile->laudio_time*cfile->fps)))) {
	mainw->firstticks=mainw->currticks;
      }
      mainw->startticks+=(gint64)((cfile->frameno-oframeno)*mainw->period);
      if ((mainw->fixed_fps==-1.&&cfile->frameno!=oframeno)||(mainw->fixed_fps>0.&&(mainw->currticks-last_display_ticks)/U_SEC>=1./mainw->fixed_fps)) {
	load_frame_image(cfile->frameno,oframeno);
	last_display_ticks=mainw->currticks;
      }
    }
  }
  
  // paused
  if (G_UNLIKELY(cfile->play_paused)) {
    mainw->startticks=mainw->currticks+mainw->deltaticks;
    if (mainw->ext_playback&&mainw->fixed_fps==-1.) (*vid_playback_plugin->send_keycodes)();
  }
  if (visible) {
    // fixes a problem with opening preview with bg generator
    if (cfile->proc_ptr==NULL) {
      if (mainw->cancelled==CANCEL_NONE) mainw->cancelled=CANCEL_NO_PROPOGATE;
    }
    else {
      gdouble est_time,fraction_done,timesofar;
      gchar *prog_label;

      if (mainw->framedraw_spinbutton!=NULL) gtk_spin_button_set_range(GTK_SPIN_BUTTON(mainw->framedraw_spinbutton),1,cfile->proc_ptr->frames_done);

      // set the progress bar %
      if (mainw->opening_frames!=-1) {
	//prog_label=g_strdup_printf(_("\n%d%% Frames opened. Time remaining: %u sec"),(gint)(fraction_done*100.),(guint)(est_time+.5));
	prog_label=g_strdup_printf(_("\n%d / %d Frames opened."),mainw->opening_frames,cfile->frames);
	gtk_label_set_text(GTK_LABEL(cfile->proc_ptr->label3),prog_label);
	g_free(prog_label);
      }
      if (visible&&cfile->proc_ptr->frames_done>=cfile->progress_start&&cfile->proc_ptr->frames_done<=cfile->progress_end&&cfile->progress_end>0) {
	if (progress_count++==PROG_LOOP_VAL) {
	  gettimeofday(&tv, NULL);
	  mainw->currticks=U_SECL*(tv.tv_sec-mainw->startsecs)+tv.tv_usec*ratio;
	  fraction_done=(cfile->proc_ptr->frames_done-cfile->progress_start)/(cfile->progress_end-cfile->progress_start+1.);
	  if (cfile->proc_ptr->frames_done>frames_done) gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(cfile->proc_ptr->progressbar),fraction_done);
	  timesofar=(mainw->currticks-mainw->origticks-mainw->timeout_ticks)/U_SEC;
	  est_time=timesofar/fraction_done-timesofar;
	  prog_label=g_strdup_printf(_("\n%d%% done. Time remaining: %u sec"),(gint)(fraction_done*100.),(guint)(est_time+.5));
	  gtk_label_set_text(GTK_LABEL(cfile->proc_ptr->label3),prog_label);
	  g_free(prog_label);
	  progress_count=0;
	}
      }
      else {
	gtk_progress_bar_pulse(GTK_PROGRESS_BAR(cfile->proc_ptr->progressbar));
      }
    }
  }
  if (G_LIKELY(mainw->cancelled==CANCEL_NONE)) {
    while (g_main_context_iteration(NULL,FALSE));
    return TRUE;
  }
  cancel_process(visible);
  return FALSE;
}


gboolean do_progress_dialog(gboolean visible, gboolean cancellable, const gchar *text) {
  // monitor progress, return FALSE if the operation was cancelled
  FILE *infofile=NULL;
  gboolean finished=FALSE;

  event_start=0;
  audio_start=mainw->play_start;
  accelerators_swapped=FALSE;
  frames_done=0;
  last_display_ticks=0;

  progress_count=0;

  if (!visible) {
    mainw->startticks=mainw->startsecs=mainw->currticks=0;
    if (!(cfile->event_list==NULL)) {
      // this is for audio, work out the apparent frame when we have variable fps
      audio_start=calc_variable_time(mainw->current_file,mainw->play_start)*cfile->fps;
    }
    if (mainw->clip_index==NULL) {
      mainw->clip_index=weed_malloc(sizint);
      mainw->clip_index[0]=-1;
    }
    if (mainw->frame_index==NULL) {
      mainw->frame_index=weed_malloc(sizint);
      mainw->frame_index[0]=0;
    }
  }

  if (prefs->show_player_stats) {
    mainw->fps_measure=0.;
  }

  mainw->cancelled=CANCEL_NONE;   
  mainw->error=FALSE;
  clear_mainw_msg();
  if (!mainw->preview) mainw->timeout_ticks=0;

  if (visible) {
    mainw->is_processing=TRUE;
    desensitize();
    procw_desensitize();
    cfile->proc_ptr=create_processing (text);

    gtk_progress_bar_set_pulse_step(GTK_PROGRESS_BAR(cfile->proc_ptr->progressbar),.01);
    if (mainw->show_procd) gtk_widget_show(cfile->proc_ptr->processing);
    
    cfile->proc_ptr->frames_done=0;
    
    if (cancellable) {
      gtk_widget_show (cfile->proc_ptr->cancel_button);
    }
    if (cfile->opening&&capable->has_sox&&mainw->playing_file==-1) {
      gtk_widget_show (cfile->proc_ptr->preview_button);
      if (mainw->preview_box!=NULL) gtk_tooltips_set_tip (mainw->tooltips, mainw->p_playbutton,_ ("Preview"), NULL);
      gtk_tooltips_set_tip (mainw->tooltips, mainw->m_playbutton,_ ("Preview"), NULL);
      // gtk_widget_remove_accelerator (mainw->playall, mainw->accel_group, GDK_p, 0);
      //gtk_widget_add_accelerator (cfile->proc_ptr->preview_button, "clicked", mainw->accel_group, GDK_p, 0, 0);
      accelerators_swapped=TRUE;
    }
  }

  if (cfile->next_event!=NULL) event_start=get_event_timecode(cfile->next_event);

  if (!visible) {
    cfile->last_frameno=cfile->frameno=mainw->play_start;
    mainw->deltaticks=0;
  }
  gettimeofday(&tv, NULL);
  
  // we subtract this from every calculation to make the numbers smaller
  mainw->startsecs=tv.tv_sec;
  
#ifdef ENABLE_JACK
  if (!mainw->foreign&&!visible) {
    if (prefs->audio_player==AUD_PLAYER_JACK&&cfile->achans>0&&cfile->laudio_time>0.&&!mainw->is_rendering&&!(cfile->opening&&!mainw->preview)&&mainw->jackd!=NULL&&mainw->jackd->playing_file>-1) audio_seek_frame(mainw->jackd,mainw->play_start);
  }
#endif
  // mainw->origticks is our base for quantising (and is constant for each playback session)
  mainw->origticks=mainw->firstticks=mainw->startticks=tv.tv_usec*ratio;

  if (mainw->multitrack!=NULL&&!mainw->multitrack->is_rendering) mainw->origticks-=get_event_timecode(mainw->multitrack->pb_start_event); // playback start from middle of multitrack

  //try to open info file - or if internal_messaging is TRUE, we get mainw->msg
  // from the mainw->progress_fn function

  do {
    while (!mainw->internal_messaging&&((!visible&&(mainw->mute||prefs->audio_player==AUD_PLAYER_JACK))||!g_file_test(cfile->info_file,G_FILE_TEST_EXISTS))) {
      // just pulse the progress bar
      if (G_UNLIKELY(!process_one(visible))) return FALSE;
      if (prefs->audio_player==AUD_PLAYER_JACK) sched_yield();
      if (!mainw->foreign&&(visible||(!mainw->mute&&prefs->audio_player!=AUD_PLAYER_JACK))) {
	g_usleep(prefs->sleep_time);
      }
    }
    if (!mainw->internal_messaging) {
      if ((infofile=fopen(cfile->info_file,"r"))) {
	// OK, now we might have some frames
	fgets(mainw->msg,512,infofile);
	fclose(infofile);
      }
    }
    else (*mainw->progress_fn)(FALSE);

#ifdef DEBUG
    g_print("msg %s\n",mainw->msg);
#endif

    if (visible&&!accelerators_swapped&&cancellable&&!cfile->nopreview) {
      gtk_widget_show (cfile->proc_ptr->pause_button);
      gtk_widget_show (cfile->proc_ptr->preview_button);
      gtk_widget_grab_default (cfile->proc_ptr->preview_button);
      if (mainw->preview_box!=NULL) gtk_tooltips_set_tip (mainw->tooltips, mainw->p_playbutton,_ ("Preview"), NULL);
      gtk_tooltips_set_tip (mainw->tooltips, mainw->m_playbutton,_ ("Preview"), NULL);
      //gtk_widget_remove_accelerator (mainw->playall, mainw->accel_group, GDK_p, 0);
      //gtk_widget_add_accelerator (cfile->proc_ptr->preview_button, "clicked", mainw->accel_group, GDK_p, 0, 0);
      accelerators_swapped=TRUE;
    }

    if (strncmp(mainw->msg,"completed",9)&&strncmp(mainw->msg,"error",5)&&(visible||((strncmp(mainw->msg,"video_ended",11)||mainw->whentostop!=STOP_ON_VID_END)&&(strncmp(mainw->msg,"audio_ended",11)||mainw->preview||mainw->whentostop!=STOP_ON_AUD_END)))) {
      // processing not yet completed...
      if (visible) {
	// last frame processed ->> will go from cfile->start to cfile->end
	cfile->proc_ptr->frames_done=atoi(mainw->msg);
      }
      if (!process_one(visible)) return FALSE;
      g_usleep(prefs->sleep_time);
    }
    else finished=TRUE;
  } while (!finished);

#ifdef DEBUG
  g_print ("exit pt 3 %s\n",mainw->msg);
#endif

  //play/operation ended
  if (visible) {
    if (cfile->clip_type==CLIP_TYPE_DISK&&(mainw->cancelled!=CANCEL_NO_MORE_PREVIEW||!cfile->opening)) {
      unlink(cfile->info_file);
    }
    if (mainw->preview_box!=NULL&&!mainw->preview) gtk_tooltips_set_tip (mainw->tooltips, mainw->p_playbutton,_ ("Play all"), NULL);
    if (accelerators_swapped) {
      if (!mainw->preview) gtk_tooltips_set_tip (mainw->tooltips, mainw->m_playbutton,_ ("Play all"), NULL);
      //gtk_widget_remove_accelerator (cfile->proc_ptr->preview_button, mainw->accel_group, GDK_p, 0);
      //gtk_widget_add_accelerator (mainw->playall, "activate", mainw->accel_group, GDK_p, 0, GTK_ACCEL_VISIBLE);
    }
    if (cfile->proc_ptr!=NULL) {
      gtk_widget_destroy(cfile->proc_ptr->processing);
      g_free(cfile->proc_ptr);
      cfile->proc_ptr=NULL;
    }
    mainw->is_processing=FALSE;
    if (!(cfile->menuentry==NULL)) {
      // note - for operations to/from clipboard (file 0) we
      // should manually call sensitize() after operation
      sensitize();
    }
  }
  else {
    if (prefs->show_player_stats) {
      if (mainw->fps_measure>0.) {
	gettimeofday(&tv, NULL);
	mainw->fps_measure/=(gdouble)(U_SECL*(tv.tv_sec-mainw->startsecs)+tv.tv_usec*ratio-mainw->origticks)/U_SEC;
      }
    }
    mainw->is_processing=TRUE;
  }
  
  // get error message (if any)
  if (!strncmp(mainw->msg,"error",5)) {
    int i;
    gchar **array;
    gint numtok=get_token_count (mainw->msg,'|');
    
    array=g_strsplit(mainw->msg,"|",numtok);
    g_snprintf(mainw->msg,512,"\n\n");
    for (i=1;i<numtok;i++) {
      g_strappend(mainw->msg,512,_(array[i]));
      g_strappend(mainw->msg,512,"\n");
    }
    g_strfreev(array);
    mainw->error=TRUE;
  }
#ifdef DEBUG
  g_print("exiting progress dialogue\n");
#endif
  return TRUE;
}



void 
do_auto_dialog (const gchar *text, gint type) {
  // type 0 = normal auto_dialog
  // type 1 = countdown dialog for audio recording

  FILE *infofile=NULL;
  gint count=0;
  guint64 time=0,end_time=1;
  gchar *label_text;
  gint time_rem,last_time_rem=10000000;

  cfile->proc_ptr=create_processing (text);
  gtk_widget_hide (cfile->proc_ptr->stop_button);
  gtk_window_set_modal (GTK_WINDOW (cfile->proc_ptr->processing), TRUE);
     
  gtk_progress_bar_set_pulse_step(GTK_PROGRESS_BAR(cfile->proc_ptr->progressbar),.01);
  gtk_widget_show(cfile->proc_ptr->processing);
  
  if (type==0) {
    clear_mainw_msg();
    count=100000./prefs->sleep_time;  // don't want to flash too fast...
  }
  else {
    gtk_widget_show(cfile->proc_ptr->stop_button);
    gtk_widget_show(cfile->proc_ptr->pause_button);
    gtk_widget_show(cfile->proc_ptr->cancel_button);
    if (mainw->rec_end_time!=-1.) {
      while (g_main_context_iteration(NULL,FALSE));
      gettimeofday(&tv, NULL);
      time=tv.tv_sec*1000000.+tv.tv_usec; // time in microseconds
      end_time=time+mainw->rec_end_time*1000000.;
    }
  }

  while ((type==1&&mainw->cancelled==CANCEL_NONE)||(type==0&&!(infofile=fopen(cfile->info_file,"r")))) {
    gtk_progress_bar_pulse(GTK_PROGRESS_BAR(cfile->proc_ptr->progressbar));
    while (g_main_context_iteration(NULL,FALSE));
    g_usleep(prefs->sleep_time);
    if (type==1&&mainw->rec_end_time!=-1.) {
      gettimeofday(&tv, NULL);
      time=tv.tv_sec*1000000.+tv.tv_usec; // time in microseconds
      time_rem=(gint)((gdouble)(end_time-time)/1000000.+.5);
      if (time_rem>=0&&time_rem<last_time_rem) {
	label_text=g_strdup_printf(_("\nTime remaining: %d sec"),time_rem);
	gtk_label_set_text(GTK_LABEL(cfile->proc_ptr->label2),label_text);
	g_free(label_text);
	last_time_rem=time_rem;
      }
    }
  }
  
  if (type==0) {
    fgets(mainw->msg,512,infofile);
    fclose(infofile);
    if (cfile->clip_type==CLIP_TYPE_DISK) unlink(cfile->info_file);

    while (count>0) {
      gtk_progress_bar_pulse(GTK_PROGRESS_BAR(cfile->proc_ptr->progressbar));
      while (g_main_context_iteration(NULL,FALSE));
      g_usleep(prefs->sleep_time);
      count--;
    }
  }

  if (cfile->proc_ptr!=NULL) {
    gtk_widget_destroy(cfile->proc_ptr->processing);
    g_free(cfile->proc_ptr);
    cfile->proc_ptr=NULL;
  }

  if (type==1) return;

  // get error message (if any)
  if (!strncmp(mainw->msg,"error",5)) {
    gchar **array=g_strsplit(mainw->msg,"|",5);
    g_snprintf(mainw->msg,512,"\n\n%s\n\n%s\n\n%s\n\n%s\n",array[1],array[2],array[3],array[4]);
    g_strfreev(array);
    mainw->error=TRUE;
  }
}




void
too_many_files(void) {
  gchar *warn=g_strdup_printf(_ ("\nSorry, LiVES can only open %d files at once.\nPlease close a file and then try again."),MAX_FILES);
  do_error_dialog(warn);
  g_free(warn);
}

void
tempdir_warning (void) {
    gchar *com=g_strdup_printf(_ ("LiVES was unable to write to its temporary directory.\n\nThe current temporary directory is:\n\n%s\n\nPlease make sure you can write to this directory."),prefs->tmpdir);
    if (mainw!=NULL&&mainw->is_ready) {
      do_error_dialog(com);
    }
    g_free(com);
}


void
do_encoder_sox_error(void) {
  gchar *msg;
  msg=g_strdup_printf (_ ("Audio resampling is required for this format.\nPlease install 'sox'\nOr switch to another encoder format in Tools | Preferences | Encoding\n"));
  d_print (msg);
  g_free (msg);
}


void do_encoder_acodec_error (void) {
  do_error_dialog (_ ("\n\nThis encoder/format cannot use the requested audio codec.\nPlease set the audio codec in Tools|Preferences|Encoding\n"));
}


void do_layout_scrap_file_error(GtkWindow *window) {
  do_error_dialog_with_check_transient(_("This layout includes generated frames.\nIt cannot be saved, you must render it to a clip first.\n"),TRUE,0,window);
}




gboolean 
rdet_suggest_values (gint width, gint height, gdouble fps, gint fps_num, gint fps_denom, gint arate, gboolean anr, gboolean ignore_fps) {
  gchar *msg1=g_strdup_printf (_ ("\n\nDue to restrictions in the %s format\n"),prefs->encoder.of_name);
  gchar *msg2=g_strdup ("");
  gchar *msg3=g_strdup ("");
  gchar *msg4=g_strdup ("");
  gchar *msg5=g_strdup ("");
  gchar *msg6=g_strdup ("");
  gchar *msg7=g_strdup ("");
  gchar *msg_a;

  gboolean ret;

  GtkWidget *prep_dialog;

  if ((fps>0.&&fps!=rdet->fps)||(fps_denom>0&&(fps_num*1.)/(fps_denom*1.)!=rdet->fps)||(!anr&&(rdet->width!=width||rdet->height!=height)&&height*width>0)||arate!=rdet->arate) {
    g_free (msg2);
    msg2=g_strdup (_ ("LiVES recommends the following settings:\n\n"));
    if (arate>0&&arate!=rdet->arate) {
      g_free (msg3);
      msg3=g_strdup_printf (_ ("Use an audio rate of %d Hz\n"),arate);
    }
    if (!ignore_fps) {
      if (fps>0&&fps!=rdet->fps) {
	g_free (msg4);
	msg4=g_strdup_printf (_ ("Set video rate to %.3f frames per second\n"),fps);
      }
      else if (fps_denom>0&&fps_num/fps_denom!=rdet->fps) {
	g_free (msg4);
	msg4=g_strdup_printf (_ ("Set video rate to %d:%d frames per second\n"),fps_num,fps_denom);
      }
    }
    if (!anr&&((rdet->width!=width||rdet->height!=height)&&height*width>0)) {
      g_free (msg5);
      msg5=g_strdup_printf (_ ("Set video size to %d x %d pixels\n"),width,height);
      mainw->fx1_bool=TRUE;
    }
  }
  if (anr) {
    if ((rdet->width!=width||rdet->height!=height)&&height*width>0) {
      g_free (msg6);
      g_free (msg7);
      msg6=g_strdup_printf (_ ("\nYou may wish to:\n"));
      msg7=g_strdup_printf (_ ("Set video size to %d x %d pixels\n"),width,height);
    }
    else anr=FALSE;
  }
  msg_a=g_strconcat (msg1,msg2,msg3,msg4,msg5,msg6,msg7,NULL);
  g_free (msg1);
  g_free (msg2);
  g_free (msg3);
  g_free (msg4);
  g_free (msg5);
  g_free (msg6);
  g_free (msg7);
  prep_dialog=create_encoder_prep_dialog(msg_a,NULL,anr);
  g_free (msg_a);
  ret=(gtk_dialog_run(GTK_DIALOG (prep_dialog))==GTK_RESPONSE_OK);
  gtk_widget_destroy (prep_dialog);
  return ret;
}



gboolean 
do_encoder_restrict_dialog (gint width, gint height, gdouble fps, gint fps_num, gint fps_denom, gint arate, gboolean anr) {
  gchar *msg1=g_strdup_printf (_ ("\n\nDue to restrictions in the %s format\n"),prefs->encoder.of_name);
  gchar *msg2=g_strdup ("");
  gchar *msg3=g_strdup ("");
  gchar *msg4=g_strdup ("");
  gchar *msg5=g_strdup ("");
  gchar *msg6=g_strdup ("");
  gchar *msg7=g_strdup ("");
  gchar *msg_a,*msg_b;

  gboolean ret;

  GtkWidget *prep_dialog;

  if ((arate>0&&arate!=cfile->arate)||(fps>0.&&fps!=cfile->fps)||(fps_denom>0&&(fps_num*1.)/(fps_denom*1.)!=cfile->fps)||(!anr&&(cfile->hsize!=width||cfile->vsize!=height)&&height*width>0)) {
    g_free (msg2);
    msg2=g_strdup (_ ("LiVES must:\n"));
    if (arate>0&&arate!=cfile->arate) {
      g_free (msg3);
      msg3=g_strdup_printf (_ ("resample audio to %d Hz\n"),arate);
    }
    if (fps>0&&fps!=cfile->fps) {
      g_free (msg4);
      msg4=g_strdup_printf (_ ("resample video to %.3f frames per second\n"),fps);
    }
    else if (fps_denom>0&&fps_num/fps_denom!=cfile->fps) {
      g_free (msg4);
      msg4=g_strdup_printf (_ ("resample video to %d:%d frames per second\n"),fps_num,fps_denom);
    }
    if (!anr&&((cfile->hsize!=width||cfile->vsize!=height)&&height*width>0)) {
      g_free (msg5);
      msg5=g_strdup_printf (_ ("resize video to %d x %d pixels\n"),width,height);
      mainw->fx1_bool=TRUE;
    }
  }
  if (anr) {
    if ((cfile->hsize!=width||cfile->vsize!=height)&&height*width>0) {
      g_free (msg6);
      g_free (msg7);
      msg6=g_strdup_printf (_ ("\nYou may wish to:\n"));
      msg7=g_strdup_printf (_ ("resize video to %d x %d pixels\n"),width,height);
    }
    else anr=FALSE;
  }
  msg_a=g_strconcat (msg1,msg2,msg3,msg4,msg5,msg6,msg7,NULL);
  if (mainw->save_all) {
    msg_b=g_strdup (_ ("\nYou will be able to undo these changes afterwards.\n\nClick `OK` to proceed, `Cancel` to abort.\n\n"));
  }
  else {
    msg_b=g_strdup (_ ("\nChanges applied to the selection will not be permanent.\n\n"));
  }
  g_free (msg1);
  g_free (msg2);
  g_free (msg3);
  g_free (msg4);
  g_free (msg5);
  g_free (msg6);
  g_free (msg7);
  prep_dialog=create_encoder_prep_dialog(msg_a,msg_b,anr);
  g_free (msg_a);
  g_free (msg_b);
  ret=(gtk_dialog_run(GTK_DIALOG (prep_dialog))==GTK_RESPONSE_OK);
  gtk_widget_destroy (prep_dialog);
  return ret;
}


void
perf_mem_warning(void) {
  do_error_dialog(_ ("\n\nLiVES was unable to record a performance. There is currently insufficient memory available.\nTry recording for just a selection of the file."));
}

gint
do_clipboard_fps_warning(void) {
  if (prefs->warning_mask&WARN_MASK_FPS) {
    return TRUE;
  }
  return do_warning_dialog_with_check(_ ("The playback speed (fps), or the audio rate\n of the clipboard does not match\nthe playback speed or audio rate of the clip you are inserting into.\n\nThe insertion will be adjusted to fit into the clip.\n\nPlease press Cancel to abort the insert, or OK to continue."),WARN_MASK_FPS);
}



gboolean
do_comments_dialog (void) {
  gboolean response;

  commentsw=create_comments_dialog();
  if ((response=(gtk_dialog_run(GTK_DIALOG (commentsw->comments_dialog))==GTK_RESPONSE_OK))) {
    g_snprintf (cfile->title,256,"%s",gtk_entry_get_text (GTK_ENTRY (commentsw->title_entry)));
    g_snprintf (cfile->author,256,"%s",gtk_entry_get_text (GTK_ENTRY (commentsw->author_entry)));
    g_snprintf (cfile->comment,256,"%s",gtk_entry_get_text (GTK_ENTRY (commentsw->comment_entry)));
  }
  gtk_widget_destroy (commentsw->comments_dialog);
  g_free (commentsw);
  
  return response;
}


void 
do_keys_window (void) {
  do_text_window (_ ("Show Keys"),_ ("You can use the following keys during playback to control LiVES:-\n\nRecordable keys (press 'r' before playback to make a recording)\n-----------------------\nctrl-left                     skip back\nctrl-right                   skip forwards\nctrl-up                      faster/increase effect\nctrl-down                 slower/decrease effect\nctrl-enter                    reset frame rate\nctrl-space                reverse direction\nctrl-backspace         freeze frame\nn                             nervous\nctrl-page up            previous clip\nctrl-page down        next clip\n\nctrl-1                       toggle real-time effect 1\nctrl-2                       toggle real-time effect 2\n                                          ...etc...\nctrl-0                       real-time effects off\n\nk           grab keyboard for last activated effect\nm         switch effect mode (when effect has keyboard grab)\nx                     swap background/foreground\nf1                           store/switch to clip mnemonic 1\nf2                           store/switch to clip mnemonic 2\n                                          ...etc...\nf12                          clear function keys\n\n\n Other playback keys\n-----------------------------\np                             play all\ny                             play selection\nq                             stop\nf                               fullscreen\ns                              separate window\nd                             double size\ng                             ping pong loops\n"));

}



void 
do_mt_keys_window (void) {
  do_text_window (_ ("Multitrack Keys"),_ ("You can use the following keys to control the multitrack window:-\n\nctrl-left-arrow              move timeline cursor left 1 second\nctrl-right-arrow            move timeline cursor right 1 second\nshift-left-arrow            move timeline cursor left 1 frame\nshift-right-arrow          move timeline cursor right 1 frame\nctrl-up-arrow               move current track up\nctrl-down-arrow           move current track down\nctrl-page-up                select next clip\nctrl-page-down            select previous clip\nctrl-space                    select/deselect current track\nctrl-plus                       zoom in\nctrl-minus                    zoom out\nm                                 make a mark on the timeline (during playback)\nw                                 rewind to play start.\n\nFor other keys, see the menus.\n"));

}








void 
do_messages_window (void) {
  GtkTextIter start_iter,end_iter;
  gtk_text_buffer_get_start_iter(gtk_text_view_get_buffer(GTK_TEXT_VIEW(mainw->textview1)),&start_iter);
  gtk_text_buffer_get_end_iter(gtk_text_view_get_buffer(GTK_TEXT_VIEW(mainw->textview1)),&end_iter);
  do_text_window (_ ("Message History"),gtk_text_iter_get_text (&start_iter,&end_iter));
}


void 
do_text_window (const gchar *title, const gchar *text) {
  text_window *textwindow=create_text_window (title,text,NULL);
  gtk_widget_show (textwindow->dialog);
}



void 
do_firstever_dialog (void) {
  startup_message_nonfatal (g_strdup (_ ("\n\nWelcome to LiVES. If this is your first time using LiVES, please check all the\nPreferences using Control-P. In particular make sure the temporary directory is located somewhere with plenty of diskspace.\nHave fun !\n")));
}


void 
do_upgrade_error_dialog (void) {
  startup_message_nonfatal (g_strdup (_("After upgrading/installing, you may need to adjust the <prefix_dir> setting in your ~/.lives file")));
}


void do_rendered_fx_dialog(void) {
    gchar *msg=g_strdup_printf(_("\n\nLiVES could not find any rendered effect plugins.\nPlease make sure you have them installed in\n%s%s%s\n(maybe you need to set the value of <prefix_dir> in your ~/.lives file)\n"),prefs->prefix_dir,PLUGIN_DIR,PLUGIN_RENDERED_EFFECTS_BUILTIN);
    do_error_dialog_with_check(msg,WARN_MASK_RENDERED_FX);
    g_free(msg);
}

void do_audio_import_error(void) {
  do_error_dialog(_ ("Sorry, unknown audio type.\n\n (Filenames must end in .mp3, .ogg, .wav, .mod, .xm or .it)"));
  d_print(_ ("failed (unknown type)\n"));
}


gboolean prompt_remove_layout_files(void) {
  return (do_warning_dialog(_("\nDo you wish to remove the layout files associated with this set ?\nClick OK to remove them, or Cancel to leave them.\n(They will not be usable without the set).\n")));
}


gboolean do_set_duplicate_warning (gchar *new_set) {
  gchar *msg=g_strdup_printf(_("\nA set entitled %s already exists.\nClick OK to add the current clips and layouts to the existing set.\nClick Cancel to pick a new name.\n"),new_set);
  gboolean retcode=do_warning_dialog_with_check(msg,WARN_MASK_DUPLICATE_SET);
  g_free(msg);
  return retcode;
}

gboolean do_layout_alter_frames_warning(void) {
  return do_warning_dialog(_("\nFrames from this clip are used in some multitrack layouts.\nAre you sure you wish to continue ?\n."));
}

gboolean do_layout_alter_audio_warning(void) {
  return do_warning_dialog(_("\nAudio from this clip is used in some multitrack layouts.\nAre you sure you wish to continue ?\n."));
}


gboolean do_set_rename_old_layouts_warning(gchar *new_set) {
  gchar *msg=g_strdup_printf(_("\nSome old layouts for the set %s already exist.\nIt is recommended that you delete them.\nClick OK to delete them, Cancel to leave them on the disk.\n"),new_set);
  gboolean retcode=do_warning_dialog(msg);
  g_free(msg);
  return retcode;
}

void do_mt_undo_mem_error(void) {
  do_error_dialog(_("\nLiVES was unable to reserve enough memory for multitrack undo.\nEither close some other applications, or reduce the undo memory\nusing Preferences/Multitrack/Undo Memory\n"));
}

void do_mt_undo_buf_error(void) {
  do_error_dialog(_("\nOut of memory for undo.\nYou may need to increase the undo memory\nusing Preferences/Multitrack/Undo Memory\n"));
}

void do_mt_set_mem_error(gboolean has_mt, gboolean trans) {
  gchar *msg1=(_("\nLiVES was unable to reserve enough memory for the multitrack undo buffer.\n"));
  gchar *msg2;
  gchar *msg3=(_("or enter a smaller value.\n"));
  gchar *msg;
  if (has_mt) msg2=(_("Try again from the clip editor, try closing some other applications\n"));
  else msg2=(_("Try closing some other applications\n"));

  msg=g_strdup_printf("%s%s%s",msg1,msg2,msg3);

  if (!trans) do_blocking_error_dialog(msg);
  else do_error_dialog_with_check_transient(msg,TRUE,0,GTK_WINDOW(prefsw->prefs_dialog));
  g_free(msg);
}


void do_mt_audchan_error(lives_mt *mt, gint warn_mask) {
  do_error_dialog_with_check_transient(_("Multitrack is set to 0 audio channels, but this layout has audio.\nYou should adjust the audio settings from the Tools menu.\n"),FALSE,warn_mask,GTK_WINDOW(mt->window));
}

void do_mt_no_audchan_error(lives_mt *mt) {
  do_error_dialog_with_check_transient(_("The current layout has audio, so audio channels may not be set to zero.\n"),FALSE,0,GTK_WINDOW(mt->window));
}

void do_mt_no_jack_error(lives_mt *mt, gint warn_mask) {
  do_error_dialog_with_check_transient(_("Multitrack audio preview is only available with the jack audio player.\nYou can set this in Tools|Preferences|Playback."),FALSE,warn_mask,GTK_WINDOW(mt->window));
}

gboolean do_mt_rect_prompt(lives_mt *mt) {
  return do_warning_dialog_with_check_transient(_("Errors were detected in the layout (which may be due to transferring from another system).\nShould I try to repair the disk copy of the layout ?\n"),0,GTK_WINDOW(mt->window));
}

void do_audrate_error_dialog(void) {
  do_error_dialog(_ ("\n\nAudio rate must be greater than 0.\n"));
}

gboolean do_event_list_warning(lives_mt *mt) {
  return do_warning_dialog_with_check_transient(_("\nEvent list will be very large\nand may take a long time to display.\nAre you sure you wish to view it ?\n"),0,mt!=NULL?GTK_WINDOW(mt->window):GTK_WINDOW(mainw->LiVES));
}


void do_dvgrab_error(void) {
  do_error_dialog(_ ("\n\nYou must install 'dvgrab' to use this function.\n"));
}


static void *dth2 (void *arg) {
  GtkWidget *dialog_vbox1;
  GtkWidget *vbox2;
  GtkWidget *vbox3;
  process *procw=(process*)(g_malloc(sizeof(process)));
  gchar tmp_label[256];

  // mutex lock
  pthread_mutex_lock(&mainw->gtk_mutex);
  procw->processing = gtk_dialog_new ();
  gtk_container_set_border_width (GTK_CONTAINER (procw->processing), 10);
  gtk_window_add_accel_group (GTK_WINDOW (procw->processing), mainw->accel_group);
  gtk_widget_show(procw->processing);

  gtk_widget_set_size_request (procw->processing, 320, 185);
  gtk_window_set_resizable (GTK_WINDOW (procw->processing), FALSE);
  
  gtk_widget_modify_bg(procw->processing, GTK_STATE_NORMAL, &palette->normal_back);
  gtk_window_set_title (GTK_WINDOW (procw->processing), _("LiVES: - Processing..."));
  gtk_window_set_transient_for(GTK_WINDOW(procw->processing),GTK_WINDOW(mainw->LiVES));
  gtk_window_set_position (GTK_WINDOW (procw->processing), GTK_WIN_POS_CENTER);

  dialog_vbox1 = GTK_DIALOG (procw->processing)->vbox;
  gtk_widget_show (dialog_vbox1);

  gtk_dialog_set_has_separator(GTK_DIALOG(procw->processing),FALSE);

  vbox2 = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox2);
  gtk_box_pack_start (GTK_BOX (dialog_vbox1), vbox2, TRUE, TRUE, 0);

  vbox3 = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox3);
  gtk_box_pack_start (GTK_BOX (vbox2), vbox3, TRUE, TRUE, 0);

  g_snprintf(tmp_label,256,"%s...\n",(gchar *)arg);
  procw->label = gtk_label_new (tmp_label);
  gtk_widget_show (procw->label);
  gtk_box_pack_start (GTK_BOX (vbox3), procw->label, FALSE, FALSE, 0);
  gtk_widget_modify_fg(procw->label, GTK_STATE_NORMAL, &palette->normal_fore);
  gtk_label_set_justify (GTK_LABEL (procw->label), GTK_JUSTIFY_LEFT);

  procw->progressbar = gtk_progress_bar_new ();
  gtk_widget_show (procw->progressbar);
  gtk_box_pack_start (GTK_BOX (vbox3), procw->progressbar, FALSE, FALSE, 0);
  gtk_widget_modify_fg(procw->progressbar, GTK_STATE_NORMAL, &palette->normal_fore);

  procw->label2 = gtk_label_new (_("\nPlease Wait"));
  gtk_widget_show (procw->label2);
  gtk_box_pack_start (GTK_BOX (vbox3), procw->label2, FALSE, FALSE, 0);
  gtk_label_set_justify (GTK_LABEL (procw->label2), GTK_JUSTIFY_CENTER);
  gtk_widget_modify_fg(procw->label2, GTK_STATE_NORMAL, &palette->normal_fore);

  procw->label3 = gtk_label_new (NULL);
  gtk_widget_show (procw->label3);
  gtk_box_pack_start (GTK_BOX (vbox3), procw->label3, FALSE, FALSE, 0);
  gtk_label_set_justify (GTK_LABEL (procw->label3), GTK_JUSTIFY_CENTER);
  gtk_widget_modify_fg(procw->label3, GTK_STATE_NORMAL, &palette->normal_fore);
  gtk_progress_bar_set_pulse_step(GTK_PROGRESS_BAR(procw->progressbar),.01);

  while (mainw->threaded_dialog) {
    // mutex lock
    gtk_progress_bar_pulse(GTK_PROGRESS_BAR(procw->progressbar));
    while (g_main_context_iteration(NULL,FALSE)) {
      pthread_mutex_unlock(&mainw->gtk_mutex);
      // mutex unlock
      sched_yield();
      g_usleep(prefs->sleep_time);
      pthread_mutex_lock(&mainw->gtk_mutex);
    }
  }

  gtk_widget_destroy(procw->processing);
  while (g_main_context_iteration(NULL,FALSE));
  pthread_mutex_unlock(&mainw->gtk_mutex);
  // mutex unlock

  g_free(procw);
  pthread_exit(NULL);
  return NULL;
}

static pthread_t dthread;
static gchar *thread_text;

void do_threaded_dialog(gchar *text) {
  // calling this causes a threaded progress dialog to appear
  // until end_threaded_dialog() is called
  //
  // WARNING: excercise *extreme caution* after calling this;
  // in the main thread, all drawing operations MUST be surrounded by:
  // 
  // pthread_mutex_lock(&mainw->gtk_mutex);
  // ...
  // pthread_mutex_unlock(&mainw->gtk_mutex);
  // 
  // otherwise the app can crash very badly !


  thread_text=g_strdup(text);
  mainw->threaded_dialog=TRUE;
  pthread_create(&dthread,NULL,dth2,thread_text);
}


void end_threaded_dialog(void) {
  if (mainw->threaded_dialog) {
    mainw->threaded_dialog=FALSE;
    pthread_join(dthread,NULL);
    g_free(thread_text);
  }
}

void 
response_ok (GtkButton *button, gpointer user_data) {
  gtk_dialog_response (GTK_DIALOG (gtk_widget_get_toplevel(GTK_WIDGET(button))), GTK_RESPONSE_OK);
}

void 
response_cancel (GtkButton *button, gpointer user_data) {
  gtk_dialog_response (GTK_DIALOG (gtk_widget_get_toplevel(GTK_WIDGET(button))), GTK_RESPONSE_CANCEL);
}


inline void d_print_cancelled(void) {
  d_print(_("cancelled.\n"));
}

inline void d_print_failed(void) {
  d_print(_("failed.\n"));
}

inline void d_print_done(void) {
  d_print(_("done.\n"));
}
