// effects.c
// LiVES (lives-exe)
// (c) G. Finch 2003
// Released under the GPL 2.0 or higher
// see file ../COPYING for licensing details

#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>

#include "main.h"
#include "effects.h"
#include "paramwindow.h"
#include "support.h"

//////////// Effects ////////////////


#ifdef HAVE_YUV4MPEG
#include "yuv4mpeg.h"
#endif


#include "rte_window.h"

static int framecount;

//// some pixel utils


static inline void 
make_black (guchar *pixel) {
  pixel[0]=pixel[1]=pixel[2]=(guchar)0;
}

static inline void 
make_white (guchar *pixel) {
  pixel[0]=pixel[1]=pixel[2]=(guchar)255;
}


void 
nine_fill (guchar *new_data, gint i, gint j, guchar *old_data, gint x, gint y, gint rowstride) {
  // fill nine pixels with the centre colour
  new_data[i-rowstride+j-3]=new_data[i-rowstride]=new_data[i-rowstride+3]=new_data[i+j-3]=new_data[i+j]=new_data[i+j+3]=new_data[i+rowstride+j-3]=new_data[i+rowstride+j]=new_data[i+rowstride+j+3]=old_data[x+y];
  new_data[i-rowstride+j-2]=new_data[i-rowstride+1]=new_data[i-rowstride+4]=new_data[i+j-2]=new_data[i+j+1]=new_data[i+j+4]=new_data[i+rowstride+j-2]=new_data[i+rowstride+j+1]=new_data[i+rowstride+j+4]=old_data[x+y+1];
  new_data[i-rowstride+j-1]=new_data[i-rowstride+2]=new_data[i-rowstride+5]=new_data[i+j-1]=new_data[i+j+2]=new_data[i+j+5]=new_data[i+rowstride+j-1]=new_data[i+rowstride+j+2]=new_data[i+rowstride+j+5]=old_data[x+y+2];
}


///////////////////////////////////////////////////


// Rendered effects




gboolean  
do_effect(lives_rfx_t *rfx, gboolean is_preview) {
  // returns FALSE if the user cancelled
  gint oundo_start=cfile->undo_start;
  gint oundo_end=cfile->undo_end;
  gchar effectstring[128];
  gdouble old_pb_fps=cfile->pb_fps;

  gchar *text;
  gchar *fxcommand=NULL,*cmd;
  gint current_file=mainw->current_file;

  if (is_preview) {
    if (rfx->props&RFX_PROPS_MAY_RESIZE) return TRUE; // don't know how to handle resized frames
    cfile->progress_start=cfile->undo_start=1;
    cfile->progress_end=cfile->undo_end=cfile->frames;
  }
  else if (rfx->num_in_channels!=2) {
    cfile->progress_start=cfile->undo_start=cfile->start;
    cfile->progress_end=cfile->undo_end=cfile->end;
  }
  if (!mainw->internal_messaging) {
    gchar *pdefault;
    if (rfx->num_in_channels==2) {
      // transition has a few extra bits
      pdefault=g_strdup_printf ("%s %d %d %d %d %d %d \"%s/%s\"",cfile->handle,rfx->status,cfile->progress_start,cfile->progress_end,cfile->hsize,cfile->vsize,clipboard->start,prefs->tmpdir,clipboard->handle);
    }
    else {
      pdefault=g_strdup_printf ("%s %d %d %d %d %d",cfile->handle,rfx->status,cfile->progress_start,cfile->progress_end,cfile->hsize,cfile->vsize);
    }
    // and append params
    if (is_preview) {
      cmd=g_strdup("pfxrender");
      mainw->show_procd=FALSE;
    }
    else cmd=g_strdup("fxrender");
    fxcommand=g_strconcat ("smogrify ",cmd,"_",rfx->name," ", pdefault, param_marshall (rfx, FALSE), NULL);
    g_free(cmd);
    g_free (pdefault);
  }

  unlink(cfile->info_file);

  if (!mainw->internal_messaging) {
    system(fxcommand);
    g_free (fxcommand);
  }
  else {
    if (mainw->num_tr_applied>0&&mainw->blend_file>0&&mainw->files[mainw->blend_file]->clip_type!=CLIP_TYPE_GENERATOR) {
      mainw->files[mainw->blend_file]->frameno=mainw->files[mainw->blend_file]->start-1;
      // sigh, start off forwards i suppose...:-)
      mainw->blend_file_step=1;
    }
  }
  mainw->effects_paused=FALSE;
  mainw->cancel_type=CANCEL_HUP;

  if (is_preview) {
    cfile->undo_start=oundo_start;
    cfile->undo_end=oundo_end;
    return TRUE;
  }

  if (rfx->props&RFX_PROPS_MAY_RESIZE) {
    text=g_strdup_printf(_ ("%s all frames..."),_ (rfx->action_desc));
  }
  else if (rfx->num_in_channels==2) {
    text=g_strdup_printf(_ ("%s clipboard into frames %d to %d..."),_ (rfx->action_desc),cfile->progress_start,cfile->progress_end);
    } else {
    text=g_strdup_printf(_ ("%s frames %d to %d..."),_ (rfx->action_desc),cfile->start,cfile->end);
  }
  d_print(text);
  g_free(text);

  cfile->redoable=cfile->undoable=FALSE;
  gtk_widget_set_sensitive (mainw->redo, FALSE);
  gtk_widget_set_sensitive (mainw->undo, FALSE);

  cfile->undo_action=UNDO_EFFECT;

  if (rfx->props&RFX_PROPS_MAY_RESIZE) {
    cfile->ohsize=cfile->hsize;
    cfile->ovsize=cfile->vsize;
    mainw->resizing=TRUE;
    cfile->nokeep=TRUE;
  }

  // 'play' as fast as we possibly can
  cfile->pb_fps=1000000.;


  if (rfx->num_in_channels==2) {
    g_snprintf (effectstring,128,_ ("%s clipboard with selection"),_ (rfx->action_desc));
  }
  else {
    g_snprintf (effectstring,128,_ ("%s frames %d to %d"),_ (rfx->action_desc),cfile->undo_start,cfile->undo_end);
  }

  if (!do_progress_dialog(TRUE,TRUE,effectstring)||mainw->error) {
    mainw->show_procd=TRUE;
    mainw->cancel_type=CANCEL_KILL;
    if (mainw->error) {
      do_error_dialog (mainw->msg);
      d_print_failed();
    }
    cfile->undo_start=oundo_start;
    cfile->undo_end=oundo_end;
    cfile->pb_fps=old_pb_fps;
    mainw->internal_messaging=FALSE;
    mainw->resizing=FALSE;
    cfile->nokeep=FALSE;
    return FALSE;
  }

  mainw->cancel_type=CANCEL_KILL;
  mainw->resizing=FALSE;
  cfile->nokeep=FALSE;
  cfile->changed=TRUE;
  gtk_widget_set_sensitive (mainw->undo, TRUE);
  cfile->undoable=TRUE;
  cfile->pb_fps=old_pb_fps;
  mainw->internal_messaging=FALSE;
  gtk_widget_set_sensitive (mainw->select_last, TRUE);
  set_undoable (_ (rfx->menu_text),TRUE);
  mainw->show_procd=TRUE;

  if (rfx->props&RFX_PROPS_MAY_RESIZE) {
    // get new frame size
    gchar **array;
    gint numtok=get_token_count (mainw->msg,'|');
    
    if (numtok>1) {
      array=g_strsplit(mainw->msg,"|",numtok);
      // [0] is "completed"
      cfile->hsize=atoi (array[1]);
      cfile->vsize=atoi (array[2]);
      g_strfreev(array);
    }
    if (cfile->hsize==cfile->ohsize&&cfile->vsize==cfile->ovsize) cfile->undo_action=UNDO_EFFECT;
    else {
      save_clip_value(mainw->current_file,CLIP_DETAILS_WIDTH,&cfile->hsize);
      save_clip_value(mainw->current_file,CLIP_DETAILS_HEIGHT,&cfile->vsize);
      cfile->undo_action=UNDO_RESIZABLE;
      mainw->current_file=0; // force resize on switch_to_file()
    }
  }

  if (rfx->num_in_channels!=2) {
    switch_to_file (mainw->current_file,current_file);
    d_print_done();
  }
  return TRUE;
}



// realtime fx


  

gint 
realfx_progress (gboolean reset) {
  static int i;
  GError *error=NULL;
  gchar fname[256];
  gchar oname[256];
  GdkPixbuf *image;
  gchar *com;
  gint64 frameticks;

  // this is called periodically from do_processing_dialog for internal effects

  if (reset==TRUE) {
    i=cfile->start;
    clear_mainw_msg();
    return 1;
  }

  if (mainw->effects_paused) return 1;

  // sig_progress...
  g_snprintf (mainw->msg,256,"%d",i);
  // load, effect, save frame

  g_snprintf(fname,256,"%s/%s/%08d.%s",prefs->tmpdir,cfile->handle,i,prefs->image_ext);
  g_snprintf(oname,256,"%s/%s/%08d.mgk",prefs->tmpdir,cfile->handle,i);

  image=gdk_pixbuf_new_from_file(fname,&error);

  if (image==NULL) {
    g_snprintf (mainw->msg,256,"error|missing image %d",i);
    return 1;
  }

  frameticks=(i-cfile->start+1.)/cfile->fps*U_SECL;

  image=on_rte_apply (image, NULL, cfile->hsize, cfile->vsize, (weed_timecode_t)frameticks);

  if (!strcmp (prefs->image_ext,"jpg")) {
    gdk_pixbuf_save (image, oname, "jpeg", &error,"quality", "100", NULL);
  }
  else if (!strcmp (prefs->image_ext,"png")) {
    gdk_pixbuf_save (image, oname, "png", &error, NULL);
  }
  else {
    //gdk_pixbuf_save_to_callback(...);
  }
  gdk_pixbuf_unref (image);
  
  if (++i>cfile->end) {
    com=g_strdup_printf ("smogrify mv_mgk %s %d %d",cfile->handle,cfile->start,cfile->end);
    system (com);
    g_free (com);
    mainw->internal_messaging=FALSE;
  }
  return 1;
}



void
on_realfx_activate                   (GtkMenuItem     *menuitem,
				      gpointer         user_data)
{
  mainw->internal_messaging=TRUE;
  framecount=0;
  mainw->progress_fn=&realfx_progress;
  mainw->progress_fn (TRUE);
  invalidate_pixel_buffers();
  do_effect (&mainw->rendered_fx[0],FALSE);
}




GdkPixbuf *on_rte_apply (GdkPixbuf *pixbuf, weed_plant_t *filter_map, int opwidth, int opheight, weed_timecode_t tc) {
  // TODO - create layers and pull frames from filters
  // realtime effects
  int rowstride;
  int height;
  int width;
  weed_plant_t **layers;
  GdkPixbuf *blendr=NULL;
  int i;

  if (mainw->foreign) return NULL;

  layers=(weed_plant_t **)g_malloc(3*sizeof(weed_plant_t *));

  layers[2]=NULL;

  width=gdk_pixbuf_get_width(pixbuf);
  height=gdk_pixbuf_get_height(pixbuf);
  rowstride=gdk_pixbuf_get_rowstride(pixbuf);
  
  layers[0]=weed_plant_new(WEED_PLANT_CHANNEL);
  weed_set_int_value(layers[0],"width",width);
  weed_set_int_value(layers[0],"height",height);
  weed_set_int_value(layers[0],"rowstrides",rowstride);
  weed_set_int_value(layers[0],"current_palette",WEED_PALETTE_RGB24);
  
  if (pixbuf_to_layer(layers[0],pixbuf)) {
    mainw->do_not_free=gdk_pixbuf_get_pixels(pixbuf);
    mainw->free_fn=lives_free_with_check;
  }
  g_object_unref(pixbuf);
  mainw->do_not_free=NULL;
  mainw->free_fn=free;

  if (mainw->num_tr_applied&&mainw->blend_file!=mainw->current_file&&mainw->blend_file!=-1) {
    blendr=get_blend_frame(tc);
  }
  if (blendr!=NULL&&gdk_pixbuf_get_bits_per_sample(blendr)==8&&gdk_pixbuf_get_n_channels(blendr)==3) {
    
    width=gdk_pixbuf_get_width(blendr);
    height=gdk_pixbuf_get_height(blendr);
    rowstride=gdk_pixbuf_get_rowstride(blendr);
    
    layers[1]=weed_plant_new(WEED_PLANT_CHANNEL);
    weed_set_int_value(layers[1],"width",width);
    weed_set_int_value(layers[1],"height",height);
    weed_set_int_value(layers[1],"rowstrides",rowstride);
    weed_set_int_value(layers[1],"current_palette",WEED_PALETTE_RGB24);
    if (pixbuf_to_layer(layers[1],blendr)) {
      mainw->do_not_free=gdk_pixbuf_get_pixels(blendr);
      mainw->free_fn=lives_free_with_check;
    }
    g_object_unref(blendr);
    mainw->do_not_free=NULL;
    mainw->free_fn=free;
  }
  else layers[1]=NULL;

  pixbuf=weed_apply_effects(layers,mainw->filter_map,opwidth,opheight,tc,mainw->pchains);

  // all our pixel_data will have been free'd already
  for (i=0;layers[i]!=NULL;i++) weed_plant_free(layers[i]);

  g_free(layers);

  return pixbuf;
}


void 
invalidate_pixel_buffers (void) {
  // invalidate all pixel buffers

  // we call this when we start playing, and when we switch to a different sized clip
  // and also when we stop playing

  if (mainw->blend_file>-1&&mainw->num_tr_applied&&(mainw->files[mainw->blend_file]==NULL||(mainw->files[mainw->blend_file]->clip_type==CLIP_TYPE_DISK&&(!mainw->files[mainw->blend_file]->frames||!mainw->files[mainw->blend_file]->is_loaded)))) {
    // invalid blend file
    mainw->blend_file=mainw->current_file;
  }
  if (!(mainw->uyvy_frame==NULL)) {
    g_free(mainw->uyvy_frame);
    mainw->uyvy_frame=NULL;
  }
  if (!(mainw->yuv_frame==NULL)) {
    g_free(mainw->yuv_frame);
    mainw->yuv_frame=NULL;
  }
  return;
}


GdkPixbuf *get_blend_frame(weed_timecode_t tc) {
  // TODO - remove - just use pull_frame
  GError *error=NULL;
  GdkPixbuf *blendp=NULL;
  file *blend_file;
  gboolean osc_block;

  if (mainw->blend_file==-1||mainw->files[mainw->blend_file]==NULL) return NULL;
  blend_file=mainw->files[mainw->blend_file];
  
  blend_file->frameno+=mainw->blend_file_step;
  
  if (blend_file->frameno>blend_file->end) {
    if (mainw->ping_pong) {
      blend_file->frameno=blend_file->end;
      mainw->blend_file_step=-1;
    }
    else {
      blend_file->frameno=blend_file->start;
      mainw->blend_file_step=1;
    }
  }
  else if (blend_file->frameno<blend_file->start) {
    if (mainw->ping_pong) {
      blend_file->frameno=blend_file->start;
      mainw->blend_file_step=1;
    }
    else {
      blend_file->frameno=blend_file->end;
      mainw->blend_file_step=-1;
    }
  }
  
  switch (blend_file->clip_type) {
  case CLIP_TYPE_DISK:
    blendp=gdk_pixbuf_new_from_file(g_strdup_printf ("%s/%s/%08d.%s",prefs->tmpdir,mainw->files[mainw->blend_file]->handle,mainw->files[mainw->blend_file]->frameno,prefs->image_ext),&error);
    break;
#ifdef HAVE_YUV4MPEG
  case CLIP_TYPE_YUV4MPEG:
    blendp=gdk_pixbuf_new_from_yuv4m (blend_file->ext_src,&error);
    break;
#endif

  case CLIP_TYPE_GENERATOR:
    osc_block=mainw->osc_block;
    mainw->osc_block=TRUE;
    blendp=gdk_pixbuf_new_from_generator ((weed_plant_t *)blend_file->ext_src,tc);
    mainw->osc_block=osc_block;
    break;
  }
  if (error!=NULL) g_error_free(error); // returns NULL on error
  return blendp;
}


////////////////////////////////////////////////////////////////////
// keypresses





gboolean rte_on_off_callback (GtkAccelGroup *group, GObject *obj, guint keyval, GdkModifierType mod, gpointer user_data)
{
// this is the callback which happens when a rte is keyed
  gint key=GPOINTER_TO_INT(user_data);
  guint new_rte=GU641<<(key-1);

  if (key==EFFECT_NONE) {
    // switch up/down keys to default (fps change)
    weed_deinit_all();
  }
  else {
    mainw->rte^=new_rte;
    if (mainw->rte&new_rte) {
      // switch is ON
      // WARNING - if we start playing because a generator was started, we block here
      mainw->osc_block=TRUE;
      mainw->last_grabable_effect=key-1;

      if (rte_window!=NULL) rtew_set_keych(key-1,TRUE);
      if (!(weed_init_effect(key-1))) {
	// ran out of instance slots, no effect assigned, or some other error
	mainw->rte^=new_rte;
	if (rte_window!=NULL&&group!=NULL) rtew_set_keych(key-1,FALSE);
      }
      mainw->osc_block=FALSE;
      return TRUE;
    }
    else {
      // effect is OFF
      mainw->osc_block=TRUE;
      weed_deinit_effect(key-1);
      if (mainw->rte&(GU641<<(key-1))) mainw->rte^=(GU641<<(key-1));
      if (rte_window!=NULL&&group!=NULL) rtew_set_keych(key-1,FALSE);
      mainw->osc_block=FALSE;
    }
  }
  if (mainw->current_file>0&&cfile->play_paused) {
    load_frame_image (cfile->frameno,cfile->last_frameno);
  }
  return TRUE;
}



gboolean rte_on_off_callback_hook (GtkToggleButton *button, gpointer user_data) {
  rte_on_off_callback (NULL, NULL, 0, 0, user_data);
  return TRUE;
}


gboolean grabkeys_callback (GtkAccelGroup *group, GObject *obj, guint keyval, GdkModifierType mod, gpointer user_data) {
  // assign the keys to the last key-grabable effect 
  mainw->rte_keys=mainw->last_grabable_effect;
  mainw->osc_block=TRUE;
  if (rte_window!=NULL) {
    if (group!=NULL) rtew_set_keygr(mainw->rte_keys);
  }
  mainw->blend_factor=weed_get_blend_factor(mainw->rte_keys);
  mainw->osc_block=FALSE;
  return TRUE;
}


gboolean grabkeys_callback_hook (GtkToggleButton *button, gpointer user_data) {
  if (!gtk_toggle_button_get_active(button)) return TRUE;
  mainw->last_grabable_effect=GPOINTER_TO_INT(user_data);
  grabkeys_callback (NULL, NULL, 0, 0, user_data);
  return TRUE;
}


gboolean rtemode_callback (GtkAccelGroup *group, GObject *obj, guint keyval, GdkModifierType mod, gpointer user_data) {
  if (mainw->rte_keys==-1) return TRUE;
  rte_key_setmode(0,-1);
  mainw->blend_factor=weed_get_blend_factor(mainw->rte_keys);
  return TRUE;
}


gboolean rtemode_callback_hook (GtkToggleButton *button, gpointer user_data) {
  gint key_mode=GPOINTER_TO_INT(user_data);
  int modes=rte_getmodespk();
  gint key=(gint)(key_mode/modes);
  gint mode=key_mode-key*modes;

  if (!gtk_toggle_button_get_active(button)) return TRUE;

  rte_key_setmode(key+1,mode);
  return TRUE;
}


gboolean swap_fg_bg_callback (GtkAccelGroup *group, GObject *obj, guint keyval, GdkModifierType mod, gpointer user_data) {
  gint old_file=mainw->current_file;
  gint blend_step=mainw->blend_file_step;

  if (mainw->playing_file<0||!mainw->num_tr_applied||mainw->noswitch||mainw->blend_file==-1||mainw->blend_file==mainw->current_file||mainw->preview||mainw->noswitch) {
    return TRUE;
  }
  mainw->blend_file_step=cfile->pb_fps>0?1:-1;
  do_quick_switch (mainw->blend_file);
  if ((cfile->pb_fps>0&&blend_step<0)||(cfile->pb_fps<0&&blend_step>0)) cfile->pb_fps*=-1;
  mainw->blend_file=old_file;

  rte_swap_fg_bg();
  return TRUE;

  // TODO - something with fps matching
  // **TODO - for weed, invert all transition parameters for any active effects
}






//////////////////////////////////////////////////////////////



