// osc.c
// LiVES (lives-exe)
// (c) G. Finch 2004
// Released under the GPL 2.0 or higher
// see file ../COPYING for licensing details

// partly based on some code by N. Elburg


#include "main.h"

#ifdef ENABLE_OSC
#include <netinet/in.h>

#include "osc.h"
#include "htmsocket.h"
#include "callbacks.h"
#include "effects.h"
#include "support.h"
#include "rte_window.h"

void *status_socket;
void *notify_socket;

static lives_osc *livesOSC=NULL;

#define OSC_STRING_SIZE 255

#define FX_MAX FX_KEYS_MAX_VIRTUAL-1


/* convert a big endian 32 bit string to an int for internal use */

static int toInt(const char* b) {
  if (G_BYTE_ORDER==G_LITTLE_ENDIAN) {
    return (( (int) b[3] ) & 0xff ) + ((((int) b[2]) & 0xff) << 8) + ((((int) b[1]) & 0xff) << 16) +
      ((((int) b[0] ) & 0xff) << 24);
  }
    return (( (int) b[0] ) & 0xff ) + ((((int) b[1]) & 0xff) << 8) + ((((int) b[2]) & 0xff) << 16) +
      ((((int) b[3] ) & 0xff) << 24);
}


/* convert a big endian 32 bit float to a float for internal use */
static float toFloat(const char* b) {
  if (G_BYTE_ORDER==G_LITTLE_ENDIAN) {
    float fl;
    guchar *rev=g_malloc(4);
    rev[0]=b[3];
    rev[1]=b[2];
    rev[2]=b[1];
    rev[3]=b[0];
    fl=*(float *)rev;
    g_free(rev);
    return fl;
  }
  return *(float *)b;
}


// wrapper for correct typing
void *lives_malloc(int size) {
  return g_malloc(size);
}

static gboolean using_types;
static gint osc_header_len;
static gint offset;

inline gint pad4(gint val) {
  return (gint)((val+4)/4)*4;
}

static gboolean lives_osc_check_arguments(int arglen, const void *vargs, const gchar *check_pattern, gboolean calc_header_len) {
  // check if using type tags and get header_len
  // should be called from each cb that uses parameters
  const char *args=vargs;
  gint header_len;

  osc_header_len=0;
  offset=0;

  if (arglen<4||args[0] != 0x2c) return (!(using_types=FALSE)); // comma
  using_types=TRUE;

  header_len=pad4(strlen(check_pattern));

  if (arglen<header_len) return FALSE;
  if (!strncmp (check_pattern,++args,strlen (check_pattern))) {
    if (calc_header_len) osc_header_len=header_len;
    return TRUE;
  }
  return FALSE;
}



static void lives_osc_parse_char_argument(const void *vargs, gchar *dst)
{
  const char *args = (char*)vargs;
  strncpy(dst, args+osc_header_len+offset,1);
  offset+=4;
}



static void lives_osc_parse_string_argument(const void *vargs, gchar *dst)
{
  const char *args = (char*)vargs;
  g_snprintf(dst, OSC_STRING_SIZE, "%s", args+osc_header_len+offset);
  offset+=pad4(strlen (dst));
}



static void lives_osc_parse_int_argument(const void *vargs, gint *arguments)
{
  const char *args = (char*)vargs;
  arguments[0] = toInt( args + osc_header_len + offset );
  offset+=4;
}

static void lives_osc_parse_float_argument(const void *vargs, gfloat *arguments)
{
  const char *args = (char*)vargs;
  arguments[0] = toFloat( (const char *)(args + osc_header_len + offset) );
  offset+=4;
}






/* memory allocation functions of libOMC_dirty (OSC) */


void *_lives_osc_time_malloc(int num_bytes) {
  return g_malloc( num_bytes );
}

void *_lives_osc_rt_malloc(int num_bytes) {
  return g_malloc( num_bytes );
}


// status returns

gboolean lives_status_send (gchar *msg) {
  return SendHTMSocket (status_socket,strlen (msg),msg);
}



void lives_osc_close_status_socket (void) {
  if (status_socket!=NULL) CloseHTMSocket (status_socket);
  status_socket=NULL;
}

void lives_osc_close_notify_socket (void) {
  if (notify_socket!=NULL) CloseHTMSocket (notify_socket);
  notify_socket=NULL;
}



///////////////////////////////////// CALLBACKS FOR OSC ////////////////////////////////////////


/* /video/play */
void lives_osc_cb_play_forward(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {

  if (mainw->playing_file==-1&&mainw->current_file>0) on_playall_activate(NULL,NULL);

}

void lives_osc_cb_play_backward(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {

  if (mainw->current_file<0||cfile->clip_type!=CLIP_TYPE_DISK) return;
  dirchange_callback(NULL,NULL,0,0,GINT_TO_POINTER(TRUE));

}

/* /video/stop */
void lives_osc_cb_stop(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {
  if (mainw->playing_file>-1) on_stop_activate(NULL,NULL);
}


void lives_osc_cb_set_fps(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {
  gint fpsi;
  gfloat fps;
  if (lives_osc_check_arguments (arglen,vargs,"i",TRUE)) {
    lives_osc_parse_int_argument(vargs,&fpsi);
    fps=(float)fpsi;
  }
  else {
    if (!lives_osc_check_arguments (arglen,vargs,"f",TRUE)) return;
    lives_osc_parse_float_argument(vargs,&fps);
  }
  if (mainw->playing_file>-1) gtk_spin_button_set_value(GTK_SPIN_BUTTON(mainw->spinbutton_pb_fps),(gdouble)(fps));

}


void lives_osc_cb_fx_reset(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {

  if (!mainw->osc_block) rte_on_off_callback(NULL,NULL,0,0,GINT_TO_POINTER(0));

}

void lives_osc_cb_fx_map_clear(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {
  if (!mainw->osc_block) on_clear_all_clicked(NULL,NULL);
}

void lives_osc_cb_fx_map(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {
  int effect_key;
  gchar effect_name[OSC_STRING_SIZE];

  if (!lives_osc_check_arguments (arglen,vargs,"is",TRUE)) return;
  lives_osc_parse_int_argument(vargs,&effect_key);
  lives_osc_parse_string_argument(vargs,effect_name);
  if (!mainw->osc_block) weed_add_effectkey(effect_key,effect_name,FALSE); // allow partial matches
}

void lives_osc_cb_fx_enable(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {

  int effect_key;
  gint grab=mainw->last_grabable_effect;
  if (!lives_osc_check_arguments (arglen,vargs,"i",TRUE)) return;
  lives_osc_parse_int_argument(vargs,&effect_key);

  if (!(mainw->rte&(GU641<<(effect_key-1)))) {
    if (!mainw->osc_block) rte_on_off_callback(NULL,NULL,0,0,GINT_TO_POINTER(effect_key));
  }
  mainw->last_grabable_effect=grab;
}

void lives_osc_cb_fx_disable(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {

  int effect_key;
  if (!lives_osc_check_arguments (arglen,vargs,"i",TRUE)) return;
  lives_osc_parse_int_argument(vargs,&effect_key);
  if (mainw->rte&(GU641<<(effect_key-1))) {
    if (!mainw->osc_block) rte_on_off_callback(NULL,NULL,0,0,GINT_TO_POINTER(effect_key));
  }
}

void lives_osc_cb_clip_select(void *context, int arglen, const void *vargs, OSCTimeTag when,
	NetworkReturnAddressPtr ra) {
  // switch fg clip

  int clip;
  if (mainw->current_file<1||mainw->preview||mainw->is_processing) return; // TODO

  if (!lives_osc_check_arguments (arglen,vargs,"i",TRUE)) return;
  lives_osc_parse_int_argument(vargs,&clip);

  if (clip>0&&clip<MAX_FILES-1) {
    if (mainw->files[clip]!=NULL&&mainw->files[clip]->clip_type==CLIP_TYPE_DISK) {
      if (mainw->playing_file>0) {
	mainw->pre_src_file=clip;
	mainw->new_clip=clip;
      }
      else if (mainw->playing_file==-1) {
	switch_to_file((mainw->current_file=0),clip);
      }
    }
  }
}


void lives_osc_cb_quit(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {

  if (mainw->was_set) mainw->leave_files=TRUE;
  lives_exit();

}


void lives_osc_cb_open_status_socket(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {
  gchar host[OSC_STRING_SIZE];
  int port;

  if (!lives_osc_check_arguments (arglen,vargs,"si",TRUE)) {
    if (!lives_osc_check_arguments (arglen,vargs,"i",TRUE)) return;
    g_snprintf (host,OSC_STRING_SIZE,"localhost");
  }
  else lives_osc_parse_string_argument(vargs,host);
  lives_osc_parse_int_argument(vargs,&port);

  if (!(status_socket=OpenHTMSocket (host,port))) g_printerr ("Unable to open status socket !\n");

}

void lives_osc_cb_close_status_socket(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {
  lives_osc_close_status_socket();
}






void lives_osc_cb_clip_count(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {

  if (status_socket==NULL) return;
  lives_status_send (g_strdup_printf ("%d\n",mainw->clips_available));

}

void lives_osc_cb_clip_goto(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {

  int frame;
  if (mainw->current_file<1||mainw->preview||mainw->playing_file<1) return;

  if (!lives_osc_check_arguments (arglen,vargs,"i",TRUE)) return;
  lives_osc_parse_int_argument(vargs,&frame);

  if (frame<1||frame>cfile->frames||cfile->clip_type!=CLIP_TYPE_DISK) return;

  cfile->last_frameno=cfile->frameno=frame;

}


void lives_osc_cb_clip_get_current(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {

  if (status_socket==NULL) return;
  lives_status_send (g_strdup_printf ("%d\n",mainw->current_file>0?mainw->current_file:0));

}


void lives_osc_cb_clip_set_start(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {

  int frame;
  if (mainw->current_file<1||mainw->preview||mainw->is_processing) return;

  if (!lives_osc_check_arguments (arglen,vargs,"i",TRUE)) return;

  lives_osc_parse_int_argument(vargs,&frame);
  if (frame<1||frame>cfile->frames||cfile->clip_type!=CLIP_TYPE_DISK) return;

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(mainw->spinbutton_start),frame);
}

void lives_osc_cb_clip_set_end(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {

  int frame;
  if (mainw->current_file<1||mainw->preview||mainw->is_processing) return;

  lives_osc_check_arguments (arglen,vargs,"i",TRUE);
  lives_osc_parse_int_argument(vargs,&frame);
  if (frame<1||frame>cfile->frames||cfile->clip_type!=CLIP_TYPE_DISK) return;

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(mainw->spinbutton_end),frame);
}


void lives_osc_cb_clip_select_all(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {

  if (mainw->current_file<1||mainw->preview||mainw->is_processing) return;
  if (cfile->clip_type!=CLIP_TYPE_DISK||!cfile->frames) return;

  on_select_all_activate (NULL,NULL);
}

void lives_osc_cb_clip_isvalid(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {

  int clip;
  if (status_socket==NULL) return;

  lives_osc_check_arguments (arglen,vargs,"i",TRUE);
  lives_osc_parse_int_argument(vargs,&clip);

  if (clip>0&&clip<MAX_FILES&&mainw->files[clip]!=NULL) lives_status_send ("1\n");
  else lives_status_send ("0\n");

}

void lives_osc_cb_rte_count(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {
  // count realtime effects - (only those assigned to keys for now)
  if (status_socket==NULL) return;

  // TODO - return 32
  lives_status_send (g_strdup_printf ("%d\n",prefs->rte_keys_virtual));

}

void lives_osc_cb_rteuser_count(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {
  // count realtime effects
  if (status_socket==NULL) return;

  lives_status_send (g_strdup_printf ("%d\n",FX_MAX));

}


void lives_osc_cb_fssepwin_enable(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {
  if (!mainw->sep_win) {
    on_sepwin_pressed (NULL,NULL);
  }

 if (!mainw->fs) {
      on_full_screen_pressed (NULL,NULL);
  }
}


void lives_osc_cb_fssepwin_disable(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {

  if (mainw->fs) {
      on_full_screen_pressed (NULL,NULL);
  }
  if (mainw->sep_win) {
    on_sepwin_pressed (NULL,NULL);
  }
}

void lives_osc_cb_op_fps_set(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {

  // TODO - make float, show messages
  int fps;
  gdouble fpsd;
  if (mainw->fixed_fps!=-1) return;
  if (!lives_osc_check_arguments (arglen,vargs,"i",TRUE)) return;
  lives_osc_parse_int_argument(vargs,&fps);
  fpsd=(gdouble)(fps*1.);
  if (fps>0&&fpsd<=FPS_MAX) {
    mainw->fixed_fps=fpsd;
    d_print (g_strdup_printf (_("Syncing to external framerate of %.8f frames per second.\n"),fpsd));
  }
  else if (fps==0) mainw->fixed_fps=-1.;

}


void lives_osc_cb_op_nodrope(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {
  
  mainw->noframedrop=TRUE;

}


void lives_osc_cb_op_nodropd(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {
  
  mainw->noframedrop=FALSE;

}


void lives_osc_cb_rte_setmode(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {
  int effect_key;
  int mode;

  if (!lives_osc_check_arguments (arglen,vargs,"ii",TRUE)) return;
  lives_osc_parse_int_argument(vargs,&effect_key);
  lives_osc_parse_int_argument(vargs,&mode);
  if (effect_key<1||effect_key>FX_MAX||mode<1||mode>rte_getmodespk()) return;
  if (!mainw->osc_block) rte_key_setmode (effect_key,mode-1);

}

void lives_osc_cb_rte_setparam(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {

  int effect_key;
  int pnum;
  int valuei;
  float valuef;

  if (!lives_osc_check_arguments (arglen,vargs,"ii",FALSE)) return;
  osc_header_len=8; // i'm not sure why this is 8 instead of 4

  lives_osc_parse_int_argument(vargs,&effect_key);
  lives_osc_parse_int_argument(vargs,&pnum);
  //g_print("pt bx %d %d\n",effect_key,pnum);
  if (effect_key<1||effect_key>FX_MAX) return;
  //g_print("key %d pnum %d\n",effect_key,pnum);

  if (!mainw->osc_block) {
    int error;
    weed_plant_t *inst=rte_keymode_get_instance(effect_key,rte_key_getmode(effect_key-1));
    weed_plant_t **in_params;
    weed_plant_t *tparam;
    weed_plant_t *tparamtmpl;
    int hint,cspace;
    int nparams;
    int valuesi[4];
    float valuesf[4];
    double valuesd[4];

    if (!weed_plant_has_leaf(inst,"in_parameters")) return;
    nparams=weed_leaf_num_elements(inst,"in_parameters");
    if (pnum>=nparams) return;

    in_params=weed_get_plantptr_array(inst,"in_parameters",&error);

    tparam=in_params[pnum];
    tparamtmpl=weed_get_plantptr_value(tparam,"template",&error);
    hint=weed_get_int_value(tparamtmpl,"hint",&error);

    switch (hint) {
    case WEED_HINT_INTEGER:
      if (!lives_osc_check_arguments (arglen,vargs,"iii",TRUE)) return;
      offset=8; // check resets it to zero...doh
      lives_osc_parse_int_argument(vargs,&valuei);
      weed_set_int_value(tparam,"value",valuei);
      break;

    case WEED_HINT_FLOAT:
      if (!lives_osc_check_arguments (arglen,vargs,"iif",TRUE)) return;
      offset=8; // check resets it to zero
      lives_osc_parse_float_argument(vargs,&valuef);
      weed_set_double_value(tparam,"value",(double)valuef);
      break;

    case WEED_HINT_COLOR:
    cspace=weed_get_int_value(tparamtmpl,"colorspace",&error);
    switch (cspace) {
    case WEED_COLORSPACE_RGB:
      if (weed_leaf_seed_type(tparamtmpl,"default")==WEED_SEED_INT) {
	if (!lives_osc_check_arguments (arglen,vargs,"iiiii",TRUE)) return;
	offset=8; // check resets it to zero
	lives_osc_parse_int_argument(vargs,&valuesi[0]);
	lives_osc_parse_int_argument(vargs,&valuesi[1]);
	lives_osc_parse_int_argument(vargs,&valuesi[2]);
	weed_set_int_array(tparam,"value",3,valuesi);
      }
      else {
	if (!lives_osc_check_arguments (arglen,vargs,"iifff",TRUE)) return;
	offset=8; // check resets it to zero
	lives_osc_parse_float_argument(vargs,&valuesf[0]);
	lives_osc_parse_float_argument(vargs,&valuesf[1]);
	lives_osc_parse_float_argument(vargs,&valuesf[2]);
	valuesd[0]=(double)valuesf[0];
	valuesd[1]=(double)valuesf[1];
	valuesd[2]=(double)valuesf[2];
	weed_set_double_array(tparam,"value",3,valuesd);
      }
      break;
    default:
      // TODO
      return;
    }
    default:
      // TODO
      return;
    }
    //update_visual_params(rfx,FALSE);
    weed_free(in_params);
  }
}


void lives_osc_cb_rte_getmode(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {

  int effect_key;
  if (status_socket==NULL) return;

  if (!lives_osc_check_arguments (arglen,vargs,"i",TRUE)) return;
  lives_osc_parse_int_argument(vargs,&effect_key);

  if (effect_key<1||effect_key>FX_MAX) {
    lives_status_send ("0\n");
    return;
  }

  lives_status_send (g_strdup_printf ("%d\n",rte_key_getmode (effect_key)));

}


void lives_osc_cb_rte_get_keyfxname(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {
  int effect_key;
  int mode;
  if (status_socket==NULL) return;

  if (!lives_osc_check_arguments (arglen,vargs,"ii",TRUE)) return;
  lives_osc_parse_int_argument(vargs,&effect_key);
  lives_osc_parse_int_argument(vargs,&mode);
  if (effect_key<1||effect_key>FX_MAX||mode<1||mode>rte_getmodespk()) return;
  lives_status_send (g_strdup_printf ("%s\n",rte_keymode_get_filter_name (effect_key,mode-1)));
}

void lives_osc_cb_rte_getmodespk(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {

  int effect_key;
  if (status_socket==NULL) return;

  if (!lives_osc_check_arguments (arglen,vargs,"i",TRUE)) {
    if (lives_osc_check_arguments (arglen,vargs,"",TRUE)) {
      lives_status_send (g_strdup_printf ("%d\n",rte_getmodespk ()));
    }
    return;
  }

  lives_osc_parse_int_argument(vargs,&effect_key);

  if (effect_key>FX_MAX||effect_key<1) {
    lives_status_send ("0\n");
    return;
  }

  lives_status_send (g_strdup_printf ("%d\n",rte_getmodespk ()));

}


void lives_osc_cb_swap(void *context, int arglen, const void *vargs, OSCTimeTag when, NetworkReturnAddressPtr ra) {
 swap_fg_bg_callback (NULL,NULL,0,0,NULL);
}



static struct 
{
  char	 *descr;
  char	 *name;
  void	 (*cb)(void *ctx, int len, const void *vargs, OSCTimeTag when,	NetworkReturnAddressPtr ra);
  int		 leave;
} osc_methods[] = 
  {
    { "/video/play",		"play",		lives_osc_cb_play_forward,			5	},	
    { "/video/stop",		"stop",	        lives_osc_cb_stop,				5	},
    { "/video/set/fps",	       "fps",	lives_osc_cb_set_fps,			12	},
    { "/video/reverse",		"reverse",	lives_osc_cb_play_backward,		5	},
    { "/effect_key/map",		"map",	lives_osc_cb_fx_map,			25	},
    { "/effect_key/map/clear",		"clear",	lives_osc_cb_fx_map_clear,			32	},
    { "/effect_key/reset",		"reset",	lives_osc_cb_fx_reset,			25	},
    { "/effect_key/enable",		"enable",	lives_osc_cb_fx_enable,		        25	},
    { "/effect_key/disable",		"disable",	lives_osc_cb_fx_disable,		        25	},
    { "/effect_key/count",		"count",	lives_osc_cb_rte_count,		        25	},
    { "/effect_key/set/parameter",		"parameter",	lives_osc_cb_rte_setparam,		        31	},
    { "/effect_key/set/mode",		"mode",	lives_osc_cb_rte_setmode,		        31	},
    { "/effect_key/get/mode",		"mode",	lives_osc_cb_rte_getmode,		        26	},
    { "/effect_key/get/name",		"name",	lives_osc_cb_rte_get_keyfxname,		        26	},
    { "/effect_key/get/maxmode",		"maxmode",	lives_osc_cb_rte_getmodespk,		        26	},
    { "/effect_key/get/userkeys",		"userkeys",	lives_osc_cb_rteuser_count,		        26	},
    { "/clip/select",		"select",	lives_osc_cb_clip_select,			1	},
    { "/lives/quit",	         "quit",	        lives_osc_cb_quit,			21	},
    { "/app/quit",	         "quit",	           lives_osc_cb_quit,			22	},
    { "/broadcast/quit",	         "quit",	        lives_osc_cb_quit,			23	},
    { "/lives/open_status_socket",	         "open_status_socket",	        lives_osc_cb_open_status_socket,			21	},
    { "/app/open_status_socket",	         "open_status_socket",	        lives_osc_cb_open_status_socket,			22	},
    { "/clip/count",	         "count",	        lives_osc_cb_clip_count,			1  },
    { "/clip/goto",	         "goto",	        lives_osc_cb_clip_goto,			1	},
    { "/clip/is_valid",	         "is_valid",	        lives_osc_cb_clip_isvalid,			1	},
    { "/clip/select_all",	         "select_all",	        lives_osc_cb_clip_select_all,			1	},
    { "/clip/get/current",	 "current",	        lives_osc_cb_clip_get_current,			24	},
    { "/clip/set/start",	 "start",	        lives_osc_cb_clip_set_start,			13	},
    { "/clip/set/end",	 "end",	        lives_osc_cb_clip_set_end,			13	},
    { "/output/fullscreen/enable",		"enable",	lives_osc_cb_fssepwin_enable,		28	},
      { "/output/fullscreen/disable",		"disable",	lives_osc_cb_fssepwin_disable,       	28	},
      { "/output/set/fps",		"fps",	lives_osc_cb_op_fps_set,       	29	},
      { "/output/nodrop/enable",		"fps",	lives_osc_cb_op_nodrope,       	30	},
      { "/output/nodrop/disable",		"fps",	lives_osc_cb_op_nodropd,       	30	},
      { "/lives/swap",		"swap",	lives_osc_cb_swap,       	21	},
      
    
    /*
      { "/video/freeze",		"freeze",	lives_osc_cb_play_freeze,		0	},
      { "/video/slower",		"slower",	lives_osc_cb_play_slower,		0	},
      { "/video/faster",		"faster",	lives_osc_cb_play_faster,		0	},
      
      { "/video/slow",		"slow",		lives_osc_cb_set_dup,			0	},
      { "/video/goto_start",		"goto_start",	lives_osc_cb_skip_to_start,		0	},
      { "/video/goto_end",		"goto_end",	lives_osc_cb_skip_to_end,			0	},
      { "/video/set_frame",		"set_frame",	lives_osc_cb_set_frame,			0	},
      { "/video/prev_frame",		"prev_frame",	lives_osc_cb_prev_frame,			0	},
      { "/video/next_frame",		"next_frame",	lives_osc_cb_next_frame,			0	},
      { "/video/next_second",		"next_second",	lives_osc_cb_next_second,			0	},
      { "/video/prev_second",		"prev_second",	lives_osc_cb_prev_second,			0	},
      
      { "/clip/del",			"del",		lives_osc_cb_clip_del,			1	},
      { "/clip/goto_start",		"goto_start",	lives_osc_cb_select_start,			1	},
      { "/clip/goto_end",		"goto_end",	lives_osc_cb_select_end,			1	},
      { "/clip/set/looptype",		"looptype",	lives_osc_cb_clip_set_looptype,		13	},
      { "/clip/set/fps",		"fps",	lives_osc_cb_clip_set_fps,		13	},
      { "/clip/set/marker",		"marker",	lives_osc_cb_clip_set_marker,		13	},
      { "/clip/set/slow",		"slow",		lives_osc_cb_clip_set_dup,			13	},
      { "/clip/set/nomarker",		"nomarker",	lives_osc_cb_clip_clear_marker,		13	},
      { "/clip/rec/start",		"start",	lives_osc_cb_clip_record_start,		14	},
      { "/clip/rec/stop",		"stop",		lives_osc_cb_clip_record_stop,		14	},
      
      { "/stream/select",		"select",	lives_osc_cb_tag_select,			2	},
      { "/stream/new/v4l",		"v4l",		lives_osc_cb_tag_new_v4l,			19	},
      { "/stream/new/y4m",		"y4m",		lives_osc_cb_tag_new_y4m,			19	},
      //	{ "/stream/new/avformat",	"avformat",	lives_osc_cb_tag_new_avformat,		19	},
      
      { "/stream/rec/o_start",	"o_start",	lives_osc_cb_tag_record_offline_start,	16	},
      { "/stream/rec/o_stop",		"o_stop",	lives_osc_cb_tag_record_offline_stop,	16	},
      { "/stream/rec/start",		"start",	lives_osc_cb_tag_record_start,		16	},
      { "/stream/rec/stop",		"stop",		lives_osc_cb_tag_record_stop,		16	},
      
      { "/chain/fade_out",		"fade_out",	lives_osc_cb_chain_fade_out,		4	},
      { "/chain/enable",		"enable",	lives_osc_cb_chain_enable,			4	},
      { "/chain/disable",		"disable",	lives_osc_cb_chain_disable,		4	},
      { "/chain/opacity",		"opacity",	lives_osc_cb_chain_manual_fade,		4	},
      
      { "/entry/disable",		"disable",	lives_osc_cb_chain_entry_disable_video,	9	},
      { "/entry/enable",		"enable",	lives_osc_cb_chain_entry_enable_video,	9	},
      { "/entry/del",			"del",		lives_osc_cb_chain_entry_del,		9	},
      { "/entry/select",		"select",	lives_osc_cb_chain_entry_select,		9	},
      { "/entry/defaults",		"defaults",	lives_osc_cb_chain_entry_default,		9	},
      { "/entry/preset",		"preset",	lives_osc_cb_chain_entry_preset,		9	},
      { "/entry/set",			"set",		lives_osc_cb_chain_entry_set,		9	},
      { "/entry/channel",		"channel",	lives_osc_cb_chain_entry_channel,		9	},
      { "/entry/source",		"source",	lives_osc_cb_chain_entry_source,		9	},
      
      { "/arg/set",			"set",		lives_osc_cb_chain_entry_set_arg_val,	8	},
      
      
      { "/cl/load",			"load",		lives_osc_cb_load_cliplist,		6 	},
      { "/cl/save",			"save",		lives_osc_cb_save_cliplist,		6	}, */
    
    { NULL,					NULL,		NULL,							0	},
  };

static struct
{
  char *comment;
  char *name;	
  int  leave;
  int  att; 
  int  it;
} osc_cont[] = 
  {
    {	"/video/",	 	"video",	 5, -1,0   	},
    {	"/clip/", 		"clip",		 1, -1,0	},
    {	"/stream/",		"stream",	 2, -1,0	},
    {	"/record/", 		"record",	 3, -1,0	},
    {	"/effect/" , 		"effect",	 4, -1,0	},
    {	"/effect_key/" , 		"effect_key",	 25, -1,0	},
    {	"/effect_key/get/" , 		"get",	 26, 25,0	},
    {	"/effect_key/set/" , 		"set",	 31, 25,0	},
    {	"/effect_key/map/" , 		"map",	 32, 25,0	},
    {	"/lives/" , 		"lives",	 21, -1,0	},
    {	"/app/" , 		"app",	         22, -1,0	},
    {	"/broadcast/" , 	"broadcast",	 23, -1,0	},
    {	"/output/" , 	"output",	 27, -1,0	},
    {	"/output/fullscreen/" , 	"fullscreen",	 28, 27,0	},
    {	"/output/set/" , 	"set",	 29, 27 ,0	},
    {	"/output/nodrop/" , 	"nodrop",	 30, 27 ,0	},
    {	"/video/set/" , 	"set",	 12, 5,0	},
    {	"/clip/set/",   		"set",		 13, 1,0	},
    {	"/clip/get/",   		"get",		 24, 1,0	},

    /*
      {	"/clip/rec", 		"rec",		 14, 1,0	},
      // {	"/clip/chain", 		"chain",	 15, 1,0	},
      // {	"/clip/entry",  	"entry",	 18, 1,0	},
      {	"/stream/rec",		"rec",		 16, 2,0	},
      // {	"/stream/chain",	"chain",	 17, 2,0	},
      // { 	"/stream/entry",	"entry",	 18, 2,0	},
      {	"/stream/new",  	"new",	     	 19, 2,0	},
      {	"/out/",		"out",		 5, -1,0	},
      {	"/cl",			"cl",		 6, -1,0	},
      {	"/el",			"el",		 7, -1,0	},
      { 	"/arg",			"arg",		 8, -1,0	},
      {	"/entry",		"entry",	 9, -1,0	},
      // {	"<n>",			"%d",	 	20, 9 ,20	},
      // {	"<n>",			"%d",		40, 8 ,10	}, */	
    {	NULL,			NULL,		0, -1,0		},
  };


int lives_osc_build_cont( lives_osc *o )
{ 
  /* Create containers /video , /clip, /chain and /tag */
  int i;
  for( i = 0; osc_cont[i].name != NULL ; i ++ )
    {
      if ( osc_cont[i].it == 0 )
	{
	  o->cqinfo.comment = osc_cont[i].comment;
	  
	  // add a container to a leaf
	  if ( ( o->leaves[ osc_cont[i].leave ] =
		 OSCNewContainer( osc_cont[i].name,
				  (osc_cont[i].att == -1 ? o->container : o->leaves[ osc_cont[i].att ] ),
				  &(o->cqinfo) ) ) == 0 )
	    {
	      if(osc_cont[i].att == - 1)
		{
		  g_printerr( "Cannot create container %d (%s) \n",
			      i, osc_cont[i].name );
		  return 0;
		}
	      else
		{
		  g_printerr( "Cannot add branch %s to  container %d)\n",
			      osc_cont[i].name, osc_cont[i].att );  
		  return 0;
		}
	    }
	}
      else
	{
	  int n = osc_cont[i].it;
	  int j;
	  int base = osc_cont[i].leave;
	  char name[50];
	  char comment[50];
	  
	  for ( j = 0; j < n ; j ++ )
	    {
	      sprintf(name, "N%d", j);	
	      sprintf(comment, "<%d>", j);
	      g_printerr( "Try cont.%d  '%s', %d %d\n",j, name,
			  base + j, base );	
	      o->cqinfo.comment = comment;
	      if ((	o->leaves[ base + j ] = OSCNewContainer( name,
								 o->leaves[ osc_cont[i].att ] ,
								 &(o->cqinfo) ) ) == 0 )
		{
		  g_printerr( "Cannot auto numerate container %s \n",
			      osc_cont[i].name );
		  return 0;
		  
		}
	    }
	}
    }
  return 1;
}

int	lives_osc_attach_methods( lives_osc *o )
{
  int i;
  for( i = 0; osc_methods[i].name != NULL ; i ++ )
    {
      o->ris.description = osc_methods[i].descr;
      OSCNewMethod( osc_methods[i].name, 
		    o->leaves[ osc_methods[i].leave ],
		    osc_methods[i].cb , 
		    &(o->osc_args[0]),
		    &(o->ris));
    }
  return 1;
}	





/* initialization, setup a UDP socket and invoke OSC magic */
lives_osc* lives_osc_allocate(int port_id) {
  lives_osc *o = (lives_osc*)g_malloc(sizeof(lives_osc));
  o->osc_args = (osc_arg*)g_malloc(50 * sizeof(*o->osc_args));
  o->rt.InitTimeMemoryAllocator = _lives_osc_time_malloc;
  o->rt.RealTimeMemoryAllocator = _lives_osc_rt_malloc;
  o->rt.receiveBufferSize = 1024;
  o->rt.numReceiveBuffers = 100;
  o->rt.numQueuedObjects = 100;
  o->rt.numCallbackListNodes = 200;
  o->leaves = (OSCcontainer*) g_malloc(sizeof(OSCcontainer) * 100);
  o->t.initNumContainers = 100;
  o->t.initNumMethods = 200;
  o->t.InitTimeMemoryAllocator = lives_malloc;
  o->t.RealTimeMemoryAllocator = lives_malloc;
  
  if(OSCInitReceive( &(o->rt))==FALSE) {
    d_print( _ ("Cannot initialize OSC receiver\n"));
    return NULL;
  } 
  o->packet = OSCAllocPacketBuffer();
  
  if(NetworkStartUDPServer( o->packet, port_id) != TRUE) {
    d_print( g_strdup_printf (_ ("WARNING: Cannot start OSC server at UDP port %d\n"),port_id));
  }
  else {
    d_print( g_strdup_printf (_ ("Started OSC server at UDP port %d\n"),port_id));
  }
  
  /* Top level container / */
  o->container = OSCInitAddressSpace(&(o->t));
  
  OSCInitContainerQueryResponseInfo( &(o->cqinfo) );
  o->cqinfo.comment = "Video commands";
  
  if( !lives_osc_build_cont( o ))
    return NULL;
  
  OSCInitMethodQueryResponseInfo( &(o->ris));
  if( !lives_osc_attach_methods( o ))
    return NULL;
  
  return o;
}	







void lives_osc_dump()
{
  OSCPrintWholeAddressSpace();
}

 




// CALL THIS PERIODICALLY, will read all queued messages and call callbacks


/* get a packet */
static int lives_osc_get_packet(lives_osc *o) {
  //OSCTimeTag tag;
  struct timeval tv;
  tv.tv_sec=0;
  tv.tv_usec = 0;
  
  /* see if there is something to read , this is effectivly NetworkPacketWaiting */
  // if(ioctl( o->sockfd, FIONREAD, &bytes,0 ) == -1) return 0;
  // if(bytes==0) return 0;
  if(NetworkPacketWaiting(o->packet)==TRUE) {
    /* yes, receive packet from UDP */
    if(NetworkReceivePacket(o->packet) == TRUE) {
      /* OSC must accept this packet (OSC will auto-invoke it, see source !) */
      OSCAcceptPacket(o->packet);
      /* Is this really productive ? */
      OSCBeProductiveWhileWaiting();
      // here we call the callbacks
      
      /* tell caller we had 1 packet */	
      return 1;
    }
  }
  return 0;
}



void lives_osc_free(lives_osc *c)
{
  if(c==NULL) return;
  if(c->leaves) free(c->leaves);
  if(c) free(c);
  c = NULL;
}




////////////////////////////// API public functions /////////////////////



gboolean lives_osc_init(guint udp_port) {

  if (livesOSC!=NULL) {
    // TODO - make this work...
    /*  OSCPacketBuffer packet=livesOSC->packet;
	if (shutdown (packet->returnAddr->sockfd,SHUT_RDWR)) {
	d_print( g_strdup_printf (_ ("Cannot shut down OSC/UDP server\n"));
	}
    */
    if (NetworkStartUDPServer( livesOSC->packet, udp_port) != TRUE) {
      d_print( g_strdup_printf (_ ("Cannot start OSC/UDP server at port %d \n"),udp_port));
    }
  }
  else { 
    livesOSC=lives_osc_allocate(udp_port);
    status_socket=NULL;
    notify_socket=NULL;
  }
  if (livesOSC==NULL) return FALSE;

  gtk_timeout_add(KEY_RPT_INTERVAL,&lives_osc_poll,NULL);

  return TRUE;
}


gboolean lives_osc_poll(gpointer data) {
  // data is always NULL
  // must return TRUE
  if (!mainw->osc_block&&mainw->multitrack==NULL) lives_osc_get_packet(livesOSC);
  return TRUE;
}

void lives_osc_end(void) {
  if (notify_socket!=NULL) {
    //lives_notify_send ("lives::quit\n");
    lives_osc_close_notify_socket();
  }
  if (status_socket!=NULL) {
    lives_osc_close_status_socket();
  }

  if (livesOSC!=NULL) lives_osc_free(livesOSC);
}

#endif
