// plugins.c
// LiVES
// (c) G. Finch 2003 <salsaman@xs4all.nl>
// released under the GNU GPL 2.0 or higher
// see file ../COPYING or www.gnu.org for licensing details

#include <dlfcn.h>

#include "main.h"
#include "resample.h"
#include "support.h"
#include "weed-utils.h"
#include "effects-weed.h"

static gboolean list_plugins;


GList *
get_plugin_list (gchar *plugin_type, gboolean allow_nonex, gchar *plugdir) {
  // returns a GList * of plugins of type plugin_type
  // returns empty list if there are no plugins of that type
  gchar *com;
  if (plugin_type==PLUGIN_THEMES) {
    com=g_strdup_printf ("smogrify list_plugins 0 1 \"%s\" 0",g_filename_from_utf8 (THEME_DIR,-1,NULL,NULL,NULL));
  }
  else if (!strcmp (plugin_type,PLUGIN_RENDERED_EFFECTS_CUSTOM_SCRIPTS)||!strcmp (plugin_type,PLUGIN_RENDERED_EFFECTS_TEST_SCRIPTS)||!strcmp (plugin_type,PLUGIN_RENDERED_EFFECTS_CUSTOM)||!strcmp (plugin_type,PLUGIN_RENDERED_EFFECTS_TEST)) {
    // look in home
    com=g_strdup_printf ("smogrify list_plugins %d 0 \"%s\" 1",allow_nonex,plugin_type);
  }
  else if (!strcmp(plugin_type,PLUGIN_EFFECTS_WEED)) {
    com=g_strdup_printf ("smogrify list_plugins 1 1 \"%s\" 2",g_filename_from_utf8(plugdir,-1,NULL,NULL,NULL));
  }
  else {
    com=g_strdup_printf ("smogrify list_plugins %d 0 \"%s%s\" 0",allow_nonex,PLUGIN_DIR,plugin_type);
  }
  list_plugins=TRUE;
  return get_plugin_result (com,"|",FALSE);
}


// this is an internal function
GList *
get_plugin_result (gchar *command, gchar *delim, gboolean allow_blanks) {
  gchar **array;
  gint bytes=0,pieces;
  int outfile_fd,i;
  gchar *msg,*buf;
  gint error;
  GList *list=NULL;
  gint count=30000000/prefs->sleep_time;  // timeout of 30 seconds

  gchar *outfile;
  gchar *com;
  gchar buffer[65536];

  pthread_mutex_lock(&mainw->gtk_mutex);

  outfile=g_strdup_printf ("%s/.smogplugin.%d",prefs->tmpdir,getpid());
  com=g_strconcat (command," >",outfile,NULL);

  mainw->error=FALSE;

  if ((error=system (com))!=0&&error!=126*256&&error!=256) {
    if (list_plugins==FALSE) {
      gchar *msg2;
      g_free (com);
      if (mainw->is_ready) {
	if ((outfile_fd=open(outfile,O_RDONLY))) {
	  bytes=read (outfile_fd,&buffer,65535);
	  close (outfile_fd);
	  unlink (outfile);
	  memset (buffer+bytes,0,1);
	}
	msg=g_strdup_printf (_("\nPlugin error: %s failed with code %d"),command,error/256);
	if (bytes) {
	  msg2=g_strconcat (msg,g_strdup_printf (_ (" : message was %s\n"),buffer),NULL);
	}
	else {
	  msg2=g_strconcat (msg,"\n",NULL);
	}
	d_print (msg2);
	g_free (msg2);
	g_free (msg);
      }
    }
    g_free (outfile);
    pthread_mutex_unlock(&mainw->gtk_mutex);
    return list;
  }
  g_free (com);
  if (!g_file_test (outfile, G_FILE_TEST_EXISTS)) {
    g_free (outfile);
    g_free (command);
    pthread_mutex_unlock(&mainw->gtk_mutex);
    return NULL;
  }
  pthread_mutex_unlock(&mainw->gtk_mutex);

  while (!(outfile_fd=open(outfile,O_RDONLY))&&(count-->0||list_plugins==TRUE)) {
    g_usleep (prefs->sleep_time);
  }
  
  if (!count) {
    g_printerr (_("Plugin timed out on message %s\n"),command);
    g_free (command);
    g_free (outfile);
    g_free(buffer);
    return list;
  }
  g_free (command);
  
  bytes=read (outfile_fd,&buffer,65535);
  close (outfile_fd);
  unlink (outfile);
  g_free (outfile);
  memset (buffer+bytes,0,1);

#ifdef DEBUG_PLUGINS
  g_print("plugin msg: %s %d\n",buffer,error);
#endif
  
  if (error==256) {
    mainw->error=TRUE;
    g_snprintf (mainw->msg,512,buffer);
    return list;
  }
  
  pthread_mutex_lock(&mainw->gtk_mutex);
  pieces=get_token_count (buffer,delim[0]);
  array=g_strsplit(buffer,delim,pieces);
  for (i=0;i<pieces;i++) {
    if ((buf=g_strdup(array[i]))!=NULL) {
      buf=(g_strchomp (g_strchug(buf)));
      if (strlen (buf)||allow_blanks) {
	list=g_list_append (list, buf);
      }
    }
  }
  g_strfreev (array);
  pthread_mutex_unlock(&mainw->gtk_mutex);
  return list;
}


GList *
plugin_request_with_blanks (const gchar *plugin_type, const gchar *plugin_name, const gchar *request) {
  // allow blanks in a list
  return plugin_request_common(plugin_type, plugin_name, request, "|", TRUE);
}

GList *
plugin_request (const gchar *plugin_type, const gchar *plugin_name, const gchar *request) {
  return plugin_request_common(plugin_type, plugin_name, request, "|", FALSE);
}

GList *
plugin_request_by_line (const gchar *plugin_type, const gchar *plugin_name, const gchar *request) {
  return plugin_request_common(plugin_type, plugin_name, request, "\n", FALSE);
}
GList *
plugin_request_by_space (const gchar *plugin_type, const gchar *plugin_name, const gchar *request) {
  return plugin_request_common(plugin_type, plugin_name, request, " ", FALSE);
}



GList *
plugin_request_common (const gchar *plugin_type, const gchar *plugin_name, const gchar *request, gchar *delim, gboolean allow_blanks) {
  // returns a GList of responses to -request, or NULL on error
  // by_line says whether we split on '\n' or on '|'
  GList *empty_list=NULL;
  gchar *com;

  if (plugin_type!=NULL) {
    // some types live in home directory...
    if (!strcmp (plugin_type,PLUGIN_RENDERED_EFFECTS_CUSTOM)||!strcmp (plugin_type,PLUGIN_RENDERED_EFFECTS_TEST)) {
      com=g_strdup_printf ("%s/%s/%s %s",capable->home_dir,plugin_type,plugin_name,request);
    }
    else {
      com=g_strdup_printf ("%s%s%s/%s %s",prefs->prefix_dir,PLUGIN_DIR,plugin_type,plugin_name,request);
    }
    if (plugin_name==NULL||!strlen(plugin_name)) {
      return empty_list;
    }
  }
  else com=g_strdup (request);

  list_plugins=FALSE;
  return get_plugin_result (com,delim,allow_blanks);
}




void 
do_plugin_error(const gchar *plugin_type, const gchar *plugin_name) {
  gchar *msg;

  if (plugin_name==NULL) {
    msg=g_strdup_printf(_("LiVES was unable to find its %s plugins. Please make sure you have the plugins installed in\n%s%s%s"),plugin_type,prefs->prefix_dir,PLUGIN_DIR,plugin_type);
    if (rdet!=NULL) do_error_dialog_with_check_transient(msg,FALSE,0,GTK_WINDOW(rdet->dialog));
    else do_error_dialog(msg);
    g_free(msg);
    return;
  }
  msg=g_strdup_printf(_("LiVES did not receive a response from the %s plugin called '%s'.\nPlease make sure you have that plugin installed correctly in\n%s%s%s\nor switch to another plugin using Tools|Preferences|Encoding\n"),plugin_type,plugin_name,prefs->prefix_dir,PLUGIN_DIR,plugin_type);
  if (rdet!=NULL) do_error_dialog_with_check_transient(msg,FALSE,0,GTK_WINDOW(rdet->dialog));
  else do_error_dialog(msg);
  g_free(msg);
}






gboolean 
open_vid_playback_plugin (const gchar *name) {
  // this is called on startup or when the user selects a new playback plugin

  // TODO - allow user to select framerate, if a list is returned

  gchar *plugname=g_strdup_printf ("%s%s%s/%s",prefs->prefix_dir,PLUGIN_DIR,PLUGIN_VID_PLAYBACK,name);
  guint cap;
  GModule *GM=g_module_open (plugname,0);
  gboolean OK=TRUE;
  gchar *msg;
  gchar **array;

  gchar *(*get_fps_list)(void);
  gboolean (*set_fps)(gdouble fps);



  if (GM==NULL) {
    gchar *msg=g_strdup_printf (_("\n\nFailed to open playback module %s\nError was %s\n"),plugname,g_module_error());
    set_pref ("vid_playback_plugin","none");
    do_error_dialog (msg);
    g_free (msg);
    g_free(plugname);
    return FALSE;
  }

  if (vid_playback_plugin!=NULL) {
    g_module_close (vid_playback_plugin->gmodule);
    if (vid_playback_plugin->fps_list!=NULL) g_free(vid_playback_plugin->fps_list);
    g_free (vid_playback_plugin);
    mainw->fixed_fps=-1.;
  }

  vid_playback_plugin=(_vid_playback_plugin *) g_malloc (sizeof(_vid_playback_plugin));
  if (g_module_symbol (GM,"version",(gpointer)&vid_playback_plugin->version)==FALSE) {
    OK=FALSE;
  }
  if (g_module_symbol (GM,"set_palette",(gpointer)&vid_playback_plugin->set_palette)==FALSE) {
    OK=FALSE;
  }
  if (g_module_symbol (GM,"set_keyfun",(gpointer)&vid_playback_plugin->set_keyfun)==FALSE) {
    OK=FALSE;
  }
  if (g_module_symbol (GM,"get_capabilities",(gpointer)&vid_playback_plugin->get_capabilities)==FALSE) {
    OK=FALSE;
  }
  if (g_module_symbol (GM,"init_screen",(gpointer)&vid_playback_plugin->init_screen)==FALSE) {
    OK=FALSE;
  }
  if (g_module_symbol (GM,"render_frame",(gpointer)&vid_playback_plugin->render_frame)==FALSE) {
    OK=FALSE;
  }
  if (g_module_symbol (GM,"free_frame",(gpointer)&vid_playback_plugin->free_frame)==FALSE) {
    OK=FALSE;
  }
  if (g_module_symbol (GM,"send_keycodes",(gpointer)&vid_playback_plugin->send_keycodes)==FALSE) {
    OK=FALSE;
  }
  if (g_module_symbol (GM,"exit_screen",(gpointer)&vid_playback_plugin->exit_screen)==FALSE) {
    OK=FALSE;
  }
  if (!OK) {
    g_free (vid_playback_plugin);
    vid_playback_plugin=NULL;
    g_module_close (GM);
    g_free(plugname);
    return FALSE;
  }

  // try different palettes until we get TRUE
  // TODO - use Weed palettes
  if (!(*vid_playback_plugin->set_palette)("UYVY")) {
    if (!(*vid_playback_plugin->set_palette)("RGB24")) {
      if (!(*vid_playback_plugin->set_palette)("RGBA32")) {
	if (!(*vid_playback_plugin->set_palette)("YUV420")) {
	  g_free (vid_playback_plugin);
	  vid_playback_plugin=NULL;
	  g_module_close (GM);
	  d_print (_("failed to agree a palette\n"));
	  return FALSE;
	}
	mainw->ext_palette=EXTP_YUV420;
      } else {
	mainw->ext_palette=EXTP_RGBA32;
      }
    }
    else {
      mainw->ext_palette=EXTP_RGB24;
    }
  }
  else {
    mainw->ext_palette=EXTP_UYVY;
  }
  cap=(*vid_playback_plugin->get_capabilities)();
  vid_playback_plugin->can_resize=cap&1;
  if (!vid_playback_plugin->can_resize&&mainw->ext_palette==EXTP_UYVY) {
    // YUV must resize
    if (!(*vid_playback_plugin->set_palette)("RGB24")) {
      g_free (vid_playback_plugin);
      vid_playback_plugin=NULL;
      g_module_close (GM);
      d_print (_("YUV palettes must resize\n"));
      return FALSE;
    }
    mainw->ext_palette=EXTP_RGB24;
    cap=(*vid_playback_plugin->get_capabilities)();
    vid_playback_plugin->can_resize=cap&1;
  }
  vid_playback_plugin->fwidth=vid_playback_plugin->fheight=-1;
  vid_playback_plugin->can_return=cap&2;
  if (!(vid_playback_plugin->local_display=!(cap&8))) {
    // no display
    vid_playback_plugin->fwidth=vid_playback_plugin->fheight=0;
  }

  if (cap&4) {
    // fixed fps
    if (g_module_symbol (GM,"get_fps_list",(gpointer)&get_fps_list)) {
      vid_playback_plugin->fps_list=g_strdup((*get_fps_list)());
      array=g_strsplit (vid_playback_plugin->fps_list,"|",-1);
      if (get_token_count (array[0],':')>1) {
	mainw->fixed_fps=get_ratio_fps(array[0]);
      }
      else mainw->fixed_fps=g_strtod(array[0],NULL);
      g_strfreev(array);

      // TODO - for no_display, prompt also for (fixed) screen size
      // usually we don't want to start a new stream when we switch clips
    }
    else mainw->fixed_fps=prefs->default_fps;
    if (g_module_symbol (GM,"set_fps",(gpointer)&set_fps)) {
      if (!((*set_fps) (mainw->fixed_fps))) {
	d_print (_("Unable to set framerate of plugin\n"));
      }
    }
  }
  else vid_playback_plugin->fps_list=NULL;

  (*vid_playback_plugin->set_keyfun)(&pl_key_function);

  g_snprintf (vid_playback_plugin->name,256,"%s",name);
  vid_playback_plugin->gmodule=GM;
  cached_key=cached_mod=0;
  msg=g_strdup_printf(_("*** Using %s plugin for fs playback, agreed to use palette type %d. ***\n"),name,mainw->ext_palette);
  d_print (msg);
  g_free (msg);
  g_free(plugname);
  return TRUE;
}


void vid_playback_plugin_exit (void) {
  // external plugin
  if (!(mainw->uyvy_frame==NULL)) {
    g_free(mainw->uyvy_frame);
    mainw->uyvy_frame=NULL;
  }
  if (!(mainw->yuv_frame==NULL)) {
    g_free(mainw->yuv_frame);
    mainw->yuv_frame=NULL;
  }
  (*vid_playback_plugin->exit_screen)(mainw->ptr_x,mainw->ptr_y);
  mainw->ext_playback=FALSE;
  mainw->ext_keyboard=FALSE;
}

gboolean 
check_encoder_restrictions (gboolean get_extension) {
  gchar **checks;
  gchar **array=NULL;
  gchar **array2;
  gint pieces,numtok;
  gboolean calc_aspect=FALSE;
  gchar aspect_buffer[512];
  gint hblock=1,vblock=1;
  int i,r;
  GList *ofmt_all=NULL;
  gboolean sizer=FALSE;

  // for auto resizing/resampling
  gdouble best_fps=0.;
  gint best_arate=0;
  gint width,owidth;
  gint height,oheight;

  gdouble best_fps_delta=0.;
  gint best_arate_delta=0;
  gboolean allow_aspect_override=FALSE;

  gint best_fps_num=0,best_fps_denom=0;
  gdouble fps;
  gint arate,achans,asampsize;

  if (rdet==NULL) {
    width=owidth=cfile->hsize;
    height=oheight=cfile->vsize;
    fps=cfile->fps;
  }
  else {
    width=owidth=rdet->width;
    height=oheight=rdet->height;
    fps=rdet->fps;
    rdet->suggestion_followed=FALSE;
  }
  // TODO - allow lists for size
  g_snprintf (prefs->encoder.of_restrict,5,"none");
  if (!((ofmt_all=plugin_request_by_line(PLUGIN_ENCODERS,prefs->encoder.name,"get_formats"))==NULL)) {
    // get any restrictions for the current format
    for (i=0;i<g_list_length(ofmt_all);i++) {
      if ((numtok=get_token_count (g_list_nth_data (ofmt_all,i),'|'))>2) {
	array=g_strsplit (g_list_nth_data (ofmt_all,i),"|",-1);
	if (!strcmp(array[0],prefs->encoder.of_name)) {
	  if (numtok>4) {
	    g_snprintf(prefs->encoder.of_def_ext,16,"%s",array[4]);
	  }
	  else {
	    memset (prefs->encoder.of_def_ext,0,1);
	  }
	  if (numtok>3) {
	    g_snprintf(prefs->encoder.of_restrict,128,"%s",array[3]);
	  }
	  else {
	    g_snprintf(prefs->encoder.of_restrict,128,"none");
	  }
	  prefs->encoder.of_allowed_acodecs=atoi (array[2]);
	  g_list_free (ofmt_all);
	  break;
	}
      }
    }
  }

  if (get_extension) return TRUE; // just wanted file extension
  
  if (rdet==NULL&&mainw->save_with_sound&&prefs->encoder.audio_codec!=AUDIO_CODEC_NONE) {
    if (!(prefs->encoder.of_allowed_acodecs&(1<<prefs->encoder.audio_codec))) {
      do_encoder_acodec_error();
      return FALSE;
    }
  }

  if (!strcmp(prefs->encoder.of_restrict,"none")) {
    return TRUE;
  }

  if (rdet==NULL) {
    arate=cfile->arate;
    achans=cfile->achans;
    asampsize=cfile->asampsize;
  }
  else {
    arate=rdet->arate;
    achans=rdet->achans;
    asampsize=rdet->asamps;
  }

  pieces=get_token_count (prefs->encoder.of_restrict,',');
  checks=g_strsplit(prefs->encoder.of_restrict,",",pieces);

  for (r=0;r<pieces;r++) {
    // check each restriction in turn

    if (!strncmp (checks[r],"fps=",4)) {
      gdouble allowed_fps;
      gint mbest_num=0,mbest_denom=0;
      gint numparts;
      gchar *fixer;

      best_fps_delta=1000000000.;
      array=g_strsplit(checks[r],"=",2);
      numtok=get_token_count (array[1],';');
      array2=g_strsplit(array[1],";",numtok);
      for (i=0;i<numtok;i++) {
	mbest_num=mbest_denom=0;
	if ((numparts=get_token_count (array2[i],':'))>1) {
	  gchar **array3=g_strsplit(array2[i],":",2);
	  mbest_num=atoi (array3[0]);
	  mbest_denom=atoi (array3[1]);
	  g_strfreev (array3);
	  if (mbest_denom==0) continue;
	  allowed_fps=(mbest_num*1.)/(mbest_denom*1.);
	}
	else allowed_fps=g_strtod (array2[i],NULL);

	// convert to 8dp
	fixer=g_strdup_printf ("%.8f %.8f",allowed_fps,fps);
	g_free (fixer);

	if (allowed_fps>=fps) {
	  if (allowed_fps-fps<best_fps_delta) {
	    best_fps_delta=allowed_fps-fps;
	    if (mbest_denom>0) {
	      best_fps_num=mbest_num;
	      best_fps_denom=mbest_denom;
	      best_fps=0.;
	      if (rdet==NULL) cfile->ratio_fps=TRUE;
	      else rdet->ratio_fps=TRUE;
	    }
	    else {
	      best_fps_num=best_fps_denom=0;
	      best_fps=allowed_fps;
	      if (rdet==NULL) cfile->ratio_fps=FALSE;
	      else rdet->ratio_fps=FALSE;
	    }
	  }
	}
	else if ((best_fps_denom==0&&allowed_fps>best_fps)||(best_fps_denom>0&&allowed_fps>(best_fps_num*1.)/(best_fps_denom*1.))) {
	  best_fps_delta=fps-allowed_fps;
	  if (mbest_denom>0) {
	    best_fps_num=mbest_num;
	    best_fps_denom=mbest_denom;
	    best_fps=0.;
	    if (rdet==NULL) cfile->ratio_fps=TRUE;
	    else rdet->ratio_fps=TRUE;
	  }
	  else {
	    best_fps=allowed_fps;
	    best_fps_num=best_fps_denom=0;
	    if (rdet==NULL) cfile->ratio_fps=FALSE;
	    else rdet->ratio_fps=FALSE;
	  }
	}
	if (best_fps_delta<(.0005*prefs->ignore_tiny_fps_diffs)) best_fps_delta=0.;
	if (best_fps_delta==0.) break;
      }
      g_strfreev(array2);
      continue;
    }


    if (!strncmp (checks[r],"size=",5)) {
      // TODO - allow list for size
      array=g_strsplit(checks[r],"=",2);
      array2=g_strsplit(array[1],"x",2);
      width=atoi (array2[0]);
      height=atoi (array2[1]);
      g_strfreev(array2);
      sizer=TRUE;
      continue;
    }

    if (!strncmp (checks[r],"arate=",6)&&(mainw->save_with_sound||rdet!=NULL)&&prefs->encoder.audio_codec!=AUDIO_CODEC_NONE&&(arate*achans*asampsize)) {
      // we only perform this test if we are encoding with audio
      // find next highest allowed rate from list,
      // if none are higher, use the highest
      gint allowed_arate;
      best_arate_delta=1000000000;
      
      array=g_strsplit(checks[r],"=",2);
      numtok=get_token_count (array[1],';');
      array2=g_strsplit(array[1],";",numtok);
      for (i=0;i<numtok;i++) {
	allowed_arate=atoi (array2[i]);
	if (allowed_arate>=arate) {
	  if (allowed_arate-arate<best_arate_delta) {
	    best_arate_delta=allowed_arate-arate;
	    best_arate=allowed_arate;
	  }
	}
	else if (allowed_arate>best_arate) best_arate=allowed_arate;
      }
      g_strfreev(array2);
      
      if (!capable->has_sox) {
	do_encoder_sox_error();
	g_strfreev(array);
	g_strfreev(checks);
	return FALSE;
      }
      continue;
    }

    if (!strncmp (checks[r],"hblock=",7)) {
      // width must be a multiple of this
      array=g_strsplit(checks[r],"=",2);
      hblock=atoi (array[1]);
      width=(gint)(width/hblock+.5)*hblock;
      continue;
    }
    
    if (!strncmp (checks[r],"vblock=",7)) {
      // height must be a multiple of this
      array=g_strsplit(checks[r],"=",2);
      vblock=atoi (array[1]);
      height=(gint)(height/vblock+.5)*vblock;
      continue;
    }
    
    if (!strncmp (checks[r],"aspect=",7)) {
      // we calculate the nearest smaller frame size using aspect, 
      // hblock and vblock
      calc_aspect=TRUE;
      array=g_strsplit(checks[r],"=",2);
      g_snprintf (aspect_buffer,512,"%s",array[1]);
      continue;
    }
  }
  
  /// end restrictions
  if (!(array==NULL)) {
    g_strfreev(array);
    array=NULL;
  }

  g_strfreev(checks);

  if (calc_aspect&&!sizer) {
    // we calculate this last, after getting hblock and vblock sizes
    gchar **array3;
    gdouble allowed_aspect;

    // aspect restricted encoders don't seem to like odd frame sizes
    if (hblock==1) hblock++;
    if (vblock==1) vblock++;

    numtok=get_token_count (aspect_buffer,';');
    array2=g_strsplit(aspect_buffer,";",numtok);
    
    height=width=0;

    for (i=0;i<numtok;i++) {
      array3=g_strsplit(array2[i],":",2);
      allowed_aspect=(gdouble)(atoi (array3[0]))/(gdouble)(atoi (array3[1]));
      g_strfreev(array3);
      minimise_aspect_delta (allowed_aspect,hblock,vblock,owidth,oheight,&width,&height);
    }
    g_strfreev(array2);

    if (((owidth*1.)/(hblock*1.))==(int)((owidth*1.)/(hblock*1.))&&((oheight*1.)/(vblock*1.))==(int)((oheight*1.)/(vblock*1.))) allow_aspect_override=TRUE;
  }

  // fps can't be altered if we have a multitrack event_list
  if (mainw->multitrack!=NULL&&mainw->multitrack->event_list!=NULL) best_fps_delta=0.;

  if (sizer) allow_aspect_override=FALSE;
  if (((width!=owidth||height!=oheight)&&width*height>0)||(best_fps_delta>0.)||(best_arate_delta>0&&best_arate>0)) {
    gboolean ofx1_bool=mainw->fx1_bool;
    mainw->fx1_bool=FALSE;
    if (!capable->has_convert&&rdet==NULL) {
      if (allow_aspect_override) {
	width=owidth;
	height=oheight;
      }
      else {
	do_error_dialog (_ ("Unable to resize, please install imageMagick\n"));
	return FALSE;
      }
    }

    if (rdet!=NULL) {
      rdet->arate=(gint)atoi (gtk_entry_get_text(GTK_ENTRY(resaudw->entry_arate)));
      rdet->achans=(gint)atoi (gtk_entry_get_text(GTK_ENTRY(resaudw->entry_achans)));
      rdet->asamps=(gint)atoi (gtk_entry_get_text(GTK_ENTRY(resaudw->entry_asamps)));
      if (width!=rdet->width||height!=rdet->height||best_fps_delta!=0.||best_arate!=rdet->arate) {
	if (rdet_suggest_values(width,height,best_fps,best_fps_num,best_fps_denom,best_arate,allow_aspect_override,(best_fps_delta==0.))) {
	  gchar *arate_string;
	  rdet->width=width;
	  rdet->height=height;
	  rdet->arate=best_arate;
	  if (best_fps_delta>0.) {
	    if (best_fps_denom>0) {
	      rdet->fps=(best_fps_num*1.)/(best_fps_denom*1.);
	    }
	    else rdet->fps=best_fps;
	    gtk_spin_button_set_value(GTK_SPIN_BUTTON(rdet->spinbutton_fps),rdet->fps);
	  }
	  gtk_spin_button_set_value(GTK_SPIN_BUTTON(rdet->spinbutton_width),rdet->width);
	  gtk_spin_button_set_value(GTK_SPIN_BUTTON(rdet->spinbutton_height),rdet->height);
	  arate_string=g_strdup_printf("%d",best_arate);
	  gtk_entry_set_text (GTK_ENTRY (resaudw->entry_arate),arate_string);
	  g_free(arate_string);
	  rdet->suggestion_followed=TRUE;
	  return TRUE;
	}
      }
      return FALSE;
    }

    if (do_encoder_restrict_dialog (width,height,best_fps,best_fps_num,best_fps_denom,best_arate,allow_aspect_override)) {
      if (!mainw->fx1_bool) {
	width=owidth;
	height=oheight;
      }
      if (!auto_resample_resize (width,height,best_fps,best_fps_num,best_fps_denom,best_arate)) {
	mainw->fx1_bool=ofx1_bool;
	return FALSE;
      }
    }
    else {
      mainw->fx1_bool=ofx1_bool;
      return FALSE;
    }
  }
  return TRUE;
}


gboolean check_rfx_for_lives (lives_rfx_t *rfx) {
  // check that an RFX is suitable for loading (cf. check_for_lives in effects-weed.c)
  if (rfx->num_in_channels==2&&rfx->props&RFX_PROPS_MAY_RESIZE) {
    d_print (g_strdup_printf (_ ("Failed to load %s, transitions may not resize.\n"),rfx->name));
    return FALSE;
  }
  return TRUE;
}


void render_fx_get_params (lives_rfx_t *rfx, const gchar *plugin_name, gshort status) {
  // create lives_param_t array from plugin supplied values
  GList *parameter_list;
  int param_idx,i;
  lives_param_t *cparam;
  gchar **param_array;
  gchar *line;
  gint len;

  switch (status) {
  case RFX_STATUS_BUILTIN:
    parameter_list=plugin_request_by_line (PLUGIN_RENDERED_EFFECTS_BUILTIN,plugin_name,"get_parameters");
    break;
  case RFX_STATUS_CUSTOM:
    parameter_list=plugin_request_by_line (PLUGIN_RENDERED_EFFECTS_CUSTOM,plugin_name,"get_parameters");
    break;
  default:
    parameter_list=plugin_request_by_line (PLUGIN_RENDERED_EFFECTS_TEST,plugin_name,"get_parameters");
    break;
  }

  if (parameter_list==NULL) {
    rfx->num_params=0;
    rfx->params=NULL;
    return;
  }

  pthread_mutex_lock(&mainw->gtk_mutex);
  rfx->num_params=g_list_length (parameter_list);
  rfx->params=g_malloc (rfx->num_params*sizeof(lives_param_t));
  
  for (param_idx=0;param_idx<rfx->num_params;param_idx++) {
    // TODO - error check
    line=g_list_nth_data(parameter_list,param_idx);

    len=get_token_count (line,(unsigned int)rfx->delim[0]);
    param_array=g_strsplit(line,rfx->delim,-1);
    g_free (line);
    cparam=&rfx->params[param_idx];
    cparam->name=g_strdup (param_array[0]);
    cparam->label=g_strdup (param_array[1]);
    cparam->desc=NULL;
    cparam->use_mnemonic=TRUE;
    cparam->interp_func=NULL;
    cparam->display_func=NULL;
    cparam->hidden=0;
    cparam->wrap=FALSE;
    cparam->transition=FALSE;
    cparam->step_size=1.;
    cparam->copy_to=-1;
    cparam->group=0;
    cparam->max=0.;
    cparam->changed=FALSE;

#ifdef DEBUG_RENDER_FX_P
    g_printerr("Got parameter %s\n",cparam->name);
#endif
    cparam->dp=0;
    cparam->list=NULL;
    
    if (!strncmp (param_array[2],"num",3)) {
      cparam->dp=atoi (param_array[2]+3);
      cparam->type=LIVES_PARAM_NUM;
    }
    else if (!strncmp (param_array[2],"bool",4)) {
      cparam->type=LIVES_PARAM_BOOL;
    }
    else if (!strncmp (param_array[2],"colRGB24",8)) {
      cparam->type=LIVES_PARAM_COLRGB24;
    }
    else if (!strncmp (param_array[2],"string",8)) {
      cparam->type=LIVES_PARAM_STRING;
    }
    else if (!strncmp (param_array[2],"string_list",8)) {
      cparam->type=LIVES_PARAM_STRING_LIST;
    }
    
    if (cparam->dp) {
      gdouble val=g_strtod (param_array[3],NULL);
      cparam->value=g_malloc(sizdbl);
      cparam->def=g_malloc(sizdbl);
      set_double_param(cparam->def,val);
      set_double_param(cparam->value,val);
      cparam->min=g_strtod (param_array[4],NULL);
      cparam->max=g_strtod (param_array[5],NULL);
      if (len>6) {
	cparam->step_size=g_strtod(param_array[6],NULL);
	if (cparam->step_size==0.) cparam->step_size=1./lives_10pow(cparam->dp);
	else if (cparam->step_size<0.) {
	  cparam->step_size=-cparam->step_size;
	  cparam->wrap=TRUE;
	}
      }
    }
    else if (cparam->type==LIVES_PARAM_COLRGB24) {
      gshort red=(gshort)atoi (param_array[3]);
      gshort green=(gshort)atoi (param_array[4]);
      gshort blue=(gshort)atoi (param_array[5]);
      cparam->value=g_malloc(sizeof(lives_colRGB24_t));
      cparam->def=g_malloc(sizeof(lives_colRGB24_t));
      set_colRGB24_param(cparam->def,red,green,blue);
      set_colRGB24_param(cparam->value,red,green,blue);
    }
    else if (cparam->type==LIVES_PARAM_STRING) {
      cparam->value=g_strdup(_ (param_array[3]));
      cparam->def=g_strdup(_ (param_array[3]));
      if (len>4) cparam->max=(gdouble)atoi (param_array[4]);
      if (cparam->max==0.||cparam->max>RFX_MAXSTRINGLEN) cparam->max=RFX_MAXSTRINGLEN;
    }
    else if (cparam->type==LIVES_PARAM_STRING_LIST) {
      cparam->value=g_malloc(sizint);
      cparam->def=g_malloc(sizint);
      *(int *)cparam->def=atoi (param_array[3]);
      if (len>3) {
	cparam->list=array_to_string_list (param_array,3,len);
      }
      else {
	set_int_param (cparam->def,0);
      }
      set_int_param (cparam->value,get_int_param (cparam->def));
    }
    else {
      // int or bool
      gint val=atoi (param_array[3]);
      cparam->value=g_malloc(sizint);
      cparam->def=g_malloc(sizint);
      set_int_param(cparam->def,val);
      set_int_param(cparam->value,val);
      if (cparam->type==LIVES_PARAM_BOOL) {
	cparam->min=0;
	cparam->max=1;
	if (len>4) cparam->group=atoi (param_array[4]);
      }
      else {
	cparam->min=(gdouble)atoi (param_array[4]);
	cparam->max=(gdouble)atoi (param_array[5]);
	if (len>6) {
	  cparam->step_size=(gdouble)atoi(param_array[6]);
	  if (cparam->step_size==0.) cparam->step_size=1.;
	  else if (cparam->step_size<0.) {
	    cparam->step_size=-cparam->step_size;
	    cparam->wrap=TRUE;
	  }
	}
      }
    }
    
    for (i=0;i<MAX_PARAM_WIDGETS;i++) {
      cparam->widgets[i]=NULL;
    }
    cparam->onchange=FALSE;
    g_strfreev (param_array);
  }
  g_list_free (parameter_list);
  pthread_mutex_unlock(&mainw->gtk_mutex);
}


GList *array_to_string_list (gchar **array, gint offset, gint len) {
  // build a GList from an array.
  int i;

  gchar *string;
  GList *slist=NULL;

  for (i=offset+1;i<len;i++) {
    string=subst (L2U8(array[i]),"\\n","\n");

    // omit a last empty string
    if (i<len-1||strlen (string)) {
      slist=g_list_append (slist, string);
    }
  }

  return slist;
}



void sort_rfx_array (lives_rfx_t *in, gint num) {
  // sort rfx array into UTF-8 order by menu entry
  int i;
  int start=1,min_val=0;
  gboolean used[num];
  gint sorted=1;
  gchar *min_string=NULL;
  lives_rfx_t *rfx;

  for (i=0;i<num;i++) {
    used[i]=FALSE;
  }

  rfx=mainw->rendered_fx=(lives_rfx_t *)g_malloc ((num+1)*sizeof(lives_rfx_t));

  rfx->name=g_strdup (in[0].name);
  rfx->menu_text=g_strdup (in[0].menu_text);
  rfx->action_desc=g_strdup (in[0].action_desc);
  rfx->props=in[0].props;
  rfx->num_params=0;
  rfx->min_frames=1;
  rfx->params=NULL;
  rfx->source=NULL;
  rfx->is_template=FALSE;
  rfx->extra=NULL;

  while (sorted<=num) {
    for (i=start;i<=num;i++) {
      if (!used[i-1]) {
	if (min_string==NULL) {
	  min_string=g_utf8_collate_key (in[i].menu_text,strlen (in[i].menu_text));
	  min_val=i;
	}
	else if (strcmp (min_string,g_utf8_collate_key(in[i].menu_text,strlen (in[i].menu_text)))==1) {
	  g_free (min_string);
	  min_string=g_utf8_collate_key (in[i].menu_text,strlen (in[i].menu_text));
	  min_val=i;
	}
      }
    }
    rfx_copy (&in[min_val],&mainw->rendered_fx[sorted++],FALSE);
    used[min_val-1]=TRUE;
    if (min_string!=NULL) g_free (min_string);
    min_string=NULL;
  }
  for (i=0;i<=num;i++) {
    rfx_free(&in[i]);
  }
}


void rfx_copy (lives_rfx_t *src, lives_rfx_t *dest, gboolean full) {
  // Warning, does not copy all fields (full will do that)
  dest->name=g_strdup (src->name);
  dest->menu_text=g_strdup (src->menu_text);
  dest->action_desc=g_strdup (src->action_desc);
  dest->min_frames=src->min_frames;
  dest->num_in_channels=src->num_in_channels;
  dest->status=src->status;
  dest->props=src->props;
  dest->is_template=src->is_template;
  memcpy(dest->delim,src->delim,2);
  if (!full) return;
}


void rfx_free(lives_rfx_t *rfx) {
  int i;

  g_free(rfx->name);
  g_free(rfx->menu_text);
  g_free(rfx->action_desc);
  for (i=0;i<rfx->num_params;i++) {
    if (rfx->params[i].type==LIVES_PARAM_UNDISPLAYABLE) continue;
    g_free(rfx->params[i].name);
    if (rfx->params[i].def!=NULL) g_free(rfx->params[i].def);
    if (rfx->params[i].value!=NULL) g_free(rfx->params[i].value);
    if (rfx->params[i].label!=NULL) g_free(rfx->params[i].label);
    if (rfx->params[i].desc!=NULL) g_free(rfx->params[i].desc);
    if (rfx->params[i].list!=NULL) g_list_free(rfx->params[i].list);
  }
  if (rfx->params!=NULL) {
    g_free(rfx->params);
  }
  if (rfx->extra!=NULL) {
    free(rfx->extra);
  }
  if (rfx->is_template) {
    int error;
    weed_plant_t *filter=weed_get_plantptr_value(rfx->source,"filter_class",&error);
    if (weed_plant_has_leaf(filter,"deinit_func")) {
      weed_deinit_f filter_deinit=weed_get_voidptr_value(filter,"deinit_func",&error);
      if (filter_deinit!=NULL) (*filter_deinit)(rfx->source);
    }
    weed_free_instance(rfx->source);
  }
}


void rfx_free_all (void) {
  int i;
  for (i=0;i<=mainw->num_rendered_effects_builtin+mainw->num_rendered_effects_custom+mainw->num_rendered_effects_test;i++) {
    rfx_free(&mainw->rendered_fx[i]);
  }
  g_free(mainw->rendered_fx);
}


void param_copy (lives_param_t *src, lives_param_t *dest, gboolean full) {
  // rfxbuilder.c uses this to copy params to a temporary copy and back again

  dest->name=g_strdup (src->name);
  dest->label=g_strdup (src->label);
  dest->group=src->group;
  dest->onchange=src->onchange;
  dest->type=src->type;
  dest->dp=src->dp;
  dest->min=src->min;
  dest->max=src->max;
  dest->step_size=src->step_size;
  dest->wrap=src->wrap;
  dest->list=NULL;

  switch (dest->type) {
  case LIVES_PARAM_BOOL:
    dest->dp=0;
  case LIVES_PARAM_NUM:
    if (!dest->dp) {
      dest->def=g_malloc (sizint);
      memcpy (dest->def,src->def,sizint);
    }
    else {
      dest->def=g_malloc (sizdbl);
      memcpy (dest->def,src->def,sizdbl);
    }
    break;
  case LIVES_PARAM_COLRGB24:
    dest->def=g_malloc (sizeof(lives_colRGB24_t));
    memcpy (dest->def,src->def,sizeof(lives_colRGB24_t));
    break;
  case LIVES_PARAM_STRING:
    dest->def=g_strdup (src->def);
    break;
  case LIVES_PARAM_STRING_LIST:
    dest->def=g_malloc (sizint);
    set_int_param (dest->def,get_int_param (src->def));
    if (src->list!=NULL) dest->list=g_list_copy (src->list);
    break;
  }
  if (!full) return;
  // TODO - copy value, copy widgets

}



void **store_rfx_params (lives_rfx_t *rfx) {
  int i;
  void **store=(void **)g_malloc (rfx->num_params*sizeof(void *));
  // store current values

  for (i=0;i<rfx->num_params;i++) {
    switch (rfx->params[i].type) {
    case LIVES_PARAM_BOOL:
    case LIVES_PARAM_NUM:
      if (rfx->params[i].dp) {
	store[i]=g_malloc (sizdbl);
	memcpy (store[i],rfx->params[i].value,sizdbl);
      }
      else {
	store[i]=g_malloc (sizint);
	memcpy (store[i],rfx->params[i].value,sizint);
      }
      break;
    case LIVES_PARAM_COLRGB24:
      store[i]=g_malloc (3*sizint);
      memcpy (store[i],rfx->params[i].value,3*sizint);
      break;
    case LIVES_PARAM_STRING:
      store[i]=g_strdup (rfx->params[i].value);
      break;
    case LIVES_PARAM_STRING_LIST:
      store[i]=g_malloc (sizint);
      memcpy (store[i],rfx->params[i].value,sizint);
      break;
    default:
      store[i]=NULL;
    }
  }
  return store;
}

void set_rfx_params_from_store (lives_rfx_t *rfx, void **param_store) {
    // check each parameter value against the stored value. If it has changed
    // then change the default. This gives a user controllable way of setting
    // defaults in the init trigger

  int i;

  lives_colRGB24_t rgbs,rgbv,rgbd;
  
  for (i=0;i<rfx->num_params;i++) {
    switch (rfx->params[i].type) {
    case LIVES_PARAM_BOOL:
    case LIVES_PARAM_NUM:
      if (rfx->params[i].dp) {
	if (get_double_param (param_store[i])!=get_double_param (rfx->params[i].value)) set_double_param (rfx->params[i].def,get_double_param (rfx->params[i].value));
      }
      else {
	if (get_int_param (param_store[i])!=get_int_param (rfx->params[i].value)) set_int_param (rfx->params[i].def,get_int_param (rfx->params[i].value));
      }
      break;
    case LIVES_PARAM_COLRGB24:
      get_colRGB24_param (param_store[i],&rgbs);
      get_colRGB24_param (rfx->params[i].value,&rgbv);
      get_colRGB24_param (rfx->params[i].def,&rgbd);
      if (rgbs.red!=rgbv.red) rgbd.red=rgbv.red;
      if (rgbs.green!=rgbv.green) rgbd.green=rgbv.green;
      if (rgbs.blue!=rgbv.blue) rgbd.blue=rgbv.blue;
      set_colRGB24_param (rfx->params[i].def,rgbd.red,rgbd.green,rgbd.blue);
      break;
    case LIVES_PARAM_STRING:
      if (strcmp (param_store[i],rfx->params[i].value)) {
	g_free (rfx->params[i].def);
	rfx->params[i].def=g_strdup (rfx->params[i].value);
      }
      break;
    case LIVES_PARAM_STRING_LIST:
      if (get_int_param (param_store[i])!=get_int_param (rfx->params[i].value)) set_int_param (rfx->params[i].def,get_int_param (rfx->params[i].value));
      break;
    }
  }
}

void rfx_params_store_free (lives_rfx_t *rfx, void **store) {
  int i;
  for (i=0;i<rfx->num_params;i++) {
    if (store[i]!=NULL) g_free (store[i]);
  }
}


gboolean get_bool_param(void *value) {
  gboolean ret;
  memcpy(&ret,value,sizint);
  return ret;
}

gint get_int_param(void *value) {
  gint ret;
  memcpy(&ret,value,sizint);
  return ret;
}

gdouble get_double_param(void *value) {
  gdouble ret;
  memcpy(&ret,value,sizdbl);
  return ret;
}

void get_colRGB24_param(void *value, lives_colRGB24_t *rgb) {
  memcpy(rgb,value,sizeof(lives_colRGB24_t));
}

void get_colRGBA32_param(void *value, lives_colRGBA32_t *rgba) {
  memcpy(rgba,value,sizshrt*4);
}

void set_bool_param(void *value, const gboolean _const) {
  set_int_param(value,!!_const);
}

void set_int_param(void *value, const gint _const) {
  memcpy(value,&_const,sizint);
}
void set_double_param(void *value, const gdouble _const) {
  memcpy(value,&_const,sizdbl);

}

void set_colRGB24_param(void *value, gshort red, gshort green, gshort blue) {
  lives_colRGB24_t *rgbp=(lives_colRGB24_t *)value;

  if (red<0) red=0;
  if (red>255) red=255;
  if (green<0) green=0;
  if (green>255) green=255;
  if (blue<0) blue=0;
  if (blue>255) blue=255;

  memcpy(&rgbp->red,&red,sizshrt);
  memcpy(&rgbp->green,&green,sizshrt);
  memcpy(&rgbp->blue,&blue,sizshrt);

}

void set_colRGBA32_param(void *value, gshort red, gshort green, gshort blue, gshort alpha) {
  lives_colRGBA32_t *rgbap=(lives_colRGBA32_t *)value;
  memcpy(&rgbap->red,&red,sizshrt);
  memcpy(&rgbap->green,&green,sizshrt);
  memcpy(&rgbap->blue,&blue,sizshrt);
  memcpy(&rgbap->alpha,&alpha,sizshrt);
}




///////////////////////////////////////////////////////////////



gint find_rfx_plugin_by_name (gchar *name, gshort status) {
  int i;
  for (i=1;i<mainw->num_rendered_effects_builtin+mainw->num_rendered_effects_custom+mainw->num_rendered_effects_test;i++) {
    if (mainw->rendered_fx[i].name!=NULL&&!strcmp (mainw->rendered_fx[i].name,name)&&mainw->rendered_fx[i].status==status) return (gint)i;
  }
  return -1;
}



lives_param_t *weed_params_to_rfx(gint npar, weed_plant_t *plant, gboolean show_reinits) {
  int i,j;
  lives_param_t *rpar=g_malloc(npar*sizeof(lives_param_t));
  int param_hint;
  char **list;
  GList *gtk_list=NULL;
  gchar *string;
  int error;
  int vali;
  double vald;
  weed_plant_t *gui=NULL;
  int listlen;
  int cspace,*cols=NULL,red_min=0,red_max=255,green_min=0,green_max=255,blue_min=0,blue_max=255,*maxi=NULL,*mini=NULL;
  double *colsd;
  double red_mind=0.,red_maxd=1.,green_mind=0.,green_maxd=1.,blue_mind=0.,blue_maxd=1.,*maxd=NULL,*mind=NULL;
  int flags=0;
  gboolean col_int;

  weed_plant_t *wtmpl;
  weed_plant_t **wpars=NULL,*wpar=NULL;

  wpars=weed_get_plantptr_array(plant,"in_parameters",&error);

  for (i=0;i<npar;i++) {
    wpar=wpars[i];
    wtmpl=weed_get_plantptr_value(wpar,"template",&error);

    if (weed_plant_has_leaf(wtmpl,"flags")) flags=weed_get_int_value(wtmpl,"flags",&error);
    else flags=0;

    rpar[i].flags=flags;

    if (weed_plant_has_leaf(wtmpl,"gui")) gui=weed_get_plantptr_value(wtmpl,"gui",&error);

    rpar[i].group=0;

    rpar[i].use_mnemonic=FALSE;
    rpar[i].interp_func=rpar[i].display_func=NULL;
    rpar[i].hidden=0;
    rpar[i].step_size=1.;
    rpar[i].copy_to=-1;
    rpar[i].transition=FALSE;
    rpar[i].wrap=FALSE;
    rpar[i].reinit=FALSE;

    if (flags&WEED_PARAMETER_VARIABLE_ELEMENTS&&!(flags&WEED_PARAMETER_ELEMENT_PER_CHANNEL)) {
      rpar[i].hidden|=HIDDEN_MULTI;
      rpar[i].multi=PVAL_MULTI_ANY;
    }
    else if (flags&WEED_PARAMETER_ELEMENT_PER_CHANNEL) {
      rpar[i].hidden|=HIDDEN_MULTI;
      rpar[i].multi=PVAL_MULTI_PER_CHANNEL;
    }
    else rpar[i].multi=PVAL_MULTI_NONE;

    param_hint=weed_get_int_value(wtmpl,"hint",&error);

    if (gui!=NULL) {
      if (weed_plant_has_leaf(gui,"copy_value_to")) {
	int copyto=weed_get_int_value(gui,"copy_value_to",&error);
	int param_hint2,flags2=0;
	weed_plant_t *wtmpl2;
	if (copyto==i||copyto<0) copyto=-1;
	if (copyto>-1) {
	  wtmpl2=weed_get_plantptr_value(wpars[copyto],"template",&error);
	  if (weed_plant_has_leaf(wtmpl2,"flags")) flags2=weed_get_int_value(wtmpl2,"flags",&error);
	  param_hint2=weed_get_int_value(wtmpl2,"hint",&error);
	  if (param_hint==param_hint2&&((flags2&WEED_PARAMETER_VARIABLE_ELEMENTS)||(flags&WEED_PARAMETER_ELEMENT_PER_CHANNEL&&flags2&WEED_PARAMETER_ELEMENT_PER_CHANNEL)||weed_leaf_num_elements(wtmpl,"default")==weed_leaf_num_elements(wtmpl2,"default"))) rpar[i].copy_to=copyto;
	}
      }
    }

    rpar[i].dp=0;
    rpar[i].min=0.;
    rpar[i].max=0.;
    rpar[i].list=NULL;

    if (flags&WEED_PARAMETER_REINIT_ON_VALUE_CHANGE) {
      rpar[i].reinit=TRUE;
      if (!show_reinits) rpar[i].hidden|=HIDDEN_NEEDS_REINIT;
    }

    ///////////////////////////////

    switch (param_hint) {
    case WEED_HINT_SWITCH:
      if (weed_plant_has_leaf(wtmpl,"default")&&weed_leaf_num_elements(wtmpl,"default")>1) {
	rpar[i].hidden|=HIDDEN_MULTI;
      }
      rpar[i].type=LIVES_PARAM_BOOL;
      rpar[i].value=g_malloc(sizint);
      rpar[i].def=g_malloc(sizint);
      if (weed_plant_has_leaf(wtmpl,"host_default")) vali=weed_get_boolean_value(wtmpl,"host_default",&error); 
      else if (weed_leaf_num_elements(wtmpl,"default")>0) vali=weed_get_boolean_value(wtmpl,"default",&error);
      else vali=weed_get_boolean_value(wtmpl,"new_default",&error);
      set_int_param(rpar[i].def,vali);
      vali=weed_get_boolean_value(wpar,"value",&error);
      set_int_param(rpar[i].value,vali);
      if (weed_plant_has_leaf(wtmpl,"group")) rpar[i].group=weed_get_int_value(wtmpl,"group",&error);
      break;
    case WEED_HINT_INTEGER:
      if (weed_plant_has_leaf(wtmpl,"default")&&weed_leaf_num_elements(wtmpl,"default")>1) {
	rpar[i].hidden|=HIDDEN_MULTI;
      }
      rpar[i].type=LIVES_PARAM_NUM;
      rpar[i].value=g_malloc(sizint);
      rpar[i].def=g_malloc(sizint);
      if (weed_plant_has_leaf(wtmpl,"host_default")) vali=weed_get_int_value(wtmpl,"host_default",&error); 
      else if (weed_leaf_num_elements(wtmpl,"default")>0) vali=weed_get_int_value(wtmpl,"default",&error);
      else vali=weed_get_int_value(wtmpl,"new_default",&error);
      set_int_param(rpar[i].def,vali);
      vali=weed_get_int_value(wpar,"value",&error);
      set_int_param(rpar[i].value,vali);
      rpar[i].min=(gdouble)weed_get_int_value(wtmpl,"min",&error);
      rpar[i].max=(gdouble)weed_get_int_value(wtmpl,"max",&error);
      if (weed_plant_has_leaf(wtmpl,"wrap")&&weed_get_boolean_value(wtmpl,"wrap",&error)==WEED_TRUE) rpar[i].wrap=TRUE;
      if (gui!=NULL) {
	if (weed_plant_has_leaf(gui,"choices")) {
	  listlen=weed_leaf_num_elements(gui,"choices");
	  list=weed_get_string_array(gui,"choices",&error);
	  for (j=0;j<listlen;j++) {
	    gtk_list=g_list_append(gtk_list,g_strdup(list[j]));
	    weed_free(list[j]);
	  }
	  weed_free(list);
	  rpar[i].list=g_list_copy(gtk_list);
	  g_list_free(gtk_list);
	  gtk_list=NULL;
	  rpar[i].type=LIVES_PARAM_STRING_LIST;
	}
	else if (weed_plant_has_leaf(gui,"step_size")) rpar[i].step_size=(gdouble)weed_get_int_value(gui,"step_size",&error);
	if (rpar[i].step_size==0.) rpar[i].step_size=1.;
      }
      break;
    case WEED_HINT_FLOAT:
      if (weed_plant_has_leaf(wtmpl,"default")&&weed_leaf_num_elements(wtmpl,"default")>1) {
	rpar[i].hidden|=HIDDEN_MULTI;
      }
      rpar[i].type=LIVES_PARAM_NUM;
      rpar[i].value=g_malloc(sizdbl);
      rpar[i].def=g_malloc(sizdbl);
      if (weed_plant_has_leaf(wtmpl,"host_default")) vald=weed_get_double_value(wtmpl,"host_default",&error); 
      else if (weed_leaf_num_elements(wtmpl,"default")>0) vald=weed_get_double_value(wtmpl,"default",&error);
      else vald=weed_get_double_value(wtmpl,"new_default",&error);
      set_double_param(rpar[i].def,vald);
      vald=weed_get_double_value(wpar,"value",&error);
      set_double_param(rpar[i].value,vald);
      rpar[i].min=weed_get_double_value(wtmpl,"min",&error);
      rpar[i].max=weed_get_double_value(wtmpl,"max",&error);
      if (weed_plant_has_leaf(wtmpl,"wrap")&&weed_get_boolean_value(wtmpl,"wrap",&error)==WEED_TRUE) rpar[i].wrap=TRUE;
      rpar[i].step_size=0.;
      if (gui!=NULL) {
	if (weed_plant_has_leaf(gui,"step_size")) rpar[i].step_size=weed_get_double_value(gui,"step_size",&error);
	if (weed_plant_has_leaf(gui,"decimals")) rpar[i].dp=weed_get_int_value(gui,"decimals",&error);
      }
      if (rpar[i].dp==0) rpar[i].dp=2;
      if (rpar[i].step_size==0.) rpar[i].step_size=1./lives_10pow(rpar[i].dp);
      break;
    case WEED_HINT_TEXT:
      if (weed_plant_has_leaf(wtmpl,"default")&&weed_leaf_num_elements(wtmpl,"default")>1) {
	rpar[i].hidden|=HIDDEN_MULTI;
      }
      rpar[i].type=LIVES_PARAM_STRING;
      if (weed_plant_has_leaf(wtmpl,"host_default")) string=weed_get_string_value(wtmpl,"host_default",&error); 
      else if (weed_leaf_num_elements(wtmpl,"default")>0) string=weed_get_string_value(wtmpl,"default",&error);
      else string=weed_get_string_value(wtmpl,"new_default",&error);
      rpar[i].def=g_strdup(string);
      weed_free(string);
      string=weed_get_string_value(wpar,"value",&error);
      rpar[i].value=g_strdup(string);
      weed_free(string);
      rpar[i].max=0.;
      if (gui!=NULL&&weed_plant_has_leaf(gui,"maxchars")) {
	rpar[i].max=(gdouble)weed_get_int_value(gui,"maxchars",&error);
	if (rpar[i].max<0.) rpar[i].max=0.;
      }
      break;
    case WEED_HINT_COLOR:
      cspace=weed_get_int_value(wtmpl,"colorspace",&error);
      switch (cspace) {
      case WEED_COLORSPACE_RGB:
	if (weed_leaf_num_elements(wtmpl,"default")>3) {
	  rpar[i].hidden|=HIDDEN_MULTI;
	}
	rpar[i].type=LIVES_PARAM_COLRGB24;
	rpar[i].value=g_malloc(3*sizint);
	rpar[i].def=g_malloc(3*sizint);

	if (weed_leaf_seed_type(wtmpl,"default")==WEED_SEED_INT) {
	  if (weed_plant_has_leaf(wtmpl,"host_default")) cols=weed_get_int_array(wtmpl,"host_default",&error); 
	  else if (weed_leaf_num_elements(wtmpl,"default")>0) cols=weed_get_int_array(wtmpl,"default",&error);
	  else cols=weed_get_int_array(wtmpl,"new_default",&error);
	  if (weed_leaf_num_elements(wtmpl,"max")==1) {
	    red_max=green_max=blue_max=weed_get_int_value(wtmpl,"max",&error);
	  }
	  else {
	    maxi=weed_get_int_array(wtmpl,"max",&error);
	    red_max=maxi[0];
	    green_max=maxi[1];
	    blue_max=maxi[2];
	  }
	  if (weed_leaf_num_elements(wtmpl,"min")==1) {
	    red_min=green_min=blue_min=weed_get_int_value(wtmpl,"min",&error);
	  }
	  else {
	    mini=weed_get_int_array(wtmpl,"min",&error);
	    red_min=mini[0];
	    green_min=mini[1];
	    blue_min=mini[2];
	  }
	  if (cols[0]<red_min) cols[0]=red_min;
	  if (cols[1]<green_min) cols[1]=green_min;
	  if (cols[2]<blue_min) cols[2]=blue_min;
	  if (cols[0]>red_max) cols[0]=red_max;
	  if (cols[1]>green_max) cols[1]=green_max;
	  if (cols[2]>blue_max) cols[2]=blue_max;
	  cols[0]=(cols[0]-red_min)/(red_max-red_min)*255;
	  cols[1]=(cols[1]-green_min)/(green_max-green_min)*255;
	  cols[2]=(cols[2]-blue_min)/(blue_max-blue_min)*255;
	  col_int=TRUE;
	}
	else {
	  if (weed_plant_has_leaf(wtmpl,"host_default")) colsd=weed_get_double_array(wtmpl,"host_default",&error); 
	  else if (weed_leaf_num_elements(wtmpl,"default")>0) colsd=weed_get_double_array(wtmpl,"default",&error);
	  else colsd=weed_get_double_array(wtmpl,"default",&error);
	  if (weed_leaf_num_elements(wtmpl,"max")==1) {
	    red_maxd=green_maxd=blue_maxd=weed_get_double_value(wtmpl,"max",&error);
	  }
	  else {
	    maxd=weed_get_double_array(wtmpl,"max",&error);
	    red_maxd=maxd[0];
	    green_maxd=maxd[1];
	    blue_maxd=maxd[2];
	  }
	  if (weed_leaf_num_elements(wtmpl,"min")==1) {
	    red_mind=green_mind=blue_mind=weed_get_double_value(wtmpl,"min",&error);
	  }
	  else {
	    mind=weed_get_double_array(wtmpl,"min",&error);
	    red_mind=mind[0];
	    green_mind=mind[1];
	    blue_mind=mind[2];
	  }
	  if (colsd[0]<red_mind) colsd[0]=red_mind;
	  if (colsd[1]<green_mind) colsd[1]=green_mind;
	  if (colsd[2]<blue_mind) colsd[2]=blue_mind;
	  if (colsd[0]>red_maxd) colsd[0]=red_maxd;
	  if (colsd[1]>green_maxd) colsd[1]=green_maxd;
	  if (colsd[2]>blue_maxd) colsd[2]=blue_maxd;
	  cols=weed_malloc(3*sizshrt);
	  cols[0]=(colsd[0]-red_mind)/(red_maxd-red_mind)*255.+.5;
	  cols[1]=(colsd[1]-green_mind)/(green_maxd-green_mind)*255.+.5;
	  cols[2]=(colsd[2]-blue_mind)/(blue_maxd-blue_mind)*255.+.5;
	  col_int=FALSE;
	}
	set_colRGB24_param(rpar[i].def,cols[0],cols[1],cols[2]);
	if (col_int) {
	  weed_free(cols);
	  cols=weed_get_int_array(wpar,"value",&error);
	  if (cols[0]<red_min) cols[0]=red_min;
	  if (cols[1]<green_min) cols[1]=green_min;
	  if (cols[2]<blue_min) cols[2]=blue_min;
	  if (cols[0]>red_max) cols[0]=red_max;
	  if (cols[1]>green_max) cols[1]=green_max;
	  if (cols[2]>blue_max) cols[2]=blue_max;
	  cols[0]=(cols[0]-red_min)/(red_max-red_min)*255;
	  cols[1]=(cols[1]-green_min)/(green_max-green_min)*255;
	  cols[2]=(cols[2]-blue_min)/(blue_max-blue_min)*255;
	}
	else {
	  colsd=weed_get_double_array(wpar,"value",&error);
	  if (colsd[0]<red_mind) colsd[0]=red_mind;
	  if (colsd[1]<green_mind) colsd[1]=green_mind;
	  if (colsd[2]<blue_mind) colsd[2]=blue_mind;
	  if (colsd[0]>red_maxd) colsd[0]=red_maxd;
	  if (colsd[1]>green_maxd) colsd[1]=green_maxd;
	  if (colsd[2]>blue_maxd) colsd[2]=blue_maxd;
	  cols[0]=(colsd[0]-red_mind)/(red_maxd-red_mind)*255.+.5;
	  cols[1]=(colsd[1]-green_mind)/(green_maxd-green_mind)*255.+.5;
	  cols[2]=(colsd[2]-blue_mind)/(blue_maxd-blue_mind)*255.+.5;
	}
	set_colRGB24_param(rpar[i].value,(gshort)cols[0],(gshort)cols[1],(gshort)cols[2]);
	weed_free(cols);
	if (maxi!=NULL) weed_free(maxi);
	if (mini!=NULL) weed_free(mini);
	if (maxd!=NULL) weed_free(maxd);
	if (mind!=NULL) weed_free(mind);
	break;
      }
      break;

    default:
      rpar[i].type=LIVES_PARAM_UNKNOWN; // TODO - try to get default
    }

    string=weed_get_string_value(wtmpl,"name",&error);
    rpar[i].name=g_strdup(string);
    rpar[i].label=g_strdup(string);
    weed_free(string);

    if (weed_plant_has_leaf(wtmpl,"description")) {
      string=weed_get_string_value(wtmpl,"description",&error);
      rpar[i].desc=g_strdup(string);
      weed_free(string);
    }
    else rpar[i].desc=NULL;

    // gui part /////////////////////

    if (gui!=NULL) {
      if (weed_plant_has_leaf(gui,"label")) {
	string=weed_get_string_value(gui,"label",&error);
	rpar[i].label=g_strdup(string);
	weed_free(string);
      }
      if (weed_plant_has_leaf(gui,"use_mnemonic")) rpar[i].use_mnemonic=weed_get_boolean_value(gui,"use_mnemonic",&error);
      if (weed_plant_has_leaf(gui,"hidden")) rpar[i].hidden|=((weed_get_boolean_value(gui,"hidden",&error)==WEED_TRUE)*HIDDEN_GUI);
      if (weed_plant_has_leaf(gui,"display_func")) rpar[i].display_func=weed_get_voidptr_value(gui,"display_func",&error);
      if (weed_plant_has_leaf(gui,"interp_func")) rpar[i].interp_func=weed_get_voidptr_value(gui,"interp_func",&error);
    }

    for (j=0;j<MAX_PARAM_WIDGETS;j++) {
      rpar[i].widgets[j]=NULL;
    }
    rpar[i].onchange=FALSE;
  }

  for (i=0;i<npar;i++) {
    if (rpar[i].copy_to!=-1) weed_leaf_copy(weed_inst_in_param(plant,i,FALSE),"value",weed_inst_in_param(plant,rpar[i].copy_to,FALSE),"value");
  }

  weed_free(wpars);

  return rpar;
}



lives_rfx_t *weed_to_rfx (weed_plant_t *plant, gboolean show_reinits) {
  // return an RFX for a weed effect
  int error;
  weed_plant_t *filter;

  gchar *string;
  lives_rfx_t *rfx=g_malloc(sizeof(lives_rfx_t));
  rfx->is_template=FALSE;
  if (weed_get_int_value(plant,"type",&error)==WEED_PLANT_FILTER_INSTANCE) filter=weed_get_plantptr_value(plant,"filter_class",&error);
  else {
    filter=plant;
    plant=weed_instance_from_filter(filter);
    if (weed_plant_has_leaf(filter,"init_func")) {
      weed_init_f filter_init=weed_get_voidptr_value(filter,"init_func",&error);
      update_host_info(plant);
      if (filter_init!=NULL) (*filter_init)(plant);
    }
    rfx->is_template=TRUE;
  }
  string=weed_get_string_value(filter,"name",&error);
  rfx->name=g_strdup(string);
  rfx->menu_text=g_strdup(string);
  weed_free(string);
  rfx->action_desc=g_strdup("no action");
  rfx->min_frames=0;
  rfx->num_in_channels=enabled_in_channels(plant,FALSE);
  rfx->status=RFX_STATUS_WEED;
  rfx->props=0;
  rfx->menuitem=NULL;
  if (!weed_plant_has_leaf(filter,"in_parameter_templates")||weed_get_plantptr_value(filter,"in_parameter_templates",&error)==NULL) rfx->num_params=0;
  else rfx->num_params=weed_leaf_num_elements(filter,"in_parameter_templates");
  if (rfx->num_params>0) rfx->params=weed_params_to_rfx(rfx->num_params,plant,show_reinits);
  else rfx->params=NULL;
  rfx->source=(void *)plant;
  rfx->extra=NULL;
  return rfx;
}



GList *get_external_window_hints(lives_rfx_t *rfx) {
  GList *hints=NULL;

  if (rfx->status==RFX_STATUS_WEED) {
    int i,error;
    int num_hints;
    weed_plant_t *gui;
    weed_plant_t *inst=rfx->source;
    weed_plant_t *filter=weed_get_plantptr_value(inst,"filter_class",&error);
    char *string,**rfx_strings,*delim;
   
    if (!weed_plant_has_leaf(filter,"gui")) return NULL;
    gui=weed_get_plantptr_value(filter,"gui",&error);

    string=weed_get_string_value(gui,"layout_scheme",&error);
    if (strcmp(string,"RFX")) {
      weed_free(string);
      return NULL;
    }
    weed_free(string);

    if (!weed_plant_has_leaf(gui,"rfx_delim")) return NULL;
    delim=weed_get_string_value(gui,"rfx_delim",&error);
    g_snprintf(rfx->delim,2,"%s",delim);
    weed_free(delim);

    if (!weed_plant_has_leaf(gui,"rfx_strings")) return NULL;

    num_hints=weed_leaf_num_elements(gui,"rfx_strings");

    if (num_hints==0) return NULL;
    rfx_strings=weed_get_string_array(gui,"rfx_strings",&error);

    for (i=0;i<num_hints;i++) {
      hints=g_list_append(hints,g_strdup(rfx_strings[i]));
      weed_free(rfx_strings[i]);
    }
    weed_free(rfx_strings);
  }
  return hints;
}
