/****************************************************************************
** $Id: torclient.cpp,v 1.55 2006/10/21 13:46:08 hoganrobert Exp $
**
** Copyright (C) 1992-2000 Trolltech AS.  All rights reserved.
**
** This file is part of an example program for Qt.  This example
** program may be used, distributed and modified without limitation.
**
*****************************************************************************/

#include <qsocket.h>
#include <qtextstream.h>
#include <qstringlist.h>
#include <qregexp.h>
#include "torclient.h"
#include "tork.h"
#include "torkconfig.h"
#include "dndlistview.h"
#include <qtimer.h>
#include <klocale.h>
#include <assert.h>
#include <qfile.h>

#include "crypto.h"

#ifdef HAVE_GEOIP_H
# include <GeoIP.h>
#else
# include "GeoIP-1.4.0/libGeoIP/GeoIP.h"
#endif

Client::Client( const QString &host, Q_UINT16 port ) 
{
        // create the socket and connect various of its signals
        socket = new QSocket( this );
        connect( socket, SIGNAL(connected()),
                SLOT(socketConnected()) );
        connect( socket, SIGNAL(connectionClosed()),
                SLOT(socketConnectionClosed()) );
        connect( socket, SIGNAL(readyRead()),
                SLOT(socketReadyRead()) );
        connect( socket, SIGNAL(error(int)),
                SLOT(socketError(int)) );

        // connect to the server
//         infoText->append( tr("Trying to connect to the server\n") );
        socket->connectToHost( host, port );
        m_expectingCircuitInfo= false;
        m_expectingStreamInfo= false;
        m_expectingOrconnInfo= false;
        m_expectingGuardsInfo= false;
        m_expectingDirStatus= false;
        m_expectingServerInfo= false;
        m_controllerWorking= false;
        authorityread = false;
        m_firstloadcomplete = false;
        QTimer *timer = new QTimer( this );
        connect( timer, SIGNAL( timeout() ), this, SLOT( slotCheckTorNet() ) );
        timer->start( 180000, FALSE );

}


void Client::fetchServerInfo( const QString & server)
{

    QString fp = getFPFromFPDigest(server);
    fp.replace("$","");

    kdDebug() << "fp" << fp << endl;
    sendToServer("GETINFO dir/server/fp/" + fp);
}

void Client::fetchServerInfoByNick( const QString & server)
{

    
    QString fp = getFPFromNickName(server);
    fp.replace("$","");

    kdDebug() << "fp" << fp << endl;
    sendToServer("GETINFO dir/server/fp/" + fp);
}

void Client::slotCheckTorNet()
{

    sendToServer("GETINFO dir/status/all");
}

void Client::createService(const QString &dir, const QString &port)
{
    sendToServer(QString("setconf hiddenservicedir=\"%1\" hiddenserviceport=\"%2\"").arg(dir).arg(port));
}

void Client::authenticate()
{

    if (TorkConfig::cookieAuthentication()){
        if (!readCookie())
            emit displayError("Sorry!", "ccookie error");
    }else if (!TorkConfig::hashedControlPassword().isEmpty())
        sendToServer(QString("AUTHENTICATE \"%1\"").arg(TorkConfig::hashedControlPassword()));
    else
        sendToServer("AUTHENTICATE");
    bandwidth();

}


bool Client::readCookie()
{

    QString hex;
    char hx[2];

    QFile inf(QString("%1/.tor/control_auth_cookie").arg(getenv("HOME")));
    if ( inf.open(IO_ReadOnly) ) {
        QByteArray array = inf.readAll();
        inf.close();
        if (array.size() != 32)
            return false;
        for ( unsigned int i = 0; i < array.size(); i++ ) {
            sprintf(hx,"%02x",array[i]);
            hex += QString(hx).right(2);
        }
        kdDebug() << "Cookie: " << hex << endl;
    
        sendToServer(QString("AUTHENTICATE %1").arg(hex));
    
        return true;

    }

    return false; 

}

void Client::readRouters()
{

    QFile inf(QString("%1/.tor/cached-status/7EA6EAD6FD83083C538F44038BBFA077587DD755").arg(getenv("HOME")));
    if ( inf.open(IO_ReadOnly) ) {
        QTextStream stream( &inf );
        QString line;
        while ( !stream.atEnd() ) {
            line = stream.readLine(); // line of text excluding '\n'
//            kdDebug() << line << endl;
            parseDirStatus(line);
        }
        inf.close();
    }

 
}

void Client::newIdentity()
{
    
    sendToServer("signal newnym");

}

void Client::bandwidth()
{
    sendToServer("usefeature verbose_names");
    sendToServer("GETINFO dir/status/all");
/*    sendToServer("GETINFO entry-guards");*/
    sendToServer("GETINFO circuit-status");
    sendToServer("GETINFO stream-status");
    sendToServer("GETINFO orconn-status");
    sendToServer("SETEVENTS CIRC  STREAM  ORCONN  NOTICE  WARN  ERR  NEWDESC  ADDRMAP AUTHDIR_NEWDESCS BW");

}

void Client::socketReadyRead()
{
    QString line;
    // read from the server
    while ( socket->canReadLine() ) {
        
        line = (socket->readLine()).stripWhiteSpace();
        //kdDebug() << line << endl;
        if (line.contains("250 OK"))
            m_controllerWorking = true;

        if (m_expectingCircuitInfo){
            if (line != "."){
                parseEvent("CIRC",line);
            }else
                m_expectingCircuitInfo = false;
        }

        if (m_expectingOrconnInfo){
            if (line != "."){
                parseEvent("ORCONN",line);
            }else
                m_expectingOrconnInfo = false;
        }

        if (m_expectingStreamInfo){
            if (line != "."){
                parseEvent("STREAM",line);
            }else
                m_expectingStreamInfo = false;
        }

        if (m_expectingServerInfo){
            if (line != "."){
                parseEvent("SERVER",line);
            }else
                m_expectingServerInfo = false;
        }

        if (m_expectingGuardsInfo){
            if (line != "."){
                parseEvent("GUARDS",line);
                emit whatImDoing(i18n("Ready for use."));
            }else{
                m_expectingGuardsInfo = false;
            }

        }

        if (m_expectingDirStatus){
            if (line.contains("SIGNATURE"))
                authorityread = true;
            if (line != "."){
                if (!authorityread)
                    parseEvent("DIRSTATUS",line);
            }else{
                sendToServer("GETINFO entry-guards");
                emit shouldIApplySettings();
                m_firstloadcomplete = true;
                m_expectingDirStatus = false;
                authorityread = false;
            }
        }


        if (line.contains("250+circuit-status="))
            m_expectingCircuitInfo= true;
        if (line.contains("250+orconn-status="))
            m_expectingOrconnInfo= true;
        if (line.contains("250+stream-status="))
            m_expectingStreamInfo= true;
        if (line.contains("250+entry-guards="))
            m_expectingGuardsInfo= true;
        if (line.contains("250+dir/server/fp/"))
            m_expectingServerInfo= true;
        if (line.contains("250+dir/status/all=")){
            m_expectingDirStatus= true;
            emit whatImDoing("Inspecting the Tor network..");
        }


//         if (line.contains("250-network-status=")){
//             kdDebug() << "netwrok statsu" << endl;
//             line.replace("250-network-status=","");
//             storeServers(line);
//         }

        QString code = line.section(" ",0,0);

        //kdDebug() << code << endl;
        if (code == "650"){
            QString eventType = line.section(" ",1,1);
            QString eventInfo = line.section(" ",2);
            if (eventInfo.contains("circuit_testing_failed"))
                emit serverError();

            parseEvent(eventType,eventInfo);
        }

        if (code == "552"){
            QString eventInfo = line.section(" ",1);
            emit displayError("Sorry!", eventInfo);
        }

    }
}

void Client::parseEvent(const QString &type, const QString &info)
{

    //kdDebug() << type << endl;
    if (type == "STREAM")
        parseStream(info);
    else if (type == "ORCONN")
        parseORConn(info);
    else if (type == "CIRC")
        parseCircuit(info);
    else if (type == "GUARDS")
        parseGuards(info);
    else if (type == "SERVER")
        parseServer(info);
    else if (type == "DIRSTATUS")
        parseDirStatus(info);
    else if (type == "BW")
        parseBW(info);
    else if ((type == "WARN") || (type == "NOTICE") || (type == "ERR"))
        parseInfo(type,info);
}

void Client::parseBW(const QString &info)
{

    QString in = info.section(" ",0,0);
    QString out = info.section(" ",1,1);

    //kdDebug() << "streamID " << streamID << endl;
    emit bwUpdate(in, out);

}


void Client::parseStream(const QString &info)
{
    QString streamID = info.section(" ",0,0);
    QString status = info.section(" ",1,1);
    QString circID = info.section(" ",2,2);
    QString Target = info.section(" ",3,3);

    //kdDebug() << "streamID " << streamID << endl;
    emit streamStatusUpdate(streamID, status,circID, Target);

}

void Client::parseServer(const QString &info)
{
    //kdDebug() << m_statustip << endl;

    if (info.left(7) == "router "){
        GeoIP * gi;
        QString ip = info.section(" ",2,2);
        gi = GeoIP_new(GEOIP_STANDARD);
        int country_id = 0;
        country_id = GeoIP_id_by_name(gi, ip);
        QString cc = GeoIP_country_name[country_id];
        GeoIP_delete(gi);


        m_statustip = i18n("<b>Name:</b> %1<br>").arg(info.section(" ",1,1));
        m_statustip.append(i18n("<b>IP:</b> %1 <b>Port:</b> %2<br>").arg(ip).arg(info.section(" ",3,3)));
        m_statustip.append(i18n("<b>Country:</b> %1 <br>").arg(cc));
    }else if (info.left(8) == "platform"){
        m_statustip.append(i18n("<b>Version:</b> %1 <b>OS:</b> %2<br>").arg(info.section(" ",1,2)).arg(info.section(" ",4)));
    }else if (info.left(9) == "published"){
        m_statustip.append(i18n("<b>Published:</b> %1 <br>").arg(info.section(" ",1)));
    }else if (info.left(6) == "uptime"){
        kdDebug() << m_statustip << endl;
        //from the clever ktorrent
        KLocale* loc = KGlobal::locale();
        QTime t;
        int nsecs = info.section(" ",1).toInt();
        int ndays = (nsecs) / 86400;
        t = t.addSecs(nsecs % 86400);
        QString s = loc->formatTime(t,true,true);
        if (ndays > 0)
            s = i18n("1 day ","%n days ",ndays) + s;

        m_statustip.append(i18n("<b>Up Time:</b> %1 minutes<br>").arg(s));
        emit displayServer("Server Info", m_statustip);
    }


    //kdDebug() << "streamID " << streamID << endl;

}




void Client::parseGuards(const QString &info)
{
    kdDebug() << "guardinfo " << info << endl;

    QString fp_identity = info.section(" ",0,0);
    QString status = info.section(" ",1,1);
    QRegExp rx("(\\$[A-Z0-9]{40})");
    rx.search(fp_identity);
    QString server = getNickNameFromFP(rx.cap(0));

    entryGuards[fp_identity] = status;
    kdDebug() << "server " << server << endl;

    if (!server.isEmpty()){
        emit ORStatusUpdate(server, status);
    }
}

void Client::parseCircuit(const QString &info)
{
    if (info.contains("FAILED"))
        emit displayError("Circuit Failed.", "Circuit: " + info.section(" ",2,2));

    QString circuitID = info.section(" ",0,0).stripWhiteSpace();
    QString status = info.section(" ",1,1).stripWhiteSpace();
    QString path = info.section(" ",2,2).stripWhiteSpace();
    path.replace(QRegExp("(\\$[A-Z0-9]{40})(~|=)"),"");

    //kdDebug() << "circuitID " << circuitID << endl;
    emit circuitStatusUpdate(circuitID, status, path);

    //updateCandidateServers(path);
}

void Client::parseORConn(const QString &info)
{
    QString serverID = info.section(" ",0,0);
    QString status = info.section(" ",1,1);

    if (serverID.startsWith("$")){
        QString server = getNickNameFromFP(serverID);
        if (!server.isEmpty())
            serverID = server;
    }

    if (!status.contains("NEW")){
        serverID.replace(QRegExp("^[A-Z0-9$=~]{42}"),"");
        emit ORStatusUpdate(serverID, status);
    }
}

void Client::parseInfo(const QString &type,const QString &info)
{

    kdDebug() << info << endl;
    QString message = info;
    message.replace(QRegExp("^[a-zA-Z0-9_]+\\(\\):"),"");
//     QString summary = info.section(":",0,0);
//     QString data = info.section(":",1);

    emit infoUpdate(type, message, QString());

}

void Client::updateCandidateServers(const QString &path)
{

   	QStringList servers = QStringList::split(",", path);
    //kdDebug() << servers << endl;
    QStringList existingServers = TorkConfig::serversHistory();
	for ( QStringList::Iterator it = servers.begin(); it != servers.end(); ++it )
	{
		if ((*it).isEmpty())
			continue;
        if (existingServers.find(*it) == existingServers.end())
            existingServers.append(*it);
	}
    TorkConfig::setServersHistory(existingServers);

    //kdDebug() << existingServers << endl;
    TorkConfig::writeConfig();
}

void Client::attemptAttach(const QString &circid, const QString &streamid)
{

    kdDebug() << "attempting attach" << endl;
    QStringList streams = QStringList::split( " ", streamid);
    for ( QStringList::Iterator it = streams.begin(); it != streams.end(); ++it )
	{
		if ((*it).isEmpty())
			continue;
        sendToServer(QString("ATTACHSTREAM %1 %2").arg(*it).arg(circid));
    }

}

void Client::attemptExtendCircuit(const QString &circid, const QString &serverlist, bool usefp)
{

    kdDebug() << "attempting extend" << endl;

    QStringList servers = QStringList::split( " ", serverlist);
    QStringList circuitlist;
    for ( QStringList::Iterator it = servers.begin(); it != servers.end(); ++it )
	{
		if ((*it).isEmpty())
			continue;
        if (usefp)
            circuitlist.append(getFPFromFPDigest((*it)));
        else
            circuitlist.append((*it));
    }

    QString circuit = circuitlist.join(",");
    sendToServer(QString("EXTENDCIRCUIT %1 %2").arg(circid).arg(circuit));
}

void Client::attemptCreateCircuit(const QString &serverlist, bool usefp)
{

    QStringList servers = QStringList::split( " ", serverlist);
    QStringList circuitlist;
    for ( QStringList::Iterator it = servers.begin(); it != servers.end(); ++it )
	{
		if ((*it).isEmpty())
			continue;
        if (usefp)
            circuitlist.append(getFPFromFPDigest((*it)));
        else
            circuitlist.append((*it));
    }

    QString circuit = circuitlist.join(",");
    sendToServer(QString("EXTENDCIRCUIT 0 %1").arg(circuit));
}

void Client::attemptCloseStream(const QString &streamid)
{

    kdDebug() << "attempting close" << endl;
    sendToServer(QString("CLOSESTREAM %1 1").arg(streamid));
}

void Client::attemptAttachStreams( bool attachStreams)
{

    kdDebug() << "setting __LeaveStreamsUnattached" << endl;
    sendToServer(QString("SETCONF __LeaveStreamsUnattached=%1").arg(attachStreams));
}

void Client::attemptCloseCircuit(const QString &circuitid)
{

    kdDebug() << "attempting close" << endl;
    sendToServer(QString("CLOSECIRCUIT %1").arg(circuitid));
}


void Client::updatePrevConfig(PrevConfig::PrevConfigList prevlist)
{

    m_previtems = prevlist;
}


void Client::applySettingsToRunningTor()
{

    //FIXME: use function pointers and a list to do this


    switch (TorkConfig::quickConfigure()) {
            case 0 : //Tor client and server with default settings
                return;
            case 1 : //Tor client with default settings
                return;
            case 2 : //Tor server with default settings
                return;
            case 3 : //Use custom settings
                break;
    }

    KConfigSkeletonItem::List items = TorkConfig::self()->items();
    KConfigSkeletonItem::List::ConstIterator it;
    
    for( it = items.begin(); it != items.end(); ++it ) {
        if (elementShouldBeUsed((*it))){
            if (noSpecialProcessing((*it))){
                PrevConfig::PrevConfigList::iterator mit;
                QVariant oldvalue;
                for( mit = m_previtems.begin(); mit != m_previtems.end(); ++mit ) {
                    if ((*mit).name() == (*it)->name()){
                        oldvalue = (*mit).property();
                        continue;
                    }
                }


                if ( (*it)->property().type() == QVariant::String ) {
                    if ((oldvalue !=(*it)->property())){
                        ( sendToServer(QString("SETCONF %1=%2").arg((*it)->name()).arg((*it)->property().toString())));
                    }
                }else if ( (*it)->property().type() == QVariant::StringList ) {
                    if ((oldvalue !=(*it)->property())){
                        ( sendToServer(QString("SETCONF %1=\"%2\"").arg((*it)->name()).arg( (*it)->property().toStringList().join(","))));
                    }
                }else if ( (*it)->property().type() == QVariant::Int ) {
                    if ((oldvalue !=(*it)->property())){
                        ( sendToServer(QString("SETCONF %1=%2").arg((*it)->name()).arg( (*it)->property().toString())));
                    }
                }else if ( (*it)->property().type() == QVariant::Bool ) {
                    if ((oldvalue !=(*it)->property())){
                        ( sendToServer(QString("SETCONF %1=%2").arg((*it)->name()).arg( (*it)->property().toInt())));
                    }
                }

            }
        }
        //if ( p.type() == QVariant::Bool ) {
    }

    if ((TorkConfig::httpProxyPort() > 0) && (!TorkConfig::httpProxyHost().isEmpty()))  
        ( sendToServer(QString("SETCONF HttpProxy=%1:%2").arg(TorkConfig::httpProxyHost()).arg(TorkConfig::httpProxyPort()))) ;
    if ((TorkConfig::httpsProxyPort() > 0) && (!TorkConfig::httpsProxyHost().isEmpty()))  
        ( sendToServer(QString("SETCONF HttpsProxy=%1:%2").arg(TorkConfig::httpsProxyHost()).arg(TorkConfig::httpsProxyPort()))) ;

    if ((!TorkConfig::httpProxyAuthenticatorUserName().isEmpty()) && (!TorkConfig::httpProxyAuthenticatorPassword().isEmpty()))
        ( sendToServer(QString("SETCONF HttpProxyAuthenticator=%1:%2").arg(TorkConfig::httpProxyAuthenticatorUserName()).arg(TorkConfig::httpProxyAuthenticatorPassword())));

    if ((!TorkConfig::httpsProxyAuthenticatorUserName().isEmpty()) && (!TorkConfig::httpsProxyAuthenticatorPassword().isEmpty()))  
        ( sendToServer(QString("SETCONF HttpsProxyAuthenticator=%1:%2").arg(TorkConfig::httpsProxyAuthenticatorUserName() ).arg(TorkConfig::httpsProxyAuthenticatorPassword())));

    if ((!TorkConfig::sOCKSBindAddressHost().isEmpty()) && (TorkConfig::sOCKSBindAddressPort()  > -1))
        ( sendToServer(QString("SETCONF SOCKSListenAddress=%1:%2").arg(TorkConfig::sOCKSBindAddressHost()).arg( TorkConfig::sOCKSBindAddressPort()))) ;

    if ((TorkConfig::sOCKSBindAddressHost().isEmpty()) && (TorkConfig::sOCKSBindAddressPort()  > -1))
        ( sendToServer(QString("SETCONF SOCKSPort=%2").arg(TorkConfig::sOCKSBindAddressPort()))) ;

    emit copyOldConfig();
    emit makeTorkStoppable();
}

bool Client::elementShouldBeUsed(const KConfigSkeletonItem* it)
{


    if ((((!TorkConfig::defaultMaxMinOptions())) && ((*it).group() == "MaxMin")) ||
       (((!TorkConfig::defaultRunningNormalOptions())) && ((*it).group() == "RunningNormal")) ||
       ((!(TorkConfig::defaultServerIP()))  && ((*it).group() == "DefaultServerAddress")) ||
       (((*it).group() == "FirewallsProxies")) ||
       (((*it).group() == "RunningSpecial")) ||
       (((*it).group() == "Servers")) ||
       (((*it).group() == "MyServer")) ||
       (((*it).group() == "Usability")) ||
       (((*it).group() == "UsingTor")) ||
       (((*it).group() == "MyHiddenServices")) ||
       ((!(TorkConfig::defaultServerPerformance())) && ((*it).group() == "ServerPerformance")))
            return true;


    return false;
}

bool Client::noSpecialProcessing(const KConfigSkeletonItem* it)
{

    if ((*it).group() == "DefaultServerAddress"){
        if ((*it).name() == "ORPort")
            ( sendToServer(QString("SETCONF %1=%2").arg((*it).name()).arg( (*it).property().toString())));
        else{
            if (!(TorkConfig::defaultServerIP())){
                    if (TorkConfig::dynDnsServerIP()){
                        QString myIP = getPublicIP();
                        ( sendToServer(QString("SETCONF Address=%2").arg(myIP)));
                    }
            }

        }
        return false; 
    }

    if (((*it).name() == "BandwidthBurst") || ((*it).name() == "BandwidthRate")){
        ( sendToServer(QString("SETCONF %1=%2KB").arg((*it).name()).arg( (*it).property().toString())));
        return false;
    }

    if ((*it).name() == "MaxAdvertisedBandwidth"){
        ( sendToServer(QString("SETCONF %1=%2GB").arg((*it).name()).arg( (*it).property().toString())));
        return false;
    }

    if ((*it).name() == "AccountingMax"){
        ( sendToServer(QString("SETCONF %1=\"%2 bytes\"").arg((*it).name()).arg( ((*it).property().toInt() * 1024 * 1024))));
        return false;
    }

    if ((*it).name() == "AccountingStart"){
       if ((*it).property().toString() == "day")
            ( sendToServer(QString("SETCONF %1=\"%2 00:00\"").arg((*it).name()).arg( (*it).property().toString())));
       else
            ( sendToServer(QString("SETCONF %1=\"%2 1 00:00\"").arg((*it).name()).arg( (*it).property().toString())));
       return false;
    }


    if ((*it).name() == "KeepalivePeriod"){
        if (!TorkConfig::reachableAddresses().isEmpty()){
            ( sendToServer(QString("SETCONF %1=%2").arg((*it).name()).arg( ((*it).property().toInt() * 60)))) ;
        }
        return false;
    }

    if ((*it).name() == "TrackHostExits"){
        if (!TorkConfig::trackHostExits().isEmpty()){
            ( sendToServer(QString("SETCONF %1=%2").arg((*it).name()).arg( ((*it).property().toStringList().join(","))))) ;
            if (TorkConfig::trackHostExitsExpire() > 0)  
                ( sendToServer(QString("SETCONF TrackHostExitsExpire=%2").arg((TorkConfig::trackHostExitsExpire() * 60)))) ;
        }
        return false;
    }


    if ((*it).name() == "SOCKSBindAddressMany"){

        if (!TorkConfig::sOCKSBindAddressMany().isEmpty()){
            QStringList socksbind = TorkConfig::sOCKSBindAddressMany();
            for ( QStringList::Iterator it = (socksbind).begin(); it != (socksbind).end(); it++ )
            {
                if ((*it).isEmpty())
                    continue;
                ( sendToServer(QString("SETCONF SOCKSListenAddress=%2").arg((*it)))) ;
            }
        }
        return false;
    }

    if ((*it).name() == "ExitPolicy"){
        if (TorkConfig::middleMan())
            ( sendToServer(QString("SETCONF ExitPolicy=%1").arg(("\"reject *:*\"")))) ;
        else
            ( sendToServer(QString("SETCONF %1=\"%2\"").arg((*it).name()).arg( (*it).property().toStringList().join(","))));
        return false;
    }


    if ((*it).name() == "HiddenServices"){
        QStringList hiddenServices = TorkConfig::hiddenServices();
        QString allservices;
        for ( QStringList::Iterator it = (hiddenServices).begin(); it != (hiddenServices).end(); it++ )
        {
            if ((*it).isEmpty())
                continue;
            allservices += (QString("HiddenServiceDir=\"%1\" HiddenServicePort=\"%2 %3\" ").arg((*it).section("\n",-1)).arg((*it).section("\n",-4,-4)).arg((*it).section("\n",-3,-3))) ;
        }
        ( sendToServer(QString("SETCONF %1").arg(allservices))) ;

        return false;
    }

    return true;
}



// void Client::storeServers(const QString &line)
// {
// 
//     QStringList networkServers = QStringList::split( " ", line);
//     TorkConfig::setActiveServers(networkServers);
// 
//     for ( QStringList::Iterator it = networkServers.begin(); it != networkServers.end(); ++it )
// 	{
// 		if ((*it).isEmpty())
// 			continue;
//            kdDebug() << (*it) << endl;
//            QStringList tokens = QStringList::split( "=", (*it));
//            serverTofp_identity[tokens[0]] = tokens[1];
//            fp_identityToServer[tokens[1]] = tokens[0];
// //             kdDebug() << tokens[0]).arg( serverTofp_identity[tokens[0]] << endl;
// 
// 
//     }
//     sendToServer("GETINFO entry-guards");
//     //emit updateActiveServers(serverTofp_identity.keys());
// 
// }


QString Client::getPublicIP()
{


    Http http("checkip.dyndns.org");
    if (http.request("/")) {
        QString publicip = http.body().replace("Current IP Address:","");
        //kdDebug() << publicip << endl;
        if (QHostAddress().setAddress(publicip))
            return publicip.stripWhiteSpace();
    }

    return "";
}
Client::~Client()
{

}

void Client::parseDirStatus(const QString &info)
{

    if (info.left(2) == "r "){
        //kapp->processEvents();
/*        char buf[256];
        char hexdigest[HEX_DIGEST_LEN+1];*/
        ds_identity = info.section(" ",2,2);
        ds_ip = info.section(" ",6,6);
/*        digest_from_base64(buf, ds_identity);
        base16_encode(hexdigest, HEX_DIGEST_LEN+1, buf, DIGEST_LEN);*/
        //kdDebug() << "BUFFF:" << hexdigest <<  " " << strlen(ds_identity) << endl;

        ds_server = info.section(" ",1,1);

        storeServer(ds_server,ds_identity);
    }

    if (info.left(2) == "s "){
        ds_statuses = info;
/*        if (info.contains("Guard"))
            parseGuards("$"+ds_fp_identity + " " + ds_statuses);*/
        emit updateServerStatus(ds_ip, ds_identity, ds_server, ds_statuses, m_firstloadcomplete);

    }

}

bool Client::isControllerWorking()
{
    return m_controllerWorking;
 
}

// FP is the 40 byte expression of type $A6DS3... you see in network-status and guardian-info
// FPDigest is the base64 digest of an FP provided in a dirstatus document
// Since we map server nicknames to FPDigests we need to convert FPDigests to FPs where necessary

// QString Client::getFPDigestFromFP(const QString &fp)
// {
//     char identity64[BASE64_DIGEST_LEN+1];
//     char digest[DIGEST_LEN];
//     QString FP = fp;
//     FP.replace("$","");
// 
//     base16_decode(digest, DIGEST_LEN, FP, strlen(FP));
//     digest_to_base64(identity64, digest);
//     //kdDebug() << "identity64" << identity64 << endl;
//     return identity64;
// }
// 
// QString Client::getNickNameFromFPDigest(const QString &fpdigest)
// {
// 
//     return fp_identityToServer[fpdigest];
// 
// }
// 
// QString Client::getNickNameFromFP(const QString &fp)
// {
//     QString fpdigest = getFPDigestFromFP(fp);
// 
//     return fp_identityToServer[fpdigest];
// }
// 
// 
// bool Client::isControllerWorking()
// {
//     return m_controllerWorking;
//  
// }
// 
// 
// 
// QString Client::getFPFromNickName(const QString &nickname)
// {
//     char buf[256];
//     char hexdigest[HEX_DIGEST_LEN+1];
// 
//     QString fp = serverTofp_identity[nickname];
// 
//     if (fp.isEmpty())
//         return QString();
//     if (!digest_from_base64(buf, fp))
//         base16_encode(hexdigest, HEX_DIGEST_LEN+1, buf, DIGEST_LEN);
// 
//     return hexdigest;
// }
// 
// QString Client::getFPFromFPDigest(const QString &fp)
// {
//     char buf[256];
//     char hexdigest[HEX_DIGEST_LEN+1];
// 
//     digest_from_base64(buf, fp);
//     base16_encode(hexdigest, HEX_DIGEST_LEN+1, buf, DIGEST_LEN);
// 
//     return hexdigest;
// }




#include "torclient.moc"
