/***************************************************************************
 *   Copyright (C) 2004-2006 by Pere Constans
 *   constans@molspaces.com
 *   cb2Bib version 0.6.1
 *   See LICENSE file that comes with this distribution
 ***************************************************************************/
#include "citeidmaker.h"
#include "cb2bibbase.h"
#include "c2butils.h"
#include "cb2bib_par.h"

#include <qlineedit.h>
#include <qregexp.h>
#include <qsettings.h>


citeIDMaker::citeIDMaker(cb2bibbase *c2bMain, QObject *parent, const char *name)
        : QObject(parent, name)
{
    c2b = c2bMain;
    m_author = "";
    m_journal = "";
    m_pages = "";
    m_volume = "";
    m_year = "";
    readSettings();
    connect( c2b, SIGNAL( newSettingAvailable() ), this, SLOT( readSettings() ) );
}

citeIDMaker::~citeIDMaker()
{}


void citeIDMaker::makeCiteID()
{
    /*! \page citeid_placeholders Predefined cite ID placeholders

    - <tt><<author_all_initials>></tt> Takes capitalized initials of all authors in cite.

    - <tt><<author_first>></tt> Takes first author last name/s.

    - <tt><<journal_initials>></tt> Takes capitalized initials of journal name.

    - <tt><<pages_first>></tt> First page.

    - <tt><<volume>></tt> Volume number.

    - <tt><<year_abbreviated>></tt> Last two digits from year.

    - <tt><<year_full>></tt> All digits from year.

    */

    QString cid = m_citeidpattern;
    cid.replace( "<<author_first>>", m_author );
    cid.replace( "<<author_all_initials>>", m_author );
    cid.replace( "<<journal_initials>>", m_journal );
    cid.replace( "<<pages_first>>", m_pages );
    cid.replace( "<<volume>>", m_volume );
    cid.replace( "<<year_abbreviated>>", m_year );
    cid.replace( "<<year_full>>", m_year );
    cid.remove( " " );
    c2b->id->setText( cid );
    if( cid.isEmpty() )
        cid = "(none)";
    c2b->setCaption( tr( QString("%1 - cb2Bib").arg(cid) ) );
}

void citeIDMaker::makeCiteID_author_first()
{
    m_author = c2b->authors->text();
    QRegExp rx( "([-'\\s\\w]+)(\\sand|$)" );
    rx.setMinimal( TRUE );
    rx.search( m_author );
    m_author = rx.cap(1);
    if ( m_author.contains( QRegExp ("[^A-Za-z]") ) )
        m_author = c2btoAscii( m_author );
    makeCiteID();
}

void citeIDMaker::makeCiteID_author_all_initials()
{
    m_author = c2b->authors->text();
    m_author.remove( QRegExp("\\b\\w\\b") );
    m_author.remove( " and " );
    m_author.remove( QRegExp("[a-z]") );
    if ( m_author.contains( QRegExp ("[^A-Z]") ) )
        m_author = c2btoAscii( m_author );
    makeCiteID();
}

void citeIDMaker::makeCiteID_journal_initials()
{
    m_journal = c2b->journal->text();
    m_journal.remove( QRegExp ("[^A-Z]") );
    makeCiteID();
}

void citeIDMaker::makeCiteID_pages_first()
{
    m_pages = firstPage( c2b->pages->text() );
    makeCiteID();
}

void citeIDMaker::makeCiteID_volume()
{
    m_volume = c2b->volume->text();
    makeCiteID();
}

void citeIDMaker::makeCiteID_year_abbreviated()
{
    m_year = c2b->year->text().right(2);
    makeCiteID();
}

void citeIDMaker::makeCiteID_year_full()
{
    m_year = c2b->year->text();
    makeCiteID();
}

void citeIDMaker::readSettings()
{
    QSettings settings;
    settings.insertSearchPath( QSettings::Windows, WINDOWS_REGISTRY );
    m_citeidpattern = settings.readEntry( APP_KEY + "citeidpattern", APP_CITE_ID_PATTERN );
    setPatternConnections();
}

void citeIDMaker::setPatternConnections()
{
    disconnect( c2b->authors, SIGNAL( textChanged(const QString& ) ), this, 0 );
    disconnect( c2b->journal, SIGNAL( textChanged(const QString& ) ), this, 0 );
    disconnect( c2b->pages, SIGNAL( textChanged(const QString& ) ), this, 0 );
    disconnect( c2b->volume, SIGNAL( textChanged(const QString& ) ), this, 0 );
    disconnect( c2b->year, SIGNAL( textChanged(const QString& ) ), this, 0 );

    // authors
    if( m_citeidpattern.contains( "<<author_first>>" ) )
        connect( c2b->authors, SIGNAL( textChanged(const QString& ) ),
                 this, SLOT( makeCiteID_author_first() ) );
    else if( m_citeidpattern.contains( "<<author_all_initials>>" ) )
        connect( c2b->authors, SIGNAL( textChanged(const QString& ) ),
                 this, SLOT( makeCiteID_author_all_initials() ) );

    // journal
    if( m_citeidpattern.contains( "<<journal_initials>>" ) )
        connect( c2b->journal, SIGNAL( textChanged(const QString& ) ),
                 this, SLOT( makeCiteID_journal_initials() ) );

    // pages
    if( m_citeidpattern.contains( "<<pages_first>>" ) )
        connect( c2b->pages, SIGNAL( textChanged(const QString& ) ),
                 this, SLOT( makeCiteID_pages_first() ) );

    // volume
    if( m_citeidpattern.contains( "<<volume>>" ) )
        connect( c2b->volume, SIGNAL( textChanged(const QString& ) ),
                 this, SLOT( makeCiteID_volume() ) );

    // year
    if( m_citeidpattern.contains( "<<year_abbreviated>>" ) )
        connect( c2b->year, SIGNAL( textChanged(const QString& ) ),
                 this, SLOT( makeCiteID_year_abbreviated() ) );
    else if( m_citeidpattern.contains( "<<year_full>>" ) )
        connect( c2b->year, SIGNAL( textChanged(const QString& ) ),
                 this, SLOT( makeCiteID_year_full() ) );
}
