/***************************************************************************
 *   Copyright (C) 2004-2006 by Pere Constans
 *   constans@molspaces.com
 *   cb2Bib version 0.6.1
 *   See LICENSE file that comes with this distribution
 ***************************************************************************/
#include "journaldb.h"

JournalDB::JournalDB( const QString &dbfile )
{
    if ( dbfile.isEmpty() )
    {
        qDebug( QString("cb2Bib: No journal file especified.") );
        return;
    }
    nitems = 0;
    QFile file( dbfile );
    if ( ! file.open( IO_ReadOnly ) )
    {
        qDebug (QString("cb2Bib: Could not open %1 journal file for reading.").arg(dbfile));
        return;
    }
    QTextStream stream( &file );
    QString line;

    while ( !stream.atEnd() )
    {
        line = stream.readLine();
        if ( !( line.isEmpty() || line.contains(QRegExp("^#")) ) )
        {
            QStringList spLine = QStringList::split( "|", line );

            nitems++;
            JCode += spLine[0];

            QString dum = spLine[1];
            JAbbrev += dum;
            dum.replace( QRegExp( "\\W" ), "" );  //  Removes whitespaces also
            JAbbrev_simp += dum;

            dum = spLine[2];
            JExtended += dum;
            dum.replace( QRegExp( "\\W" ), "" );  //  Removes whitespaces also
            JExtended_simp += dum;
        }
    }
    file.close();
}


JournalDB::~JournalDB()
{
    JCode.clear();
    JAbbrev.clear();
    JExtended.clear();
    JAbbrev_simp.clear();
    JExtended_simp.clear();
}

QString JournalDB::retrieve( QString JQuery )
{

    /*! \page journalproc Processing of journal names

     cb2Bib processes journal names according to its editable database,
     stored at <tt>abbreviations.txt</tt>. See \ref c2bconf_files and
     \ref c2bconf_output.

     Journal names processing is performed in the following situations 
     whenever a string is recognized as 'journal'. Additionally, 
     it is also processed by pressing <b>Intro Key</b> at the journal 
     edit line.

    <p>&nbsp;</p>

            - Retrieves Journal name, <b>abbreviated form</b>, if found.

            - If Journal name is not found in the database, returns 
            input Journal name.

            - Search is case unsensitive.

            - Warning: Journal codes can be duplicated. If duplicated, 
            returns input Journal name.

    */

    QString query = JQuery;
    query.replace( QRegExp( "\\W" ), "" );  //  Removes whitespaces also
    QRegExp JQrx = QRegExp( QString("^%1$").arg(query), FALSE );
    int jfound = 0;
    int foundat = -1;

    for ( int i = 0; i < nitems; i++ )
    {
        if( JCode[i].contains( JQrx ) )
        {
            jfound++;
            foundat = i;
        }
        if( JAbbrev_simp[i].contains( JQrx ) )
            if( ++jfound == 1 )
                return( JAbbrev[i] );
        if( JExtended_simp[i].contains( JQrx ) )
            if( ++jfound == 1 )
                return( JAbbrev[i] );
    }
    if( jfound == 1 && foundat != -1 )
        return( JAbbrev[foundat] );
    else
        return( JQuery );
}

QString JournalDB::retrieveFull( const QString& JQuery )
{
    /*! \page journalproc
    <p>&nbsp;</p>

    - Retrieves Journal name, <b>full form</b>, if found.

    - If Journal name is not found in the database, returns 
    input Journal name.

    - Search is case unsensitive.

    - Warning: Journal codes can be duplicated. If duplicated, 
    returns input Journal name.

    */

    QString query = JQuery;
    query.replace( QRegExp( "\\W" ), "" );  //  Removes whitespaces also
    QRegExp JQrx = QRegExp( QString("^%1$").arg(query), FALSE );
    int jfound = 0;
    int foundat = -1;

    for ( int i = 0; i < nitems; i++ )
    {
        if( JCode[i].contains( JQrx ) )
        {
            jfound++;
            foundat = i;
        }
        if( JAbbrev_simp[i].contains( JQrx ) )
            if( ++jfound == 1 )
                return( JExtended[i] );
        if( JExtended_simp[i].contains( JQrx ) )
            if( ++jfound == 1 )
                return( JExtended[i] );
    }
    if( jfound == 1 && foundat != -1 )
        return( JExtended[foundat] );
    else
        return( JQuery );
}
