
#include <qlabel.h>
#include <qhbox.h>
#include <qvbox.h>
#include <qlistview.h>
#include <qtoolbutton.h>
#include <qlayout.h>
#include <qgrid.h>
#include <qframe.h>
#include <qlabel.h>
#include <qdockarea.h>
#include <qfileinfo.h>
#include <qtimer.h>
#include <qdir.h>

#include <klocale.h>
#include <kfiledialog.h>
#include <kurlcombobox.h>
#include <kurlcompletion.h>
#include <kdiroperator.h>
#include <ktoolbar.h>
#include <kdebug.h>
#include <kactioncollection.h>
#include <kcombobox.h>
#include <kiconloader.h>

#include "diroperator.h"

DirOperator::DirOperator(QWidget *parent, const char *name)
: QWidget(parent, name), akt_sng(NULL)
{
	QGridLayout *mainGrid = new QGridLayout(this,0,0);

	QVBox *geo = new QVBox(this);
	mainGrid->addWidget(geo,0,0);

	QWidget *hb = new QWidget(geo);
	hb->setFixedHeight(30);
	hb->setMinimumWidth(300);

	da = new QDockArea(Qt::Horizontal,QDockArea::Normal,hb);
	da->setFixedHeight(30);
	da->setFixedWidth(180);

	tb = new KToolBar(da);
	tb->setFlat(false);
	tb->setIconText( KToolBar::IconOnly );
	tb->setIconSize( 16 );
	tb->setEnableContextMenu( false );
	tb->setMovingEnabled(false);

	cmbPath = new KURLComboBox( KURLComboBox::Directories, true,geo, "path combo" );
	cmbPath->setFixedHeight(20);
	cmbPath->setCompletionObject(new KURLCompletion(KURLCompletion::DirCompletion));
	cmbPath->setAutoDeleteCompletionObject( true );
	cmbPath->listBox()->installEventFilter( this );

	connect( cmbPath, SIGNAL( urlActivated( const KURL&  )),
		this,  SLOT( cmbPathActivated( const KURL& ) ));
	connect( cmbPath, SIGNAL( returnPressed( const QString&  )),
		this,  SLOT( cmbPathReturnPressed( const QString& ) ));

	dir = new KDirOperator(QDir::homeDirPath(), geo);
	dir->setView(KFile::Simple);
	dir->setMode(KFile::Files);

	connect(dir, SIGNAL(urlEntered(const KURL&)),
		this, SLOT(dirUrlEntered(const KURL&)) );

	connect(dir, SIGNAL(finishedLoading()),
		this, SLOT(dirFinishedLoading()) );

	connect(dir, SIGNAL(fileSelected(const KFileItem *)),
		this, SLOT(selectionChange(const KFileItem *)) );

	KAction *ac1 = dir->actionCollection()->action("up");
	KAction *ac2 = dir->actionCollection()->action("back");
	KAction *ac3 = dir->actionCollection()->action("forward");
	KAction *ac4 = dir->actionCollection()->action("home");
	KAction *ac5 = dir->actionCollection()->action("detailed view");
	KAction *ac6 = dir->actionCollection()->action("short view");
	KAction *ac8 = dir->actionCollection()->action("reload");

	ac1->setShortcut( KShortcut( ALT + SHIFT + Key_Up ) );
	ac2->setShortcut( KShortcut( ALT + SHIFT + Key_Left ) );
	ac3->setShortcut( KShortcut( ALT + SHIFT + Key_Right ));
	ac4->setShortcut( KShortcut( CTRL + ALT + Key_Home ) );
	ac8->setShortcut( KShortcut( ALT + Key_F5 ) );

	if ( ac1 ) ac1->plug( tb );
	if ( ac2 ) ac2->plug( tb );
	if ( ac3 ) ac3->plug( tb );
	if ( ac4 ) ac4->plug( tb );
	if ( ac5 ) ac5->plug( tb );
	if ( ac6 ) ac6->plug( tb );
	if ( ac8 ) ac8->plug( tb );

	btnFilter = new QToolButton( hb );
	btnFilter->setFixedSize(25,25);
	btnFilter->move(183,0);
	btnFilter->setIconSet( SmallIconSet("filter" ) );
	btnFilter->setToggleButton( true );
	connect( btnFilter, SIGNAL( clicked() ),this,SLOT( btnFilterClick()));

	filter = new KHistoryCombo( true, hb, "filter");
	filter->setFixedWidth(150);
	filter->setFixedHeight(25);
	filter->move(210,0);
	connect(filter,SIGNAL(activated(const QString&)),
		SLOT(slotFilterChange(const QString&)));
	connect(filter,SIGNAL(returnPressed(const QString&)),filter,
		SLOT(addToHistory(const QString&) ) );

	timer = new QTimer(this);
}

DirOperator::~DirOperator(){}

void DirOperator::dirFinishedLoading() {cmbPath->setURL(dir->url());}

void DirOperator::slotFilterChange( const QString & nf )
{
	QString f = nf.stripWhiteSpace();
	bool empty = f.isEmpty() || f == "*";
	if ( empty ) {
		dir->clearFilter();
		filter->lineEdit()->setText( QString::null );
	}
	else {
		dir->setNameFilter( f );
		lastFilter = f;
	}
	btnFilter->setOn( !empty );
	dir->updateDir();
	btnFilter->setEnabled( !( empty && lastFilter.isEmpty() ) );
}
bool IsReadable ( const KURL& url )
{
	if ( !url.isLocalFile() ) return true;
	QDir dir (url.path());
	return dir.exists ();
}

void DirOperator::setDir( KURL u )
{
	KURL newurl;

	if ( !u.isValid() ) newurl.setPath( QDir::homeDirPath() );
	else newurl = u;

	QString pathstr = newurl.path(+1);
	newurl.setPath(pathstr);

	if(!IsReadable ( newurl )) newurl.cd(QString::fromLatin1(".."));
	if(!IsReadable (newurl)) newurl.setPath( QDir::homeDirPath() );

	dir->setURL(newurl, true);
}

void DirOperator::cmbPathActivated(const KURL& u) {cmbPathReturnPressed( u.url());}

void DirOperator::cmbPathReturnPressed( const QString& u )
{
	KURL typedURL( u );
	if ( typedURL.hasPass() )
	typedURL.setPass( QString::null );

	QStringList urls = cmbPath->urls();
	urls.remove( typedURL.url() );
	urls.prepend( typedURL.url() );
	cmbPath->setURLs( urls, KURLComboBox::RemoveBottom );
	dir->setFocus();
	dir->setURL( KURL(u), true );
}

void DirOperator::dirUrlEntered( const KURL& u ) {cmbPath->setURL( u );}

void DirOperator::btnFilterClick()
{
	if ( !btnFilter->isOn() ) {
		slotFilterChange( QString::null );
	}
	else {
		filter->lineEdit()->setText( lastFilter );
		slotFilterChange( lastFilter );
	}
}


void DirOperator::selectionChange(const KFileItem *s)
{
	emit selectionChange();
	QString file = (new QFileInfo((s->url()).pathOrURL()))->absFilePath();

	// detect double klicks (hack)
	if( timer->isActive() && akt_sng->absFilePath() == file) {
		timer->stop();
		emit doubleKlick();
	}
	else timer->start( 500 , TRUE ); // 5/10 seconds single-shot

	akt_sng = new QFileInfo((s->url()).pathOrURL());
}

void DirOperator::clearSelection()
{
	if(!this->hasMouse()) {
		const QString a;
		dir->setCurrentItem(a);
	}
}

const KFileItemList *DirOperator::getSelectedItems() {return dir->selectedItems();}

void DirOperator::setURL(const QString url) {dir->setURL(url,true);}

void DirOperator::selectNext()
{
	const KFileItemList *ll = (dir->view())->selectedItems();
	for ( KFileItemList::Iterator it = ll->begin(); it != ll->end(); ++it )  {
		if (dir->isSelected(static_cast<KFileItem *>(*it))) {
			const KFileItem *kfi = dir->view()->nextItem(static_cast<KFileItem *>(*it));
			if (kfi) {
				if (!kfi->isDir()) {
					akt_sng = new QFileInfo((kfi->url()).pathOrURL());
					(dir->view())->clearSelection();
					(dir->view())->setSelected(kfi,true);
					(dir->view())->ensureItemVisible(kfi);
				}
			}
		}
	}
}

void DirOperator::selectPrev()
{
	const KFileItemList *ll = (dir->view())->selectedItems();
	for ( KFileItemList::Iterator it = ll->begin(); it != ll->end(); ++it )  {
		if (dir->isSelected(static_cast<KFileItem *>(*it))) {
			const KFileItem *kfi = dir->view()->prevItem(static_cast<KFileItem *>(*it));
			if (kfi) {
				if (!kfi->isDir()) {
					akt_sng = new QFileInfo((kfi->url()).pathOrURL());
					(dir->view())->clearSelection();
					(dir->view())->setSelected(kfi,true);
					(dir->view())->ensureItemVisible(kfi);
				}
			}
		}
	}
}

void DirOperator::enable()
{
	dir->setDisabled(false);
	da->setDisabled(false);
	tb->setDisabled(false);

	cmbPath->setDisabled(false);
	cmbPath->setEditable(true);

	connect( cmbPath, SIGNAL( urlActivated( const KURL&  )),
		this,  SLOT( cmbPathActivated( const KURL& ) ));
	connect( cmbPath, SIGNAL( returnPressed( const QString&  )),
		this,  SLOT( cmbPathReturnPressed( const QString& ) ));
	connect(dir, SIGNAL(urlEntered(const KURL&)),
		this, SLOT(dirUrlEntered(const KURL&)) );
	connect(dir, SIGNAL(finishedLoading()),
		this, SLOT(dirFinishedLoading()) );
	connect(dir, SIGNAL(fileSelected(const KFileItem *)),
		this, SLOT(selectionChange(const KFileItem *)) );

	btnFilter->setDisabled(false);
	connect( btnFilter, SIGNAL( clicked() ),
		this,SLOT( btnFilterClick()));

	connect(filter,SIGNAL(activated(const QString&)),
		this,SLOT(slotFilterChange(const QString&)));
	connect(filter,SIGNAL(returnPressed(const QString&)),
		filter,SLOT(addToHistory(const QString&) ) );
}


void DirOperator::disable()
{
	dir->setDisabled(true);
	da->setDisabled(true);
	tb->setDisabled(true);

	cmbPath->setDisabled(true);
	cmbPath->setEditable(false);

	disconnect( cmbPath, SIGNAL( urlActivated( const KURL&  )),
		this,  SLOT( cmbPathActivated( const KURL& ) ));
	disconnect( cmbPath, SIGNAL( returnPressed( const QString&  )),
		this,  SLOT( cmbPathReturnPressed( const QString& ) ));
	disconnect(dir, SIGNAL(urlEntered(const KURL&)),
		this, SLOT(dirUrlEntered(const KURL&)) );
	disconnect(dir, SIGNAL(finishedLoading()),
		this, SLOT(dirFinishedLoading()) );
	disconnect(dir, SIGNAL(fileSelected(const KFileItem *)),
		this, SLOT(selectionChange(const KFileItem *)) );

	btnFilter->setDisabled(true);
	disconnect( btnFilter, SIGNAL( clicked() ),
		this,SLOT( btnFilterClick()));

	disconnect(filter,SIGNAL(activated(const QString&)),
		this,SLOT(slotFilterChange(const QString&)));
	disconnect(filter,SIGNAL(returnPressed(const QString&)),
		filter,SLOT(addToHistory(const QString&) ) );
}

