/***************************************************************************
                          madlib.cpp  -  description
                             -------------------
    begin                : april 10 2006
    copyright            : (C) 2006 by Martin Keil
    email                : martin-keil@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// This code was orginally written in C by Bertrand Petit.
// (c) 2001--2004 Bertrand Petit

#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <string.h>
#include <limits.h>
#include <math.h> /* for pow() and log10() */

#include <qstring.h>
#include <qfileinfo.h>

#include <kdebug.h>

#include "madlib.h"

madlib::madlib()   {first_header = new mad_header; }
madlib::~madlib(){}


bstdfile* madlib::NewBstdFile(FILE *fp)
{
	bstdfile_t *BstdFile;

	// Allocate the bstdfile structure. 
	BstdFile=(bstdfile_t *)malloc(sizeof(bstdfile_t));
	if(BstdFile==NULL) {
		errno=ENOMEM;
		return(NULL);
	}

	// Initialize the structure to safe defaults.
	BstdFile->live=BstdFile->buffer;
	BstdFile->live_size=0;
	BstdFile->eof=0;
	BstdFile->error=0;
	BstdFile->fp=fp;

	// Return the new bfile.
	return(BstdFile);
}

int madlib::BstdFileDestroy(bstdfile_t *BstdFile)
{
	if(BstdFile==NULL)
	{
		errno=EBADF;
		return(1);
	}
	free(BstdFile);
	return(0);
}


size_t madlib::BstdRead(void *UserBuffer, size_t ElementSize, size_t ElementsCount, bstdfile_t *BstdFile)
{
	size_t	RequestSize=ElementSize*ElementsCount,
			FeededSize=0,
			ReadSize,
			ObtainedSize;
	int OldErrno=errno;

	/* Check the validity of the arguments. */
	if(BstdFile==NULL)
	{
		errno=EBADF;
		return((size_t)0);
	}
	if(UserBuffer==NULL)
	{
		errno=EFAULT;
		return((size_t)0);
	}
	if(RequestSize<1)
	{
		errno=EINVAL;
		return((size_t)0);
	}

	/* Return immediately if an exceptional situation exists. */
	if(BstdFile->eof)
		return((size_t)0);
	if(BstdFile->error)
	{
		errno=BstdFile->error;
		return((size_t)0);
	}

	/* The easy case. */
	if(RequestSize==0U)
		return((size_t)0);

	/* First feed the target buffer from the BstdFile buffer if it has
	 * some meat to be feeded on.
	 */
	if(BstdFile->live_size>0)
	{
		/* If there is more data in the buffer than requested by the
		 * user then we feed him directly from our buffer without a
		 * read operation.
		 */
		if(BstdFile->live_size>RequestSize)
		{
			memcpy(UserBuffer,BstdFile->live,RequestSize);
			BstdFile->live+=RequestSize;
			BstdFile->live_size-=RequestSize;
			return(RequestSize);
		}
		/* Else we drain our buffer. */
		else
		{
			memcpy(UserBuffer,BstdFile->live,BstdFile->live_size);
			UserBuffer=(char *)UserBuffer+BstdFile->live_size;
			FeededSize=BstdFile->live_size;
			BstdFile->live=BstdFile->buffer;
			BstdFile->live_size=0;
		}
	}

	/* If the user request was not yet fulfilled we then read from the
     * file the remaining data requested by the user.
	 */
	if(FeededSize<RequestSize)
	{
		ReadSize=RequestSize-FeededSize;
		ObtainedSize=fread(UserBuffer,1,ReadSize,BstdFile->fp);
		FeededSize+=ObtainedSize;

		/* If an error occurs we return the amount of data that was
		 * feeded from the buffer and store the error condition for a
		 * later call. If our buffer was empty and we thus have
		 * transferred no data to the user buffer then we directly
		 * return the error.
		 */
		if(ObtainedSize==0U)
		{
			if(feof(BstdFile->fp))
				BstdFile->eof=1;
			else
			{
				BstdFile->error=errno;
				errno=OldErrno;
			}
			if(FeededSize!=0)
				return(FeededSize);
			else
				return(0U);
		}
	}

	/* Fill again our buffer. In case of error, or end of file, that
	 * error is recorded but we still report the amount of data that
	 * was feeded to the user buffer.
	 */
	ObtainedSize=fread(BstdFile->buffer,1,BFILE_BUFSIZE,BstdFile->fp);
	if(ObtainedSize==0)
	{
		if(feof(BstdFile->fp)) BstdFile->eof=1;
		else
		{
			BstdFile->error=errno;
			errno=OldErrno;
		}
	}
	else
	{
		BstdFile->live=BstdFile->buffer;
		BstdFile->live_size=ObtainedSize;
	}

	/* Eventually return the number ob bytes feeded to the user
     * buffer.
	 */
	return(FeededSize);
}

/****************************************************************************
 * Print human readable informations about an audio MPEG frame.				*
 ****************************************************************************/

int ref_header;
int equal_frames;

static int PrintFrameInfo(FILE *fp, struct mad_header *Header, struct mad_header *first_header)
{
	equal_frames = 0;
	ref_header = Header->layer + Header->mode + Header->emphasis + Header->samplerate;

	first_header->layer              = Header->layer;
	first_header->mode               = Header->mode;
	first_header->samplerate         = Header->samplerate;
	first_header->mode_extension     = Header->mode_extension;
	first_header->bitrate            = Header->bitrate;
	first_header->emphasis           = Header->emphasis;

	const char *Layer, *Mode, *Emphasis;

	/* Convert the layer number to it's printed representation. */
	switch(Header->layer)
	{
		case MAD_LAYER_I:
			Layer="I";
			break;
		case MAD_LAYER_II:
			Layer="II";
			break;
		case MAD_LAYER_III:
			Layer="III";
			break;
		default:
			Layer="(unexpected layer value)";
			break;
	}

	/* Convert the audio mode to it's printed representation. */
	switch(Header->mode)
	{
		case MAD_MODE_SINGLE_CHANNEL:
			Mode="single channel";
			break;
		case MAD_MODE_DUAL_CHANNEL:
			Mode="dual channel";
			break;
		case MAD_MODE_JOINT_STEREO:
			Mode="joint (MS/intensity) stereo";
			break;
		case MAD_MODE_STEREO:
			Mode="normal LR stereo";
			break;
		default:
			Mode="(unexpected mode value)";
			break;
	}

	/* Convert the emphasis to it's printed representation. Note that
	 * the MAD_EMPHASIS_RESERVED enumeration value appeared in libmad
	 * version 0.15.0b.
	 */
	switch(Header->emphasis)
	{
		case MAD_EMPHASIS_NONE:
			Emphasis="no";
			break;
		case MAD_EMPHASIS_50_15_US:
			Emphasis="50/15 us";
			break;
		case MAD_EMPHASIS_CCITT_J_17:
			Emphasis="CCITT J.17";
			break;
#if (MAD_VERSION_MAJOR>=1) || \
	((MAD_VERSION_MAJOR==0) && (MAD_VERSION_MINOR>=15))
		case MAD_EMPHASIS_RESERVED:
			Emphasis="reserved(!)";
			break;
#endif
		default:
			Emphasis="(unexpected emphasis value)";
			break;
	}

	return(ferror(fp));
}

/****************************************************************************
 * Return an error string associated with a mad error code.					*
 ****************************************************************************/
/* Mad version 0.14.2b introduced the mad_stream_errorstr() function.
 * For previous library versions a replacement is provided below.
 */
#if (MAD_VERSION_MAJOR>=1) || ((MAD_VERSION_MAJOR==0) && \
 (((MAD_VERSION_MINOR==14) && \
       (MAD_VERSION_PATCH>=2)) || \
      (MAD_VERSION_MINOR>14)))
#define MadErrorString(x) mad_stream_errorstr(x)
#else
static const char *MadErrorString(const struct mad_stream *Stream)
{
	switch(Stream->error)
	{
		/* Generic unrecoverable errors. */
		case MAD_ERROR_BUFLEN:
			return("input buffer too small (or EOF)");
		case MAD_ERROR_BUFPTR:
			return("invalid (null) buffer pointer");
		case MAD_ERROR_NOMEM:
			return("not enough memory");

		/* Frame header related unrecoverable errors. */
		case MAD_ERROR_LOSTSYNC:
			return("lost synchronization");
		case MAD_ERROR_BADLAYER:
			return("reserved header layer value");
		case MAD_ERROR_BADBITRATE:
			return("forbidden bitrate value");
		case MAD_ERROR_BADSAMPLERATE:
			return("reserved sample frequency value");
		case MAD_ERROR_BADEMPHASIS:
			return("reserved emphasis value");

		/* Recoverable errors */
		case MAD_ERROR_BADCRC:
			return("CRC check failed");
		case MAD_ERROR_BADBITALLOC:
			return("forbidden bit allocation value");
		case MAD_ERROR_BADSCALEFACTOR:
			return("bad scalefactor index");
		case MAD_ERROR_BADFRAMELEN:
			return("bad frame length");
		case MAD_ERROR_BADBIGVALUES:
			return("bad big_values count");
		case MAD_ERROR_BADBLOCKTYPE:
			return("reserved block_type");
		case MAD_ERROR_BADSCFSI:
			return("bad scalefactor selection info");
		case MAD_ERROR_BADDATAPTR:
			return("bad main_data_begin pointer");
		case MAD_ERROR_BADPART3LEN:
			return("bad audio data length");
		case MAD_ERROR_BADHUFFTABLE:
			return("bad Huffman table select");
		case MAD_ERROR_BADHUFFDATA:
			return("Huffman data overrun");
		case MAD_ERROR_BADSTEREO:
			return("incompatible block_type for JS");

		/* Unknown error. This switch may be out of sync with libmad's
		 * defined error codes.
		 */
		default:
			return("Unknown error code");
	}
}
#endif

#define INPUT_BUFFER_SIZE	(5*8192)
#define OUTPUT_BUFFER_SIZE	8192 /* Must be an integer multiple of 4. */


/** To reject bin Files, we compare any headers each other */
static void compare_headers(struct mad_header *Header)
{
	int this_header = Header->layer + Header->mode + Header->emphasis + Header->samplerate;
	if(this_header == ref_header) equal_frames++;

	//kdDebug() << "layer: " << Header->layer << "  Mode: " << Header->mode << "  Emphasis: " << Header->emphasis << "  Samplerate: " << Header->samplerate << "  Bitrate: " << Header->bitrate/1000 << " KBps" <<  endl;

}

bool madlib::MpegAudioDecoder(const QString *file, long &length_ms, int &b_rate, int &s_rate, int &channels)
{
	struct mad_stream Stream;
	struct mad_frame Frame;
	struct mad_synth Synth;
	mad_timer_t Timer;
	unsigned char InputBuffer[INPUT_BUFFER_SIZE+MAD_BUFFER_GUARD], *GuardPtr=NULL;
	bool Status = true;
	unsigned long FrameCount=0;
	bstdfile_t *BstdFile;

	/* First the structures used by libmad must be initialized. */
	mad_stream_init(&Stream);
	mad_frame_init(&Frame);
	mad_synth_init(&Synth);
	mad_timer_reset(&Timer);

	kdDebug() << "madlib: " << *file << endl;
	QFileInfo f((QFile::encodeName(*file).data()));
	FILE *ip = fopen(f.filePath(),"r");
	if (!ip) {
		kdDebug() << "madlib: fopen fails: " << *file <<  "\n\n";  
		return false;
	}

	BstdFile=NewBstdFile(ip);
	if(BstdFile==NULL) {
		kdDebug() << "madlib: Can't create a new bstdfile" << endl;
		return( false );
	}

	/* This is the decoding loop. */
	do
	{
		/* The input bucket must be filled if it becomes empty or if
		 * it's the first execution of the loop.
		 */
		if(Stream.buffer==NULL || Stream.error==MAD_ERROR_BUFLEN)
		{
			size_t ReadSize, Remaining;
			unsigned char	*ReadStart;

			if(Stream.next_frame!=NULL)
			{
				Remaining=Stream.bufend-Stream.next_frame;
				memmove(InputBuffer,Stream.next_frame,Remaining);
				ReadStart=InputBuffer+Remaining;
				ReadSize=INPUT_BUFFER_SIZE-Remaining;
			}
			else
				ReadSize=INPUT_BUFFER_SIZE,
					ReadStart=InputBuffer,
					Remaining=0;

			ReadSize=BstdRead(ReadStart,1,ReadSize,BstdFile);
			if(ReadSize<=0)
			{
				if(ferror(ip))
				{
					kdDebug() << "madlib: read error on bit-stream" << endl;
					Status = false;
				}
				if(feof(ip))
					;
				break;
			}

			if(BstdFileEofP(BstdFile))
			{
				GuardPtr=ReadStart+ReadSize;
				memset(GuardPtr,0,MAD_BUFFER_GUARD);
				ReadSize+=MAD_BUFFER_GUARD;
			}

			mad_stream_buffer(&Stream,InputBuffer,ReadSize+Remaining);
			Stream.error=MAD_ERROR_NONE;
		}

		if(mad_header_decode(&Frame.header,&Stream)) {
			if(MAD_RECOVERABLE(Stream.error)) {
				//if(Stream.error!=MAD_ERROR_LOSTSYNC || Stream.this_frame!=GuardPtr)
					//kdDebug() << "madlib: recoverable frame level error " << MadErrorString(&Stream) << endl;
				continue;
			}
			else
				if(Stream.error==MAD_ERROR_BUFLEN)
					continue;
				else
				{
					kdDebug() << "madlib: unrecoverable frame level error " << MadErrorString(&Stream) << endl;;
					Status = false;
					break;
				}
		}

		if(FrameCount==0)
			if(PrintFrameInfo(stderr,&Frame.header,first_header)) {
				Status = false;
				break;
			}

		compare_headers(&Frame.header);

		FrameCount++;
		mad_timer_add(&Timer,Frame.header.duration);
	} while(1);

	/* The input file was completely read; the memory allocated by our
	 * reading module must be reclaimed.
	 */
	BstdFileDestroy(BstdFile);
	mad_synth_finish(&Synth);
	mad_frame_finish(&Frame);
	mad_stream_finish(&Stream);


	const char *md;
	switch(first_header->mode)
	{
		case MAD_MODE_SINGLE_CHANNEL:
			md = "single channel";
			break;
		case MAD_MODE_DUAL_CHANNEL:
			md = "dual channel";
			break;
		case MAD_MODE_JOINT_STEREO:
			md = "joint (MS/intensity) stereo";
			break;
		case MAD_MODE_STEREO:
			md = "normal LR stereo";
			break;
		default:
			md = "(unexpected mode value)";
			break;
	}

	// compare_headers 
	int quality = (int) ((float) equal_frames * 100 / (float) FrameCount);
	if (quality < 90 ) { //  ----->  90% equal frames
		Status = false; 
		kdDebug() << "madlib:  mp3file seems to be faulty: " << quality << endl;
	}

	if (Status == false ) return false;
	else {
		/* Accounting report and return values if no error occurred. */
		char Buffer[80];
		mad_timer_string(Timer,Buffer,"%lu:%02lu.%03u",
			MAD_UNITS_MINUTES,MAD_UNITS_MILLISECONDS,0);
		kdDebug() << "madlib: " << FrameCount << " frames  Playtime: "
			<< Buffer << "  Mode: " << md << "   Bitrate: " <<
			first_header->bitrate/1000 << " KBps" << endl;

		if ( MAD_MODE_SINGLE_CHANNEL == first_header->mode) channels = 1;
		else channels = 2;

		b_rate      = first_header->bitrate/1000;
		s_rate      = first_header->samplerate;
		length_ms   = mad_timer_count( Timer, MAD_UNITS_MILLISECONDS );
		return true;
	}
}
