
/* 
 * This file is part of the K3b project.
 * Copyright (C) 1998-2004 Sebastian Trueg <trueg@k3b.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * See the file "COPYING" for the exact licensing terms.
 */



#include "myprocess.h"

#include <qstringlist.h>
#include <qsocketnotifier.h>

#include <kdebug.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>



class MyProcess::Private
{
public:
  QString unfinishedStdoutLine;
  QString unfinishedStderrLine;

  int dupStdoutFd;
  int dupStdinFd;

  bool rawStdin;
  bool rawStdout;

  int in[2];
  int out[2];
};


MyProcess::MyProcess()
  : KProcess(),
    m_bSplitStdout(false),
    m_suppressEmptyLines(true)
{
  d = new Private();
  d->dupStdinFd = d->dupStdoutFd = -1;
  d->rawStdout = d->rawStdin = false;
  d->in[0] = d->in[1] = -1;
  d->out[0] = d->out[1] = -1;
}

MyProcess::~MyProcess()
{
  delete d;
}


MyProcess& MyProcess::operator<<( const QString& arg )
{
  static_cast<KProcess*>(this)->operator<<( arg );
  return *this;
}

MyProcess& MyProcess::operator<<( const char* arg )
{
  static_cast<KProcess*>(this)->operator<<( arg );
  return *this;
}

MyProcess& MyProcess::operator<<( const QCString& arg )
{
  static_cast<KProcess*>(this)->operator<<( arg );
  return *this;
}

MyProcess& MyProcess::operator<<( const QStringList& args )
{
  static_cast<KProcess*>(this)->operator<<( args );
  return *this;
}


bool MyProcess::start( RunMode run, Communication com )
{
  if( com & Stderr ) {
    connect( this, SIGNAL(receivedStderr(KProcess*, char*, int)),
	     this, SLOT(slotSplitStderr(KProcess*, char*, int)) );
  }
  if( com & Stdout ) {
    connect( this, SIGNAL(receivedStdout(KProcess*, char*, int)),
	     this, SLOT(slotSplitStdout(KProcess*, char*, int)) );
  }

  return KProcess::start( run, com );
}


void MyProcess::slotSplitStdout( KProcess*, char* data, int len )
{
  if( m_bSplitStdout )
    splitOutput( data, len, true );
}


void MyProcess::slotSplitStderr( KProcess*, char* data, int len )
{
  splitOutput( data, len, false );
}


void MyProcess::splitOutput( char* data, int len, bool stdout )
{
  //
  // The stderr splitting is mainly used for parsing of messages
  // That's why we simplify the data before proceeding
  //

  QString buffer;
  for( int i = 0; i < len; i++ ) {
    if( data[i] == '\b' ) {
      while( data[i] == '\b' )  // we replace multible backspaces with a single line feed
	i++;
      buffer += '\n';
    }
    if( data[i] == '\r' )
      buffer += '\n';
    else if( data[i] == '\t' )  // replace tabs with a single space
      buffer += " ";
    else
      buffer += data[i];
  }

  QStringList lines = QStringList::split( '\n', buffer, !m_suppressEmptyLines );

  // in case we suppress empty lines we need to handle the following separately
  // to make sure we join unfinished lines correctly
  if( m_suppressEmptyLines && buffer[0] == '\n' )
    lines.prepend( QString::null );

  QString* unfinishedLine = ( stdout ? &d->unfinishedStdoutLine : &d->unfinishedStderrLine );

  if( !unfinishedLine->isEmpty() ) {
    lines.first().prepend( *unfinishedLine );
    *unfinishedLine = "";

    //kdDebug() << "(MyProcess)           joined line: '" << (lines.first()) << "'" << endl;
  }

  QStringList::iterator it;

  // check if line ends with a newline
  // if not save the last line because it is not finished
  QChar c = buffer.right(1).at(0);
  bool hasUnfinishedLine = ( c != '\n' && c != '\r' && c != QChar(46) );  // What is unicode 46?? It is printed as a point
  if( hasUnfinishedLine ) {
    //kdDebug() << "(MyProcess) found unfinished line: '" << lines.last() << "'" << endl;
    //kdDebug() << "(MyProcess)             last char: '" << buffer.right(1) << "'" << endl;
    *unfinishedLine = lines.last();
    it = lines.end();
    --it;
    lines.remove(it);
  }

  for( it = lines.begin(); it != lines.end(); ++it ) {
    QString& str = *it;

    // just to be sure since something above does not do this right
    if( str.isEmpty() )
      continue;

    if( stdout )
      emit stdoutLine( str );
    else
      emit stderrLine( str );
  }
}


int MyProcess::setupCommunication( Communication comm )
{
  if( KProcess::setupCommunication( comm ) ) {

    //
    // Setup our own socketpair
    //

    if( d->rawStdin || d->dupStdinFd ) {
      if( socketpair(AF_UNIX, SOCK_STREAM, 0, d->in) == 0 ) {
	fcntl(d->in[0], F_SETFD, FD_CLOEXEC);
	fcntl(d->in[1], F_SETFD, FD_CLOEXEC);
      }
      else
	return 0;
    }

    if( d->rawStdout || d->dupStdoutFd ) {
      if( socketpair(AF_UNIX, SOCK_STREAM, 0, d->out) == 0 ) {
	fcntl(d->out[0], F_SETFD, FD_CLOEXEC);
	fcntl(d->out[1], F_SETFD, FD_CLOEXEC);
      }
      else {
	if( d->rawStdin || d->dupStdinFd ) {
	  close(d->in[0]);
	  close(d->in[1]);
	}
	return 0;
      }
    }

    return 1;
  }
  else
    return 0;
}


void MyProcess::commClose()
{
  if( d->rawStdin || d->dupStdinFd ) {
    close(d->in[1]);
    d->in[1] = -1;
  }
  if( d->rawStdout || d->dupStdoutFd ) {
    close(d->out[0]);
    d->out[0] = -1;
  }

  KProcess::commClose();
}


int MyProcess::commSetupDoneP()
{
  int ok = KProcess::commSetupDoneP();

  if( d->rawStdin || d->dupStdinFd )
    close(d->in[0]);
  if( d->rawStdout || d->dupStdoutFd )
    close(d->out[1]);
  d->in[0] = d->out[1] = -1;

  return ok;
}


int MyProcess::commSetupDoneC()
{
  int ok = KProcess::commSetupDoneC();

  if( d->dupStdoutFd != -1 ) {
    if( ::dup2( d->dupStdoutFd, STDOUT_FILENO ) < 0 ) {
      kdDebug() << "(MyProcess) Error while dup( " << d->dupStdoutFd << ", " << STDOUT_FILENO << endl;
      ok = 0;
    }
  }
  else if( d->rawStdout ) {
    if( ::dup2( d->out[1], STDOUT_FILENO ) < 0 ) {
      kdDebug() << "(MyProcess) Error while dup( " << d->out[1] << ", " << STDOUT_FILENO << endl;
      ok = 0;
    }
  }

  if( d->dupStdinFd != -1 ) {
    if( ::dup2( d->dupStdinFd, STDIN_FILENO ) < 0 ) {
      kdDebug() << "(MyProcess) Error while dup( " << d->dupStdinFd << ", " << STDIN_FILENO << endl;
      ok = 0;
    }
  }
  else if( d->rawStdin ) {
    if( ::dup2( d->in[0], STDIN_FILENO ) < 0 ) {
      kdDebug() << "(MyProcess) Error while dup( " << d->in[0] << ", " << STDIN_FILENO << endl;
      ok = 0;
    }
  }

  return ok;
}



int MyProcess::stdinFd() const
{
  if( d->rawStdin )
    return d->in[1];
  else if( d->dupStdinFd != -1 )
    return d->dupStdinFd;
  else
    return -1;
}

int MyProcess::stdoutFd() const
{
  if( d->rawStdout )
    return d->out[0];
  else if( d->dupStdoutFd != -1 )
    return d->dupStdoutFd;
  else
    return -1;
}


void MyProcess::dupStdout( int fd )
{
  writeToFd( fd );
}

void MyProcess::dupStdin( int fd )
{
  readFromFd( fd );
}


void MyProcess::writeToFd( int fd )
{
  d->dupStdoutFd = fd;
  if( fd != -1 )
    d->rawStdout = false;
}

void MyProcess::readFromFd( int fd )
{
  d->dupStdinFd = fd;
  if( fd != -1 )
    d->rawStdin = false;
}


void MyProcess::setRawStdin(bool b)
{
  if( b ) {
    d->rawStdin = true;
    d->dupStdinFd = -1;
  }
  else
    d->rawStdin = false;
}


void MyProcess::setRawStdout(bool b)
{
  if( b ) {
    d->rawStdout = true;
    d->dupStdoutFd = -1;
  }
  else
    d->rawStdout = false;
}


MyProcess::OutputCollector::OutputCollector( KProcess* p )
  : m_process(0)
{
  setProcess( p );
}

void MyProcess::OutputCollector::setProcess( KProcess* p )
{
  if( m_process )
    m_process->disconnect( this );

  m_process = p;
  if( p ) {
    connect( p, SIGNAL(receivedStdout(KProcess*, char*, int)), 
	     this, SLOT(slotGatherOutput(KProcess*, char*, int)) );
    connect( p, SIGNAL(receivedStderr(KProcess*, char*, int)), 
	     this, SLOT(slotGatherOutput(KProcess*, char*, int)) );
  }

  m_gatheredOutput = "";
}

void MyProcess::OutputCollector::slotGatherOutput( KProcess*, char* data, int len )
{
  m_gatheredOutput.append( QString::fromLatin1( data, len ) );
}
