
#include <config.h>

#include <mad.h>

#include <taglib/fileref.h>
#include <taglib/tag.h>
#include <taglib/tstring.h>
#include <taglib/audioproperties.h>
#include <taglib/mpegfile.h>
#include <taglib/vorbisfile.h>
#include <taglib/flacfile.h>
#include <taglib/flacproperties.h>   // flacfiles: sampleWidth() 

#include <qfileinfo.h>
#include <kdebug.h>
#include <kmimetype.h>

#include "song.h"
#include "tools.h"
#include "madlib.h"

#define BLOCK ( 100 * 1024) // 100 kb

// taglib erkennt keine mp3mono files.
// taglib erkennt öfters keine properties.

using namespace TagLib;

static const char mp3Type[]  = "audio/x-mp3";
static const char oggType[]  = "application/ogg";
static const char flacType[] = "audio/x-flac";

Song::Song(const QString &fileName, File *file, QString format) :

m_info(fileName),
m_fileName(fileName),
m_format(format)
{
	m_title    = TStringToQString(file->tag()->title()).stripWhiteSpace();
	m_artist   = TStringToQString(file->tag()->artist()).stripWhiteSpace();
	m_album    = TStringToQString(file->tag()->album()).stripWhiteSpace();
	m_genre    = TStringToQString(file->tag()->genre()).stripWhiteSpace();
	m_comment  = TStringToQString(file->tag()->comment()).stripWhiteSpace();
	m_track    = file->tag()->track();
	m_year     = file->tag()->year();
	m_seconds  = file->audioProperties()->length();
	m_bitrate  = file->audioProperties()->bitrate();
	m_length   = file->audioProperties()->length();
	m_samplerate = file->audioProperties()->sampleRate();
	m_channels = file->audioProperties()->channels();
	m_samplewidth = 0;   // only for flacfiles

	if(m_title.isEmpty())
	m_title = m_info.baseName(true);

	if (format == "mp3" || format == "ogg" || format == "flac")
		m_format = format;

	if(format == "flac")
		m_samplewidth = (static_cast<FLAC::Properties *> (file->audioProperties()))->sampleWidth();
}

Song::~Song() {}


Song *Song::createTag(const QString &fileName)
{
	/** Find sound file format: mp3, ogg or flac */
	KMimeType::Ptr result = KMimeType::findByFileContent(fileName);

	/** find binary files and check them by libmad */
	bool unix_file;
	Tools *t = new Tools();
	if(t->is_soundfile(fileName)) unix_file = true;
	else {
		kdDebug() << "Tools: Can't find audio header: " << fileName << endl;  
		unix_file = false;
	}

	if ((!result->is(oggType)) && (!result->is(flacType))) {   // mp3type

		MPEG::File File(QFile::encodeName(fileName).data(),true,AudioProperties::Accurate);
		if(!File.isValid()) {
			kdDebug() << "TagLib: no valid file" << endl;
			return NULL;
			}
		Song *s = new Song(fileName, &File, "mp3");

		if(!s) {
			kdDebug() << "TagLib: no tag found" << endl;
			return NULL;
		}

		bool taglib;
		if((s->m_length > 2) && (s->m_bitrate > 32)) taglib = true; 
		else {
			taglib = false;
			kdDebug() << "TagLib: no valid tag found" << endl;
		}

		/** to bad, give up */
		if((taglib == false) && (unix_file == false)) {
			//kdDebug() << "Give up" << endl;
			return NULL;
		}
		/** very good, continue */
		else if ((taglib == true) && (unix_file == true)) { 
			//kdDebug() << "taglib true and unix_file true" << endl;
			return s;

			}
		/** ask madlib */
		else {
			madlib *m = new madlib();

			long len;
			int r = m->MpegAudioDecoder(&fileName,len,s->m_bitrate,s->m_samplerate,s->m_channels);
			s->m_length = (len + 500) /1000;  // runden, ms auf s

			if(r != 0) return NULL;
			else {
				kdDebug() << "madlib: Bitrate: " << s->m_bitrate << " Channels: " << s->m_channels << " Samplerate: " <<  s->m_samplerate << " Playtime: " << s->m_length << endl << endl;;
				return s;
			}
		}
	}

	else if  (result->is(oggType))  {
		Vorbis::File file(QFile::encodeName(fileName).data(),true,AudioProperties::Accurate);

		if( !file.isValid() || unix_file == false ) return NULL;
		return new Song(fileName, &file, "ogg");
	}

	else if  (result->is(flacType))  {
		FLAC::File file(QFile::encodeName(fileName).data(),true,AudioProperties::Accurate);

		if( !file.isValid() || unix_file == false ) return NULL;
		Song *s = new Song(fileName, &file, "flac");

		// akode supports only flacfiles with (samplewidth == 16bit)
		// or something like that, please correct me if I'm wrong :-)
		// Reject other files 
		if(s->m_samplewidth == 16) return s; 
		else {
			kdDebug() << fileName << endl;
			kdDebug() << "TagLib: reject invalid flacfile: samplewidth: != 16 bit" << endl;
			return NULL;
		}
	}

return NULL;
}
