/***************************************************************************
 *   Copyright (C) 2005-2006 by Georg Hennig                               *
 *   Email: georg.hennig@web.de                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qfile.h>

#include <kapplication.h>
#include <kio/netaccess.h>
#include <klocale.h>
#include <kmdcodec.h>
#include <kmessagebox.h>
#include <ktempfile.h>

#include "komparejob.h"
#include "komparatorwidget.h"

#include "kfileitemext.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

KFileItemExt::KFileItemExt( const KIO::UDSEntry &_entry, const KURL &_url,
														bool _determineMimeTypeOnDemand, bool _urlIsDirectory ) :
	KFileItem( _entry, _url, _determineMimeTypeOnDemand, _urlIsDirectory )
{
	dir = 0;
	hasdupes_size = 0;
	hasdupes_path = 0;
	isdupe_size = 0;
	isdupe_path = 0;
	duplicates_size = NULL;
	duplicates_path = NULL;
	dup_size_parent = NULL;
	dup_path_parent = NULL;
	next = NULL;
	m_md5 = NULL;
	m_tmp_file = NULL;
}

KFileItemExt::KFileItemExt( mode_t _mode, mode_t _permissions,
														const KURL &_url, bool _determineMimeTypeOnDemand ) :
	KFileItem( _mode, _permissions, _url, _determineMimeTypeOnDemand )
{
	dir = 0;
	hasdupes_size = 0;
	hasdupes_path = 0;
	isdupe_size = 0;
	isdupe_path = 0;
	duplicates_size = NULL;
	duplicates_path = NULL;
	dup_size_parent = NULL;
	dup_path_parent = NULL;
	next = NULL;
	m_md5 = NULL;
	m_tmp_file = NULL;
}

KFileItemExt::KFileItemExt( const KURL &url, const QString &mimeType, mode_t mode ) :
	KFileItem( url, mimeType, mode )
{
	dir = 0;
	hasdupes_size = 0;
	hasdupes_path = 0;
	isdupe_size = 0;
	isdupe_path = 0;
	duplicates_size = NULL;
	duplicates_path = NULL;
	dup_size_parent = NULL;
	dup_path_parent = NULL;
	next = NULL;
	m_md5 = NULL;
	m_tmp_file = NULL;
}

KFileItemExt::KFileItemExt( const KFileItem &item ) :
	KFileItem( item )
{
	dir = 0;
	hasdupes_size = 0;
	hasdupes_path = 0;
	isdupe_size = 0;
	isdupe_path = 0;
	duplicates_size = NULL;
	duplicates_path = NULL;
	dup_size_parent = NULL;
	dup_path_parent = NULL;
	next = NULL;
	m_md5 = NULL;
	m_tmp_file = NULL;
}

KFileItemExt::KFileItemExt( const KFileItemExt &item ) :
	KFileItem( item )
{
	dir = item.dir;
	hasdupes_size = item.hasdupes_size;
	hasdupes_path = item.hasdupes_path;
	isdupe_size = item.isdupe_size;
	isdupe_path = item.isdupe_path;
	duplicates_size = item.duplicates_size;
	duplicates_path = item.duplicates_path;
	dup_size_parent = item.dup_size_parent;
	dup_path_parent = item.dup_path_parent;
	next = item.next;
	m_md5 = item.m_md5;
	m_tmp_file = item.m_tmp_file;
}

KFileItemExt::~KFileItemExt()
{
	delete m_md5;
	m_md5=NULL;

	// We don't need to delete the duplicates, as duplicates also appera in "next".
	delete next;
	next=NULL;
}

void KFileItemExt::emitMessage( const QString &message, const QString &title, KomparatorWidget *parent )
{
	QString *new_message = new QString( message );
	QString *new_title = new QString( title );

	QPair< QString *, QString * > *resultpair = new QPair< QString *, QString * >;

	*resultpair = qMakePair( new_message, new_title );

	QCustomEvent *event = new QCustomEvent( MESSAGE_ERROR );
	event->setData( resultpair );
	QApplication::postEvent( parent, event );
}

QCString KFileItemExt::MD5( KomparatorWidget *parent, bool force_recalculate )
{
	if ( force_recalculate )
	{
		delete m_md5;
		m_md5 = NULL;
	}

	if ( !m_md5 )
	{
		const int BUFFER_SIZE = 512*1024;
		char buffer[BUFFER_SIZE];
		int read_data;

		if ( isLocalFile() )
		{
			QFile file;
			file.setName( url().directory( false )+url().fileName() );
			if ( !file.open( IO_Raw | IO_ReadOnly ) )
			{
				emitMessage( i18n( "Error getting md5 sum of local file %1." ).arg( url().directory( false )+url().fileName() ),
					i18n( "Error while calculating md5 checksum" ), parent );
				return QCString( "d41d8cd98f00b204e9800998ecf8427e" ); // checksum of empty file.
			}

			m_md5 = new KMD5();
			while ( ( read_data = file.readBlock( buffer, BUFFER_SIZE ) ) > 0 )
			{
				m_md5->update( buffer, read_data );

				if ( parent->kompare->cancel() ) break; // m_md5 sum will be invalid, but cancel will delete it anyway.
			}

			file.close();
		}
		else
		{
			m_tmp_file = NULL;
			m_download_error = -1;

			m_mutex.lock();

			QCustomEvent *event = new QCustomEvent( DOWNLOAD_FILE );
			event->setData( this );
			QApplication::postEvent( parent, event );

			while ( m_download_error == -1 )
			{
				m_mutex.unlock();
				usleep( 50 );
				m_mutex.lock();
			}
			usleep( 50 );

			m_mutex.unlock();

			if ( parent->kompare->cancel() )
			{
				m_tmp_file->close();
				m_tmp_file->unlink();

				QCustomEvent *event2 = new QCustomEvent( DELETE_FILE );
				event2->setData( m_tmp_file );
				QApplication::postEvent( parent, event2 );

				m_tmp_file = NULL;

				return QCString( "d41d8cd98f00b204e9800998ecf8427e" ); // checksum of empty file.
			}

			m_md5 = new KMD5();

			if ( m_download_error == 0 ) // everything went well.
			{
				QFile file( m_tmp_file->name() );
				m_tmp_file->close();
				if ( !file.open( IO_Raw | IO_ReadOnly ) )
				{
					emitMessage( i18n( "Opening temporary file %1 failed. MD5 checksum won't be correct." ).arg( m_tmp_file->name() ),
						i18n( "Error while calculating md5 checksum" ), parent );
				}
				else
				{
					while ( ( read_data = file.readBlock( buffer, BUFFER_SIZE ) ) > 0 )
					{
						m_md5->update( buffer, read_data );

						if ( parent->kompare->cancel() ) break; // m_md5 sum will be invalid, but cancel will delete it anyway.
					}

					file.close();
				}
			}
			else // download errors are handled by komparatorwidget.
			{
				m_tmp_file->close();
			}
			m_tmp_file->unlink();

			QCustomEvent *event2 = new QCustomEvent( DELETE_FILE );
			event2->setData( m_tmp_file );
			QApplication::postEvent( parent, event2 );

			m_tmp_file = NULL;
		}
	}

	return m_md5->hexDigest();
}

QCString KFileItemExt::MD5( bool force_recalculate )
{
	if ( force_recalculate )
	{
		delete m_md5;
		m_md5 = NULL;
	}

	if ( !m_md5 )
	{
		const int BUFFER_SIZE = 512*1024;
		char buffer[BUFFER_SIZE];
		int read_data;

		if ( isLocalFile() )
		{
			QFile file;
			file.setName( url().directory( false )+url().fileName() );
			if ( !file.open( IO_Raw | IO_ReadOnly ) )
			{
				KMessageBox::error( 0,
					i18n( "Error getting md5 sum of local file %1." ).arg( url().directory( false )+url().fileName() ),
					i18n( "Error while calculating md5 sum" ) );
				return QCString( "d41d8cd98f00b204e9800998ecf8427e" ); // checksum of empty file.
			}

			m_md5 = new KMD5();
			while ( ( read_data = file.readBlock( buffer, BUFFER_SIZE ) ) > 0 )
			{
				m_md5->update( buffer, read_data );
			}

			file.close();
		}
		else
		{
			QString m_tmp_file_str;

			m_md5 = new KMD5();

			if ( KIO::NetAccess::download( this->url(), m_tmp_file_str, NULL ) ) // everything went well.
			{
				QFile file( m_tmp_file_str );
				if ( !file.open( IO_Raw | IO_ReadOnly ) )
				{
					KMessageBox::sorry( NULL, i18n( "Opening temporary file %1 failed. MD5 checksum won't be correct." ).arg( m_tmp_file_str ) );
				}
				else
				{
					while ( ( read_data = file.readBlock( buffer, BUFFER_SIZE ) ) > 0 )
					{
						m_md5->update( buffer, read_data );
					}

					file.close();
				}
			}
			else // download errors are handled by komparatorwidget.
			{
				KMessageBox::sorry( NULL, i18n( "Download of %1 to %2 failed." ).arg( this->url().url() ).arg( m_tmp_file_str ) );
			}

			KIO::NetAccess::removeTempFile( m_tmp_file_str );
		}
	}

	return m_md5->hexDigest();
}

QString KFileItemExt::getFile( KomparatorWidget *parent )
{
	QString ret = "";

	if ( isLocalFile() )
	{
		ret = url().directory( false ) + url().fileName();
	}
	else
	{
		m_tmp_file = NULL;
		m_download_error = -1;

		m_mutex.lock();

		QCustomEvent *event = new QCustomEvent( DOWNLOAD_FILE );
		event->setData( this );
		QApplication::postEvent( parent, event );

		while ( m_download_error == -1 ) // wait until the KIO::Job has finished (in komparatorwidget).
		{
			m_mutex.unlock();
			usleep( 50 );
			m_mutex.lock();
		}
		usleep( 50 );

		m_mutex.unlock();

		if ( parent->kompare->cancel() )
		{
			if ( m_tmp_file )
			{
				m_tmp_file->close();

				QCustomEvent *event2 = new QCustomEvent( DELETE_FILE );
				event2->setData( m_tmp_file );
				QApplication::postEvent( parent, event2 );
			}

			m_tmp_file = NULL;

			return "";
		}

		if ( m_download_error == 0 ) // everything went well. otherwise will fail in komparejob.
		{
			if ( m_tmp_file ) ret = m_tmp_file->name();
		}

		if ( m_tmp_file ) m_tmp_file->close();
	}

	return ret;
}

void KFileItemExt::deleteFile( KomparatorWidget *parent )
{
	if ( m_tmp_file )
	{
		QCustomEvent *event = new QCustomEvent( DELETE_FILE );
		event->setData( m_tmp_file );
		QApplication::postEvent( parent, event );

		m_tmp_file = NULL;
	}
}

void KFileItemExt::setTmpFile( KTempFile *_tmp_file )
{
	m_mutex.lock();
	m_tmp_file = _tmp_file;
	m_mutex.unlock();
}

void KFileItemExt::setDownloadError( int _download_error )
{
	m_mutex.lock();
	m_download_error = _download_error;
	m_mutex.unlock();
}
