/***************************************************************************
 *   Copyright (C) 2005-2006 by Georg Hennig                               *
 *   Email: georg.hennig@web.de                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qfileinfo.h>
#include <qgroupbox.h>
#include <qimage.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qstylesheet.h>
#include <qtooltip.h>
#include <qvbox.h>
#include <qwhatsthis.h>

#include <kiconloader.h>
#include <kio/previewjob.h>
#include <klocale.h>
#include <kprocess.h>
#include <kseparator.h>
#include <ksqueezedtextlabel.h>

#include "kfileitemext.h"
#include "komparatorwidget.h"

#include "komparedialog.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

KompareDialog::KompareDialog( QWidget *_parent, KFileItemExt *_item1, KFileItemExt *_item2,
	const QString &url1, const QString &url2, bool _md5 ) :
		KDialogBase( _parent, "komparedialog", true/*modal*/, i18n( "File compare dialog" ),
			KDialogBase::Ok | KDialogBase::Cancel | KDialogBase::User1, KDialogBase::Cancel ),
		m_parent( _parent ), m_has_kdiff3( false ), m_md5( _md5 ), m_item1( _item1 ), m_item2( _item2 ), m_result( CANCEL )
{
	m_kompare_proc = new KProcess( this );

	enableButtonSeparator( true );
	showButton( KDialogBase::User1, false );
	setButtonText( KDialogBase::User1, i18n( "Compare with KDiff3" ) );
	setButtonTip( KDialogBase::User1, i18n( "Compare the two text files with KDiff3" ) );
	setButtonWhatsThis( KDialogBase::User1, i18n( "Compare the two text files with KDiff3" ) );

	if ( isTextFile( m_item1 ) && isTextFile( m_item2 ) )
	{
		*m_kompare_proc << "whereis" << "-b"/*binaries only*/ << "kdiff3";
		connect( m_kompare_proc, SIGNAL( receivedStdout( KProcess *, char *, int ) ), this, SLOT( slotReceivedStdout( KProcess *, char *, int ) ) );
		m_kompare_proc->start( KProcess::NotifyOnExit, KProcess::AllOutput );
		connect( m_kompare_proc, SIGNAL( processExited( KProcess * ) ), this, SLOT( slotProcessExited( KProcess * ) ) );
	}

	QHBox *main_h_box = makeHBoxMainWidget();
	QVBox *left_v_box = new QVBox( main_h_box );
	left_v_box->setSpacing( 5 );
	QVBox *center_v_box = new QVBox( main_h_box );
	center_v_box->setSpacing( 5 );
	QVBox *right_v_box = new QVBox( main_h_box );
	right_v_box->setSpacing( 5 );

  createTitle( left_v_box, m_item1 );
	new KSeparator( Qt::Horizontal, left_v_box );
	QWidget *label_column1 = createBasicInfos( left_v_box, m_item1, url1 );
	new KSeparator( Qt::Horizontal, left_v_box );
	QWidget *label_column2 = createExtendedInfos( left_v_box, m_item1 );
	if ( label_column1->width() > label_column2->width() )
	{
		label_column1->setFixedWidth( label_column1->width() );
		label_column2->setFixedWidth( label_column1->width() );
	}
	else
	{
		label_column1->setFixedWidth( label_column2->width() );
		label_column2->setFixedWidth( label_column2->width() );
	}
	new KSeparator( Qt::Horizontal, left_v_box );
	createMetaInfos( left_v_box, m_item1 );

	QSize small_size = QIconSet::iconSize( QIconSet::Small ); // save original icon set size, because we'll change them.

	QImage transparent = ((KomparatorWidget*)m_parent)->transparentImage();

	new QLabel( "", center_v_box );
	m_copy_l_r = new QPushButton( center_v_box );
	m_copy_l_r->setIconSet( ((KomparatorWidget*)m_parent)->broadIconSet( SmallIcon( "editcopy" ), transparent, SmallIcon( "2rightarrow" ) ) );
	m_copy_l_r->setMaximumWidth( small_size.width()*3 );
	m_copy_l_r->setToggleButton( true );
	connect( m_copy_l_r, SIGNAL( toggled( bool ) ), this, SLOT( slotCopyLeftRight( bool ) ) );
	QToolTip::add( m_copy_l_r, i18n( "Copy the file on the left to the right side" ) );
	QWhatsThis::add( m_copy_l_r, i18n( "Copy the file on the left to the right side" ) );

	m_move_l_r = new QPushButton( center_v_box );
	m_move_l_r->setIconSet( ((KomparatorWidget*)m_parent)->broadIconSet( SmallIcon( "goto" ), transparent, SmallIcon( "2rightarrow" ) ) );
	m_move_l_r->setMaximumWidth( small_size.width()*3 );
	m_move_l_r->setToggleButton( true );
	connect( m_move_l_r, SIGNAL( toggled( bool ) ), this, SLOT( slotMoveLeftRight( bool ) ) );
	QToolTip::add( m_move_l_r, i18n( "Move the file on the left to the right side" ) );
	QWhatsThis::add( m_move_l_r, i18n( "Move the file on the left to the right side" ) );

	m_trash_l = new QPushButton( center_v_box );
	m_trash_l->setIconSet( ((KomparatorWidget*)m_parent)->broadIconSet( SmallIcon( "1rightarrow" ), transparent, SmallIcon( "edittrash" ) ) );
	m_trash_l->setMaximumWidth( small_size.width()*3 );
	m_trash_l->setToggleButton( true );
	connect( m_trash_l, SIGNAL( toggled( bool ) ), this, SLOT( slotTrashLeft( bool ) ) );
	QToolTip::add( m_trash_l, i18n( "Trash the file on the left" ) );
	QWhatsThis::add( m_trash_l, i18n( "Trash the file on the left" ) );

	m_delete_l = new QPushButton( center_v_box );
	m_delete_l->setIconSet( ((KomparatorWidget*)m_parent)->broadIconSet( SmallIcon( "1rightarrow" ), transparent, SmallIcon( "editdelete" ) ) );
	m_delete_l->setMaximumWidth( small_size.width()*3 );
	m_delete_l->setToggleButton( true );
	connect( m_delete_l, SIGNAL( toggled( bool ) ), this, SLOT( slotDeleteLeft( bool ) ) );
	QToolTip::add( m_delete_l, i18n( "Delete the file on the left" ) );
	QWhatsThis::add( m_delete_l, i18n( "Delete the file on the left" ) );
	
	QLabel *line = new QLabel( "", center_v_box );
	line->setFrameShape( QFrame::HLine );

	m_copy_r_l = new QPushButton( center_v_box );
	m_copy_r_l->setIconSet( ((KomparatorWidget*)m_parent)->broadIconSet( SmallIcon( "2leftarrow" ), transparent, SmallIcon( "editcopy" ) ) );
	m_copy_r_l->setMaximumWidth( small_size.width()*3 );
	m_copy_r_l->setToggleButton( true );
	connect( m_copy_r_l, SIGNAL( toggled( bool ) ), this, SLOT( slotCopyRightLeft( bool ) ) );
	QToolTip::add( m_copy_r_l, i18n( "Copy the file on the right to the left side" ) );
	QWhatsThis::add( m_copy_r_l, i18n( "Copy the file on the right to the left side" ) );

	m_move_r_l = new QPushButton( center_v_box );
	m_move_r_l->setIconSet( ((KomparatorWidget*)m_parent)->broadIconSet( SmallIcon( "2leftarrow" ), transparent, SmallIcon( "goto" ) ) );
	m_move_r_l->setMaximumWidth( small_size.width()*3 );
	m_move_r_l->setToggleButton( true );
	connect( m_move_r_l, SIGNAL( toggled( bool ) ), this, SLOT( slotMoveRightLeft( bool ) ) );
	QToolTip::add( m_move_r_l, i18n( "Move the file on the right to the left side" ) );
	QWhatsThis::add( m_move_r_l, i18n( "Move the file on the right to the left side" ) );

	m_trash_r = new QPushButton( center_v_box );
	m_trash_r->setIconSet( ((KomparatorWidget*)m_parent)->broadIconSet( SmallIcon( "edittrash" ), transparent, SmallIcon( "1leftarrow" ) ) );
	m_trash_r->setMaximumWidth( small_size.width()*3 );
	m_trash_r->setToggleButton( true );
	connect( m_trash_r, SIGNAL( toggled( bool ) ), this, SLOT( slotTrashRight( bool ) ) );
	QToolTip::add( m_trash_r, i18n( "Trash the file on the right" ) );
	QWhatsThis::add( m_trash_r, i18n( "Trash the file on the right" ) );

	m_delete_r = new QPushButton( center_v_box );
	m_delete_r->setIconSet( ((KomparatorWidget*)m_parent)->broadIconSet( SmallIcon( "editdelete" ), transparent, SmallIcon( "1leftarrow" ) ) );
	m_delete_r->setMaximumWidth( small_size.width()*3 );
	m_delete_r->setToggleButton( true );
	connect( m_delete_r, SIGNAL( toggled( bool ) ), this, SLOT( slotDeleteRight( bool ) ) );
	QToolTip::add( m_delete_r, i18n( "Delete the file on the right" ) );
	QWhatsThis::add( m_delete_r, i18n( "Delete the file on the right" ) );

	new QLabel( "", center_v_box );

	QIconSet::setIconSize( QIconSet::Small, small_size ); // reset icon set to original size.

	createTitle( right_v_box, m_item2 );
	new KSeparator( Qt::Horizontal, right_v_box );
	label_column1 = createBasicInfos( right_v_box, m_item2, url2 );
	new KSeparator( Qt::Horizontal, right_v_box );
	label_column2 = createExtendedInfos( right_v_box, m_item2 );
	if ( label_column1->width() > label_column2->width() )
	{
		label_column1->setFixedWidth( label_column1->width() );
		label_column2->setFixedWidth( label_column1->width() );
		
	}
	else
	{
		label_column1->setFixedWidth( label_column2->width() );
		label_column2->setFixedWidth( label_column2->width() );
	}
	new KSeparator( Qt::Horizontal, right_v_box );
	createMetaInfos( right_v_box, m_item2 );

	KFileItemList item_list;
	item_list.append( m_item1 );
	item_list.append( m_item2 );
	KIO::PreviewJob *preview_job = new KIO::PreviewJob( item_list, 150/*width*/, 150/*height*/,
		48/*iconSize*/, 0/*iconAlpha*/, false/*scale*/, true/*save*/, NULL/*enabledPlugins*/ );
	connect( preview_job, SIGNAL( gotPreview( const KFileItem *, const QPixmap & ) ), this, SLOT( slotPreview( const KFileItem *, const QPixmap & ) ) );

	if ( minimumSizeHint().width() > m_parent->width() ) resize( m_parent->width(), minimumSizeHint().height() );
	if ( minimumSizeHint().height() > m_parent->height() ) resize( width(), m_parent->height() );
}

KompareDialog::~KompareDialog()
{
}

int KompareDialog::getResult()
{
	return m_result;
}

void KompareDialog::createTitle( QWidget *local_parent, KFileItemExt *item )
{
	QHBox *title_box = new QHBox( local_parent );
	QLabel *title_pix = new QLabel( title_box );
	title_pix->setPixmap( item->pixmap( 0 ) );
	KSqueezedTextLabel *title = new KSqueezedTextLabel( KIO::decodeFileName( item->url().fileName() ), title_box );
	title->setAlignment( Qt::AlignHCenter );
}

QWidget *KompareDialog::createBasicInfos( QWidget *local_parent, KFileItemExt *item, const QString &url )
{
	QHBox *info_box = new QHBox( local_parent );
	QVBox *names_box = new QVBox( info_box );
	names_box->setSpacing( 3 );
	new QLabel( i18n( "Path:" ), names_box );
	new QLabel( i18n( "Size:" ), names_box );
	new QLabel( i18n( "Modified:" ), names_box );
	new QLabel( i18n( "Accessed:" ), names_box );
	new QLabel( i18n( "Owner:" ), names_box );
	new QLabel( i18n( "Group:" ), names_box );
	new QLabel( i18n( "Permissions:" ), names_box );
	QVBox *entries_box = new QVBox( info_box );
	entries_box->setSpacing( 3 );
	new KSqueezedTextLabel( url, entries_box );
	QString size1_str = KIO::convertSize( item->size() );
	if ( item->size() >= 1024 ) size1_str += " (" + KGlobal::locale()->formatNumber( item->size(), 0 ) + " B)";
	new QLabel( size1_str, entries_box );
	QDateTime dt1;
	dt1.setTime_t( item->time( KIO::UDS_MODIFICATION_TIME ) );
	new QLabel( KGlobal::locale()->formatDateTime( dt1 ), entries_box );
	dt1.setTime_t( item->time( KIO::UDS_ACCESS_TIME ) );
	new QLabel( KGlobal::locale()->formatDateTime( dt1 ), entries_box );
	new QLabel( item->user(), entries_box );
	new QLabel( item->group(), entries_box );
	new QLabel( item->permissionsString(), entries_box );
	return names_box;
}

QWidget *KompareDialog::createExtendedInfos( QWidget *local_parent, KFileItemExt *item )
{
	QString magicMimeComment  = QString::null;
	if ( item->isLocalFile() )
	{
		KMimeType::Ptr magicMimeType = KMimeType::findByFileContent( item->url().path() );
		if ( magicMimeType->name() != KMimeType::defaultMimeType() ) magicMimeComment = magicMimeType->comment();
	}

	QHBox *info_box = new QHBox( local_parent );
	QVBox *names_box = new QVBox( info_box );
	names_box->setSpacing( 3 );
	if ( item->isLink() ) new QLabel( i18n( "Points to:" ), names_box );
	new QLabel( i18n( "Type:" ), names_box );
	if ( !magicMimeComment.isEmpty() && magicMimeComment != item->mimeComment() ) new QLabel( i18n( "Contents:" ), names_box );
	new QLabel( i18n( "MD5 checksum:" ), names_box );
	QVBox *entries_box = new QVBox( info_box );
	if ( item->isLink() ) new KSqueezedTextLabel( item->linkDest(), entries_box );
	new QLabel( item->mimeComment(), entries_box );
	if ( !magicMimeComment.isEmpty() && magicMimeComment != item->mimeComment() ) new QLabel( magicMimeComment, entries_box );
	if ( m_md5 )
	{
		m_md5_label1 = new KSqueezedTextLabel( item->MD5(), entries_box );
	}
	else
	{
		QHBox *md5_box = new QHBox( entries_box );
		if ( item == m_item1 )
		{
			m_md5_button1 = new QPushButton( i18n( "Calculate" ), md5_box );
			connect( m_md5_button1, SIGNAL( clicked() ), SLOT( slotMD5_1() ) );
			m_md5_label1 = new KSqueezedTextLabel( md5_box );
			m_md5_label1->setMaximumWidth( 0 );
		}
		else
		{
			m_md5_button2 = new QPushButton( i18n( "Calculate" ), md5_box );
			connect( m_md5_button2, SIGNAL( clicked() ), SLOT( slotMD5_2() ) );
			m_md5_label2 = new KSqueezedTextLabel( md5_box );
			m_md5_label2->setMaximumWidth( 0 );
		}
	}

	return names_box;
}

void KompareDialog::slotMD5_1()
{
	m_md5_label1->setText( m_item1->MD5() );
	m_md5_label1->setMaximumWidth( 32767 );
	delete m_md5_button1;
	m_md5_button1 = NULL;
}

void KompareDialog::slotMD5_2()
{
	m_md5_label2->setText( m_item2->MD5() );
	m_md5_label2->setMaximumWidth( 32767 );
	delete m_md5_button2;
	m_md5_button2 = NULL;
}

void KompareDialog::createMetaInfos( QWidget *local_parent, KFileItemExt *item )
{
	QScrollView *view = new QScrollView( local_parent );
	view->setSizePolicy( QSizePolicy::Expanding, QSizePolicy::Ignored );
	view->setFrameShape( QFrame::NoFrame );
	view->setResizePolicy( QScrollView::AutoOneFit );
	QVBox *view_box = new QVBox( view->viewport() );
	view_box->setSpacing( 5 );
	view_box->setMargin( 5 );
	view->addChild( view_box );
	
	if ( item == m_item1 )
	{
		m_preview_label1 = new QLabel( view_box );
		m_preview_label1->setMaximumWidth( 0 );
		m_preview_label1->setMaximumHeight( 0 );
		m_preview_label1->setAlignment( Qt::AlignCenter );
	}
	else
	{
		m_preview_label2 = new QLabel( view_box );
		m_preview_label2->setMaximumWidth( 0 );
		m_preview_label2->setMaximumHeight( 0 );
		m_preview_label2->setAlignment( Qt::AlignCenter );
	}

	QFileInfo file_info( item->url().path() );
	KFileMetaInfo m_info = item->metaInfo();
	if ( !m_info.isValid() )
	{
		m_info = KFileMetaInfo( item->url().path( -1 ) );
		item->setMetaInfo( m_info );
	}
	
	if ( !m_info.isValid() || ( m_info.preferredKeys() ).isEmpty() ) return;

	KFileMetaInfoProvider* prov = KFileMetaInfoProvider::self();
	QStringList groupList = m_info.preferredGroups();

	const KFileMimeTypeInfo* mtinfo = prov->mimeTypeInfo( m_info.mimeType() );
	if ( !mtinfo ) return;

	for ( QStringList::Iterator git=groupList.begin(); git!=groupList.end(); ++git )
	{
		QStringList itemList = m_info.group(*git).preferredKeys();
		if (itemList.isEmpty()) continue;

		QGroupBox *group_box = new QGroupBox( 2, Qt::Horizontal, QStyleSheet::escape( mtinfo->groupInfo(*git)->translatedName() ), view_box );

		for ( QStringList::Iterator iit = itemList.begin(); iit!=itemList.end(); ++iit )
		{
			KFileMetaInfoItem item = m_info[*git][*iit];
			if ( !item.isValid() ) continue;

			QLabel *l = new QLabel( item.translatedKey() + ":", group_box );
			l->setAlignment( Qt::AlignLeft | Qt::AlignTop );
			new QLabel( item.string(), group_box );
		}
	}
}

void KompareDialog::slotPreview( const KFileItem *item, const QPixmap &pixmap )
{
	if ( item == m_item1 )
	{
		m_preview_label1->setMaximumWidth( 32767 );
		m_preview_label1->setMaximumHeight( 32767 );
		m_preview_label1->setPixmap( pixmap );
	}
	else
	{
		m_preview_label2->setMaximumWidth( 32767 );
		m_preview_label2->setMaximumHeight( 32767 );
		m_preview_label2->setPixmap( pixmap );
	}
}

void KompareDialog::slotOk()
{
	if ( m_copy_l_r->isOn() ) m_result |= COPY_LEFT_TO_RIGHT;
	if ( m_move_l_r->isOn() ) m_result |= MOVE_LEFT_TO_RIGHT;
	if ( m_trash_l->isOn() )  m_result |= TRASH_LEFT;
	if ( m_delete_l->isOn() ) m_result |= DELETE_LEFT;
	if ( m_copy_r_l->isOn() ) m_result |= COPY_RIGHT_TO_LEFT;
	if ( m_move_r_l->isOn() ) m_result |= MOVE_RIGHT_TO_LEFT;
	if ( m_trash_r->isOn() )  m_result |= TRASH_RIGHT;
	if ( m_delete_r->isOn() ) m_result |= DELETE_RIGHT;
	
	KDialogBase::slotOk();
}

void KompareDialog::slotCancel()
{
	KDialogBase::slotCancel();
}

void KompareDialog::slotUser1()
{
	if ( m_kompare_proc->isRunning() ) return;
	m_kompare_proc->clearArguments();
	*m_kompare_proc << "kdiff3" << m_item1->url().url() << m_item2->url().url();

	if ( !( m_result & DIRTY ) ) // Only calculate md5 sum if files aren't dirty.
	{
		m_item1->MD5(); // calculate md5 sum now, so we can check, whether user modified something with kdiff3.
		m_item2->MD5(); // This should be rather fast, because text files aren't usually that large.
	}
	enableButton( KDialogBase::Ok, false );
	enableButton( KDialogBase::Cancel, false );
	enableButton( KDialogBase::User1, false );
	m_kompare_proc->start( KProcess::NotifyOnExit );
}

void KompareDialog::slotReceivedStdout( KProcess *, char *buffer, int )
{
	if ( !m_has_kdiff3 )
	{
		QString buffer_str = buffer;
		buffer_str.remove( "kdiff3:" );
		if ( buffer_str.find( "/kdiff3" ) != -1 )
		{
			m_has_kdiff3 = true;
			showButton( KDialogBase::User1, true );
		}
	}
}

void KompareDialog::slotProcessExited( KProcess * )
{
	if ( !( m_result & DIRTY ) ) // Only calculate md5 sum if files aren't dirty.
	{
		QCString md5_old1 = m_item1->MD5();
		QCString md5_old2 = m_item2->MD5();
		if ( md5_old1 != m_item1->MD5( true ) ||
				md5_old2 != m_item2->MD5( true ) )
		{
			m_result |= DIRTY;
		}
	}

	enableButton( KDialogBase::Ok, true );
	enableButton( KDialogBase::Cancel, true );
	enableButton( KDialogBase::User1, true );
}

void KompareDialog::slotCopyLeftRight( bool toggled )
{
	if ( toggled )
	{
		m_move_l_r->setOn( false );
		m_trash_l->setOn( false );
		m_delete_l->setOn( false );

		m_copy_r_l->setOn( false );
		m_move_r_l->setOn( false );
		m_trash_r->setOn( false );
		m_delete_r->setOn( false );
	}
}

void KompareDialog::slotCopyRightLeft( bool toggled )
{
	if ( toggled )
	{
		m_copy_l_r->setOn( false );
		m_move_l_r->setOn( false );
		m_trash_l->setOn( false );
		m_delete_l->setOn( false );

		m_move_r_l->setOn( false );
		m_trash_r->setOn( false );
		m_delete_r->setOn( false );
	}
}

void KompareDialog::slotMoveLeftRight( bool toggled )
{
	if ( toggled )
	{
		m_copy_l_r->setOn( false );
		m_trash_l->setOn( false );
		m_delete_l->setOn( false );

		m_copy_r_l->setOn( false );
		m_move_r_l->setOn( false );
		m_trash_r->setOn( false );
		m_delete_r->setOn( false );
	}
}

void KompareDialog::slotMoveRightLeft( bool toggled )
{
	if ( toggled )
	{
		m_copy_l_r->setOn( false );
		m_move_l_r->setOn( false );
		m_trash_l->setOn( false );
		m_delete_l->setOn( false );

		m_copy_r_l->setOn( false );
		m_trash_r->setOn( false );
		m_delete_r->setOn( false );
	}
}

void KompareDialog::slotTrashRight( bool toggled )
{
	if ( toggled )
	{
		m_copy_l_r->setOn( false );
		m_move_l_r->setOn( false );
		m_trash_l->setOn( false );
		m_delete_l->setOn( false );

		m_copy_r_l->setOn( false );
		m_move_r_l->setOn( false );
		m_delete_r->setOn( false );
	}
}

void KompareDialog::slotTrashLeft( bool toggled )
{
	if ( toggled )
	{
		m_copy_l_r->setOn( false );
		m_move_l_r->setOn( false );
		m_delete_l->setOn( false );

		m_copy_r_l->setOn( false );
		m_move_r_l->setOn( false );
		m_trash_r->setOn( false );
		m_delete_r->setOn( false );
	}
}

void KompareDialog::slotDeleteRight( bool toggled )
{
	if ( toggled )
	{
		m_copy_l_r->setOn( false );
		m_move_l_r->setOn( false );
		m_trash_l->setOn( false );
		m_delete_l->setOn( false );

		m_copy_r_l->setOn( false );
		m_move_r_l->setOn( false );
		m_trash_r->setOn( false );
	}
}

void KompareDialog::slotDeleteLeft( bool toggled )
{
	if ( toggled )
	{
		m_copy_l_r->setOn( false );
		m_move_l_r->setOn( false );
		m_trash_l->setOn( false );

		m_copy_r_l->setOn( false );
		m_move_r_l->setOn( false );
		m_trash_r->setOn( false );
		m_delete_r->setOn( false );
	}
}

bool KompareDialog::isTextFile( KFileItemExt *item )
{
	if ( item->mimetype().find( "text" ) == 0 || item->mimetype() == "application/x-zerosize" )
	{
		return true;
	}

	KMimeType::Ptr magicMimeType = KMimeType::findByFileContent( item->url().path() );
	if ( magicMimeType->name().find( "text" ) == 0 )
	{
		return true;
	}

	KMimeType::Ptr current_mime_type = item->mimeTypePtr();
	while ( current_mime_type->parentMimeType() )
	{
		if ( current_mime_type->parentMimeType().find( "text" ) == 0 )
		{
			return true;
		}
		current_mime_type = KMimeType::mimeType( current_mime_type->parentMimeType() );
	}

	current_mime_type = magicMimeType;
	while ( current_mime_type->parentMimeType() )
	{
		if ( current_mime_type->parentMimeType().find( "text" ) == 0 )
		{
			return true;
		}
		current_mime_type = KMimeType::mimeType( current_mime_type->parentMimeType() );
	}
	return false;
}
