//
// C++ Implementation: SQLiteConnection
//
// Description: 
//
//
// Author: Andreas Scherf <scherfa@web.de>, (C) 2007
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "SQLiteConnection.h"
#include <iostream>
#include <kmessagebox.h>
using namespace std;

SQLiteConnection::SQLiteConnection() :DBConnection() , m_pDB(NULL)
{
}

SQLiteConnection::~SQLiteConnection()
{
}

/**
 * 
 * @return The SQLite Version ...
 */
QString SQLiteConnection::getDBVersion()
{
	QString sResult;
 	sResult.append(QString("SQLite Version %1").arg(SQLITE_VERSION));
	return sResult;
}

void SQLiteConnection::setConnectionParameter(const QString &sFilename)
{
 	m_sFilename = sFilename;	
}

void SQLiteConnection::connect()
{
	if ( !m_bConnected ) 
	{
		g_pMainUseCase->setConfigGroup( "Database" );
 
		if (!m_sFilename.isEmpty()) {
            sqlite3_open(m_sFilename , &m_pDB);
            m_sDBName=QString(m_sFilename.section( '/', -1 ));

        } else {
            sqlite3_open(g_pMainUseCase->getConfig()->readEntry( "Databasename","default.db"), &m_pDB);
            m_sDBName=g_pMainUseCase->getConfig()->readEntry( "Databasename","default.db");
        }
        m_bConnected = (m_pDB) ? true : false;
    }
}

void SQLiteConnection::disconnect()
{
	if (m_bConnected)
	{
		if (sqlite3_close(m_pDB) == SQLITE_OK)
		{
			m_pDB = NULL;
			m_bConnected=false;
		}
	}
}

bool SQLiteConnection::startTransaction()
{	
	if (m_bConnected)
	{
		char* err;
   	 	int res = sqlite3_exec(m_pDB, "BEGIN", 0, this, &err);

    	if (res == SQLITE_OK)  return true;

    	sqlite3_free(err);
	}
	return false;
}

bool SQLiteConnection::commitTransaction()
{
	if (m_bConnected)
	{
 		char* err;
		int res = sqlite3_exec(m_pDB, "COMMIT", 0, this, &err);

    	if (res == SQLITE_OK) return true;

  
  		sqlite3_free(err);
	}
    return false;
}

bool SQLiteConnection::rollbackTransaction()
{
	if (m_bConnected)
	{
 		char* err;
		int res = sqlite3_exec(m_pDB, "ROLLBACK", 0, this, &err);

    	if (res == SQLITE_OK) return true;
  		sqlite3_free(err);
	}
    return false;
}

/**
 * 
 * @param pObject 
 * @param pResultSet 
 * @param statement 
 * @return 
 */
int SQLiteConnection::query(KRunningObject *pObject,KRunningResultSet *pResultSet )
{
	char *zErrMsg=0;
	cerr << "SQLiteConnection::query " << "called." << endl;
	char **result;
	QString sStatement= createSQLStatement( pObject );
	
	int nRow,nColumns;

    int nResult = sqlite3_get_table(m_pDB,sStatement.latin1(),&result,&nRow,&nColumns,&zErrMsg);	
	QString sDebug;
	if (nResult == SQLITE_OK && nRow > 0)
	{
		pResultSet->setColumns( nColumns);
		pResultSet->setRows( nRow);
		
		for ( long row=0;row<=nRow;row++)
		{
			if (row > 0)  // First Row are Header Names ..
			{
				for ( long column=0;column<nColumns;column++) 
				{	
					pResultSet->setValue(row-1,QString(result[column]),QVariant(result[((row*nColumns)+column)]));
					sDebug +="[row]: "+QString("%1").arg(row)+" ["+ QString(result[column]) + "]: "+ QString(result[((row*nColumns)+column)]) +"\n";
				}
			}
		}
		sqlite3_free_table(result);
	}
	else if (nResult != SQLITE_OK)
	{
		KMessageBox::detailedError (NULL, zErrMsg, sStatement,"SQL::query");
	}
	
		
	//KMessageBox::detailedError (NULL, sStatement, sDebug,"SQL::query");
	
	return nRow;
}

int SQLiteConnection::insert( const QString& /* statement */, const QString& /* table */ )
{
	return 0;
}

/**
 * 
 * @return 
 */
QStringList SQLiteConnection::getTableNames() 
{
    char **res;
    int ntab,ncol;
    QStringList slResult;
    if (!m_bConnected) return slResult;

    int nResult=sqlite3_get_table (m_pDB,"SELECT name FROM sqlite_master WHERE type='table' ORDER BY name;", &res,&ntab,&ncol,NULL);
    if (nResult == SQLITE_OK && ntab > 0 )	
	{
        for (int i=0 ;i<ntab;i++) 
		{
            slResult.append(res[i]);
        }
    } 
	else	
	{	
        qDebug ("SQLiteConnection\ngetTableNames: Could not get list of tables: %d",nResult);
    }
    sqlite3_free_table (res);

    return slResult;
}

/**
 * 
 * @param table 
 * @return 
 */
bool SQLiteConnection::createInitialTable(KRunningTableObject* pTable)
{
	bool bReturn = false;
	bool bArgumentListStart = true;

	KRunningSqlFields vTableHeaders=pTable->getHeaderVector();
	KRunningSqlFields::iterator it;
	QString sParameter;
	QString sTable = QString ("create table %1 (").arg(pTable->getTableName());
    for ( it = vTableHeaders.begin(); it != vTableHeaders.end(); ++it )
	{
		if (!bArgumentListStart)
		{
			sTable += QString(",");
		}
		bArgumentListStart = false;

		sParameter = QString ("%1").arg((*it).getName());
		if ((*it).isPrimaryKey())
		{
			sParameter +=" INTEGER PRIMARY KEY ";
		}
		else
		{
			switch ((*it).getType())
			{
				case QVariant::String:
				sParameter +=" text ";
				break;
				case QVariant::LongLong:
				sParameter +=" int ";
				break;
				case QVariant::Int:
				sParameter +=" int2 ";
				break;
				default:
				sParameter +=" text ";
				break; 
			}
		}
		if ((*it).isUnique())
		{
			sParameter+=" UNIQUE ";
		}
       	sTable +=sParameter;
	}
	sTable+=QString(")");
#ifdef DEBUG
	cerr << sTable << endl;
#endif
	bReturn = createTable( sTable);
	delete pTable;
	return bReturn;
}

bool SQLiteConnection::createTable(const QString& sTable)
{	
	char *zErrMsg=0;
	if (m_pDB && !sTable.isEmpty())
	{
		sqlite3_exec(m_pDB, sTable, 0, 0, &zErrMsg);
	}
	return true;
}

long SQLiteConnection::getTableCount()
{
	return getTableNames().size();
}

QString SQLiteConnection::getTableRows(const QString& sTablename)
{
	char **result;
	int nRow,nColumns;
	QString sResult;
	QString sStatement=QString("select COUNT(*) from %1").arg(sTablename);
	int nResult = sqlite3_get_table(m_pDB,sStatement.latin1(),&result,&nRow,&nColumns,NULL);
	if (nResult == SQLITE_OK && nRow > 0) 
	{
		for (int i=(nColumns);i<((nRow+1)*nColumns);i+=nColumns) 
		{
			sResult.append(QString(result[i]));
		}
	}
	sqlite3_free_table(result);
	return sResult;
}

/**
 * 
 * @param pKRunningObject 
 * @return 
 */
bool SQLiteConnection::insertValues(KRunningObject *pKRunningObject)
{
	char *zErrMsg=0;
	bool bArgumentListStart = true;
	bool bReturn = false;
	
	QString sColumns,sValues;
	KRunningSqlFields map=pKRunningObject->getFields();
	KRunningSqlFields::iterator it;
	
	for ( it = map.begin(); it != map.end(); ++it ) 
	{
		if (!(*it).isHeader() && (*it).getValue().isValid() && !((*it).isPrimaryKey() && pKRunningObject->getAutoIndex()))
		{
			if (!bArgumentListStart)
			{
				sColumns += QString(", ");
				sValues += QString(", ");
			}
			bArgumentListStart = false;
			
			sColumns+=(*it).getName();
			if ((*it).getType()==QVariant::String)
			{
				sValues+=QString("'"+(*it).getValue().toString()+"'"); 
			}
			else
			{
				sValues+=(*it).getValue().toString();
			}
		}
	}
	QString sStatement = QString ("INSERT INTO %1 (%2) VALUES (%3)").arg(pKRunningObject->getTableName()).arg(sColumns).arg(sValues);
	
 	if (sqlite3_exec(m_pDB, sStatement.latin1(),  0, 0, &zErrMsg )==SQLITE_OK)
	{
		pKRunningObject->setNumber(sqlite3_last_insert_rowid(m_pDB));
		bReturn = true;
	}
	else
	{
		pKRunningObject->setNumber(-1);
		cerr << "FEHLER in 'SQLiteConnection::insertValues' : "<< zErrMsg << endl;
	}
	return bReturn;
}

/**
 * 
 * @param pKRunningObject 
 * @return 
 */
bool SQLiteConnection::updateValues(KRunningObject *pKRunningObject,const QString& sFilter)
{
	char *zErrMsg=0;
	bool bArgumentListStart = true;
	bool bReturn = false;
	QString sStatement=QString("UPDATE %1 SET ").arg(pKRunningObject->getTableName());
	
	KRunningSqlFields map=pKRunningObject->getFields();
	KRunningSqlFields::iterator it;
	
	for ( it = map.begin(); it != map.end(); ++it ) 
	{
		if (!(*it).isHeader() && (*it).getValue().isValid())
		{
			if (!bArgumentListStart)
			{
				sStatement += QString(", ");
			}
			bArgumentListStart = false;
			sStatement+=(*it).getName()+"=";
			if ((*it).getType()==QVariant::String)
			{
				sStatement += QString("'"+(*it).getValue().toString()+"'");
			}
			else
			{
				sStatement+=(*it).getValue().toString();
			}
		}
	}
	if (sFilter.isEmpty())
	{
		sStatement +=QString(" WHERE number=%1").arg(pKRunningObject->getNumber());
	}
	else
	{
		sStatement +=sFilter;
	}
#ifdef DEBUG
	cerr << "Statement : " << sStatement << endl;
#endif
	
 	if (sqlite3_exec(m_pDB, sStatement.latin1(),  0, 0, &zErrMsg )==SQLITE_OK)
	{
		bReturn = true;
	}
	else
	{
		cerr << "FEHLER in 'SQLiteConnection::updateValues' : "<< zErrMsg << endl;
	}
	
	return bReturn;
}

bool SQLiteConnection::deleteValues( KRunningObject *pObject)
{
	bool bReturn = false;
	char *zErrMsg=0;
	
	QString sStatement=QString("DELETE FROM %1 ").arg(pObject->getTableName());
	if (!pObject->toWhereString().isEmpty())
	{
		sStatement += QString ("where %1 ").arg(pObject->toWhereString()); 
	}

	if (sqlite3_exec(m_pDB, sStatement.latin1(),  0, 0, &zErrMsg )==SQLITE_OK)
	{
		bReturn = true;
	}
	else
	{
		cerr << "FEHLER in 'SQLiteConnection::deleteValues' : "<< zErrMsg << endl;
	}
	return bReturn;
}

QString SQLiteConnection::createSQLStatement(KRunningObject *pObject)
{
	QString sResult;
	if (pObject->toIndexString().isEmpty()) // Select while table
	{
		sResult= QString ("select * from %1 ").arg(pObject->toFromString());
	}
	else
	{
		sResult = QString ("select %1 from %2 ").arg(pObject->toIndexString()).arg(pObject->toFromString());
	}
	
	if (!pObject->toWhereString().isEmpty())
	{
		sResult += QString ("where %1 ").arg(pObject->toWhereString()); 
	}
	
	if (!pObject->toOrderString().isEmpty())
	{
		sResult += QString ("order by %1").arg(pObject->toOrderString()); 
	}
	return sResult;
}






