/***************************************************************************
 *   Copyright (C) 2003 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <qpainter.h>
#include <qtooltip.h>
#include <qpopupmenu.h>
#include <qcursor.h>
#include <qfont.h>
#include <qfontmetrics.h>
#include <qpaintdevicemetrics.h>
#include <qbitmap.h>
#include <klocale.h>
#include <kaction.h>
#include <kprinter.h>
#include <kiconloader.h>
#include <klineeditdlg.h>
#include <kmessagebox.h>
#include <kdebug.h>
#include "configuration.h"
#include "paletteview.h"
#include "flosspalette.h"
#include "kxstitch.h"
#include "kxstitchview.h"
#include "kxstitchdoc.h"
#include "patterncanvas.h"
#include "loadpalettedlg.h"
#include "patternpropertiesdialog.h"

const uchar swapCursor[] =
{ 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x08, 0x00,
  0x00, 0x00, 0x18, 0x00,
  0x00, 0x00, 0x3f, 0x80,
  0x00, 0x00, 0x7f, 0xc0,
  0x3f, 0xff, 0x3f, 0xe0,
  0x3f, 0xff, 0x19, 0xe0,
  0x3f, 0xff, 0x08, 0xe0,
  0x3f, 0xff, 0x00, 0xe0,
  0x3f, 0xff, 0x00, 0xe0,
  0x3f, 0xff, 0x03, 0xf8,
  0x3f, 0xff, 0x01, 0xf0,
  0x3f, 0xff, 0x00, 0xe0,
  0x3f, 0xff, 0x00, 0x40,
  0x3f, 0xff, 0x00, 0x00,
  0x3f, 0xff, 0xff, 0x80,
  0x3f, 0xff, 0x00, 0x80,
  0x3f, 0xff, 0x00, 0x80,
  0x3f, 0xff, 0x00, 0x80,
  0x00, 0x10, 0x00, 0x80,
  0x00, 0x10, 0x00, 0x80,
  0x00, 0x10, 0x00, 0x80,
  0x00, 0x10, 0x00, 0x80,
  0x00, 0x10, 0x00, 0x80,
  0x00, 0x10, 0x00, 0x80,
  0x00, 0x10, 0x00, 0x80,
  0x00, 0x10, 0x00, 0x80,
  0x00, 0x10, 0x00, 0x80,
  0x00, 0x1f, 0xff, 0x80,
  0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00 };

const uchar swapCursorMask[] =
{ 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1c, 0x00,
  0x00, 0x00, 0x3c, 0x00,
  0x00, 0x00, 0x7f, 0xc0,
  0x00, 0x00, 0xff, 0xe0,
  0x7f, 0xff, 0xff, 0xf0,
  0x7f, 0xff, 0xff, 0xf0,
  0x7f, 0xff, 0xff, 0xf0,
  0x7f, 0xff, 0xbf, 0xf0,
  0x7f, 0xff, 0x9d, 0xf0,
  0x7f, 0xff, 0x87, 0xfc,
  0x7f, 0xff, 0x87, 0xfc,
  0x7f, 0xff, 0x87, 0xfc,
  0x7f, 0xff, 0x83, 0xf8,
  0x7f, 0xff, 0x81, 0xf0,
  0x7f, 0xff, 0xff, 0xe0,
  0x7f, 0xff, 0xff, 0xc0,
  0x7f, 0xff, 0xff, 0xc0,
  0x7f, 0xff, 0xff, 0xc0,
  0x7f, 0xff, 0xff, 0xc0,
  0x7f, 0xff, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x00, 0x00, 0x00 };

const uchar replaceCursor[] =
{ 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x08, 0x00,
  0x00, 0x00, 0x18, 0x00,
  0x00, 0x00, 0x3f, 0x80,
  0x00, 0x00, 0x7f, 0xc0,
  0x3f, 0xff, 0x3f, 0xe0,
  0x3f, 0xff, 0x19, 0xe0,
  0x3f, 0xff, 0x08, 0xe0,
  0x3f, 0xff, 0x00, 0xe0,
  0x3f, 0xff, 0x00, 0xe0,
  0x3f, 0xff, 0x00, 0xe0,
  0x3f, 0xff, 0x00, 0xe0,
  0x3f, 0xff, 0x00, 0xe0,
  0x3f, 0xff, 0x00, 0xe0,
  0x3f, 0xff, 0x00, 0x00,
  0x3f, 0xff, 0xff, 0x80,
  0x3f, 0xff, 0x00, 0x80,
  0x3f, 0xff, 0x00, 0x80,
  0x3f, 0xff, 0x00, 0x80,
  0x00, 0x10, 0x00, 0x80,
  0x00, 0x10, 0x00, 0x80,
  0x00, 0x10, 0x00, 0x80,
  0x00, 0x10, 0x00, 0x80,
  0x00, 0x10, 0x00, 0x80,
  0x00, 0x10, 0x00, 0x80,
  0x00, 0x10, 0x00, 0x80,
  0x00, 0x10, 0x00, 0x80,
  0x00, 0x10, 0x00, 0x80,
  0x00, 0x1f, 0xff, 0x80,
  0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00 };

const uchar replaceCursorMask[] =
{ 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1c, 0x00,
  0x00, 0x00, 0x3c, 0x00,
  0x00, 0x00, 0x7f, 0xc0,
  0x00, 0x00, 0xff, 0xe0,
  0x7f, 0xff, 0xff, 0xf0,
  0x7f, 0xff, 0xff, 0xf0,
  0x7f, 0xff, 0xff, 0xf0,
  0x7f, 0xff, 0xbf, 0xf0,
  0x7f, 0xff, 0x9d, 0xf0,
  0x7f, 0xff, 0x81, 0xf0,
  0x7f, 0xff, 0x81, 0xf0,
  0x7f, 0xff, 0x81, 0xf0,
  0x7f, 0xff, 0x81, 0xf0,
  0x7f, 0xff, 0x81, 0xf0,
  0x7f, 0xff, 0xff, 0xe0,
  0x7f, 0xff, 0xff, 0xc0,
  0x7f, 0xff, 0xff, 0xc0,
  0x7f, 0xff, 0xff, 0xc0,
  0x7f, 0xff, 0xff, 0xc0,
  0x7f, 0xff, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x3f, 0xff, 0xc0,
  0x00, 0x00, 0x00, 0x00 };

PaletteView::PaletteView(KXStitchDoc* doc,QWidget *parent)
  : QFrame(parent),
    QToolTip(this),
    m_doc(doc),
    m_waitingToSwap(false),
    m_waitingToReplace(false),
    m_rows(1),
    m_cols(1),
    m_symbols(false)
{
  KIconLoader* loader = KGlobal::iconLoader();
  KActionCollection* actionCollection = ((KXStitchApp*)topLevelWidget())->actionCollection();
  m_savePalette = new KAction(i18n("Save the current palette..."), 0, this, SLOT(slotSavePalette()), actionCollection, "savepalette");
  m_loadPalette = new KAction(i18n("Load a palette..."), 0, this, SLOT(slotLoadPalette()), actionCollection, "loadpalette");
  m_createPalette = new KAction(i18n("Create a palette..."), 0, this, SLOT(slotCreatePalette()), actionCollection, "createpalette");
  new KToggleAction(i18n("Show symbols"), 0, this, SLOT(slotShowSymbols()), actionCollection, "showsymbols");
  new KAction(i18n("Clear unused colors"), 0, this, SLOT(slotClearUnused()), actionCollection, "clearunused");
  new KAction(i18n("Swap colors"), 0, this, SLOT(slotPrepareToSwap()), actionCollection, "swapcolors");
  new KAction(i18n("Replace color"), 0, this, SLOT(slotPrepareToReplace()), actionCollection, "replacecolor");
  new KAction(i18n("Palette Manager"), QIconSet(loader->loadIcon("appearance.png",KIcon::User)), 0, this, SLOT(slotPaletteManager()), actionCollection, "palettemanager");
}

PaletteView::~PaletteView()
{
  // nothing to do here
}

int PaletteView::print(KPrinter *printer, QPainter *painter, int width, int height, int pages)
{
  FlossPalette *palette = m_doc->palette();
  UsageMap usageMap = m_doc->flossUsage();
  int flossStrands = KXSConfig().Palette_FlossStrands;
  QString paletteName = QString(i18n("%1 Flosses")).arg(palette->getScheme());
  QFont font = painter->font();
  QFont symbols = font;
  font.setFixedPitch(true);
  symbols.setFamily(KXSConfig().Editor_SymbolFont);
  QFontMetrics fm = painter->fontMetrics();
  int dy = fm.lineSpacing();
  int y = dy; // top of page
  pages++;

  int wSymbol = 0;
  int wName = 0;
  int wStrands = 0;
  int wDescription = 0;
  int wLength = 0;
  QValueVector<int>::size_type floss;
  for (floss = 0 ; floss < m_flosses.count() ; floss++)
  {
    Floss *f = palette->flossAt(m_flosses[floss]);
    QChar s = palette->symbolAt(m_flosses[floss]);
    wSymbol = (fm.width(s)>wSymbol)?fm.width(s):wSymbol;
    wName = (fm.width(f->name)>wName)?fm.width(f->name):wName;
    wDescription = (fm.width(f->description)>wDescription)?fm.width(f->description):wDescription;
  }

  // check if the bold headings would be wider than the text going under each one
  font.setBold(true);
  painter->setFont(font);
  fm = painter->fontMetrics();
  if (fm.width(i18n("Symbol"))>wSymbol)
    wSymbol = fm.width("Symbol");
  if (fm.width(i18n("Name"))>wName)
    wName = fm.width("Name");
  wStrands = fm.width("Strands");
  if (fm.width(i18n("Description"))>wDescription)
    wDescription = fm.width("Description");
  wLength = fm.width(i18n("Length(m)"));

  // widen the fields to allow some spacing
  wSymbol += 20;
  wName += 20;
  wStrands += 20;
  wDescription += 20;

  for (floss = 0 ; floss < m_flosses.count() ; floss++)
  {
    if (y == dy)
    {
      font.setBold(true);
      painter->setFont(font);
      painter->drawText(0,y,paletteName);
      y += dy*2;
      if (printer->option("kde-kxstitch-flossusage") == "true")
      {
        painter->drawText(0,y,QString(i18n("Strand numbers shown are for stitches / backstitches.")));
        if (printer->option("kde-kxstitch-seperatebackstitching") == "true")
          painter->drawText(0,y+dy,QString(i18n("Lengths are for Stitches, Backstitches (Total).")));
        else
          painter->drawText(0,y+dy,QString(i18n("Lengths are totals for all stitches.")));
        y += dy*3;
        if (printer->option("kde-kxstitch-stitchcount") == "true")
        {
          if (printer->option("kde-kxstitch-combinefractionals") == "true")
            painter->drawText(0,y,QString(i18n("Stitch counts are Full, Fractionals, Knots and Backstitches.")));
          else
            painter->drawText(0,y,QString(i18n("Stitch counts are Full, Half, Quarter, 3 Quarter, Knots and Backstitches.")));
          y += dy*2;
        }
      }
      painter->drawText(0,y,i18n("Symbol"));
      painter->drawText(wSymbol,y,i18n("Name"));
      painter->drawText(wSymbol+wName,y,i18n("Strands"));
      painter->drawText(wSymbol+wName+wStrands,y,i18n("Description"));
      if (printer->option("kde-kxstitch-flossusage") == "true")
      {
        painter->drawText(wSymbol+wName+wStrands+wDescription,y,i18n("Length(m)"));
      }
      font.setBold(false);
      painter->setFont(font);
      y += dy;
    }
    Floss *f = palette->flossAt(m_flosses[floss]);
    QChar s = palette->symbolAt(m_flosses[floss]);
    int stitchStrands = palette->stitchStrandsAt(m_flosses[floss]);
    int backstitchStrands = palette->backstitchStrandsAt(m_flosses[floss]);
    double stitchLengthDivisor = (double)flossStrands/(double)stitchStrands;
    double backstitchLengthDivisor = (double)flossStrands/(double)backstitchStrands;
    /** set the font for the symbol */
    painter->setFont(symbols);
    painter->drawText(0,y,s);
    /** set the font for the text */
    painter->setFont(font);
    painter->drawText(wSymbol,y,f->name);
    painter->drawText(wSymbol+wName,y,QString("%1 / %2").arg(stitchStrands).arg(backstitchStrands));
    painter->drawText(wSymbol+wStrands+wName,y,f->description);
    if (printer->option("kde-kxstitch-flossusage") == "true")
    {
      double combinedLength = usageMap[m_flosses[floss]].stitchLength/stitchLengthDivisor + usageMap[m_flosses[floss]].backstitchLength/backstitchLengthDivisor;
      int fieldWidth = fm.width("99999 ");
      int lengthsFieldWidth = fieldWidth*3;
      if (printer->option("kde-kxstitch-seperatebackstitching") == "true")
        painter->drawText(wSymbol+wStrands+wName+wDescription,y,QString("%1 %2 (%3)").arg(usageMap[m_flosses[floss]].stitchLength/stitchLengthDivisor,0,'f',2).arg(usageMap[m_flosses[floss]].backstitchLength/backstitchLengthDivisor,0,'f',2).arg(combinedLength,0,'f',2));
      else
        painter->drawText(wSymbol+wStrands+wName+wDescription,y,QString("%1").arg(combinedLength,0,'f',2));
      if (printer->option("kde-kxstitch-stitchcount") == "true")
      {
        if (printer->option("kde-kxstitch-combinefractionals") == "true")
        {
          int fractionals = usageMap[m_flosses[floss]].half + usageMap[m_flosses[floss]].quarter + usageMap[m_flosses[floss]].threeQuarter;
          painter->drawText(wSymbol+wStrands+wName+wDescription+lengthsFieldWidth,y,QString("%1").arg(usageMap[m_flosses[floss]].full));
          painter->drawText(wSymbol+wStrands+wName+wDescription+lengthsFieldWidth+fieldWidth,y,QString("%1").arg(fractionals));
          painter->drawText(wSymbol+wStrands+wName+wDescription+lengthsFieldWidth+fieldWidth*2,y,QString("%1").arg(usageMap[m_flosses[floss]].knots));
          painter->drawText(wSymbol+wStrands+wName+wDescription+lengthsFieldWidth+fieldWidth*3,y,QString("%1").arg(usageMap[m_flosses[floss]].backstitches));
        }
        else
        {
          painter->drawText(wSymbol+wStrands+wName+wDescription+lengthsFieldWidth,y,QString("%1").arg(usageMap[m_flosses[floss]].full));
          painter->drawText(wSymbol+wStrands+wName+wDescription+lengthsFieldWidth+fieldWidth,y,QString("%1").arg(usageMap[m_flosses[floss]].half));
          painter->drawText(wSymbol+wStrands+wName+wDescription+lengthsFieldWidth+fieldWidth*2,y,QString("%1").arg(usageMap[m_flosses[floss]].quarter));
          painter->drawText(wSymbol+wStrands+wName+wDescription+lengthsFieldWidth+fieldWidth*3,y,QString("%1").arg(usageMap[m_flosses[floss]].threeQuarter));
          painter->drawText(wSymbol+wStrands+wName+wDescription+lengthsFieldWidth+fieldWidth*4,y,QString("%1").arg(usageMap[m_flosses[floss]].knots));
          painter->drawText(wSymbol+wStrands+wName+wDescription+lengthsFieldWidth+fieldWidth*5,y,QString("%1").arg(usageMap[m_flosses[floss]].backstitches));
        }
      }
    }
    y += dy;
    if (y > height)
    {
      y = dy;
      painter->drawText(0,height+dy,width,dy,Qt::AlignHCenter,QString(i18n("Page %1")).arg(pages++));
      printer->newPage();
    }
  }

  painter->drawText(0,height+dy,width,dy,Qt::AlignHCenter,QString(i18n("Page %1")).arg(pages));
  return pages;
}

void PaletteView::setCurrentColor(int c)
{
  if (m_waitingToSwap)
  {
    m_doc->swapColors(c);
    ((KXStitchView *)(parent()->parent()))->resetView();
    m_waitingToSwap = false;
    setCursor(ArrowCursor);
  }
  else if (m_waitingToReplace)
  {
    m_doc->replaceColor(c);
    ((KXStitchView *)(parent()->parent()))->resetView();
    m_waitingToReplace = false;
    setCursor(ArrowCursor);
  }
  m_doc->palette()->setCurrentColor(c);
  update();
}

void PaletteView::configure()
{
  if (m_symbols)
    repaint();
}

void PaletteView::drawContents(QPainter *p)
{
  // TODO maybe make this double buffered
  m_flosses.clear();
  QFont f = p->font();
  f.setFamily(KXSConfig().Editor_SymbolFont);
  FlossPalette *palette = m_doc->palette();
  int colors = palette->colors();
  m_flosses.resize(colors);
  QMap<int,Floss*>& colorMap = palette->colorMap();
  QMapIterator<int,Floss*> it;
  int floss;
  for (floss = 0,it = colorMap.begin() ; it != colorMap.end() ; ++it)
    m_flosses[floss++] = it.key();
  bool exchanged;
  do
  {
    exchanged = false;
    for (floss = 0 ; floss < colors-1 ; floss++)
    {
      QString s1(colorMap[m_flosses[floss]]->name);
      QString s2(colorMap[m_flosses[floss+1]]->name);
      int l1 = s1.length();
      int l2 = s2.length();
      if (((s1 > s2) && (l1 >= l2)) || (l1 > l2))
      {
        int temp = m_flosses[floss];
        m_flosses[floss] = m_flosses[floss+1];
        m_flosses[floss+1] = temp;
        exchanged = true;
      }
    }
  } while (exchanged);
  int current = palette->currentColor();
  int szHeight;
  m_cols = 5;
  while (true)
  {
    while ((m_cellWidth = contentsRect().width()/m_cols) > 40) m_cols++;
    m_rows=(palette->colors())/m_cols + 1;
    szHeight = contentsRect().height()/m_rows;
    m_cellHeight = (szHeight > m_cellWidth)?m_cellWidth:szHeight;
    if ((m_cellWidth-m_cellHeight) > m_cellHeight)
      m_cols++;
    else
      break;
  }
  if (palette->colors())
  {
    if (current==-1)
    {
      current = m_flosses[0];
      palette->setCurrentColor(current);
    }
    ((KXStitchApp*)topLevelWidget())->slotStateChanged("palette_empty",KXMLGUIClient::StateReverse);
  }
  else
  {
    ((KXStitchApp*)topLevelWidget())->slotStateChanged("palette_empty");
  }
  f.setPixelSize(m_cellWidth<?m_cellHeight);
  p->setFont(f);
  for (int i = 0 ; i < palette->colors() ; i++)
  {
    Floss *f = colorMap[m_flosses[i]];
    int x=(i%m_cols)*m_cellWidth;
    int y=(i/m_cols)*m_cellHeight;
    QRect r(x,y,m_cellWidth,m_cellHeight);

    p->fillRect(r,f->color);
    if (m_flosses[i] == current)
    {
      p->setPen(black);
      p->drawLine(r.topLeft(),r.topRight());
      p->drawLine(r.topLeft(),r.bottomLeft());
      p->setPen(white);
      p->drawLine(r.topRight(),r.bottomRight());
      p->drawLine(r.bottomLeft(),r.bottomRight());
    }
    else
    {
      p->setPen(black);
      p->drawLine(r.topRight(),r.bottomRight());
      p->drawLine(r.bottomLeft(),r.bottomRight());
      p->setPen(white);
      p->drawLine(r.topLeft(),r.topRight());
      p->drawLine(r.topLeft(),r.bottomLeft());
    }
    if (m_symbols)
    {
      if(qGray(f->color.rgb()) < 128)
        p->setPen(white);
      else
        p->setPen(black);
      p->drawText(r, AlignCenter, palette->symbolAt(m_flosses[i]), 1);
    }
  }
}

void PaletteView::contextMenuEvent(QContextMenuEvent* e)
{
  KXStitchApp *app = (KXStitchApp *)topLevelWidget();
  QPopupMenu *context = (QPopupMenu *)app->guiFactory()->container("palette",app);
  context->popup(e->globalPos());
  e->accept();
}

void PaletteView::mousePressEvent(QMouseEvent *e)
{
  if (e->button() & LeftButton)
  {
    QPoint p = e->pos();
    int xCell = (p.x()/m_cellWidth);
    int yCell = (p.y()/m_cellHeight);
    QValueVector<int>::size_type i = yCell*m_cols+xCell;
    if (i < m_flosses.count())
      setCurrentColor(m_flosses[i]);
    e->accept();
  }
}

void PaletteView::maybeTip(const QPoint &p)
{
  FlossPalette *palette = m_doc->palette();
  // find rect point is in, and index of floss
  int xCell = p.x()/m_cellWidth;
  int yCell = p.y()/m_cellHeight;
  QValueVector<int>::size_type i = yCell*m_cols+xCell;
  if (palette->colors())
  {
    if (i < m_flosses.count())
    {
      Floss *f = palette->flossAt(m_flosses[i]);
      QString s = QString("%1 %2").arg(f->name).arg(f->description);
      int x=(i%m_cols)*m_cellWidth;
      int y=(i/m_cols)*m_cellHeight;
      QRect r(x,y,m_cellWidth,m_cellHeight);
      tip(r,s);
    }
  }
  else
    tip(contentsRect(),i18n("No colors in palette"));
}

void PaletteView::slotSavePalette()
{
  QString schemeName;
  bool ok;
  schemeName = KLineEditDlg::getText(i18n("Scheme Name"),"",&ok,this);
  if (ok)
  {
    if (((KXStitchApplication*)kapp)->scheme(schemeName))
      if (KMessageBox::warningContinueCancel(this,i18n("This scheme name already exists.\nDo you want to overwrite it?"),i18n("Scheme Exists"),i18n("Overwrite")) == KMessageBox::Cancel)
        ok = false;
    if (ok)
      if (m_doc->palette()->savePalette(schemeName))
        m_doc->m_propertiesDialog->setScheme(schemeName);
  }
}

void PaletteView::slotLoadPalette()
{
  SelectScheme  dlg(this);
  dlg.FlossScheme->insertStringList(((KXStitchApplication*)kapp)->schemes());
  if (dlg.exec() == QDialog::Accepted)
  {
    QString schemeName = dlg.FlossScheme->currentText();
    m_doc->palette()->loadPalette(schemeName);
    update();
  }
}

void PaletteView::slotCreatePalette()
{
  QString schemeName;
  bool    ok = false;
  while (!ok)
  {
    schemeName = KLineEditDlg::getText(i18n("Scheme Name"),"",&ok,this);
    if (!ok)
      break; // user pressed cancel
    if (((KXStitchApplication*)kapp)->scheme(schemeName))
    {
      KMessageBox::sorry(this,i18n("This scheme name already exists.\nPlease choose another name?"),i18n("Scheme Exists"));
      ok = false;
    }
    if (ok)
    {
      if (((KXStitchApplication*)kapp)->createScheme(schemeName))
      {
        m_doc->palette()->setScheme(schemeName);
        m_doc->m_propertiesDialog->setScheme(schemeName);
      }
      else
        KMessageBox::sorry(this,i18n("This scheme could not be created"),i18n("Error Creating Scheme"));
    }
  }
}

void PaletteView::slotShowSymbols()
{
  m_symbols = !m_symbols;
  update();
}

void PaletteView::slotPrepareToSwap()
{
  setCursor(QCursor(QBitmap(32,32,swapCursor),QBitmap(32,32,swapCursorMask),-1,-1));
  m_waitingToSwap = true;
  m_waitingToReplace = false;
}

void PaletteView::slotPrepareToReplace()
{
  setCursor(QCursor(QBitmap(32,32,replaceCursor),QBitmap(32,32,replaceCursorMask),-1,-1));
  m_waitingToReplace = true;
  m_waitingToSwap = false;
}

void PaletteView::slotAddColor()
{
/*
  if (m_doc->palette()->addColors())
  {
    update();
  }
*/
}
void PaletteView::slotRemoveColors()
{
/*
  if (m_doc->removeColors())
  {
    update();
  }
*/
}

void PaletteView::slotClearUnused()
{
  m_doc->removeUnusedColors();
  update();
}

void PaletteView::slotPaletteManager()
{
  if (m_doc->paletteManager())
  {
    update();
  }
}
