/***************************************************************************
 *   Copyright (C) 2003 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#ifndef STITCH_H
#define STITCH_H

#include <qptrqueue.h>
#include <qpoint.h>

/** Store information about individual stitches.
    @author Stephen P. Allewell
    <pre>
    Stitches encoded as follows

      1\/2
      4/\8

    Type      binary  decimal
    Delete    00000000  0
    TLQtr     00000001  1
    TRQtr     00000010  2
    BLQtr     00000100  4
    BTHalf    00000110  6
    TL3Qtr    00000111  7
    BRQtr     00001000  8
    TBHalf    00001001  9
    TR3Qtr    00001011  11
    BL3Qtr    00001101  13
    BR3Qtr    00001110  14
    Full      00001111  15
    FRKnot    00010000  16
    </pre>
  */
class Stitch
{
public:
  enum Type {Delete=0,TLQtr=1,TRQtr=2,BLQtr=4,BTHalf=6,TL3Qtr=7,BRQtr=8,TBHalf=9,TR3Qtr=11,BL3Qtr=13,BR3Qtr=14,Full=15,FRKnot=16};
  typedef QPtrQueue<Stitch> Queue;

/** Constructor
    @param t StitchType of stitch to be stored.
    @param i Index into the floss table.
  */
  Stitch(Type t, int i);

  Type  type;
  int   floss;
};

/** Stores the start and end coordinates and the color of a backstitch
    @author Stephen P. Allewell
  */
class BackStitch
{
public:
/** Constructor
    @param s Start coordinate
    @param e End coordinate
    @param i Index into the color palette
  */
  BackStitch(QPoint s, QPoint e, int i);
/** Tests to see if a coordinate is the start or end of the current backstitch
    @return TRUE if p is the start or end of this backstitch, FALSE otherwise.
    @param p Coordinate to be checked.
  */
  bool contains(QPoint p);

  QPoint  start;
  QPoint  end;
  int     floss;
};

/** Store information about a french knot
    @author Stephen P. Allewell
  */
class Knot
{
public:
/** Constructor
    @param p The coordinates of the knot.
    @param i The index into the floss table.
  */
  Knot(QPoint p, int i);

  QPoint  pos;
  int     floss;
};

#endif
