/***************************************************************************
*   Copyright (C) 2005 by Adam Treat                                      *
*   treat@kde.org                                                         *
*                                                                         *
*   Copyright (C) 2004 by Scott Wheeler                                   *
*   wheeler@kde.org                                                       *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/

#ifndef DATATABLESEARCH_H
#define DATATABLESEARCH_H

#include <qregexp.h>
#include <qstring.h>
#include <qstringlist.h>

class DataField;

class DataTable;
typedef QValueList<DataTable *> DataTableList;

typedef QValueList<int> ColumnList;

class DataTableSearch
{
public:
    class Component;
    typedef QValueList<Component> ComponentList;
    enum MatchMode
    {
        Contains /*Default*/ = 0,
        DoesNotContain = 1,
        Equals = 2,
        DoesNotEqual = 3,
        IsNull = 4,
        IsNotNull = 5,
        Before = 6,
        After = 7,
        OnOrBefore = 8,
        OnOrAfter = 9,
        LessThan = 10,
        GreaterThan = 11,
        LessOrEquals = 12,
        GreaterOrEquals = 13
    };
    enum SearchMode { MatchAny = 0, MatchAll = 1 };
    enum SearchLevel { Simple = 0, Advanced = 1, Custom = 2 };

    DataTableSearch( SearchMode mode = MatchAny,
                     SearchLevel level = Simple );

    DataTableSearch( const DataTableList &dataTables,
                     const ComponentList &components,
                     const QString &name = QString::null,
                     SearchMode mode = MatchAny,
                     SearchLevel level = Simple );

    void search();
    void dataSearchPopup();
    void simpleSearchDataTable();
    void advancedSearchDataTable();
    void customSearchDataTable();
    void execQuery( DataTable* dataTable, const QString &query );

    QString constructSQLForList( DataTable* dataTable, bool isRoot = false );
    QStringList getInheritanceTree( DataTable* dataTable, DataField* dataField );

    QString searchModeSnippet();
    QString relationSnippet( const QString &query, const QString &name, const QString &key,
                             const QString &table, const QString &field, MatchMode mode = DataTableSearch::Contains );
    QString subselectSnippet( const QString &parentKey, const QString &foreignKey,
                              const QString &tableName );
    QString subselectEndSnippet();
    QString operatorSnippet( const QString &query, const QString &name, MatchMode mode = DataTableSearch::Contains );

    DataTableList dataTables() const
    {
        return m_dataTables;
    }
    void setDataTables( const DataTableList &dataTables )
    {
        m_dataTables = dataTables;
    }
    void addComponent( const Component &c )
    {
        m_originals.append( c );
    }
    void clearComponents()
    {
        m_originals.clear();
    }
    ComponentList components() const
    {
        return m_originals;
    }

    QString name() const
    {
        return m_name;
    }

    void setName( const QString &name )
    {
        m_name = name;
    }

    QString parentName() const;

    SearchMode searchMode() const
    {
        return m_mode;
    }
    void setSearchMode( SearchMode m )
    {
        m_mode = m;
    }
    SearchLevel searchLevel() const
    {
        return m_level;
    }
    void setSearchLevel( SearchLevel l )
    {
        m_level = l;
    }
    bool isNull() const
    {
        return m_originals.isEmpty();
    }
    bool isEmpty() const;

private:
    QString m_name;
    DataTableList m_dataTables;
    ComponentList m_originals;
    ComponentList m_updates;
    SearchMode m_mode;
    SearchLevel m_level;
};

class DataTableSearch::Component
{
public:

    Component();

    Component( const QString &query,
               DataTable *dataTable = 0,
               DataField *dataField = 0,
               bool prompt = false,
               DataTableSearch::MatchMode mode = DataTableSearch::Contains );

    QString query() const
    {
        return m_query;
    }

    DataTable *dataTable() const
    {
        return m_dataTable;
    }

    DataField *dataField() const
    {
        return m_dataField;
    }

    DataTableSearch::MatchMode matchMode() const
    {
        return m_mode;
    }

    bool isPrompt() const
    {
        return m_prompt;
    }

    bool operator==( const Component &v ) const;

private:
    QString m_query;
    DataTable *m_dataTable;
    DataField *m_dataField;
    bool m_prompt;
    DataTableSearch::MatchMode m_mode;
};

#endif
