/***************************************************************************
*   Copyright (C) 2005 by Adam Treat                                      *
*   treat@kde.org                                                         *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/

#include "datatableview.h"

#include "project.h"
#include "datatable.h"
#include "datafield.h"
#include "actioncollection.h"
#include "databaseconnection.h"

#include <qstyle.h>
#include <qcolor.h>
#include <qpainter.h>

#include <kaction.h>
#include <kpopupmenu.h>

using namespace ActionCollection;

DataTableView::DataTableView( DataTable *parent, const char *name )
        : QDataTable( parent, name ),
        m_dataTable( parent ),
        m_selectionColor( Qt::green ),
        m_mode( Default )
{
    setAutoEdit( false );
    setReadOnly( true );

    verticalHeader() ->hide();
    setLeftMargin( 0 );

    setNullText( "" );
    setTrueText( "true" );
    setFalseText( "false" );
    setDateFormat ( Qt::ISODate );
    QObject::connect( this, SIGNAL( doubleClicked( int, int, int, const QPoint & ) ),
                      m_dataTable, SLOT( scrollTabRight() ) );
}

DataTableView::~DataTableView()
{}

QColor DataTableView::selectionColor() const
{
    return m_selectionColor;
}

void DataTableView::setSelectionColor( const QColor &color )
{
    m_selectionColor = color;
    repaintSelections();
}

DataTableView::SelectionMode DataTableView::selectionMode() const
{
    return m_mode;
}

void DataTableView::setSelectionMode( DataTableView::SelectionMode m )
{
    m_mode = m;
    if ( m_mode == MouseOver )
    {
        viewport()->setMouseTracking( true );
        QObject::connect( this, SIGNAL( clicked( int, int, int, const QPoint & ) ),
                          m_dataTable, SLOT( associateRecord() ) );
    }
    else
    {
        viewport()->setMouseTracking( false );
        QObject::disconnect( this, SIGNAL( clicked( int, int, int, const QPoint & ) ),
                             m_dataTable, SLOT( associateRecord() ) );
    }
}

void DataTableView::columnWidthChanged( int col )
{
    int w = columnWidth( col );
    DataField *field = m_dataTable->dataField( col );
    if ( field )
        field->setWidth( w );
    QDataTable::columnWidthChanged( col );
}

void DataTableView::setColumnWidth( int col, int w )
{
    DataField * field = m_dataTable->dataField( col );
    if ( field )
        field->setWidth( w );
    QDataTable::setColumnWidth( col, w );
}

void DataTableView::slotSelectFirstRow( bool updateCursor )
{
    if ( !sqlCursor() )
        return;

    selectRow( 0 );
    if ( updateCursor ) sqlCursor() ->first();
    emit selectionChanged();
}

void DataTableView::slotSelectPrevRow( bool updateCursor )
{
    if ( !sqlCursor() )
        return;

    selectRow( currentRow() - 1 );
    if ( updateCursor ) sqlCursor() ->prev();
    emit selectionChanged();
}

void DataTableView::slotSelectNextRow( bool updateCursor )
{
    if ( !sqlCursor() )
        return;

    selectRow( currentRow() + 1 );
    if ( updateCursor ) sqlCursor() ->next();
    emit selectionChanged();
}

void DataTableView::slotSelectLastRow( bool updateCursor )
{
    if ( !sqlCursor() )
        return;

    selectRow( numRows() );
    if ( updateCursor ) sqlCursor() ->last();
    emit selectionChanged();
}

void DataTableView::slotSelectRow( int row, bool updateCursor )
{
    if ( !sqlCursor() )
        return;

    selectRow( row );
    if ( updateCursor ) sqlCursor() ->seek( row, false );
    emit selectionChanged();
}

void DataTableView::handleError( const QSqlError &error )
{
    QDataTable::handleError( error );
}

void DataTableView::paintCell( QPainter * p, int row, int col, const QRect & cr,
                           bool selected, const QColorGroup &cg )
{
    if ( selected )
    {
        QColorGroup hi(cg);
        hi.setColor( QColorGroup::Highlight, m_selectionColor );
        hi.setColor( QColorGroup::HighlightedText, Qt::black );
        QDataTable::paintCell( p, row, col, cr, selected, hi );
        return ;
    }

    int w = cr.width();
    int h = cr.height();
    int x2 = w - 1;
    int y2 = h - 1;

    QTableItem *itm = item( row, col );
    if ( itm )
    {
        p->save();
        itm->paint( p, cg, cr, selected );
        p->restore();
    }
    else
    {
        if ( row & 1 )
            p->fillRect( 0, 0, w, h, QBrush( QColor( 238, 246, 255 ) ) );
        else
            p->fillRect( 0, 0, w, h, cg.brush( QColorGroup::Base ) );
    }

    // Draw our lines
    QPen pen( p->pen() );
    int gridColor = style().styleHint( QStyle::SH_Table_GridLineColor, this );
    if ( gridColor != -1 )
    {
        const QPalette & pal = palette();
        if ( cg != colorGroup()
                && cg != pal.disabled()
                && cg != pal.inactive() )
            p->setPen( cg.mid() );
        else
            p->setPen( ( QRgb ) gridColor );
    }
    else
    {
        p->setPen( cg.mid() );
    }
    p->drawLine( x2, 0, x2, y2 );
    p->drawLine( 0, y2, x2, y2 );
    p->setPen( pen );

    if ( !sqlCursor() )
        return ;

    p->setPen( cg.text() );

    if ( sqlCursor() ->seek( row ) )
        paintField( p, sqlCursor() ->field( indexOf( col ) ), cr, selected );
}

void DataTableView::paintField( QPainter * p, const QSqlField* field, const QRect & cr, bool b )
{
    if ( !field )
        return ;

    DataRelation *relation = m_dataTable->dataRelation( field->name() );

    if ( relation )
    {
        DatabaseConnection * database = m_dataTable->project()->databaseConnection( m_dataTable->connection() );
        QSqlQuery query( "SELECT " + relation->field() +
                         " FROM " + relation->table() +
                         " WHERE " + relation->key() +
                         " = '" + field->value().toString() + "'", database->connection() );

        QString text;
        if ( query.next() )
            text = query.value( 0 ).toString();

        p->drawText( 2, 2, cr.width() - 4, cr.height() - 4, fieldAlignment( field ), text );
    }
    else
    {
        QDataTable::paintField( p, field, cr, b ) ;
    }
}

QWidget* DataTableView::beginEdit ( int, int, bool )
{
    //do nothing
    return 0L;
}

void DataTableView::sortColumn( int col, bool ascending, bool wholeRows )
{
    QDataTable::sortColumn( col, ascending, wholeRows );
    slotSelectRow( currentRow() );
}

void DataTableView::contentsContextMenuEvent( QContextMenuEvent *e )
{
    KPopupMenu *context = new KPopupMenu( this );
    action( "insert" ) ->plug( context );
    if ( !m_dataTable->isRootTable() ) action( "change" ) ->plug( context );
    action( "delete" ) ->plug( context );
    context->exec( e->globalPos() );
}

void DataTableView::contentsMouseMoveEvent( QMouseEvent * e )
{
    QDataTable::contentsMouseMoveEvent( e );
    slotSelectRow( rowAt( e->pos().y() ) );
}

#include "datatableview.moc"
