#ifndef n_TimeStamp_H
#define n_TimeStamp_H

/*
 *
 * nucleo/core/TimeStamp.H --
 *
 * Copyright (C) Nicolas Roussel
 *
 * See the file LICENSE for information on usage and redistribution of
 * this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

#include <stdint.h>

#include <string>

namespace nucleo {

  class TimeStamp {

  public:

    /**
	* This class is used to represent points in time.
	* 
	* The date+time information is stored as a single 64-bit integer
	* with millisecond precision, which makes it easy to compute
	* durations. It can also be converted to/from a UTC ISO 8601
	* string-based representation.
	* 
	* TimeStamps are described as the number of milliseconds elapsed
	* since 1970-01-01T00:00:00.000Z. The minimum date+time that can
	* be represented is 1901-12-13T20:45:52Z. The maximum one is
	* 2038-01-19T03:14:07Z. These values were chosen to remain
	* compatible with POSIX 32-bit based time_t representations.
	*
	* Useful links: 
	*     - http://en.wikipedia.org/wiki/Unix_time
	*     - http://en.wikipedia.org/wiki/ISO_8601
	*
	*/

    typedef int64_t inttype ;

    static inttype undef, min, max ;

  protected:

    inttype milliseconds ;

    static inttype now(void) ;
    static inttype string2int(std::string s) ;
    static std::string int2string(inttype i) ;

  public:

    TimeStamp(void) : milliseconds(now()) {}
    TimeStamp(inttype ms) ;
    TimeStamp(std::string s) : milliseconds(string2int(s)) {}

    bool operator<(const TimeStamp &other) const ;
    bool operator>(const TimeStamp &other) const ;
    bool operator==(const TimeStamp &other) const ;

    // the number of milliseconds since 1970-01-01T00:00:00.000Z
    inttype getAsInt(void) { return milliseconds ; }

    // a UTC ISO 8601 combined data/time string (e.g. 2006-02-25T11:59:12.113Z)
    std::string getAsString(void) { return int2string(milliseconds) ; } 

    // Use 0 for each component you don't wan't
    void getAsLocalTime(int *year, int *month, int *day,
				    int *hour, int *min, int *sec, int *msec) ;

    static inttype createAsInt(void) { return now() ; }
    static inttype createAsIntFromString(std::string s) { return string2int(s) ; }

    static std::string createAsString(void) { return int2string(now()) ; }
    static std::string createAsStringFromInt(inttype i) { return int2string(i) ; }

  } ;

}

#endif
