# -*- coding: utf-8 -*-

# Copyright (c) 2006 - 2008 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the Editor Highlighting Styles configuration page.
"""

import os

from PyQt4.QtCore import QString, pyqtSignature
from PyQt4.QtGui import QPalette

from KdeQt import KQColorDialog, KQFontDialog

from ConfigurationPageBase import ConfigurationPageBase
from Ui_EditorHighlightingStylesPage import Ui_EditorHighlightingStylesPage

import Preferences

class EditorHighlightingStylesPage(ConfigurationPageBase, Ui_EditorHighlightingStylesPage):
    """
    Class implementing the Editor Highlighting Styles configuration page.
    """
    def __init__(self, lexers):
        """
        Constructor
        
        @param lexers reference to the lexers dictionary
        """
        ConfigurationPageBase.__init__(self)
        self.setupUi(self)
        self.setObjectName("EditorHighlightingStylesPage")
        
        self.lexers = lexers
        
        # set initial values
        languages = [''] + self.lexers.keys()
        languages.sort()
        for lang in languages:
            self.lexerLanguageComboBox.addItem(lang)
        self.on_lexerLanguageComboBox_activated(QString(''))
        
    def save(self):
        """
        Public slot to save the Editor Highlighting Styles configuration.
        """
        for lexer in self.lexers.values():
            lexer.writeSettings(Preferences.Prefs.settings, "Scintilla")
        
    @pyqtSignature("QString")
    def on_lexerLanguageComboBox_activated(self, language):
        """
        Private slot to fill the style combo of the source page.
        
        @param language The lexer language (string or QString)
        """
        self.styleElementList.clear()
        self.styleGroup.setEnabled(False)
        
        if language.isEmpty():
            return
        
        try:
            self.lexer = self.lexers[unicode(language)]
        except:
            return
        
        self.styleGroup.setEnabled(True)
        self.styleElementList.addItems(self.lexer.styles)
        self.styleElementList.setCurrentRow(0)
        
    def on_styleElementList_currentRowChanged(self, index):
        """
        Private method to set up the style element part of the source page.
        
        @param index the style index.
        """
        try:
            self.style = self.lexer.ind2style[index]
        except KeyError:
            return
        
        colour = self.lexer.color(self.style)
        paper = self.lexer.paper(self.style)
        eolfill = self.lexer.eolFill(self.style)
        font = self.lexer.font(self.style)
        
        self.sampleText.setFont(font)
        pl = self.sampleText.palette()
        pl.setColor(QPalette.Text, colour)
        pl.setColor(QPalette.Base, paper)
        self.sampleText.repaint()
        self.eolfillCheckBox.setChecked(eolfill)
        
    @pyqtSignature("")
    def on_foregroundButton_clicked(self):
        """
        Private method used to select the foreground colour of the selected style 
        and lexer.
        """
        colour = KQColorDialog.getColor(self.lexer.color(self.style))
        if colour.isValid():
            pl = self.sampleText.palette()
            pl.setColor(QPalette.Text, colour)
            self.sampleText.repaint()
            if len(self.styleElementList.selectedItems()) > 1:
                for selItem in self.styleElementList.selectedItems():
                    style = self.lexer.ind2style[self.styleElementList.row(selItem)]
                    self.lexer.setColor(colour, style)
            else:
                self.lexer.setColor(colour, self.style)
        
    @pyqtSignature("")
    def on_backgroundButton_clicked(self):
        """
        Private method used to select the background colour of the selected style 
        and lexer.
        """
        colour = KQColorDialog.getColor(self.lexer.paper(self.style))
        if colour.isValid():
            pl = self.sampleText.palette()
            pl.setColor(QPalette.Base, colour)
            self.sampleText.repaint()
            if len(self.styleElementList.selectedItems()) > 1:
                for selItem in self.styleElementList.selectedItems():
                    style = self.lexer.ind2style[self.styleElementList.row(selItem)]
                    self.lexer.setPaper(colour, style)
            else:
                self.lexer.setPaper(colour, self.style)
        
    @pyqtSignature("")
    def on_allBackgroundColoursButton_clicked(self):
        """
        Private method used to select the background colour of all styles of a 
        selected lexer.
        """
        colour = KQColorDialog.getColor(self.lexer.paper(self.style))
        if colour.isValid():
            pl = self.sampleText.palette()
            pl.setColor(QPalette.Base, colour)
            self.sampleText.repaint()
            for style in self.lexer.ind2style.values():
                self.lexer.setPaper(colour, style)
        
    @pyqtSignature("")
    def on_fontButton_clicked(self):
        """
        Private method used to select the font of the selected style and lexer.
        """
        font, ok = KQFontDialog.getFont(self.lexer.font(self.style))
        if ok:
            self.sampleText.setFont(font)
            if len(self.styleElementList.selectedItems()) > 1:
                for selItem in self.styleElementList.selectedItems():
                    style = self.lexer.ind2style[self.styleElementList.row(selItem)]
                    self.lexer.setFont(font, style)
            else:
                self.lexer.setFont(font, self.style)
        
    @pyqtSignature("")
    def on_allFontsButton_clicked(self):
        """
        Private method used to change the font of all styles of a selected lexer.
        """
        font, ok = KQFontDialog.getFont(self.lexer.font(self.style))
        if ok:
            self.sampleText.setFont(font)
            for style in self.lexer.ind2style.values():
                self.lexer.setFont(font, style)
        
    def on_eolfillCheckBox_toggled(self, b):
        """
        Private method used to set the eolfill for the selected style and lexer.
        
        @param b Flag indicating enabled or disabled state.
        """
        self.lexer.setEolFill(b, self.style)
        
    def saveState(self):
        """
        Public method to save the current state of the widget.
        
        @return array containing the index of the selected lexer language (integer)
            and the index of the selected lexer entry (integer)
        """
        savedState = [
            self.lexerLanguageComboBox.currentIndex(),
            self.styleElementList.currentRow(),
        ]
        return savedState
        
    def setState(self, state):
        """
        Public method to set the state of the widget.
        
        @param state state data generated by saveState
        """
        self.lexerLanguageComboBox.setCurrentIndex(state[0])
        self.on_lexerLanguageComboBox_activated(self.lexerLanguageComboBox.currentText())
        self.styleElementList.setCurrentRow(state[1])

def create(dlg):
    """
    Module function to create the configuration page.
    
    @param dlg reference to the configuration dialog
    """
    page = EditorHighlightingStylesPage(dlg.getLexers())
    return page
