# -*- coding: utf-8 -*-

# Copyright (c) 2002 - 2008 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Package implementing the preferences interface.

The preferences interface consists of a class, which defines the default
values for all configuration items and stores the actual values. These
values are read and written to the eric4 preferences file by module
functions. On Windows the data is located in the registry, everywhere 
else it is stored in a file in a subdirectory of the users home directory.
The individual configuration data is accessed by accessor functions defined
on the module level. The module is simply imported wherever it is needed
with the statement 'import Preferences'. Do not use 'from Preferences import *'
to import it.
"""

import sys
import os
import fnmatch

from PyQt4 import QtCore, QtGui
from PyQt4 import Qsci

import QScintilla.Lexers

from Globals import settingsNameOrganization, settingsNameGlobal, settingsNameRecent

from Project.ProjectBrowser import SourcesBrowserFlag, FormsBrowserFlag, \
    ResourcesBrowserFlag, TranslationsBrowserFlag, InterfacesBrowserFlag, \
    OthersBrowserFlag, AllBrowsersFlag

class Prefs(object):
    """
    A class to hold all configuration items for the application.
    """
    # defaults for the variables window
    varDefaults = {
        "LocalsFilter" : [],
        "GlobalsFilter" : []
    }
    
    # defaults for the debugger
    debuggerDefaults = {
        "RemoteDbgEnabled" : 0,
        "RemoteHost" : "",
        "RemoteExecution" : "",
        "PassiveDbgEnabled" : 0,
        "PassiveDbgPort" : 42424,
        "AutomaticReset" : 0,
        "Autosave" : 0,
        "CustomPythonInterpreter" : 0,
        "PythonInterpreter" : "",
        "RubyInterpreter" : "/usr/bin/ruby",
        "DebugClientType" : "standard",     # supported "standard", "threaded", "custom"
        "DebugClient" : "",
        "DebugEnvironmentReplace" : 0,
        "DebugEnvironment" : "",
        "PythonRedirect" : 1,
        "PythonNoEncoding" : 0,
        "RubyRedirect" : 1,
        "ConsoleDbgEnabled" : 0,
        "ConsoleDbgCommand" : "",
        "PathTranslation" : 0,
        "PathTranslationRemote" : "",
        "PathTranslationLocal" : "",
        "NetworkInterface" : "127.0.0.1",
    }
    debuggerDefaults["AllowedHosts"] = QtCore.QStringList("127.0.0.1")
    
    uiDefaults = {
        "Language" : "System",
        "Style" : "System",
        "StyleSheet" : "",
        "ViewManager" : "tabview",
        "LayoutType" : "DockWindows",
        # allowed values are "DockWindows" and "FloatingWindows"
        "LayoutShellEmbedded" : 0,          # 0 = separate
                                            # 1 = embedded in debug browser
        "LayoutFileBrowserEmbedded" : 1,    # 0 = separate
                                            # 1 = embedded in debug browser
                                            # 2 = embedded in project browser
        "BrowsersListFoldersFirst" : 1,
        "BrowsersHideNonPublic" : 0,
        "SingleApplicationMode" : 0,
        "CaptionShowsFilename" : 1,
        "CaptionFilenameLength" : 100,
        "TopLeftByLeft" : 1, 
        "BottomLeftByLeft" : 0, 
        "TopRightByRight" : 1, 
        "BottomRightNyRight" : 0, 
        "TabViewManagerFilenameLength" : 40,
        # the order in ViewProfiles is Project-Browser, File-Browser,
        # Debug-Browser, Python-Shell, Log-Viewer, Task-Viewer,
        # Templates-Viewer
        "ViewProfiles" : {
            "edit"  : [
                        # visibility
                        [ 1,  0,  0,  1,  1,  1,  1],
                        # saved state main window with dock windows
                        "",
                        # saved states floating windows
                        ["", "", "", "", "", "", ""],
                        # saved state main window with floating windows
                        ""
                      ],
            "debug" : [
                        # visibility
                        [ 0,  0,  1,  1,  1,  1,  0], 
                        # saved state main window with dock windows
                        "",
                        # saved states floating windows
                        ["", "", "", "", "", "", ""],
                        # saved state main window with floating windows
                        ""
                      ],
        },
        "ShowSplash" : 1,
        
        "PerformVersionCheck" : 4,      # 0 = off
                                        # 1 = at startup
                                        # 2 = daily
                                        # 3 = weekly
                                        # 4 = monthly
        "UseProxy" : 0,
        "ProxyHost" : "",
        "ProxyPort" : 80,
        "ProxyUser" : "",
        "ProxyPassword" : "",
        "UseIconProvider" : 1,
        
        "PluginRepositoryUrl" : \
            "http://die-offenbachs.homelinux.org/eric/plugins/repository.xml", 
    }
    if sys.platform == "win32" or sys.platform == "darwin":
        uiDefaults["UseKDEDialogs"] = 0
    else:
        uiDefaults["UseKDEDialogs"] = 1
    viewProfilesLength = len(uiDefaults["ViewProfiles"]["edit"][2])
    
    iconsDefaults = {
        "Path" : QtCore.QStringList(),
    }
    
    # defaults for the editor settings
    editorDefaults = {
        "AutosaveInterval" : 0,
        "TabWidth" : 4,
        "IndentWidth" : 4,
        "LinenoWidth" : 4,
        "IndentationGuides" : 1,
        "LinenoMargin" : 1,
        "FoldingMargin" : 1,
        "FoldingStyle" : 1,
        "TabForIndentation" : 0,
        "TabIndents" : 1,
        "ConvertTabsOnLoad" : 0,
        "AutomaticEOLConversion" : 1,
        "ShowWhitespace" : 0,
        "ShowEOL" : 0,
        "UseMonospacedFont" : 0,
        "WrapLongLines" : 1,
        "WarnFilesize" : 512,
        "ClearBreaksOnClose" : 1,
        "StripTrailingWhitespace" : 0, 
        
        "EdgeMode" : Qsci.QsciScintilla.EdgeNone,
        "EdgeColumn" : 80,
        
        "AutoIndentation" : 1,
        "BraceHighlighting" : 1,
        "CreateBackupFile" : 0,
        "CaretLineVisible" : 0,
        "CaretWidth" : 1,
        "ColourizeSelText" : 0,
        "CustomSelectionColours" : 0,
        "ExtendSelectionToEol" : 0,
        
        "AutoPrepareAPIs" : 0,
        
        "AutoCompletionEnabled" : 0,
        "AutoCompletionCaseSensitivity" : 1,
        "AutoCompletionReplaceWord" : 0,
        "AutoCompletionShowSingle" : 0,
        "AutoCompletionSource" : Qsci.QsciScintilla.AcsDocument,
        "AutoCompletionThreshold" : 2,
        "AutoCompletionFillups" : 0,
        
        "CallTipsEnabled" : 0,
        "CallTipsVisible" : -1,
        "CallTipsStyle"   : Qsci.QsciScintilla.CallTipsNoContext,
        
        "AutoCheckSyntax" : 1,
        "AutoReopen" : 0,
        
        "MiniContextMenu" : 0,
        
        "DefaultEncoding" : "utf8",
        "DefaultOpenFilter" : "",
        "DefaultSaveFilter" : "",
        
        # All (most) lexers
        "AllFoldCompact" : 1,
        
        # Python specifics
        "PythonBadIndentation" : 1,
        "PythonFoldComment" : 1,
        "PythonFoldString" : 1,
        "PythonAutoIndent" : 1,
        
        # C++ specifics
        "CppCaseInsensitiveKeywords" : 0, 
        "CppFoldComment" : 1,
        "CppFoldPreprocessor" : 0,
        "CppFoldAtElse" : 0,
        "CppIndentOpeningBrace" : 0,
        "CppIndentClosingBrace" : 0,
        
        # SQL specifics
        "SqlFoldComment" : 1,
        "SqlBackslashEscapes" : 0,
        
        # HTML specifics
        "HtmlFoldPreprocessor" : 0,
        "HtmlCaseSensitiveTags" : 0,
        
        # Perl specifics
        "PerlFoldComment" : 1,
        
        # Bash specifics
        "BashFoldComment" : 1,
        
        # CSS specifics
        "CssFoldComment" : 1,
        
        # D specifics
        "DFoldComment" : 1,
        "DFoldAtElse" : 0,
        "DIndentOpeningBrace" : 0,
        "DIndentClosingBrace" : 0,
        
        # Povray specifics
        "PovFoldComment" : 1,
        "PovFoldDirectives" : 0,
        
        # CMake specifics
        "CMakeFoldAtElse" : 0,
        
        # VHDL specifics
        "VHDLFoldComment" : 1,
        "VHDLFoldAtElse" : 1,
        "VHDLFoldAtBegin" : 1, 
        "VHDLFoldAtParenthesis" : 1, 
    }
    
    if sys.platform == "win32":
        editorDefaults["EOLMode"] = Qsci.QsciScintilla.EolWindows
    else:
        editorDefaults["EOLMode"] = Qsci.QsciScintilla.EolUnix
    
    editorColourDefaults = {
        "CurrentMarker"        : QtGui.QColor(QtCore.Qt.yellow),
        "ErrorMarker"          : QtGui.QColor(QtCore.Qt.red),
        "MatchingBrace"        : QtGui.QColor(QtCore.Qt.green),
        "MatchingBraceBack"    : QtGui.QColor(QtCore.Qt.white),
        "NonmatchingBrace"     : QtGui.QColor(QtCore.Qt.red),
        "NonmatchingBraceBack" : QtGui.QColor(QtCore.Qt.white),
        "CallTipsBackground"   : QtGui.QColor(QtCore.Qt.white),
        "CaretForeground"      : QtGui.QColor(QtCore.Qt.black),
        "CaretLineBackground"  : QtGui.QColor(QtCore.Qt.white),
        "Edge"                 : QtGui.QColor(QtCore.Qt.lightGray),
        "SelectionBackground"  : QtGui.QColor(QtCore.Qt.black),
        "SelectionForeground"  : QtGui.QColor(QtCore.Qt.white),
    }
    
    editorOtherFontsDefaults = {
        "MarginsFont"    : "Sans Serif,10,-1,5,50,0,0,0,0,0",
        "DefaultFont"    : "Sans Serif,10,-1,5,50,0,0,0,0,0",
        "MonospacedFont" : "Courier,10,-1,5,50,0,0,0,0,0"
    }
    
    editorAPIDefaults = {}
    for language in QScintilla.Lexers.getSupportedLanguages().keys():
        editorAPIDefaults[language] = QtCore.QStringList()
    
    editorLexerAssocDefaults = QScintilla.Lexers.getDefaultLexerAssociations()
    
    editorTypingDefaults = {
        "Python/EnabledTypingAids"  : 1, 
        "Python/InsertClosingBrace" : 1,
        "Python/IndentBrace"        : 1,
        "Python/SkipBrace"          : 1,
        "Python/InsertQuote"        : 1,
        "Python/DedentElse"         : 1,
        "Python/DedentExcept"       : 1,
        "Python/Py24StyleTry"       : 1, 
        "Python/InsertImport"       : 1,
        "Python/InsertSelf"         : 1,
        "Python/InsertBlank"        : 1,
        "Python/ColonDetection"     : 1,
        
        "Ruby/EnabledTypingAids"    : 1, 
        "Ruby/InsertClosingBrace"   : 1,
        "Ruby/IndentBrace"          : 1,
        "Ruby/SkipBrace"            : 1,
        "Ruby/InsertQuote"          : 1,
        "Ruby/InsertBlank"          : 1,
        "Ruby/InsertHereDoc"        : 1, 
        "Ruby/InsertInlineDoc"      : 1, 
    }
    
    editorExporterDefaults = {
        "HTML/WYSIWYG"          : 1, 
        "HTML/Folding"          : 0, 
        "HTML/OnlyStylesUsed"   : 0, 
        "HTML/FullPathAsTitle"  : 0, 
        "HTML/UseTabs"          : 0, 
        
        "RTF/WYSIWYG"           : 1, 
        "RTF/UseTabs"           : 0, 
        "RTF/Font"              : "Courier New,10,-1,5,50,0,0,0,0,0", 
        
        "PDF/Magnification"     : 0, 
        "PDF/Font"              : "Helvetica",  # must be Courier, Helvetica or Times
        "PDF/PageSize"          : "A4",         # must be A4 or Letter
        "PDF/MarginLeft"        : 36, 
        "PDF/MarginRight"       : 36, 
        "PDF/MarginTop"         : 36, 
        "PDF/MarginBottom"      : 36, 
        
        "TeX/OnlyStylesUsed"    : 0, 
        "TeX/FullPathAsTitle"   : 0, 
    }
    
    # defaults for the printer settings
    printerDefaults = {
        "PrinterName" : "",
        "ColorMode" : 1,
        "FirstPageFirst" : 1,
        "Magnification" : -3,
        "Orientation" : 0,
        "PageSize": 0,
        "HeaderFont" : "Serif,10,-1,5,50,0,0,0,0,0"
    }
    
    # defaults for the project settings
    projectDefaults = {
        "SearchNewFiles" : 0,
        "SearchNewFilesRecursively" : 0,
        "AutoIncludeNewFiles" : 0,
        "AutoLoadSession" : 0,
        "AutoSaveSession" : 0,
        "SessionAllBreakpoints" : 0,
        "CompressedProjectFiles" : 0,
        "XMLTimestamp" : 1,
        "AutoCompileForms" : 0,
        "AutoCompileResources" : 0,
        "AutoLoadDbgProperties" : 0,
        "AutoSaveDbgProperties" : 0,
        "HideGeneratedForms" : 0,
        "FollowEditor" : 1, 
    }
    
    # defaults for the project browser flags settings
    projectBrowserFlagsDefaults = {
        "Qt4" : 
            SourcesBrowserFlag | \
            FormsBrowserFlag | \
            ResourcesBrowserFlag | \
            TranslationsBrowserFlag | \
            InterfacesBrowserFlag | \
            OthersBrowserFlag, 
        "Qt4C" : 
            SourcesBrowserFlag | \
            ResourcesBrowserFlag | \
            TranslationsBrowserFlag | \
            InterfacesBrowserFlag | \
            OthersBrowserFlag, 
        "E4Plugin" : 
            SourcesBrowserFlag | \
            FormsBrowserFlag | \
            ResourcesBrowserFlag | \
            TranslationsBrowserFlag | \
            InterfacesBrowserFlag | \
            OthersBrowserFlag, 
        "Qt" : 
            SourcesBrowserFlag | \
            FormsBrowserFlag | \
            TranslationsBrowserFlag | \
            InterfacesBrowserFlag | \
            OthersBrowserFlag, 
        "Kde" : 
            SourcesBrowserFlag | \
            InterfacesBrowserFlag | \
            OthersBrowserFlag, 
        "Django" : 
            SourcesBrowserFlag | \
            FormsBrowserFlag | \
            InterfacesBrowserFlag | \
            OthersBrowserFlag,
        # TODO: 1. move to Plugin
        "TurboGears" : 
            SourcesBrowserFlag | \
            FormsBrowserFlag | \
            InterfacesBrowserFlag | \
            OthersBrowserFlag, 
        "wxPython" :
            SourcesBrowserFlag | \
            InterfacesBrowserFlag | \
            OthersBrowserFlag, 
        "Console" : 
            SourcesBrowserFlag | \
            InterfacesBrowserFlag | \
            OthersBrowserFlag, 
        "Other" : 
            SourcesBrowserFlag | \
            InterfacesBrowserFlag | \
            OthersBrowserFlag, 
    }
    
    # defaults for the project browser colour settings
    projectBrowserColourDefaults = {
        "Highlighted"        : QtGui.QColor(QtCore.Qt.red),
        
        "VcsAdded"           : QtGui.QColor(QtCore.Qt.blue),
        "VcsConflict"        : QtGui.QColor(QtCore.Qt.red),
        "VcsModified"        : QtGui.QColor(QtCore.Qt.yellow),
        "VcsUpdate"          : QtGui.QColor(QtCore.Qt.green),
    }
    
    # defaults for the help settings
    helpDefaults = {
        "HelpViewerType" : 1,      # this coresponds with the radio button id
        "CustomViewer" : "",
        "PythonDocDir" : "",
        "QtDocDir" : "",
        "Qt4DocDir" : "",
        "PyQt4DocDir" : "",
        "SingleHelpWindow" : 1,
        "SaveGeometry" : 1,
    }
    
    # defaults for system settings
    sysDefaults = {
        "StringEncoding" : "utf8",
        "IOEncoding"     : "utf8",
    }
    
    # defaults for the shell settings
    shellDefaults = {
        "LinenoWidth" : 4,
        "LinenoMargin" : 1,
        "AutoCompletionEnabled" : 1,
        "CallTipsEnabled" : 1,
        "WrapEnabled" : 1,
        "MaxHistoryEntries" : 100,
        "SyntaxHighlightingEnabled" : 1,
    }

    # defaults for Qt related stuff
    qtDefaults = {
        "QtDir" : "",
        "ExportQtDir" : 0,
        "DisableQt3" : 0,
        "Qt4TranslationsDir" : "",
        "QtToolsPrefix3" : "", 
        "QtToolsPostfix3" : "", 
        "QtToolsPrefix4" : "", 
        "QtToolsPostfix4" : "", 
    }
    
    # defaults for corba related stuff
    corbaDefaults = {
        "omniidl" : "omniidl"
    }
    
    # defaults for user related stuff
    userDefaults = {
        "Email" : "",
        "MailServer" : "",
        "Signature" : "",
        "MailServerAuthentication" : 0,
        "MailServerUser" : "",
        "MailServerPassword" : "",
    }
    
    # defaults for vcs related stuff
    vcsDefaults = {
        "AutoClose" : 0,
        "AutoSaveFiles" : 1,
        "AutoSaveProject" : 1,
        "StatusMonitorInterval" : 30,
        "MonitorLocalStatus" : 0, 
    }
    
    # defaults for tasks related stuff
    tasksDefaults = {
        "TasksMarkers"       : "TO" + "DO:", 
        "TasksMarkersBugfix" : "FIX" + "ME:",
        # needed to keep it from being recognized as a task
        "TasksColour"          : QtGui.QColor(QtCore.Qt.black),
        "TasksBugfixColour"    : QtGui.QColor(QtCore.Qt.red),
        "TasksBgColour"        : QtGui.QColor(QtCore.Qt.white),
        "TasksProjectBgColour" : QtGui.QColor(QtCore.Qt.lightGray),
    }
    
    # defaults for templates related stuff
    templatesDefaults = {
        "AutoOpenGroups" : 1,
        "SingleDialog"   : 0,
        "ShowTooltip"    : 0,
        "SeparatorChar"  : "$",
    }
    
    # defaults for plugin manager related stuff
    pluginManagerDefaults = {
        "ActivateExternal" : 1,
        "DownloadPath" : ""
    }
    
    # defaults for the printer settings
    graphicsDefaults = {
        "Font" : "SansSerif,10,-1,5,50,0,0,0,0,0"
    }
    
    # defaults for geometry
    geometryDefaults = {
        "HelpViewerGeometry" : QtCore.QByteArray(),
        "MainGeometry"       : QtCore.QByteArray(),
    }

    # if true, revert layouts to factory defaults
    resetLayout = False

def readToolGroups(prefClass = Prefs):
    """
    Module function to read the tool groups configuration.
    
    @param prefClass preferences class used as the storage area
    @return list of tuples defing the tool groups
    """
    toolGroups = []
    groups = prefClass.settings.value("Toolgroups/Groups", 
        QtCore.QVariant(0)).toInt()[0]
    for groupIndex in range(groups):
        groupName = \
            prefClass.settings.value("Toolgroups/%02d/Name" % groupIndex).toString()
        group = [unicode(groupName), []]
        items = prefClass.settings.value("Toolgroups/%02d/Items" % groupIndex, 
            QtCore.QVariant(0)).toInt()[0]
        for ind in range(items):
            menutext = prefClass.settings.value(\
                "Toolgroups/%02d/%02d/Menutext" % (groupIndex, ind)).toString()
            icon = prefClass.settings.value(\
                "Toolgroups/%02d/%02d/Icon" % (groupIndex, ind)).toString()
            executable = prefClass.settings.value(\
                "Toolgroups/%02d/%02d/Executable" % (groupIndex, ind)).toString()
            arguments = prefClass.settings.value(\
                "Toolgroups/%02d/%02d/Arguments" % (groupIndex, ind)).toString()
            redirect = prefClass.settings.value(\
                "Toolgroups/%02d/%02d/Redirect" % (groupIndex, ind)).toString()
            
            if not menutext.isEmpty():
                if unicode(menutext) == '--':
                    tool = {
                        'menutext' : '--',
                        'icon' : '',
                        'executable' : '',
                        'arguments' : '',
                        'redirect' : 'no',
                    }
                    group[1].append(tool)
                elif not executable.isEmpty():
                    tool = {
                        'menutext' : unicode(menutext),
                        'icon' : unicode(icon),
                        'executable' : unicode(executable),
                        'arguments' : unicode(arguments),
                        'redirect' : unicode(redirect),
                    }
                    group[1].append(tool)
        toolGroups.append(group)
    currentGroup = prefClass.settings.value("Toolgroups/Current Group", 
        QtCore.QVariant(-1)).toInt()[0]
    return toolGroups, currentGroup
    
def saveToolGroups(toolGroups, currentGroup, prefClass = Prefs):
    """
    Module function to write the tool groups configuration.
    
    @param toolGroups reference to the list of tool groups
    @param currentGroup index of the currently selected tool group (integer)
    @param prefClass preferences class used as the storage area
    """
    # first step, remove all tool group entries
    prefClass.settings.remove("Toolgroups")
    
    # second step, write the tool group entries
    prefClass.settings.setValue("Toolgroups/Groups", QtCore.QVariant(len(toolGroups)))
    groupIndex = 0
    for group in toolGroups:
        prefClass.settings.setValue("Toolgroups/%02d/Name" % groupIndex,
            QtCore.QVariant(group[0]))
        prefClass.settings.setValue("Toolgroups/%02d/Items" % groupIndex,
            QtCore.QVariant(len(group[1])))
        ind = 0
        for tool in group[1]:
            prefClass.settings.setValue(\
                "Toolgroups/%02d/%02d/Menutext" % (groupIndex, ind), 
                QtCore.QVariant(tool['menutext']))
            prefClass.settings.setValue(\
                "Toolgroups/%02d/%02d/Icon" % (groupIndex, ind), 
                QtCore.QVariant(tool['icon']))
            prefClass.settings.setValue(\
                "Toolgroups/%02d/%02d/Executable" % (groupIndex, ind), 
                QtCore.QVariant(tool['executable']))
            prefClass.settings.setValue(\
                "Toolgroups/%02d/%02d/Arguments" % (groupIndex, ind), 
                QtCore.QVariant(tool['arguments']))
            prefClass.settings.setValue(\
                "Toolgroups/%02d/%02d/Redirect" % (groupIndex, ind), 
                QtCore.QVariant(tool['redirect']))
            ind += 1
        groupIndex += 1
    prefClass.settings.setValue(\
        "Toolgroups/Current Group", QtCore.QVariant(currentGroup))
    
def initPreferences():
    """
    Module function to initialize the central configuration store. 
    """
    Prefs.settings = QtCore.QSettings(QtCore.QSettings.UserScope, 
        settingsNameOrganization, settingsNameGlobal)
    if sys.platform != "win32":
        hp = QtCore.QDir.homePath()
        dn = QtCore.QDir(hp)
        dn.mkdir(".eric4")
    
def syncPreferences(prefClass = Prefs):
    """
    Module function to sync the preferences to disk.
    
    In addition to syncing, the central configuration store is reinitialized as well.
    
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("General/Configured", QtCore.QVariant(1))
    initPreferences()
    
def isConfigured(prefClass = Prefs):
    """
    Module function to check, if the the application has been configured.
    
    @param prefClass preferences class used as the storage area
    @return flag indicating the configured status (boolean)
    """
    return prefClass.settings.value("General/Configured", QtCore.QVariant(0)).toInt()[0]
    
def initRecentSettings():
    """
    Module function to initialize the central configuration store for recently
    opened files and projects. 
    
    This function is called once upon import of the module.
    """
    Prefs.rsettings = QtCore.QSettings(QtCore.QSettings.UserScope, 
        settingsNameOrganization, settingsNameRecent)
    
def getVarFilters(prefClass = Prefs):
    """
    Module function to retrieve the variables filter settings.
    
    @param prefClass preferences class used as the storage area
    @return a tuple defing the variables filter
    """
    localsFilter = eval(unicode(prefClass.settings.value("Variables/LocalsFilter", 
        QtCore.QVariant(unicode(prefClass.varDefaults["LocalsFilter"]))).toString()))
    globalsFilter = eval(unicode(prefClass.settings.value("Variables/GlobalsFilter", 
        QtCore.QVariant(unicode(prefClass.varDefaults["GlobalsFilter"]))).toString()))
    return (localsFilter, globalsFilter)
    
def setVarFilters(filters, prefClass = Prefs):
    """
    Module function to store the variables filter settings.
    
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("Variables/LocalsFilter",
        QtCore.QVariant(unicode(filters[0])))
    prefClass.settings.setValue("Variables/GlobalsFilter",
        QtCore.QVariant(unicode(filters[1])))
    
def getDebugger(key, prefClass = Prefs):
    """
    Module function to retrieve the debugger settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested debugger setting
    """
    if key in ["RemoteDbgEnabled", "PassiveDbgEnabled", \
                "PassiveDbgPort", "CustomPythonInterpreter", \
                "AutomaticReset", "DebugEnvironmentReplace", \
                "PythonRedirect", "PythonNoEncoding", "RubyRedirect", \
                "ConsoleDbgEnabled", "PathTranslation", "Autosave"]:
        return prefClass.settings.value("Debugger/" + key,
            QtCore.QVariant(prefClass.debuggerDefaults[key])).toInt()[0]
    
    if key in ["RemoteHost", "RemoteExecution", "PythonInterpreter", \
                "RubyInterpreter", "DebugClient", "DebugClientType", \
                "DebugEnvironment", "ConsoleDbgCommand", \
                "PathTranslationRemote", "PathTranslationLocal", \
                "NetworkInterface"]:
        return prefClass.settings.value("Debugger/" + key,
            QtCore.QVariant(prefClass.debuggerDefaults[key])).toString()
    
    if key in ["AllowedHosts"]:
        return prefClass.settings.value("Debugger/" + key,
            QtCore.QVariant(prefClass.debuggerDefaults[key])).toStringList()
    
def setDebugger(key, value, prefClass = Prefs):
    """
    Module function to store the debugger settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("Debugger/" + key, QtCore.QVariant(value))

def getUILanguage(prefClass = Prefs):
    """
    Module function to retrieve the language for the user interface.
    
    @param prefClass preferences class used as the storage area
    @return the language for the UI
    """
    lang = \
        prefClass.settings.value("UI/Language",
            QtCore.QVariant(prefClass.uiDefaults["Language"])).toString()
    if unicode(lang) == "None" or unicode(lang) == "" or lang is None:
        return None
    else:
        return unicode(lang)
    
def setUILanguage(lang, prefClass = Prefs):
    """
    Module function to store the language for the user interface.
    
    @param lang the language
    @param prefClass preferences class used as the storage area
    """
    if lang is None:
        prefClass.settings.setValue("UI/Language", QtCore.QVariant("None"))
    else:
        prefClass.settings.setValue("UI/Language", QtCore.QVariant(lang))

def getUILayout(prefClass = Prefs):
    """
    Module function to retrieve the layout for the user interface.
    
    @param prefClass preferences class used as the storage area
    @return the UI layout as a tuple of main layout, flag for
        an embedded shell and a value for an embedded file browser
    """
    layout = (\
        prefClass.settings.value("UI/LayoutType", 
            QtCore.QVariant(prefClass.uiDefaults["LayoutType"])).toString(), 
        prefClass.settings.value("UI/LayoutShellEmbedded", 
            QtCore.QVariant(prefClass.uiDefaults["LayoutShellEmbedded"]))\
            .toInt()[0], 
        prefClass.settings.value("UI/LayoutFileBrowserEmbedded", 
            QtCore.QVariant(prefClass.uiDefaults["LayoutFileBrowserEmbedded"]))\
            .toInt()[0], 
    )
    return layout
    
def setUILayout(layout, prefClass = Prefs):
    """
    Module function to store the layout for the user interface.
    
    @param layout the layout type
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("UI/LayoutType", 
        QtCore.QVariant(layout[0]))
    prefClass.settings.setValue("UI/LayoutShellEmbedded", 
        QtCore.QVariant(layout[1]))
    prefClass.settings.setValue("UI/LayoutFileBrowserEmbedded", 
        QtCore.QVariant(layout[2]))

def getViewManager(prefClass = Prefs):
    """
    Module function to retrieve the selected viewmanager type.
    
    @param prefClass preferences class used as the storage area
    @return the viewmanager type
    """
    return unicode(prefClass.settings.value("UI/ViewManager",
        QtCore.QVariant(prefClass.uiDefaults["ViewManager"])).toString())
    
def setViewManager(vm, prefClass = Prefs):
    """
    Module function to store the selected viewmanager type.
    
    @param vm the viewmanager type
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("UI/ViewManager", QtCore.QVariant(vm))

def getUI(key, prefClass = Prefs):
    """
    Module function to retrieve the various UI settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested UI setting
    """
    if key in ["BrowsersListFoldersFirst", "BrowsersHideNonPublic",
                "SingleApplicationMode", "TabViewManagerFilenameLength",
                "CaptionShowsFilename", "CaptionFilenameLength",
                "UseKDEDialogs", "ShowSplash", "PerformVersionCheck",
                "UseProxy", "ProxyPort", "UseIconProvider", 
                "TopLeftByLeft", "BottomLeftByLeft", 
                "TopRightByRight", "BottomRightNyRight"]:
        return prefClass.settings.value("UI/" + key,
            QtCore.QVariant(prefClass.uiDefaults[key])).toInt()[0]
    if key in ["Style", "StyleSheet", 
                "ProxyHost", "ProxyUser", "ProxyPassword", 
                "PluginRepositoryUrl"]:
        return prefClass.settings.value("UI/" + key,
                QtCore.QVariant(prefClass.uiDefaults[key])).toString()
    if key == "ViewProfiles":
        v = prefClass.settings.value("UI/ViewProfiles")
        if v.isValid():
            viewProfiles = eval(unicode(v.toString()))
            for name in ["edit", "debug"]:
                vpLength = len(viewProfiles[name][0])
                if vpLength < prefClass.viewProfilesLength:
                    viewProfiles[name][0].extend(\
                        prefClass.uiDefaults["ViewProfiles"][name][0][vpLength:])
                vpLength = len(viewProfiles[name][2])
                if vpLength < prefClass.viewProfilesLength:
                    viewProfiles[name][2].extend(\
                        prefClass.uiDefaults["ViewProfiles"][name][2][vpLength:])
        else:
            viewProfiles = prefClass.uiDefaults["ViewProfiles"]
        return viewProfiles
    
def setUI(key, value, prefClass = Prefs):
    """
    Module function to store the various UI settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    if key == "ViewProfiles":
        prefClass.settings.setValue("UI/" + key, QtCore.QVariant(unicode(value)))
    else:
        prefClass.settings.setValue("UI/" + key, QtCore.QVariant(value))
    
def getIcons(key, prefClass = Prefs):
    """
    Module function to retrieve the various Icons settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested Icons setting
    """
    dirlist = prefClass.settings.value("UI/Icons/" + key)
    if dirlist.isValid():
        return dirlist.toStringList()
    else:
        return prefClass.iconsDefaults[key]
    
def setIcons(key, value, prefClass = Prefs):
    """
    Module function to store the various Icons settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("UI/Icons/" + key, QtCore.QVariant(value))
    
def getEditor(key, prefClass = Prefs):
    """
    Module function to retrieve the various editor settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested editor setting
    """
    if key in ["DefaultEncoding", "DefaultOpenFilter", "DefaultSaveFilter"]:
        return prefClass.settings.value("Editor/" + key,
            QtCore.QVariant(prefClass.editorDefaults[key])).toString()
    else:
        return prefClass.settings.value("Editor/" + key,
            QtCore.QVariant(prefClass.editorDefaults[key])).toInt()[0]
    
def setEditor(key, value, prefClass = Prefs):
    """
    Module function to store the various editor settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("Editor/" + key, QtCore.QVariant(value))
    
def getEditorColour(key, prefClass = Prefs):
    """
    Module function to retrieve the various editor marker colours.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested editor colour
    """
    col = prefClass.settings.value("Editor/Colour/" + key)
    if col.isValid():
        return QtGui.QColor(col.toString())
    else:
        return prefClass.editorColourDefaults[key]
    
def setEditorColour(key, value, prefClass = Prefs):
    """
    Module function to store the various editor marker colours.
    
    @param key the key of the colour to be set
    @param value the colour to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("Editor/Colour/" + key, QtCore.QVariant(value.name()))
    
def getEditorOtherFonts(key, prefClass = Prefs):
    """
    Module function to retrieve the various editor fonts except the lexer fonts.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested editor font (QFont)
    """
    f = QtGui.QFont()
    f.fromString(prefClass.settings.value("Editor/Other Fonts/" + key,
        QtCore.QVariant(prefClass.editorOtherFontsDefaults[key])).toString())
    return f
    
def setEditorOtherFonts(key, font, prefClass = Prefs):
    """
    Module function to store the various editor fonts except the lexer fonts.
    
    @param key the key of the font to be set
    @param font the font to be set (QFont)
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("Editor/Other Fonts/" + key,
        QtCore.QVariant(font.toString()))
    
def getEditorAPI(key, prefClass = Prefs):
    """
    Module function to retrieve the various lists of api files.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested list of api files (QStringList)
    """
    ap = prefClass.settings.value("Editor/APIs/" + key)
    if ap.isValid():
        apis = ap.toStringList()
        if len(apis) and apis[0].isEmpty():
            return prefClass.editorAPIDefaults[key]
        else:
            return apis
    else:
        return prefClass.editorAPIDefaults[key]
    
def setEditorAPI(key, apilist, prefClass = Prefs):
    """
    Module function to store the various lists of api files.
    
    @param key the key of the api to be set
    @param apilist the list of api files (QStringList)
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("Editor/APIs/" + key, QtCore.QVariant(apilist))
    
def getEditorLexerAssocs(prefClass = Prefs):
    """
    Module function to retrieve all lexer associations.
    
    @param prefClass preferences class used as the storage area
    @return a reference to the list of lexer associations
        (dictionary of strings)
    """
    editorLexerAssoc = {}
    prefClass.settings.beginGroup("Editor/LexerAssociations")
    keyList = prefClass.settings.childKeys()
    prefClass.settings.endGroup()
    if len(keyList) == 0:
        # build from scratch
        for key in prefClass.editorLexerAssocDefaults.keys():
            editorLexerAssoc[key] = \
                QtCore.QString(prefClass.editorLexerAssocDefaults[key])
    else:
        for key in keyList:
            key = unicode(key)
            if prefClass.editorLexerAssocDefaults.has_key(key):
                defaultValue = prefClass.editorLexerAssocDefaults[key]
            else:
                defaultValue = QtCore.QString()
            editorLexerAssoc[key] = \
                prefClass.settings.value("Editor/LexerAssociations/" + key,
                    QtCore.QVariant(defaultValue)).toString()
        if len(editorLexerAssoc.keys()) < len(prefClass.editorLexerAssocDefaults.keys()):
            # new default lexer associations
            for key in prefClass.editorLexerAssocDefaults.keys():
                if not editorLexerAssoc.has_key(key):
                    editorLexerAssoc[key] = \
                        QtCore.QString(prefClass.editorLexerAssocDefaults[key])
    return editorLexerAssoc
    
def setEditorLexerAssocs(assocs, prefClass = Prefs):
    """
    Module function to retrieve all lexer associations.
    
    @param assocs dictionary of lexer associations to be set
    @param prefClass preferences class used as the storage area
    """
    # first remove lexer associations that no longer exist, than save the rest
    prefClass.settings.beginGroup("Editor/LexerAssociations")
    keyList = prefClass.settings.childKeys()
    prefClass.settings.endGroup()
    for key in keyList:
        key = unicode(key)
        if not assocs.has_key(key):
            prefClass.settings.remove("Editor/LexerAssociations/" + key)
    for key in assocs.keys():
        prefClass.settings.setValue("Editor/LexerAssociations/" + key,
            QtCore.QVariant(assocs[key]))
    
def getEditorLexerAssoc(filename, prefClass = Prefs):
    """
    Module function to retrieve a lexer association.
    
    @param filename filename used to determine the associated lexer language (string)
    @param prefClass preferences class used as the storage area
    @return the requested lexer language (string)
    """
    for pattern, language in getEditorLexerAssocs().items():
        if fnmatch.fnmatch(filename, pattern):
            return unicode(language)
    
    return ""
    
def getEditorTyping(key, prefClass = Prefs):
    """
    Module function to retrieve the various editor typing settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested editor setting
    """
    return prefClass.settings.value("Editor/Typing/" + key,
        QtCore.QVariant(prefClass.editorTypingDefaults[key])).toInt()[0]
    
def setEditorTyping(key, value, prefClass = Prefs):
    """
    Module function to store the various editor typing settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("Editor/Typing/" + key, QtCore.QVariant(value))
    
def getEditorExporter(key, prefClass = Prefs):
    """
    Module function to retrieve the various editor exporters settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested editor setting
    """
    if key in ["RTF/Font"]:
        f = QtGui.QFont()
        f.fromString(prefClass.settings.value("Editor/Exporters/" + key,
            QtCore.QVariant(prefClass.editorExporterDefaults[key])).toString())
        return f
    elif key in ["PDF/Font", "PDF/PageSize"]:
        return prefClass.settings.value("Editor/Exporters/" + key, 
            QtCore.QVariant(prefClass.editorExporterDefaults[key])).toString()
    else:
        return prefClass.settings.value("Editor/Exporters/" + key,
            QtCore.QVariant(prefClass.editorExporterDefaults[key])).toInt()[0]

def setEditorExporter(key, value, prefClass = Prefs):
    """
    Module function to store the various editor exporters settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    if key in ["RTF/Font"]:
        v = value.toString()
    else:
        v = value
    prefClass.settings.setValue("Editor/Exporters/" + key, QtCore.QVariant(v))
    
def getPrinter(key, prefClass = Prefs):
    """
    Module function to retrieve the various printer settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested printer setting
    """
    if key in ["ColorMode", "FirstPageFirst", "Magnification", 
                "Orientation", "PageSize"]:
        return prefClass.settings.value("Printer/" + key,
            QtCore.QVariant(prefClass.printerDefaults[key])).toInt()[0]
    if key in ["PrinterName"]:
        return prefClass.settings.value("Printer/" + key,
            QtCore.QVariant(prefClass.printerDefaults[key])).toString()
    if key in ["HeaderFont"]:
        f = QtGui.QFont()
        f.fromString(prefClass.settings.value("Printer/" + key,
            QtCore.QVariant(prefClass.printerDefaults[key])).toString())
        return f

def setPrinter(key, value, prefClass = Prefs):
    """
    Module function to store the various printer settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    if key in ["HeaderFont"]:
        v = value.toString()
    else:
        v = value
    prefClass.settings.setValue("Printer/" + key, QtCore.QVariant(v))

def getShell(key, prefClass = Prefs):
    """
    Module function to retrieve the various shell settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested shell setting
    """
    return prefClass.settings.value("Shell/" + key,
        QtCore.QVariant(prefClass.shellDefaults[key])).toInt()[0]

def setShell(key, value, prefClass = Prefs):
    """
    Module function to store the various shell settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("Shell/" + key, QtCore.QVariant(value))

def getProject(key, prefClass = Prefs):
    """
    Module function to retrieve the various project handling settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested project setting
    """
    return prefClass.settings.value("Project/" + key,
        QtCore.QVariant(prefClass.projectDefaults[key])).toInt()[0]
    
def setProject(key, value, prefClass = Prefs):
    """
    Module function to store the various project handling settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("Project/" + key, QtCore.QVariant(value))
    
def getProjectBrowserFlags(key, prefClass = Prefs):
    """
    Module function to retrieve the various project browser flags settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested project setting
    """
    try:
        default = prefClass.projectBrowserFlagsDefaults[key]
    except KeyError:
        default = AllBrowsersFlag
    
    return prefClass.settings.value("Project/BrowserFlags/" + key,
        QtCore.QVariant(default)).toInt()[0]
    
def setProjectBrowserFlags(key, value, prefClass = Prefs):
    """
    Module function to store the various project browser flags settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("Project/BrowserFlags/" + key, QtCore.QVariant(value))
    
def setProjectBrowserFlagsDefault(key, value, prefClass = Prefs):
    """
    Module function to store the various project browser flags settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.projectBrowserFlagsDefaults[key] = value
    
def getProjectBrowserColour(key, prefClass = Prefs):
    """
    Module function to retrieve the various project browser colours.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested project browser colour
    """
    col = prefClass.settings.value("Project/Colour/" + key)
    if col.isValid():
        return QtGui.QColor(col.toString())
    else:
        return prefClass.projectBrowserColourDefaults[key]
    
def setProjectBrowserColour(key, value, prefClass = Prefs):
    """
    Module function to store the various project browser colours.
    
    @param key the key of the colour to be set
    @param value the colour to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("Project/Colour/" + key, QtCore.QVariant(value.name()))
    
def getQt4DocDir(prefClass = Prefs):
    """
    Module function to retrieve the Qt4DocDir setting.
    
    @param prefClass preferences class used as the storage area
    @return the requested Qt4DocDir setting (string)
    """
    s = prefClass.settings.value("Help/Qt4DocDir",
        QtCore.QVariant(prefClass.helpDefaults["Qt4DocDir"])).toString()
    if s.isEmpty():
        return os.getenv("QT4DOCDIR", "")
    else:
        return unicode(s)
    
def getHelp(key, prefClass = Prefs):
    """
    Module function to retrieve the various help settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested help setting
    """
    if key in ["HelpViewerType", "SingleHelpWindow", "SaveGeometry"]:
        return prefClass.settings.value("Help/" + key,
            QtCore.QVariant(prefClass.helpDefaults[key])).toInt()[0]
    if key in ["CustomViewer", \
                "PythonDocDir", "QtDocDir", "Qt4DocDir", "PyQt4DocDir"]:
        return prefClass.settings.value("Help/" + key,
            QtCore.QVariant(prefClass.helpDefaults[key])).toString()
    
def setHelp(key, value, prefClass = Prefs):
    """
    Module function to store the various help settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("Help/" + key, QtCore.QVariant(value))
    
def getSystem(key, prefClass = Prefs):
    """
    Module function to retrieve the various system settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested system setting
    """
    if key in ["StringEncoding", "IOEncoding"]: 
        return prefClass.settings.value("System/" + key, 
            QtCore.QVariant(prefClass.sysDefaults[key])).toString()
    
def setSystem(key, value, prefClass = Prefs):
    """
    Module function to store the various system settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("System/" + key, QtCore.QVariant(value))
    
def getQtDir(prefClass = Prefs):
    """
    Module function to retrieve the QtDir setting.
    
    @param prefClass preferences class used as the storage area
    @return the requested QtDir setting (string)
    """
    if getQt("DisableQt3"):
        return ""
    
    s = prefClass.settings.value("Qt/QtDir", 
        QtCore.QVariant(prefClass.qtDefaults["QtDir"])).toString()
    if s.isEmpty():
        return os.getenv("QTDIR", "")
    else:
        return unicode(s)
    
def getQt4TranslationsDir(prefClass = Prefs):
    """
    Module function to retrieve the Qt4TranslationsDir setting.
    
    @param prefClass preferences class used as the storage area
    @return the requested Qt4TranslationsDir setting (string)
    """
    s = prefClass.settings.value("Qt/Qt4TranslationsDir", 
        QtCore.QVariant(prefClass.qtDefaults["Qt4TranslationsDir"])).toString()
    if s.isEmpty():
        return os.getenv("QT4TRANSLATIONSDIR", "")
    else:
        return unicode(s)
    
def getQt(key, prefClass = Prefs):
    """
    Module function to retrieve the various Qt settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested Qt setting
    """
    if key == "QtDir":
        return getQtDir(prefClass)
    elif key == "Qt4TranslationsDir":
        return getQt4TranslationsDir(prefClass)
    elif key in ["QtToolsPrefix3", "QtToolsPostfix3", 
                 "QtToolsPrefix4", "QtToolsPostfix4"]: 
        return prefClass.settings.value("Qt/" + key, 
            QtCore.QVariant(prefClass.qtDefaults[key])).toString()
    elif key in ["DisableQt3", "ExportQtDir"]: 
        return prefClass.settings.value("Qt/" + key, 
            QtCore.QVariant(prefClass.qtDefaults[key])).toInt()[0]
    
def setQt(key, value, prefClass = Prefs):
    """
    Module function to store the various Qt settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("Qt/" + key, QtCore.QVariant(value))
    
def getCorba(key, prefClass = Prefs):
    """
    Module function to retrieve the various corba settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested corba setting
    """
    return prefClass.settings.value("Corba/" + key,
        QtCore.QVariant(prefClass.corbaDefaults[key])).toString()
    
def setCorba(key, value, prefClass = Prefs):
    """
    Module function to store the various corba settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("Corba/" + key, QtCore.QVariant(value))
    
def getUser(key, prefClass = Prefs):
    """
    Module function to retrieve the various user settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested user setting
    """
    if key in ["MailServerAuthentication"]:
        return prefClass.settings.value("User/" + key,
            QtCore.QVariant(prefClass.userDefaults[key])).toInt()[0]
    else:
        return prefClass.settings.value("User/" + key,
            QtCore.QVariant(prefClass.userDefaults[key])).toString()
    
def setUser(key, value, prefClass = Prefs):
    """
    Module function to store the various user settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("User/" + key, QtCore.QVariant(value))
    
def getVCS(key, prefClass = Prefs):
    """
    Module function to retrieve the VCS related settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested user setting
    """
    return prefClass.settings.value("VCS/" + key,
        QtCore.QVariant(prefClass.vcsDefaults[key])).toInt()[0]
    
def setVCS(key, value, prefClass = Prefs):
    """
    Module function to store the VCS related settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("VCS/" + key, QtCore.QVariant(value))
    
def getTasks(key, prefClass = Prefs):
    """
    Module function to retrieve the Tasks related settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested user setting
    """
    if key in ["TasksColour", "TasksBugfixColour", 
               "TasksBgColour", "TasksProjectBgColour"]:
        col = prefClass.settings.value("Tasks/" + key)
        if col.isValid():
            return QtGui.QColor(col.toString())
        else:
            return prefClass.tasksDefaults[key]
    else:
        return prefClass.settings.value("Tasks/" + key,
            QtCore.QVariant(prefClass.tasksDefaults[key])).toString()
    
def setTasks(key, value, prefClass = Prefs):
    """
    Module function to store the Tasks related settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    if key in ["TasksColour", "TasksBugfixColour",
               "TasksBgColour", "TasksProjectBgColour"]:
        prefClass.settings.setValue("Tasks/" + key, QtCore.QVariant(value.name()))
    else:
        prefClass.settings.setValue("Tasks/" + key, QtCore.QVariant(value))
    
def getTemplates(key, prefClass = Prefs):
    """
    Module function to retrieve the Templates related settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested user setting
    """
    if key in ["SeparatorChar"]:
        return prefClass.settings.value("Templates/" + key,
            QtCore.QVariant(prefClass.templatesDefaults[key])).toString()
    else:
        return prefClass.settings.value("Templates/" + key,
            QtCore.QVariant(prefClass.templatesDefaults[key])).toInt()[0]
    
def setTemplates(key, value, prefClass = Prefs):
    """
    Module function to store the Templates related settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("Templates/" + key, QtCore.QVariant(value))
    
def getPluginManager(key, prefClass = Prefs):
    """
    Module function to retrieve the plugin manager related settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested user setting
    """
    if key in ["ActivateExternal"]:
        return prefClass.settings.value("PluginManager/" + key,
            QtCore.QVariant(prefClass.pluginManagerDefaults[key])).toInt()[0]
    else:
        return prefClass.settings.value("PluginManager/" + key,
            QtCore.QVariant(prefClass.pluginManagerDefaults[key])).toString()
    
def setPluginManager(key, value, prefClass = Prefs):
    """
    Module function to store the plugin manager related settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("PluginManager/" + key, QtCore.QVariant(value))
    
def getGraphics(key, prefClass = Prefs):
    """
    Module function to retrieve the Graphics related settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested user setting
    """
    if key in ["Font"]:
        f = QtGui.QFont()
        f.fromString(prefClass.settings.value("Graphics/" + key,
            QtCore.QVariant(prefClass.graphicsDefaults[key])).toString())
        return f
    else:
        return prefClass.settings.value("Graphics/" + key,
            QtCore.QVariant(prefClass.graphicsDefaults[key])).toString()
    
def setGraphics(key, value, prefClass = Prefs):
    """
    Module function to store the Graphics related settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.settings.setValue("Graphics/" + key, QtCore.QVariant(value))
    
def getGeometry(key, prefClass = Prefs):
    """
    Module function to retrieve the display geometry.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested geometry setting
    """
    v = prefClass.settings.value("Geometry/" + key)
    if v.isValid():
        return v.toByteArray()
    else:
        return prefClass.geometryDefaults[key]

def setGeometry(key, value, prefClass = Prefs):
    """
    Module function to store the display geometry.
    
    @param key the key of the setting to be set
    @param value the geometry to be set
    @param prefClass preferences class used as the storage area
    """
    if prefClass.resetLayout:
        v = prefClass.geometryDefaults[key]
    else:
        v = value
    prefClass.settings.setValue("Geometry/" + key, QtCore.QVariant(v))

def resetLayout(prefClass = Prefs):
    """
    Module function to set a flag not storing the current layout.
    
    @param prefClass preferences class used as the storage area
    """
    prefClass.resetLayout = True

def shouldResetLayout(prefClass = Prefs):
    """
    Module function to indicate a reset of the layout.
    
    @param prefClass preferences class used as the storage area
    @return flag indicating a reset of the layout (boolean)
    """
    return prefClass.resetLayout
    
def saveResetLayout(prefClass = Prefs):
    """
    Module function to save the reset layout.
    """
    if prefClass.resetLayout:
        for key in prefClass.geometryDefaults.keys():
            prefClass.settings.setValue("Geometry/" + key, 
                QtCore.QVariant(prefClass.geometryDefaults[key]))
    
from Shortcuts import readShortcuts, saveShortcuts, exportShortcuts, importShortcuts

initPreferences()
initRecentSettings()
