/***************************************************************************
                          bar.cpp  -  bar class
                             -------------------
    begin                : Wed Oct 1 2003
    copyright            : (C) 2003 by Paul Robson
    email                : autismuk@autismuk.freeserve.co.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "foosball.hxx"

//  ********************************************************************************************************
//
//                                           The constructor
//
//  ********************************************************************************************************

Bar::Bar(int p,int d,enum PlayerStyle s,int x)
{
    SetPlayers(p);
    SetDirection(d);
    SetStyle(s);
    SetPos(x);
    SetState(Waiting);
    Process();
    SetController(d > 0 ? Control1:Control2);
    NextBarMove = 0;
}

//  ********************************************************************************************************
//
//                                       Draw the Bar - the players
//
//  ********************************************************************************************************

void Bar::Draw(SGWSurface &s,Pitch &p)
{
    int i,Graphic;
    Graphic = (Direction>0) ? PLAYER1:PLAYER2;  // Work out which graphic to use
    if (Style == Goalie)
        Graphic = (Direction > 0) ? GOALIE1 : GOALIE2;
    
    for (i = 0;i < Players;i++)                 // Draw all of them.
        p.DrawLogical(s,xPos,GetYPos(i),Graphic,Swing);
}

//  ********************************************************************************************************
//
//                                Get the vertical position of a given player
//
//  ********************************************************************************************************

int  Bar::GetYPos(int n)
{
    SGWASSERT(n >= 0 && n < Players);
    return (YEXT*2)*(n+1)/(Players+1)-YEXT+BarOffset;
}

//  ********************************************************************************************************
//
//                                          Run the player FSM
//
//  ********************************************************************************************************

void Bar::Process(void)
{
    switch(State)
    {
    case Waiting:                               // Waiting to kick
        Swing = 0;
        break;

    case GoingBack:                             // Swinging back
        if (StateTimer.TimedOut()) Swing--;
        if (Swing == -2) SetState(Holding);
        break;

    case Holding:                               // Holding back, waiting to kick forwards
        Swing = -2;
        break;

    case Kicking:                               // Kicking forwards
        if (StateTimer.TimedOut()) Swing++;
        if (Swing == 2) SetState(ReturnCentre);
        break;

    case ReturnCentre:                          // Returning back to centre waiting position
        if (StateTimer.TimedOut()) Swing--;
        if (Swing == 0) SetState(Waiting);
        break;
    }
}

//  ********************************************************************************************************
//
//                               Set the position as a proportion top->bottom 0-1000
//
//  ********************************************************************************************************

void Bar::SetOffset(int n)
{
    if (n < 0) n = 0;                           // Force withing limits
    if (n > 1000) n = 1000;
    if (Style == Goalie)                        // Goalkeeper limits
    {
        if (n < 250) n = 250;
        if (n > 750) n = 750;
    }
    BarPos = n;
    BarOffset = (1000-n) * (BarLimit*2) / 1000-BarLimit;
}

//  ********************************************************************************************************
//
//                  Move Player according to current tactics set for him (her ? it ?)
//
//  ********************************************************************************************************

void Bar::Move(Ball &b)
{
    int af,bf,i,x,y;

    if (SGWGameClock() >                        // Resync if way out.
                NextBarMove+1000) NextBarMove = SGWGameClock();

    while (SGWGameClock() > NextBarMove)        // Force synchronisation of movement
    {
        NextBarMove += 25;
        switch(Controller)                      // What are we doing ?
        {
            case Control1:                      // Being moved by Player 1 or Player 2
            case Control2:
                i = (Controller == Control1) ? 1 : 2;
                SGWReadStick(i,af,bf,x,y);      // Read controller keys
                if (y < 0) SetOffset(BarPos-30);// Vertical Movement
                if (y > 0) SetOffset(BarPos+30);

                if (bf != 0)                    // Kick button pressed
                    { DrawBack(); KickForward(); }
                else
                {
                    if (af != 0)                // Trap button pressed
                    {
                        DrawBack();
                        x = GetNearest(b);
                        i = b.Distance(GetXPos(),GetYPos(x));
                        if (i < GrabRadius())
                            b.SetPos(GetXPos(),GetYPos(x));
                    }
                    else
                        KickForward();
                                                // Check for hit immobile player
                }

                if (bf == 0 && af == 0 && State == Waiting)
                {
                    x = GetNearest(b);
                    i = b.Distance(GetXPos(),GetYPos(x));
                    if (i < GrabRadius() * 2 / 3)
                                 b.SetV(-b.GetXV(),-b.GetYV());
                }
                
                if (State == Kicking)
                {
                        x = GetNearest(b);
                        i = b.Distance(GetXPos(),GetYPos(x));
                        if (i < GrabRadius())
                        {
                            b.Kick(Direction,-y,(bf != 0));
                        }
                }
                                
                break;
                
            case Wandering:                     // Wandering around waiting for the ball
                SetOffset(BarPos+BarDir*12);    // what the players do when don't have and
                if (BarPos+BarDir < 0 ||        // can't get the ball.
                    BarPos+BarDir > 1000 ||
                    EventTimer.TimedOut())
                {
                    EventTimer.ResetTimer(rand()%1000+400);
                    BarDir = (rand()%2)*2-1;
                }
                break;

            case Tracking:                      // Nearest player to the ball tracks it
                if (EventTimer.TimedOut())      // Do this every so often.
                {
                    EventTimer.ResetTimer(rand()%100+70);
                    if (Style == Goalie) EventTimer.ResetTimer(rand()%80+50);
                    
                    x = GetNearest(b);
                    BarDir=(b.GetY()<GetYPos(x))// Set the movement direction
                                          ? 1 : -1;
                }
                SetOffset(BarPos+11*BarDir);    // And move it.
                x = GetNearest(b);              // Find nearest ball and distance
                i = b.Distance(GetXPos(),GetYPos(x));
                if (i < GrabRadius())           // Within range
                {
                    i = 3;
                    if (Style == Attack) i = 7; // shoot mostly for forwards
                    if (rand()%i == 0)          // Grab or shoot ?
                    {
                        Grabber = x;            // Grabbed by this one
                        SetController(Grabbed); // Set mode
                        BarDir = (rand()%2)*2-1;// Direction to move, time to kick ?
                        EventTimer.ResetTimer(200+rand()%300);
                    }
                    else                        // Boot it
                    {
                        GetKickDirection(x,y);  // Kick it
                        b.Kick(x,y,1);
                        SetController(Tracking);
                        KickForward();                    
                    }
                }
                break;

            case Grabbed:                       // Got the ball
                if (EventTimer.TimedOut())      // Time to kick it ?
                {
                    GetKickDirection(x,y);
                    b.Kick(x,y,0);
                    SetController(Tracking);
                    KickForward();
                }
                else                            // Otherwise carry it
                    SetOffset(BarPos+BarDir*22);
                DrawBack();
                b.SetPos(GetXPos(),GetYPos(Grabber));
                break;
                
        }
        Process();                              // Process the animation state stuff
    }
}

//  ********************************************************************************************************
//
//                          Get the number of the bar player nearest the ball
//
//  ********************************************************************************************************

int Bar::GetNearest(Ball &b)
{
    int a,i,d,x,by;
    by = b.GetY();              // Target Y
    a = YEXT*32;                // Nearest so far, e.g. high
    for (i = 0;i < Players;i++) // Work through them all, find nearest
    {
       d = b.Distance(GetXPos(),GetYPos(i));
       if (d < a) a = d,x = i;
    }
    return x;
}

//  ********************************************************************************************************
//
//                               Set the controller activity
//
//  ********************************************************************************************************

void Bar::SetController(enum Controller c)
{
    Controller = c;
    BarDir = 0;                                 // Not moving, reset timer
    EventTimer.ResetTimer(1);

    switch(Controller)
    {
        case Wandering:                         // When set to wandering....
            KickForward();                      // kick forward if was swung back
            break;

        case Control1:                          // Player 1 and 2 are obeying the controller of course
        case Control2:
            break;

        case Tracking:                          // When tracking, draw back ready to trap or kick
            DrawBack();
            break;

        case Grabbed:
            break;
    }
}


//  ********************************************************************************************************
//
//                                  Get an appropriate kick direction
//
//  ********************************************************************************************************

void Bar::GetKickDirection(int &x,int &y)
{
    int Dir,Ok;
    do
    {
        Dir = (rand()%3)-1;Ok = 1;
        if (Style == Goalie && Dir == 0) Ok = 0;
        if (Dir == 0 && rand()%2 == 0) Ok = 0;
    } while (Ok == 0);
    x = Direction;y = Dir;
}

