/*
  Copyright (C) 2005 by Bram Schoenmakers
  bramschoenmakers@kde.nl

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include <qregexp.h>
#include <qtimer.h>

#include <kapplication.h>
#include <kconfig.h>
#include <kdebug.h>
#include <klineedit.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kspell.h>

#include "grid.h"
#include "l10nwidget.h"
#include "wordlist.h"

#include "game.h"

Game::Game( QWidget *parent, KSpell *spell,  const char *name )
  : QWidget( parent, name ), mRemainingTime( 0 ), mScore( 0 ), mCorrectWord( true ), mCorrected( false ), mUseDictionary( true ), mDontAskNewGame( false )
{
  mSpellChecker = spell;
  connect( mSpellChecker,
           SIGNAL( corrected( const QString &, const QString &, unsigned int ) ),
           SLOT( slotWordCorrected( const QString &, const QString &, unsigned int ) ) );

  QVBoxLayout *layout = new QVBoxLayout( this );

  mL10nWidget = new LocalisationWidget( this );
  connect( mL10nWidget, SIGNAL( configChanged( bool, bool ) ), SIGNAL( configChanged( bool, bool ) ) );
  layout->addWidget( mL10nWidget );

  mCounter = new QTimer( this );
  connect( mCounter, SIGNAL( timeout() ), SLOT( updateTime() ) );

  mLineEdit = new KLineEdit( this );
  connect( mLineEdit, SIGNAL( returnPressed() ), SLOT( slotWordEntered() ) );

  mGrid = new Grid( spell, this );
  connect( mGrid, SIGNAL( appendChar( const QChar & ) ), SLOT( appendChar( const QChar & ) ) );
  connect( mGrid, SIGNAL( removeChar( const QChar & ) ), SLOT( removeChar( const QChar & ) ) );
  connect( mGrid, SIGNAL( submit() ), SLOT( slotWordEntered() ) );
  connect( mGrid, SIGNAL( clear() ), mLineEdit, SLOT( clear() ) );

  layout->addWidget( mGrid );
  layout->addWidget( mLineEdit );

  mWordList = new WordList( this );
  layout->addWidget( mWordList );

  layout->addStretch();

  newGame();
}

Game::~Game()
{
}

void Game::newGame()
{
  if( !mDontAskNewGame && mRemainingTime > 0 )
  {
    mGrid->setCharactersVisible( false );
    if ( KMessageBox::questionYesNo(this, i18n("You did not finish this game. Do you really want to start a new game?"), QString::null, KStdGuiItem::yes(), KStdGuiItem::no(), "NewGameWarning" ) == KMessageBox::No )
    {
      mGrid->setCharactersVisible( true );
      return;
    }
  }

  readSettings();
  mRemainingTime = mTime;
  mScore = 0;
  mGrid->randomize();
  mCounter->start( 1000 );
  mLineEdit->setEnabled( true );
  mLineEdit->setFocus();
  mLineEdit->clear();
  mWordList->clear();
  mGrid->resetCells();

  emit gameStarted();

  // set statusbar
  emit secondsLeft( mRemainingTime );
  emit scoreUpdated( mScore );
}

void Game::pause()
{
  mCounter->stop();
  mGrid->setCharactersVisible( false );
  mLineEdit->setEnabled( false );
}

void Game::resume()
{
  mCounter->start( 1000 );
  mGrid->setCharactersVisible( true );
  mLineEdit->setEnabled( true );
  mLineEdit->setFocus();
}

void Game::updateTime()
{
  --mRemainingTime;
  emit secondsLeft( mRemainingTime );

  if( !mRemainingTime )
  {
    stopGame();
  }
}

void Game::stopGame()
{
  mLineEdit->clear();
  mLineEdit->setEnabled( false );
  mCounter->stop();
  mGrid->setCharactersVisible( true );
  mGrid->disableCells();

  emit secondsLeft( 0 );
  emit gameStopped();
}

QString Game::lower( QString &c1 )
{
  QString c2( c1 );
  
  c2.replace( QRegExp( QString::fromUtf8("[áàâäåã]")), "a")
    .replace( QRegExp( QString::fromUtf8("[èéêë]")), "e" )
    .replace( QRegExp( QString::fromUtf8("[íìîï]")), "i" )
    .replace( QString::fromUtf8("ñ"), "n" )
    .replace( QRegExp( QString::fromUtf8("[ôöòóõ]")), "o" )
    .replace( QRegExp( QString::fromUtf8("[ùúüû]")), "u" )
    .replace( QString::fromUtf8("ç"), "c" );

  return c2;
}

void Game::slotWordEntered()
{
  mLastWord = mLineEdit->text().stripWhiteSpace();
  if ( mUseDictionary && mSpellChecker )
  {
    mSpellChecker->checkWord( mLastWord );
  }
  else // we call it directly
  {
    slotWordCorrected( QString::null, QString::null, 0 );
  }
}

void Game::slotWordCorrected( const QString &old_word, const QString &, unsigned int )
{
  QString lastWordNoAccent = lower( mLastWord );
  bool alreadyEntered = mWordList->hasWord( lastWordNoAccent );

  if( !alreadyEntered )
  {
    bool inGrid = mGrid->hasWord( lastWordNoAccent );
    bool valid = old_word.isEmpty () && mLastWord.length() > 2 && inGrid;
    if( valid ) // word is valid
    {
      mWordList->addWord( lastWordNoAccent );
      mScore += mLastWord.length() - 2;
      emit scoreUpdated( mScore );
    }
    else if( !inGrid )
    {
      emit wordNotInGrid( mLastWord );
    }
    else // word is invalid
    {
      emit invalidWordEntered( mLastWord );
    }
  }
  else
  {
    mWordList->highlight( mLastWord );
  }

  mLineEdit->clear();
  mGrid->resetCells();
}

QStringList Game::foundWords() const
{
  return mWordList->words();
}

void Game::readSettings()
{
  mGrid->readSettings();
  mL10nWidget->readSettings();

  KConfig *config = kapp->config();

  config->setGroup( "Preferences" );
  mTime = config->readNumEntry( "GameLength", 180 );

  config->setGroup( "Localization" );
  mUseDictionary = config->readBoolEntry( "UseDictionary", true );

  config->setGroup( "Notification Messages" );
  mDontAskNewGame = config->readBoolEntry( "NewGameWarning", false );
}

void Game::setSpellChecker( KSpell *spell )
{
  mSpellChecker = spell;
  mGrid->setSpellChecker( spell );
}

void Game::appendChar( const QChar &c )
{
  if( mCounter->isActive() )
  {
    mLineEdit->setText( mLineEdit->text() + c );
  }
}

void Game::removeChar( const QChar &c )
{
  if( mLineEdit->text().right( 1 ) == c )
  {
    QString text = mLineEdit->text();
    mLineEdit->setText( text.left( text.length() - 1 ) );
  }
}

void Game::showL10nBar( bool b )
{
  b ? mL10nWidget->show() : mL10nWidget->hide();
}

#include "game.moc"
