/*
  Copyright (C) 2005 by Bram Schoenmakers
  bramschoenmakers@kde.nl

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#ifndef GAME_H
#define GAME_H

#include <qlayout.h>
#include <qstringlist.h>

class QTimer;

class KLineEdit;
class KSpell;

class Grid;
class LocalisationWidget;
class WordList;

/**
  One instance of a Boggle game.
  @author Bram Schoenmakers <bramschoenmakers@kde.nl>
*/
class Game : public QWidget
{
  Q_OBJECT
  public:
    Game( QWidget *parent, KSpell *spell = 0, const char *name = 0 );
    ~Game();

    /** Returns all words found by the player */
    QStringList foundWords() const;
    /** Reads all settings from the configuration file */
    void readSettings();

    int remainingTime() const { return mRemainingTime; }

  public slots:
    /** Starts a new game */
    void newGame();
    /** Pauses the game */
    void pause();
    /** Resumes the game from pause state */
    void resume();
    /** Stop the game, game over */
    void stopGame();

    /** Show/hides the localization bar when parameter is true/false. */
    void showL10nBar( bool );

    void setSpellChecker( KSpell * );

  signals:
    /** Emits how many seconds are left in the current game. Used to update the statusbar.*/
    // FIXME: Does it return 0 on stopped?
    void secondsLeft( int );
    void scoreUpdated( int );
    void gameStarted();
    void gameStopped();

    void invalidWordEntered( const QString & );
    void wordNotInGrid( const QString & );

    /**
       Emits when the configuration has been changed. When the first parameter
       is true, a new game will be requested. When the second parameter is true,
       KBoggle should be restarted.
    */
    void configChanged( bool, bool );

  protected slots:
    void slotWordEntered();
    void updateTime();
    void slotWordCorrected( const QString &, const QString &, unsigned int );
    void appendChar( const QChar & );
    void removeChar( const QChar & );

  protected:
    /** Removes accents from entered words. */
    QString lower( QString & );

  private:
    Grid *mGrid;
    KLineEdit *mLineEdit;
    WordList *mWordList;
    LocalisationWidget *mL10nWidget;

    KSpell *mSpellChecker;

    QTimer *mCounter;
    /** Time currently left in game */
    int mRemainingTime;
    /** Setting of how many seconds per game */
    int mTime;
    int mScore;

    bool mCorrectWord;
    bool mCorrected;

    QString mLastWord;

    bool mUseDictionary;
    bool mDontAskNewGame;
};

#endif
