/*
  Copyright (C) 2005 by Bram Schoenmakers
  bramschoenmakers@kde.nl

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include <kaction.h>
#include <kactionclasses.h>
#include <kapplication.h>
#include <kconfig.h>
#include <kdebug.h>
#include <klocale.h>
#include <kmainwindow.h>
#include <kmessagebox.h>
#include <kstatusbar.h>
#include <kspell.h>

#include "game.h"
#include "kboggle.h"
#include "settingsdialog.h"

KBoggle::KBoggle()
  : KMainWindow( 0, "KBoggle" ), mSettingsDialog( 0 ), mShowL10nBar( true ), mPaused( false )
{
    // set the shell's ui resource file
    setXMLFile("kboggleui.rc");

    mSettingsDialog = new SettingsDialog( this );
    connect( mSettingsDialog, SIGNAL( configChanged( bool, bool ) ), SLOT( configChanged( bool, bool ) ) );

    // FIXME: Don't enable spell checker if it's disabled in settings dialog.
    mSpellChecker = new KSpell( this, QString::null, this, SLOT( spellOK( KSpell * ) ), mSettingsDialog->spellConfig() );
    connect( mSpellChecker, SIGNAL( death() ), SLOT( spellNotOK() ) );

    mGame = new Game( this, mSpellChecker );
    connect( mGame, SIGNAL( gameStarted() ), SLOT( gameStarted() ) );
    connect( mGame, SIGNAL( gameStopped() ), SLOT( gameStopped() ) );

    connect( mGame, SIGNAL( secondsLeft( int ) ), SLOT( updateTime( int ) ) );
    connect( mGame, SIGNAL( wordNotInGrid( const QString & ) ), SLOT( wordNotInGrid( const QString & ) ) );
    connect( mGame, SIGNAL( invalidWordEntered( const QString & ) ), SLOT( invalidWord( const QString & ) ) );
    connect( mGame, SIGNAL( scoreUpdated( int ) ), SLOT( updateScore( int ) ) );

    connect( mGame, SIGNAL( configChanged( bool, bool ) ), SLOT( configChanged( bool, bool ) ) );

    setCentralWidget( mGame );

    new KAction( i18n("&New Game"), "filenew", 0, mGame, SLOT( newGame() ), actionCollection(), "kboggle_newgame" );
    mPauseAction = new KAction( i18n("&Pause"), "player_pause", 0, this, SLOT( togglePause() ), actionCollection(), "kboggle_pausegame" );
    mStopAction = new KAction( i18n("&Stop"), "player_stop", 0, mGame, SLOT( stopGame() ), actionCollection(), "kboggle_stopgame" );

    new KAction( i18n("&Quit"), "exit", 0, this, SLOT( close() ), actionCollection(), "kboggle_quit" );

    // TODO: mHighScores

    mL10nBarAction = new KAction( i18n("&Hide Language Bar"), "locale", 0, this, SLOT( toggleL10nBar() ), actionCollection(), "showhide_l10n" );
    KStdAction::preferences( this, SLOT( configureKBoggle() ), actionCollection() );

    statusBar()->insertItem( i18n( "Score: %1" ).arg( 0 ), 0, 0, true );
    statusBar()->insertItem( QString::null, 1, 0, true );
    statusBar()->setItemAlignment( 1, Qt::AlignRight | Qt::AlignVCenter );

    setupGUI();
    readSettings();
}

KBoggle::~KBoggle()
{
  writeSettings();
  if( mSpellChecker )
  {
    mSpellChecker->cleanUp(); // FIXME: Wait for death signal
    delete mSpellChecker;
    mSpellChecker = 0;
  }
}

void KBoggle::spellOK( KSpell * )
{
  kdDebug() << "Spellchecker is online" << endl;
}

void KBoggle::spellNotOK()
{
  KMessageBox::information( this, i18n( "Could not start the spell checker. Please check your settings in KBoggle's configuration dialog." ), QString::null, "DontShowSpellCheckerWarning" );
  delete mSpellChecker;
  mSpellChecker = 0;
  mGame->setSpellChecker( 0 );
}

void KBoggle::invalidWord( const QString &word )
{
  if( !word.isEmpty() )
  {
    // show this message for 2 seconds
    statusBar()->message( i18n("%1 is invalid").arg( word ), 2000 );
  }
}

void KBoggle::wordNotInGrid( const QString &word )
{
  if( !word.isEmpty() )
  {
    statusBar()->message( i18n("%1 does not appear in grid").arg( word ), 2000 );
  }
}

void KBoggle::updateTime( int left )
{
  statusBar()->changeItem( i18n("1 second left","%n seconds left", left ), 1 );
}

void KBoggle::updateScore( int score )
{
  statusBar()->changeItem( i18n("Score: %1").arg( score ), 0 );
}

void KBoggle::togglePause()
{
  mPaused = !mPaused;

  if( mPaused )
  {
    mPauseAction->setText( i18n("&Resume") );
    mPauseAction->setIcon( "player_play" );
    mGame->pause();
  }
  else
  {
    mPauseAction->setText( i18n("&Pause") );
    mPauseAction->setIcon( "player_pause" );
    mGame->resume();
  }
}

void KBoggle::gameStarted()
{
  mPauseAction->setEnabled( true );
  mStopAction->setEnabled( true );
  if( mPaused )
    togglePause();
  statusBar()->clear();
  updateScore( 0 );

  readSettings();
}

void KBoggle::gameStopped()
{
  statusBar()->message( i18n("You have found one word.", "You have found %n words.", mGame->foundWords().count() ) );
  mPauseAction->setEnabled( false );
  mStopAction->setEnabled( false );
}

void KBoggle::configureKBoggle()
{
  bool gameActive = mGame->remainingTime() > 0;

  if( gameActive ) mGame->pause();
  mSettingsDialog->exec();
  if( gameActive ) mGame->resume();
}

void KBoggle::configChanged( bool newgame, bool restart )
{
  if( restart )
  {
    bool active = !mPaused && mGame->remainingTime() > 0;

    if( active )
      mGame->pause();

    KMessageBox::information( this, i18n("To use the new word list settings, please restart KBoggle.") );

    if( active )
      mGame->resume();
  }

  if( newgame && mGame->remainingTime() > 0 )
  {
    if( !mPaused )
      mGame->pause();

    if( KMessageBox::questionYesNo( this, i18n("Some of the modified options will only have effect when you start a new game. Do you want to start a new game now?") ) == KMessageBox::Yes ) {
      mGame->newGame();
      return;
    }

    if( !mPaused )
      mGame->resume();
  }
  else
  {
    readSettings();
  }
}

void KBoggle::readSettings()
{
  KConfig *config = kapp->config();

  config->setGroup( "Preferences" );
  mShowL10nBar = !config->readBoolEntry( "showL10nBar", true );
  toggleL10nBar();

  mGame->readSettings();
}

void KBoggle::writeSettings()
{
  KConfig *config = kapp->config();
  config->setGroup( "Preferences" );
  config->writeEntry( "showL10nBar", mShowL10nBar );

  config->sync();
}

void KBoggle::toggleL10nBar()
{
  mShowL10nBar = !mShowL10nBar;
  if( mShowL10nBar )
  {
    mL10nBarAction->setText( i18n("&Hide Language Bar" ) );
    mGame->showL10nBar( true );
  }
  else
  {
    mL10nBarAction->setText( i18n("Show &Language Bar" ) );
    mGame->showL10nBar( false );
  }
}

#include "kboggle.moc"
