/*
  Copyright (C) 2005 by Bram Schoenmakers
  bramschoenmakers@kde.nl

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include <qcheckbox.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qspinbox.h>
#include <qwhatsthis.h>

#include <kapplication.h>
#include <kconfig.h>
#include <kdebug.h>
#include <klocale.h>
#include <ksconfig.h>

#include "cubescombo.h"
#include "settingsdialog.h"

SettingsDialog::SettingsDialog( QWidget *parent, const char *name )
  : KDialogBase( KDialogBase::Tabbed, i18n("Configuration"), KDialogBase::Ok | KDialogBase::Apply | KDialogBase::Cancel, KDialogBase::Ok, parent, name, /* modal */ true, /* separator */ true ), mCriticalModified( false ), mRestartRequired( false )
{
  QFrame *page = addPage( i18n("General") );
  QVBoxLayout *layout = new QVBoxLayout( page );

  mHighlightWords = new QCheckBox( i18n("Highlight words"), page );
  layout->addWidget( mHighlightWords );
  QWhatsThis::add( mHighlightWords, i18n("Highlights the characters of an entered word in red. The first character of the word is shown in blue.") );

  mUseMouse = new QCheckBox( i18n("Use mouse to enter words"), page );
  QWhatsThis::add( mUseMouse, i18n("<qt><p>With this option, it is possible to enter words with the mouse.</p> \
  <p><em>Left click</em> - Enter currently highlighted character<br /> \
  <em>Middle click</em> - Submit word<br /> \
  <em>Right click</em> - Erase last entered character<br /> \
  <em>Right doubleclick</em> - Erase all entered characters</p></qt>") );
  layout->addWidget( mUseMouse );

  QHBoxLayout *hlayout = new QHBoxLayout( layout );

  mTimeSpinbox = new QSpinBox( /* min */ 1, /* max */ 600, /*step*/ 1, page );
  mTimeSpinbox->setSuffix( i18n(" seconds") );
  connect( mTimeSpinbox, SIGNAL( valueChanged( int ) ), SLOT( setCriticalModified() ) );
  hlayout->addWidget( new QLabel( mTimeSpinbox, i18n( "Game length" ), page ) );
  hlayout->addSpacing( 25 );
  hlayout->addWidget( mTimeSpinbox );
  hlayout->addStretch( Qt::Horizontal );

  layout->addStretch( Qt::Vertical );

  page = addPage( i18n("Localization") );
  layout = new QVBoxLayout( page, 6);

  mUseDictionary = new QGroupBox( 1, Qt::Vertical, page );
  mUseDictionary->setTitle( i18n( "Use dictionary" ) );
  mUseDictionary->setCheckable( true );
  QWhatsThis::add( mUseDictionary, i18n("When this option is enabled, KBoggle validates each word with KSpell, KDE's spell checker. If you disable this option, a word must be at least appear in the grid to be valid.") );

  mKSConfig = new KSpellConfig( mUseDictionary, 0, 0, /* helpbutton */ false );
  connect( mKSConfig, SIGNAL( configChanged() ), SLOT( setRestartRequired() ) );
  mKSConfig->setNoRootAffix( true );
  mKSConfig->setRunTogether( false );

  layout->addWidget( mUseDictionary );

  hlayout = new QHBoxLayout( layout );

  mCubesCombo = new CubesCombo( page );
  connect( mCubesCombo, SIGNAL( activated( int ) ), SLOT( setCriticalModified() ) );
  hlayout->addWidget( new QLabel( mCubesCombo, i18n("Cube set:"), page ) );
  hlayout->addWidget( mCubesCombo );
  QWhatsThis::add( mCubesCombo, i18n( "<qt><p>Sets the language of the cubeset.</p> \
  <p>KBoggle uses 16 virtual cubes with a character on each side. The distribution of those characters differs per language. For example, the letter K appears less in English than in Dutch. That's why KBoggle tries to offer a optimized cubeset for each language. If the cubeset does not exist for your language, the English set is used.</p></qt>" ) );

  readSettings();
}

SettingsDialog::~SettingsDialog()
{
}

void SettingsDialog::slotOk()
{
  writeSettings();
  emit configChanged( mCriticalModified, mRestartRequired );
  accept();
}

void SettingsDialog::slotApply()
{
  writeSettings();
  // don't bother the user with message boxes when only hitting Apply
  emit configChanged( false, false );
}

void SettingsDialog::show()
{
  mCriticalModified = false;
  mRestartRequired = false;
  QDialog::show();
}

void SettingsDialog::readSettings()
{
  KConfig *config = kapp->config();

  config->setGroup( "Preferences" );
  mHighlightWords->setChecked( config->readBoolEntry( "HighlightWords", true ) );
  mTimeSpinbox->setValue( config->readNumEntry( "GameLength", 180 ) );
  mUseMouse->setChecked( config->readBoolEntry( "UseMouse", false ) );

  config->setGroup( "Localization" );
  mUseDictionary->setChecked( config->readBoolEntry( "UseDictionary", true ) );
  mKSConfig->setClient( config->readNumEntry( "Client", 1 ) );
  mKSConfig->setDictionary( config->readEntry( "Dictionary", "en" ) );
  mKSConfig->setEncoding( config->readNumEntry( "Encoding", 0 ) );

  mCubesCombo->readSettings();
}

void SettingsDialog::writeSettings()
{
  KConfig *config = kapp->config();

  config->setGroup( "Preferences" );
  config->writeEntry( "HighlightWords", mHighlightWords->isChecked() );
  config->writeEntry( "GameLength", mTimeSpinbox->value() );
  config->writeEntry( "UseMouse", mUseMouse->isChecked() );

  config->setGroup( "Localization" );
  config->writeEntry( "UseDictionary", mUseDictionary->isChecked() );
  config->writeEntry( "Dictionary", mKSConfig->dictionary() );
  config->writeEntry( "Client", mKSConfig->client() );
  config->writeEntry( "Encoding", mKSConfig->encoding() );

  mCubesCombo->writeSettings();

  config->sync();
}

void SettingsDialog::setCriticalModified()
{
  mCriticalModified = true;
}

void SettingsDialog::setRestartRequired()
{
  mRestartRequired = true;
}

#include "settingsdialog.moc"
