/***************************************************************************
 *   Copyright (C) 2005-2007 by Leon Pennington                            *
 *   leon@leonscape.co.uk                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "cellstate.h"

#include "partstate.h"

#include <qdom.h>

CellState::CellState()
{
    m_value = -1;
    box = col = row = 0;
    type = NORMAL;
    userPossible.clear();
    m_poss.clear();
}

const CellState& CellState::operator =( const CellState &cell )
{
    m_value = cell.m_value;
    type = cell.type;
    userPossible = cell.userPossible;
    m_poss = cell.m_poss;

    return *this;
}

void CellState::clear()
{
    m_value = -1;
    type = NORMAL;
    userPossible.clear();
    m_poss.clear();
}

bool CellState::setValue( int val )
{
    if ( val >= 0 && ( box->used( val ) || col->used( val ) || row->used( val ) ) )
        return false;

    m_value = val;
    box->valueChanged();
    col->valueChanged();
    row->valueChanged();
    return true;
}

PossibleValues CellState::possible()
{
    m_poss.clear();
    if ( m_value < 0 )
    {
        for ( int i = 0; i < 9; ++i )
        {
            if ( !box->used( i ) && !col->used( i ) && !row->used( i ) )
                m_poss[i] = true;
            else
                m_poss[i] = false;
        }
    }

    return m_poss;
}

bool CellState::isPossible( int val )
{
    //possible();
    return m_poss[val];
}

bool CellState::anyPossible()
{
    //possible();
    return !(m_poss.isEmpty());
}

void CellState::matchPossible()
{
    //possible();
    userPossible = m_poss;
}

void CellState::partChanged()
{
    possible();
}

QDomElement CellState::saveCell( QDomDocument &doc, int position )
{
    QDomElement e = doc.createElement( "cell" );
    e.setAttribute( "type", type );
    e.setAttribute( "pos", position );
    if( m_value >= 0 )
    {
        e.setAttribute( "value", m_value );
    }
    else
    {
        for( int i = 0; i < 9; ++i )
        {
            if( userPossible[i] )
            {
                QDomElement val = doc.createElement( "possible" );
                QDomText t = doc.createTextNode( QString::number(i) );
                val.appendChild( t );
                e.appendChild( val );
            }
        }
    }
    return e;
}

bool CellState::loadCell( QDomElement &e, float version )
{
    m_value = -1;
    userPossible.clear();

    if( version == 0.1 )
    {
        type = (CellType)( e.attribute( "type", "0" ).toInt() - 1 );
        if( e.text().length() > 0 )
        {
            for( unsigned i  = 0; i < e.text().length(); ++i )
            {
                if( e.text()[i].isDigit() )
                    userPossible[ e.text()[i].digitValue()] = true;
            }
        }
        else if( e.text().length() == 1 )
        {
            if( e.text()[0].isDigit() )
                m_value = e.text()[0].digitValue();
        }
    }
    else
    {
        type = (CellType)(e.attribute( "type", "0" ).toInt());
        if( e.hasAttribute( "value" ) )
        {
            m_value = e.attribute( "value", "-1" ).toLong();
        }
        else
        {
            QDomNode node = e.firstChild();
            while( node.isElement() )
            {
                QDomElement val = node.toElement();
                if( !val.isNull() && val.nodeName() == "possible" )
                {
                    bool ok;
                    int value = val.text().toLong( &ok );
                    if( ok )
                        userPossible[value] = true;
                }
                node = node.nextSibling();
            }
        }
    }
    return true;
}



