/***************************************************************************
 *   Copyright (C) 2005 by Leon Pennington                                 *
 *   leon@leonscape.co.uk                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "generator.h"
#include "solver.h"

#include <qapplication.h>

#include <kprogress.h>
#include <klocale.h>

#include <cstdlib>

const int MJR_MIN[] = { 0, 0, 0, 1, 3 };

const int MJR_MAX[] = { 0, 0, 0, 2, 5 };

const int MNR_MIN[] = { 1,3, 6, 0, 0 };

const int MNR_MAX[] = { 2, 5, 7, 100, 100 };

const unsigned MINSOLS[] = { 1, 1, 1, 1, 1 };

const int MINNUMS = 10;

const int MAXNUMS = 35;

Generator::Generator( int difficulty )
{
    m_diff = difficulty;
    std::srand( time( 0 ) );
}

GridState Generator::generate( KProgressDialog *progress )

{
    GridState grid = generateFresh();
    m_solutions.clear();
    std::list<int> removed;

    while ( m_solutions.size() < MINSOLS[m_diff] && !(progress->wasCancelled()) )
        step( grid, removed, progress );


    if ( progress->wasCancelled() )
        return GridState();
    else
    {
        m_solutions.sort();
        GridState result = m_solutions.front().grid;
        return result;
    }
}

bool Generator::step( const GridState &parent, std::list<int> removed, KProgressDialog *progress )
{
    bool found = false;
    GridState grid = parent;

    while ( grid.filledCells() > MINNUMS && m_solutions.size() < MINSOLS[m_diff] && removed.size() < 38 )
    {
        int cell = ( int )((( float )std::rand() / ( float )RAND_MAX ) * 40.0 );

        while ( contains( removed, cell ) )
            cell = ( int )((( float )std::rand() / ( float )RAND_MAX ) * 40.0 );

        removed.push_back( cell );

        grid.setValue( cell, -1 );

        if ( cell < 40 )
            grid.setValue( 80 - cell, -1 );

        Solver solve( grid );

        if ( solve.findSolution() &&
                solve.majorDifficulty() <= MJR_MAX[m_diff] &&
                solve.minorDifficulty() <= MNR_MAX[m_diff] )
        {
            if ( solve.majorDifficulty() >= MJR_MIN[m_diff] &&
                    solve.minorDifficulty() >= MNR_MIN[m_diff] &&
                    grid.filledCells() < MAXNUMS )
            {
                Solution s;
                s.grid = grid;
                s.mjrDiff = solve.majorDifficulty();
                s.mnrDiff = solve.minorDifficulty();
                m_solutions.push_back( s );
                return true;
            }
            else if ( !step( grid, removed, progress ) )
            {
                grid.setValue( cell, parent.value( cell ) );

                if ( cell < 40 )
                    grid.setValue( 80 - cell, parent.value( 80 - cell ) );
            }
        }
        else
        {
            grid.setValue( cell, parent.value( cell ) );

            if ( cell < 40 )
                grid.setValue( 80 - cell, parent.value( 80 - cell ) );
        }

        qApp->processEvents();
        if ( progress->wasCancelled() )
            return false;
        else
        {
            QString msg = i18n( "Last Difficulty %1:%2" ).arg( solve.majorDifficulty() ).arg( solve.minorDifficulty() );
            if( m_solutions.size() > 0 )
                msg += i18n( "\nFound Grid %1 of %2" ).arg( m_solutions.size() ).arg( MINSOLS[m_diff] );
            progress->setLabel( msg );
        }
    }

    return found;
}

GridState Generator::generateFresh()

{
    GridState grid;
    std::list<int> used;

    for ( int x = 0; x < 9; x += 4 )
    {
        used.clear();

        for ( int i = 0; i < 9; ++i )
        {
            int val = ( int )((( float )std::rand() / ( float )RAND_MAX ) * 9.0 );

            while ( contains( used, val ) )
                val = ( int )((( float )std::rand() / ( float )RAND_MAX ) * 9.0 );

            used.push_back( val );

            grid.setBoxValue( x, i, val );
        }
    }

    Solver solve( grid );

    solve.findSolution();
    return solve.firstSolution();
}

bool Generator::contains( std::list<int> &list, int val ) const
{
    for ( std::list<int>::iterator p = list.begin(); p != list.end(); ++p )
    {
        if ( *p == val )
            return true;
    }

    return false;
}

bool Generator::Solution::operator <( const Generator::Solution &sol ) const

{
    if ( mjrDiff != sol.mjrDiff )
        return mjrDiff > sol.mjrDiff;
    else if ( mnrDiff != sol.mnrDiff )
        return mnrDiff > sol.mnrDiff;
    else
        return grid.filledCells() > sol.grid.filledCells();
}
