/***************************************************************************
 *   Copyright (C) 2005 by Leon Pennington                                 *
 *   leon@leonscape.co.uk                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "gridstate.h"

#include "possiblevalues.h"
#include "cellstate.h"
#include "partstate.h"

#include <qdom.h>
#include <qfile.h>
#include <qstring.h>

#include <kmessagebox.h>
#include <klocale.h>

const float fileVersion = 0.2;

GridState::GridState()
{
    setup();
}

GridState::GridState( const GridState &grid )
{
    setup();
    *this = grid;
}

GridState::~GridState()

{
    delete [] m_pCells;
    delete [] m_pBoxes;
    delete [] m_pCols;
    delete [] m_pRows;
}

const GridState& GridState::operator =( const GridState &grid )
{
    for ( int i = 0; i < 81; ++i )
        m_pCells[i] = grid.m_pCells[i];

    return *this;
}

bool GridState::operator ==( const GridState &grid ) const

{
    for ( int i = 0; i < 81; ++i )
    {
        if ( m_pCells[i].value() != grid.m_pCells[i].value() )
            return false;
    }

    return true;
}

bool GridState::operator !=( const GridState &grid ) const

{
    return !( *this == grid );
}

const CellState GridState::operator []( const int idx ) const
{
    return m_pCells[idx];
}

CellState& GridState::operator[]( int idx )
{
    return m_pCells[idx];
}

void GridState::clear()
{
    for( int i = 0; i < 81; ++i )
        m_pCells[i].clear();
}

int GridState::value( int cell ) const
{
    return m_pCells[cell].value();
}

bool GridState::setValue( int cell, int value ) const
{
    return m_pCells[cell].setValue( value );
}

void GridState::setValue( int cell, int value, int type )
{
    m_pCells[cell].setValue( value );
    m_pCells[cell].type = (CellState::CellType)type;
}

void GridState::setBoxValue( int box, int cell, int value ) const
{
    m_pBoxes[box].cells[cell]->setValue( value );
}

int GridState::type( int cell ) const
{
    return m_pCells[cell].type;
}

void GridState::setType( int cell, int type )
{
    m_pCells[cell].type = (CellState::CellType)type;
}

PossibleValues GridState::possible( int cell ) const
{
    return m_pCells[cell].possible();
}

PossibleValues GridState::userPossibles( int cell ) const
{
    return m_pCells[cell].userPossible;
}

void GridState::setUserPosssibles( int cell, PossibleValues possibles )
{
    m_pCells[cell].userPossible = possibles;
}

void GridState::toggleUserPossible( int cell, int value )
{
    if( m_pCells[cell].userPossible[value] )
        m_pCells[cell].userPossible[value] = false;
    else
        m_pCells[cell].userPossible[value] = true;
}

void GridState::matchUserPossible()
{
    for( int i = 0; i < 81; ++i )
        m_pCells[i].matchPossible();
}

void GridState::clearCell( int cell )
{
    m_pCells[cell].clear();
}

void GridState::clearPossibles()
{
    for( int i =0; i < 81; ++i )
        m_pCells[i].possible().clear();
}

int GridState::simpleSolve()
{
    int i = 0;
    bool changed = true;

    for ( ;changed; ++i )
    {
        changed = false;

        for ( int i = 0; i < 81; ++i )
        {
            if ( m_pCells[i].value() < 0 )
            {
                PossibleValues poss = m_pCells[i].possible();

                if ( poss.isSingle() )
                {
                    m_pCells[i].setValue( poss.first() );
                    changed = true;
                }
            }
        }

        for ( int i = 0; i < 9; ++i )
        {
            if ( m_pBoxes[i].check() )
                changed = true;

            if ( m_pCols[i].check() )
                changed = true;

            if ( m_pRows[i].check() )
                changed = true;
        }
    }

    return i;
}

bool GridState::possible() const
{
    for ( int i = 0; i < 81; ++i )
    {
        if ( m_pCells[i].value() < 0 )
        {
            if ( m_pCells[i].possible().isEmpty() )
                return false;
        }
    }

    return true;
}

int GridState::filledCells() const

{
    int filled = 0;

    for ( int i = 0; i < 81; ++i )
    {
        if ( m_pCells[i].value() >= 0 )
            ++filled;
    }

    return filled;
}

bool GridState::saveGrid( const QString &fileName, int type )
{
    QDomDocument doc( "Grid" );
    QDomElement root = doc.createElement( "Grid" );
    root.setAttribute( "version", fileVersion );
    root.setAttribute( "type", type );
    doc.appendChild( root );
    for( int i = 0; i < 81; ++i )
    {
        QDomElement e = m_pCells[i].saveCell( doc, i );
        root.appendChild( e );
    }

    QString xml = doc.toString( 4 );

    QFile file( fileName );
    if( file.open( IO_WriteOnly ) )
    {
        QTextStream stream( &file );
        stream << xml << endl;
        file.close();
        return true;
    }
    else
        KMessageBox::error( 0, i18n( "Could not open the file %1 for writing." ).arg( fileName ) );

    return false;
}

bool GridState::loadGrid( const QString &fileName, int type )
{
    bool result = false;
    QFile file( fileName );
    if( file.open( IO_ReadOnly ) )
    {
        QDomDocument doc( "Grid" );
        if( doc.setContent( &file ) )
        {
            QDomElement root = doc.documentElement();
            float version = root.attribute( "version", QString::number( fileVersion ) ).toFloat();
            if ( version <= fileVersion )
            {
                if( version > 0.1 )
                    type = root.attribute( "type", QString::number( type ) ).toLong();
                else
                    type = 2;

                QDomNode n = root.firstChild();
                while( !n.isNull() )
                {
                    QDomElement e = n.toElement();
                    if( !e.isNull() )
                    {
                        int pos = e.attribute( "pos", "-1" ).toInt();
                        if( pos < 0 || !m_pCells[pos].loadCell( e, version ) )
                        {
                            KMessageBox::sorry( 0, i18n( "Sorry, There was an error reading this file. It maybe corrupted." ) );
                            break;
                        }
                    }
                    n = n.nextSibling();
                }

                if( n.isNull() )
                    result = true;
            }
            else
                KMessageBox::sorry( 0, i18n( "Sorry, Cannot read files with a version higher than %1." ).arg( fileVersion ) );
        }
        else
            KMessageBox::error( 0, i18n( "Could not parse the file %1." ).arg( fileName ) );
        file.close();
    }
    else
        KMessageBox::error( 0, i18n( "Could not open the file %1 for reading." ).arg( fileName ) );

    return result;
}

void GridState::setup()

{
    m_pCells = new CellState[81];
    m_pBoxes = new PartState[9];
    m_pCols = new PartState[9];
    m_pRows = new PartState[9];

    int col, row, box;

    for ( int i = 0; i < 81; ++i )
    {
        col = i % 9;
        m_pCells[i].col = &( m_pCols[ col ] );
        m_pCols[col].addCell( &m_pCells[i] );

        row = i / 9;
        m_pCells[i].row = &( m_pRows[ row ] );
        m_pRows[row].addCell( &m_pCells[i] );

        box = ( col / 3 ) + (( row / 3 ) * 3 );
        m_pCells[i].box = &( m_pBoxes[ box ] );
        m_pBoxes[box].addCell( &m_pCells[i] );
    }
}
