/***************************************************************************
 *   Copyright (C) 2005-2007 by Leon Pennington                            *
 *   leon@leonscape.co.uk                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef GRIDSTATE_H
#define GRIDSTATE_H

#include "possiblevalues.h"

//Forward declarations
class CellState;
class PartState;
class QDomDocument;
class QDomElement;
class QString;

/**
 * This class holds all the information about a grid, and also helps to
 * solve the grid.
 */

class GridState
{
public:
    /// Constructor
    GridState();
    /// Copy Constructor

    GridState( const GridState &grid );

    /// Destructor
    ~GridState();

    /// Assignment operator
    const GridState& operator=( const GridState &grid );

    /// Comparison Operators
    bool operator==( const GridState &grid ) const;
    bool operator!=( const GridState &grid ) const;

    /// Indexing operators
    const CellState operator[]( const int idx ) const;
    CellState& operator[]( int idx );

    /// Clears the gridstate
    void clear();
    /// Clears all the possibles from the cells
    void clearPossibles();

    /// Returns the calue of a cell
    int value( int cell ) const;

    /**
     * Sets a value of a cell
     * @return True if the value was valid otherwise false
     */
    bool setValue( int cell, int value ) const;

    /**
     * Sets a value of a cell and whether its fixed
     */
    void setValue( int cell, int value, int type );

    /**
     * Sets a value of a cell in a box. used in generation
     */
    void setBoxValue( int box, int cell, int value ) const;

    /// Returns the type of a cell
    int type ( int cell ) const;
    /// Sets the type of a cell
    void setType( int cell, int type );

    /// Returns a list of possible values for a cell
    PossibleValues possible( int cell ) const;

    /// Return the user set possible values
    PossibleValues userPossibles( int cell ) const;

    /// Sets the user possilbe values
    void setUserPosssibles( int cell, PossibleValues possibles );

    /// Toggles the user possible value
    void toggleUserPossible( int cell, int value );

    /// Makes the user possibles equal to the actual possibilities
    void matchUserPossible();

    /// Clears a cell
    void clearCell( int cell );

    /**
        * Does a simple scanning solve on the grid
        * @return how many loops the solver went through
        */
    int simpleSolve();

    /// Checks to see if the grid is possible
    bool possible() const;

    /// Returns whether this grid is completley filled
    bool complete() const { return( filledCells() == 81 ); }

    /// Returns the number of cells that are filled
    int filledCells() const;

    /**
     * Saves the current grid state
     * @param fileName The filename to save the grid too
     * @param type The type of game where saving
     * @return True if the save was successful
     */
    bool saveGrid( const QString &fileName, int type );

    /**
     * Loads a grid state from the specified name
     * @param fileName The filename of the grid to load
     * @param type A reference for setting the game type
     * @return True if successful
     */
    bool loadGrid( const QString &fileName, int type );

private:
    void setup();

    CellState *m_pCells;
    PartState *m_pBoxes;
    PartState *m_pCols;
    PartState *m_pRows;
};

#endif
