/***************************************************************************
 *   Copyright (C) 2005-2007 by Leon Pennington                            *
 *   leon@leonscape.co.uk                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "gridview.h"
#include "gridcell.h"
#include "gridstate.h"
#include "cellstate.h"

#include <kglobalsettings.h>
#include <klocale.h>

#include <qpainter.h>

GridView::GridView( QWidget *parent, const char *name ) : QCanvasView( parent, name )
{
    setVScrollBarMode( QScrollView::AlwaysOff );
    setHScrollBarMode( QScrollView::AlwaysOff );
    setMouseTracking( true );
    viewport()->setMouseTracking( true );
    setBackgroundColor( Qt::black );

    QCanvas* canvas = new QCanvas( visibleWidth(), visibleHeight() );
    canvas->setDoubleBuffering( true );
    canvas->setBackgroundColor( Qt::black );

    for ( int i = 0; i < 81; ++i )
    {
        m_pCells[i] = new GridCell( canvas, i );
        m_pCells[i]->show();
    }

    setCanvas( canvas );
    m_activeCell = -1;
}

void GridView::setGrid( const GridState &grid )
{
    for( int i = 0; i < 81; ++i )
        m_pCells[i]->setCell( grid[i] );
    canvas()->setAllChanged();
    canvas()->update();
}

void GridView::setCurrentCell( CellState state )
{
    m_pCells[m_activeCell]->setCell( state );
    canvas()->setChanged( m_pCells[m_activeCell]->rect() );
    canvas()->update();
}

void GridView::setCell( int cell, CellState state )
{
    m_pCells[cell]->setCell( state );
    canvas()->setChanged( m_pCells[cell]->rect() );
    canvas()->update();
}

void GridView::setCellAllPossibles( int cell, PossibleValues values )
{
    m_pCells[cell]->setAllPossibles( values );
    canvas()->setChanged( m_pCells[cell]->rect() );
    canvas()->update();
}

void GridView::clear()
{
    for( int i = 0; i < 81; ++i )
        m_pCells[i]->clear();
    canvas()->setAllChanged();
    canvas()->update();
}

void GridView::print( QPainter &painter, QRect area )
{
    painter.fillRect( area, QBrush( Qt::black ) );
    int cellSize = ( area.width() - 8 ) / 9;
    int cellOffset = ((( area.width() - 8 ) % 9 ) / 2 ) + 2;

    if( m_activeCell >= 0 ) m_pCells[m_activeCell]->setActive( false );

    QFont font = KGlobalSettings::generalFont();
    font.setPixelSize( cellSize );
    for ( int i = 0; i < 81; ++i )
    {
        m_pCells[i]->resize( area.left() + cellOffset, area.top() + cellOffset, cellSize, cellSize, font );
        painter.setPen( Qt::black ); // Forcing Color for printing
        m_pCells[i]->drawShape( painter );
    }

    viewportResizeEvent( 0 );
    if( m_activeCell >= 0 ) m_pCells[m_activeCell]->setActive( true );
}

void GridView::moveCurrentCell( int move, bool force )
{
    if( m_activeCell < 0 )
    {
        m_activeCell = 0;
        if ( force || ( m_pCells[0]->type() != GridCell::GENERATED && m_pCells[0]->type() != GridCell::FIXED ) )
        {
            m_pCells[0]->setActive( true );
            canvas()->update();
            return;
        }
    }

    int newCell = m_activeCell;
    bool done = false;
    while( !done )
    {
        switch( move )
        {
            case UP:
                newCell -= 9;
                if( newCell < 0 )
                    newCell += 81;
                break;
            case DOWN:
                newCell += 9;
                if( newCell > 80 )
                    newCell -= 81;
                break;
            case LEFT:
                newCell -= 1;
                if( newCell < 0 || ( newCell / 9 < m_activeCell / 9 ))
                    newCell += 9;
                break;
            case RIGHT:
                newCell += 1;
                if( newCell > 80 || ( newCell /  9 > m_activeCell / 9 ))
                    newCell -= 9;
                break;
        }

        if( newCell == m_activeCell )
            return;

        if( force || ( m_pCells[newCell]->type() != GridCell::GENERATED && m_pCells[newCell]->type() != GridCell::FIXED ) )
        {
            m_pCells[m_activeCell]->setActive( false );
            m_pCells[newCell]->setActive( true );
            m_activeCell = newCell;
            canvas()->update();
            done = true;
        }
    }
}

void GridView::mouseMoveEvent( QMouseEvent *e )

{
    QCanvasItemList list = canvas()->collisions( e->pos() );

    if ( list.size() > 0 )
    {
        GridCell *cell = dynamic_cast<GridCell*>( list.first() );

        if ( m_activeCell >= 0 )
            m_pCells[m_activeCell]->setActive( false );

        m_activeCell = cell->pos();

        cell->setActive( true );
    }
    else
    {
        if( m_activeCell >= 0 )
        {
            m_pCells[m_activeCell]->setActive( false );
            m_activeCell = -1;
        }
    }

    canvas()->update();
}

void GridView::viewportResizeEvent( QResizeEvent* )
{
    canvas()->resize( visibleWidth(), visibleHeight() );
    int cellWidth = ( visibleWidth() - 8 ) / 9;
    int xOffset = ((( visibleWidth() - 8 ) % 9 ) / 2 ) + 2;
    int cellHeight = ( visibleHeight() - 8 ) / 9;
    int yOffset = (( visibleHeight() - 8 ) % 9 ) / 2  + 2;

    QFont font = KGlobalSettings::generalFont();
    if( cellHeight > cellWidth )
        font.setPixelSize( cellWidth );
    else
        font.setPixelSize( cellHeight );
    for ( int i = 0; i < 81; ++i )
    {
        m_pCells[i]->resize( xOffset, yOffset, cellWidth, cellHeight, font );
    }

    canvas()->setAllChanged();
    canvas()->update();
}

#include "gridview.moc"
