/***************************************************************************
 *   Copyright (C) 2005 by Leon Pennington                                 *
 *   leon@leonscape.co.uk                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "kudus.h"
#include "gridview.h"
#include "gridcell.h"
#include "solver.h"
#include "generator.h"
#include "newdialog.h"
#include "printdialog.h"
#include "cellstate.h"

#include <qapplication.h>
#include <qlayout.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>
#include <qsignalmapper.h>

#include <kmainwindow.h>
#include <klocale.h>
#include <kstdgameaction.h>
#include <kaction.h>
#include <kedittoolbar.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <kprinter.h>
#include <kprogress.h>
#include <kglobalsettings.h>

Kudus::Kudus() : KMainWindow( 0, "Kudus" )
{
    m_modified = false;
    setupWindow();
    setupActions();
    setStandardToolBarMenuEnabled( true );
    createGUI();
    setAutoSaveSettings();
    m_solveActive = false;
    m_gameType = NONE;
}

void Kudus::gameNew()
{
    NewDialog *dlg = new NewDialog( this, m_pDesign->isChecked() && m_gameType == DESIGN );
    int code = dlg->exec();

    if ( code == QDialog::Accepted )
    {
        if( m_gameType == DESIGN && dlg->difficulty() < 0 )
        {
            m_currentGrid.clearPossibles();
            for( int i = 0; i < 81; ++i )
            {
                if( m_currentGrid[i].type == CellState::FIXED )
                    m_startGrid[i] = m_currentGrid[i];
                else
                    m_startGrid[i].clear();
            }
        }
        else
        {
            m_pDesign->setChecked( false );
            Generator gen( dlg->difficulty() );

            KProgressDialog* progress = new KProgressDialog( this, "progressDiag", i18n( "Generating Grid" ), i18n( "Generating ..." ), true );
            progress->progressBar()->hide();
            progress->show();

            m_startGrid = gen.generate( progress );
            removeChild( progress );
            delete progress;
        }

        Solver solve( m_startGrid );

        if ( solve.findSolution() )
        {
            startGrid( true );
            m_modified = false;
            m_gameType = SIMPLE;
            showTitle( solve.majorDifficulty(), solve.minorDifficulty(), i18n( "Generated" ) );
        }
        else
        {
            if( m_gameType == DESIGN && dlg->difficulty() < 0 )
                KMessageBox::sorry( this, i18n( "Could not test design, as there is no valid solution." ), i18n( "No Valid Solution" ));
            else
            {
                m_pView->clear();
                setCaption( i18n( "No Solution Set" ) );
            }
        }
    }
}

void Kudus::gameRestart()
{
    if ( m_modified )
    {
        if( m_gameType == DESIGN )
            gameClear();
        else if( m_gameType != NONE && m_pDesign->isChecked() )
        {
            startGrid();
            m_gameType = DESIGN;
        }
        else
        {
            startGrid();
            m_modified = false;
        }
    }

}

void Kudus::gameLoad()
{
    if ( m_modified )
    {
        int ret = KMessageBox::warningYesNoCancel( 0, i18n( "Do you want to save your current game progress or discard it?" ),
                  i18n( "Unsaved Game" ), i18n( "&Save" ), i18n( "&Discard" ) );

        switch ( ret )
        {
            case KMessageBox::Yes:
                gameSave();

                break;

            case KMessageBox::Cancel:
                return;

                break;
        }
    }

    QString fileName = KFileDialog::getOpenFileName( QString::null, "*.kgc|" + i18n( "Kudus Grid" ) );

    if ( !fileName.isEmpty() )
    {
        m_startGrid.clear();

        if ( m_startGrid.loadGrid( fileName, m_gameType ) )
        {
            if( m_gameType == NONE )
                m_gameType = SIMPLE;

            startGrid();
            m_modified = false;
            Solver solve( m_startGrid );

            if ( solve.findSolution() )
                showTitle( solve.majorDifficulty(), solve.minorDifficulty(), i18n( "Loaded" ) );
            else
                setCaption( i18n( "Loaded - Unsolvable" ) );
        }
    }
}

void Kudus::gameSave()
{
    QString fileName = KFileDialog::getSaveFileName( QString::null, "*.kgc|" + i18n( "Kudus Grid" ) );

    if ( !fileName.isEmpty() )
    {
        if ( !fileName.endsWith( ".kgc" ) )
            fileName += ".kgc";

        if ( QFile::exists( fileName ) )
        {
            if ( KMessageBox::warningContinueCancel( 0,
                    i18n( "A file called \"%1\" already exists.\nDo you want to overwrite it?" ).arg( fileName ),
                    QString::null, i18n( "Overwrite" ) ) == KMessageBox::Cancel )
                return;
        }

        if ( m_currentGrid.saveGrid( fileName, m_gameType ) )
            m_modified = false;
    }
}

void Kudus::gameClear()
{
    m_pView->clear();
    m_startGrid.clear();
    m_currentGrid.clear();
    m_modified = false;
    if( m_gameType != DESIGN )
        m_gameType = NONE;
    setCaption( i18n( "None - No Solution" ) );
}

void Kudus::gameDesign()
{
    switch( m_gameType )
    {
        case NONE:
            gameClear();
            m_gameType = DESIGN;
            setCaption( i18n( "Designed - No Solution" ) );
            break;
        case DESIGN:
        {
            int ret = KMessageBox::questionYesNoCancel( this, i18n( "Do you wish to test your design or exit design mode?" ), i18n( "Design Mode" ), i18n( "&Test Design" ), i18n( "&Exit Design mode" ) );
            if( ret == KMessageBox::Yes )
            {
                m_pDesign->setChecked( true );
                m_currentGrid.clearPossibles();
                for( int i = 0; i < 81; ++i )
                {
                    if( m_currentGrid[i].type == CellState::FIXED )
                        m_startGrid[i] = m_currentGrid[i];
                    else
                        m_startGrid[i].clear();
                }

                Solver solve( m_startGrid );

                if ( solve.findSolution() )
                {
                    startGrid( true );
                    m_modified = false;
                    m_gameType = SIMPLE;
                    showTitle( solve.majorDifficulty(), solve.minorDifficulty(), i18n( "Generated" ) );
                }
                else
                    KMessageBox::sorry( this, i18n( "Could not test design, as there is no valid solution." ), i18n( "No Valid Solution" ));
            }
            else if( ret == KMessageBox::No )
            {
                gameClear();
                m_gameType = NONE;
                setCaption( i18n( "None - No Solution" ) );
            }
            else
                m_pDesign->setChecked( true );
        }
            break;
        case SIMPLE:
            if( m_pDesign->isChecked() )
            {
                int ret = KMessageBox::questionYesNo( this, i18n( "Do you wish to exit the current game and start a new design?" ), i18n( "Design Mode" ) );
                if(  ret == KMessageBox::Yes )
                {
                    gameClear();
                    m_gameType = DESIGN;
                    setCaption( i18n( "Designed - No Solution" ) );
                }
                else
                    m_pDesign->setChecked( false );
            }
            else
            {
                m_pDesign->setChecked( true );
                int ret = KMessageBox::questionYesNo( this, i18n( "Do you wish to exit the test game?" ), i18n( "Design Mode" ) );
                if( ret == KMessageBox::Yes )
                {
                    m_gameType = DESIGN;
                    m_currentGrid = m_startGrid;
                    m_pView->setGrid( m_currentGrid );
                    Solver solve( m_currentGrid );
                    if( solve.findSolution() )
                        showTitle( solve.majorDifficulty(), solve.minorDifficulty(), i18n( "Designed" ) );
                    else
                        setCaption( i18n( "Designed - No Solution" ) );
                }
            }
            break;
    }
}

void Kudus::gamePrint()
{
    KPrinter printer;
    PrintDialog *dlg = new PrintDialog( this, true );
    int code = dlg->exec();

    if ( code == QDialog::Accepted )
    {
        if ( printer.setup( this ) )
        {
            QPainter painter;
            painter.begin( &printer );
            QPaintDeviceMetrics m( painter.device() );
            int maxSize = m.width() > m.height() ? m.height() : m.width();
            QPoint offset(( m.width() - maxSize ) / 2, ( m.height() - maxSize ) / 2 );

            if ( dlg->currentGrid() )
            {
                m_pView->print( painter, QRect( 0, 0, maxSize, maxSize ) );
            }
            else
            {
                int diff;
                QPoint offset2;

                switch ( dlg->numberGrids() )
                {
                    case 1:
                        if ( !genPrintGrid( painter, QRect( 0, 0, maxSize, maxSize ), dlg->difficutyLevel() ) )
                        {
                            printer.abort();
                            return;
                        }

                        break;

                    case 2:
                        if ( m.width() > m.height() )
                        {
                            diff = ( m.width() / 20 );
                            maxSize = ( m.width() / 2 ) - diff;
                            offset.setX(( m.width() - (( maxSize * 2 ) + diff ) ) / 2 );
                            offset.setY(( m.height() - maxSize ) / 2 );
                            offset2.setX( offset.x() + maxSize + diff );
                            offset2.setY( offset.y() );
                        }
                        else
                        {
                            diff = ( m.height() / 20 );
                            maxSize = ( m.height() / 2 ) - diff;
                            offset.setX(( m.width() - maxSize ) / 2 );
                            offset.setY(( m.height() - (( maxSize * 2 ) + diff ) ) / 2 );
                            offset2.setX( offset.x() );
                            offset2.setY( offset.y() + maxSize + diff );
                        }

                        if ( !genPrintGrid( painter, QRect( offset.x(), offset.y(), maxSize, maxSize ), dlg->difficutyLevel() ) ||
                                !genPrintGrid( painter, QRect( offset2.x(), offset2.y(), maxSize, maxSize ), dlg->difficutyLevel() ) )
                        {
                            printer.abort();
                            return;
                        }

                        break;

                    case 4:
                    {
                        if ( m.width() < m.height() )
                        {
                            diff = ( m.width() / 20 );
                            maxSize = ( m.width() / 2 ) - diff;
                        }
                        else
                        {
                            diff = ( m.height() / 20 );
                            maxSize = ( m.height() / 2 ) - diff;
                        }

                        offset.setX(( m.width() - (( maxSize * 2 ) + diff ) ) / 2 );

                        offset.setY(( m.height() - (( maxSize * 2 ) + diff ) ) / 2 );
                        offset2.setX( offset.x() + maxSize + diff );
                        offset2.setY( offset.y() + maxSize + diff );

                        if ( !genPrintGrid( painter, QRect( offset.x(), offset.y(), maxSize, maxSize ), dlg->difficutyLevel() ) ||
                                !genPrintGrid( painter, QRect( offset.x(), offset2.y(), maxSize, maxSize ), dlg->difficutyLevel() ) ||
                                !genPrintGrid( painter, QRect( offset2.x(), offset.y(), maxSize, maxSize ), dlg->difficutyLevel() ) ||
                                !genPrintGrid( painter, QRect( offset2.x(), offset2.y(), maxSize, maxSize ), dlg->difficutyLevel() ) )
                        {
                            printer.abort();
                            return;
                        }
                    }

                    break;
                }
            }

            painter.end();
        }
    }
}

void Kudus::moveSolve()
{
    Solver solve( m_currentGrid );

    if ( solve.findSolution() )
    {
        showTitle( solve.majorDifficulty(), solve.minorDifficulty(), i18n( "Solution" ) );
        m_solveActive = true;

        for ( int i = 0; i < 81; ++i )
        {
            if ( m_currentGrid.value( i ) < 0 && solve.solution().value( i ) >= 0 )
            {
                m_currentGrid.setValue( i, solve.solution().value( i ), CellState::GENERATED );
                m_pView->setCell( i, m_currentGrid[i] );
            }
        }

        m_solveActive = false;
    }
    else if( solve.firstSolution().complete() )
    {
        int rst = KMessageBox::questionYesNo( this, i18n( "Found several solutions to the current grid.\nDo you wish to show the first solution?" ), i18n("Multiple Solutions" ));
        if( rst == KMessageBox::Yes )
        {
            showTitle( solve.majorDifficulty(), solve.minorDifficulty(), i18n( "One of many solutions" ) );
            m_solveActive = true;
            for( int i = 0; i < 81; ++i )
            {
                if ( m_currentGrid.value( i ) < 0 && solve.firstSolution().value( i ) >= 0 )
                {
                    m_currentGrid.setValue( i, solve.firstSolution().value( i ), CellState::GENERATED );
                    m_pView->setCell( i, m_currentGrid[i] );
                }
            }
        }
    }
    else
        KMessageBox::sorry( this, i18n( "Could not find a solution." ) );

    m_modified = false;
}

void Kudus::movePencil()
{
    m_currentGrid.matchUserPossible();

    for ( int i = 0; i < 81; ++i )
    {
        if ( !m_currentGrid[i].userPossible.isEmpty() )
            m_pView->setCellAllPossibles( i, m_currentGrid[i].userPossible );
    }

    m_modified = true;
}

void Kudus::moveHint()
{
    Solver solve( m_currentGrid );

    if ( solve.findSolution() )
    {
        std::srand( time( 0 ) );

        while ( true )
        {
            int rnd = ( int )((( float )std::rand() / ( float )RAND_MAX ) * 80.0 );

            if ( m_currentGrid.value( rnd ) < 0 && solve.solution().value( rnd ) >= 0 )
            {
                m_currentGrid.setValue( rnd, solve.solution().value( rnd ), CellState::GENERATED );
                m_pView->setCell( rnd, m_currentGrid[rnd] );
                break;
            }
        }
    }
    else
        KMessageBox::sorry( this, i18n( "Could not find a solution to get a hint from." ) );

    m_modified = true;
}

void Kudus::moveCell( int move )
{
    if( m_gameType == DESIGN )
        m_pView->moveCurrentCell( move, true );
    else
        m_pView->moveCurrentCell( move, false );
}

void Kudus::moveClearCell()
{
    if( m_pView->currentCell() >= 0 )
    {
        switch( m_gameType )
        {
            case DESIGN:
            {
                m_currentGrid.clearCell( m_pView->currentCell() );
                m_pView->setCurrentCell( m_currentGrid[m_pView->currentCell()] );
                Solver solve( m_currentGrid );
                if( solve.findSolution() )
                    showTitle( solve.majorDifficulty(), solve.minorDifficulty(), i18n( "Designed" ) );
                else
                    setCaption( i18n( "Designed - No Solution" ) );
            }
                break;
            case SIMPLE:
                if( m_currentGrid.type( m_pView->currentCell() ) == CellState::NORMAL )
                {
                    m_currentGrid.clearCell( m_pView->currentCell() );
                    m_pView->setCurrentCell( m_currentGrid[m_pView->currentCell()] );
                }
                break;
            default:
                break;
        }
    }
}

void Kudus::setValue( int value )
{
    if( m_pView->currentCell() >= 0 )
    {
        switch( m_gameType )
        {
            case DESIGN:
                if( m_currentGrid[m_pView->currentCell()].isPossible( value ) )
                {
                    m_modified = true;
                    m_currentGrid.setValue( m_pView->currentCell(), value, CellState::FIXED );
                    m_pView->setCurrentCell( m_currentGrid[m_pView->currentCell()] );
                    Solver solve( m_currentGrid );
                    if( solve.findSolution() )
                        showTitle( solve.majorDifficulty(), solve.minorDifficulty(), i18n( "Designed" ) );
                    else
                        setCaption( i18n( "Designed - No Solution" ) );
                }
                break;
            case SIMPLE:
                if( m_currentGrid.type( m_pView->currentCell() ) == CellState::NORMAL )
                {
                    m_modified = true;
                    m_currentGrid.setValue( m_pView->currentCell(), value, CellState::NORMAL );
                    m_pView->setCurrentCell( m_currentGrid[m_pView->currentCell()] );
                    checkComplete();
                }
                break;
            default:
                break;
        }
    }
}

void Kudus::togglePossible( int value )
{
    if( m_pView->currentCell() >= 0 )
    {
        switch( m_gameType )
        {
            case SIMPLE:
                if( m_currentGrid.type( m_pView->currentCell() ) == CellState::NORMAL )
                {
                    m_modified = true;
                    m_currentGrid.toggleUserPossible( m_pView->currentCell(), value );
                    m_pView->setCurrentCell( m_currentGrid[ m_pView->currentCell()] );
                }
                break;
            default:
                break;
        }
    }
}

bool Kudus::queryClose()
{
    if ( m_modified )
    {
        int ret = KMessageBox::warningYesNoCancel( 0, i18n( "Do you want to save your progress or discard it?" ),
                  i18n( "Unsaved Changes" ), i18n( "&Save" ), i18n( "&Discard" ) );

        switch ( ret )
        {
            case KMessageBox::Yes:
                gameSave();

                break;

            case KMessageBox::Cancel:
                return false;

                break;
        }
    }

    return true;
}

void Kudus::optionsConfigureToolbars()
{
    saveMainWindowSettings( KGlobal::config(), autoSaveGroup() );
    KEditToolbar dlg( factory() );
    connect( &dlg, SIGNAL( newToolbarConfig() ), this, SLOT( applyNewToolbarConfig() ) );
    dlg.exec();
}

void Kudus::applyNewToolbarConfig()
{
    applyMainWindowSettings( KGlobal::config(), autoSaveGroup() );
}

void Kudus::setupWindow()
{
    m_pView = new GridView( this );
    setCentralWidget( m_pView );
    setCaption( i18n( "None - No Solution" ) );

    setMinimumSize( 300, 300 );
}

void Kudus::setupActions()
{
    // Game menu
    KStdGameAction::gameNew( this, SLOT( gameNew() ), actionCollection() );
    KStdGameAction::load( this, SLOT( gameLoad() ), actionCollection() );
    KStdGameAction::save( this, SLOT( gameSave() ), actionCollection() );
    KStdGameAction::restart( this, SLOT( gameRestart() ), actionCollection() );
    KStdGameAction::print( this, SLOT( gamePrint() ), actionCollection() );
    KStdGameAction::quit( this, SLOT( close() ), actionCollection() );
    new KAction( i18n( "Clear Grid" ), "fileclose", 0, this, SLOT( gameClear() ), actionCollection(), "game_clear" );
    m_pDesign = new KToggleAction( i18n( "Design Mode" ), "edit", 0, this, SLOT( gameDesign() ), actionCollection(), "game_design" );
    m_pDesign->setChecked( false );

    // move menu
    KStdGameAction::solve( this, SLOT( moveSolve() ), actionCollection() );
    KStdGameAction::hint( this, SLOT( moveHint() ), actionCollection() );
    new KAction( i18n( "Pencil" ), "pencil", 0, this, SLOT( movePencil() ), actionCollection(), "move_pencil" );
    QSignalMapper *moveMapper = new QSignalMapper( this );
    connect( moveMapper, SIGNAL( mapped( int ) ), SLOT( moveCell( int ) ) );
    KAction *move = new KAction( i18n( "Move Up" ), "up", Qt::Key_Up, moveMapper, SLOT( map() ), actionCollection(), "move_up" );
    moveMapper->setMapping( move, GridView::UP );
    move = new KAction( i18n( "Move Down" ), "down", Qt::Key_Down, moveMapper, SLOT( map() ), actionCollection(), "move_down" );
    moveMapper->setMapping( move, GridView::DOWN );
    move = new KAction( i18n( "Move Left" ), "back", Qt::Key_Left, moveMapper, SLOT( map() ), actionCollection(), "move_left" );
    moveMapper->setMapping( move, GridView::LEFT );
    move = new KAction( i18n( "Move Right" ), "forward", Qt::Key_Right, moveMapper, SLOT( map() ), actionCollection(), "move_right" );
    moveMapper->setMapping( move, GridView::RIGHT );
    new KAction( i18n( "Clear Cell" ), "remove", Qt::Key_Delete, this, SLOT( moveClearCell() ), actionCollection(), "move_clear_cell" );

    // Settings menu
    KStdAction::configureToolbars( this, SLOT( optionsConfigureToolbars() ), actionCollection() );
    KStdAction::keyBindings( guiFactory(), SLOT( configureShortcuts() ), actionCollection() );

    // Not on a menu
    QSignalMapper *mainMapper = new QSignalMapper( this );
    connect( mainMapper, SIGNAL( mapped( int ) ), this, SLOT( setValue( int ) ) );
    QSignalMapper *posMapper = new QSignalMapper( this );
    connect( posMapper, SIGNAL( mapped( int ) ), this, SLOT( togglePossible( int ) ) );

    for ( int i = 0; i < 9; ++i )
    {
        KAction *num = new KAction( i18n( "Enter Number %1" ).arg( i + 1 ), Qt::Key_1 + i, mainMapper, SLOT( map() ), actionCollection(), QString( "mainNum%1" ).arg( i + 1 ) );
        mainMapper->setMapping( num, i );
        num = new KAction( i18n( "Possible Number %1" ).arg( i + 1 ), Qt::SHIFT + Qt::Key_1 + i, posMapper, SLOT( map() ), actionCollection(), QString( "posNum%1" ).arg( i + 1 ) );
        posMapper->setMapping( num, i );
    }


}

void Kudus::showTitle( int mjrDiff, int mnrDiff, const QString& type )
{
    setCaption( i18n( "%1 - %2:%3" ).arg( type ).arg( mjrDiff ).arg( mnrDiff ) );
}

void Kudus::startGrid( bool fullstate )
{
    for ( int i = 0; i < 81; ++i )
    {
        if ( fullstate )
        {
            if ( m_startGrid.value( i ) >= 0 )
                m_startGrid.setType( i, CellState::FIXED );
        }

        m_pView->setCell( i, m_startGrid[i] );
    }

    m_currentGrid = m_startGrid;
}

void Kudus::checkComplete()
{
    if( m_gameType == SIMPLE )
    {
        if( m_currentGrid.complete() )
        {
            KMessageBox::information( this, i18n( "Well Done! You have completed the game!" ), i18n( "Game Complete" ) );
            m_gameType = NONE;
        }
    }
}

bool Kudus::genPrintGrid( QPainter &painter, QRect rect, int difficulty )
{
    Generator gen( difficulty );

    KProgressDialog* progress = new KProgressDialog( this, "progressDiag", i18n( "Generating Grid" ), i18n( "Generating ..." ), true );
    progress->progressBar()->hide();
    progress->show();

    GridState grid = gen.generate( progress );

    removeChild( progress );
    delete progress;

    Solver solve( grid );

    if ( solve.findSolution() )
    {
        GridView *gv = new GridView();
        painter.setPen( Qt::black );

        for ( int i = 0; i < 81; ++i )
        {
            if ( grid.value( i ) >= 0 )
                grid.setType( i, CellState::FIXED );
            gv->setCell( i, grid[i] );
        }

        gv->print( painter, rect );

        delete gv;
        return true;
    }
    else
        return false;
}

#include "kudus.moc"
