/***************************************************************************
                          animator.h  -  description
                             -------------------
    begin                : Sat Mar 29 2003
    copyright            : (C) 2003 by Fungmeista
    email                : mizunoami44@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef ANIMATOR_H
#define ANIMATOR_H 
 
class QObject;
class QTimer;

/** 
  * \author Fungmeista 
  * \brief This class provides an interface for animating a graph.
  * \note Because the QTimer class uses signals/slots, subclasses currently must include 
  *       a slotNextFrame() slot that calls this class's nextFrame() function.
  *       The Animator class itself cannot contain signals/slots, because
  *       this would require it to be a QObject subclass.  If it were a QObject subclass, 
  *       QObject subclasses would not be able to inherit it ( Multiple inheritance clashes ).  This is an instance when 
  *       using callbacks would be better than signals/slots... anyways, I'm looking for other workarounds :-)
  *
  *       Also, don't forget to call initialize() before using this class. 
  */ 
 
class Animator
{
	public:
		Animator();
		virtual ~Animator();
		
		/** Sets the value that the animation begins at and returns to after the end of a loop */
		int setAnimationMin(double);
		
		/** Sets the maximum value the animation will reach before ending */
		int setAnimationMax(double);
		
		/** Sets the step of each animation */
		int setAnimationStep(double);
		
		/**  Sets the speed between each frame in the animation in milliseconds ( 1000 ms = 1 sec ) */
		int setAnimationSpeed(int);
		
		/** Returns the value that the animation begins at and returns to after the end of a loop */
		double getAnimationMin() const{return min;}
		
		/** Returns the maximum value the animation will reach before ending */
		double getAnimationMax() const{return max;}
		
		/** Returns the step of each animation */
		double getAnimationStep() const{return step;}
		
		/** Returns the speed of the graph in milliseconds */
		int getAnimationSpeed() const{return speed;}
		
		/** Sets the value of the the animator to the given number */
		void setAnimationFrame(double n){current = n; nextFrameReady();}
		
		/** Subclasses should use this function to get the current stage of the animation
		  * in order to produce the animation
		  */
		double animatorValue() const{return current;}

		/** Begins to play the animation and will loop until pause() or stop() is called */
		void play();
		
		/** Stops a playing animation and resets the current animator value to the minimum */
		void stop();
		
		/** Stops a playing animation, but unlike stop(), the current animator value remains 
		  * what is was.  Calling play() will continue the animation from where it left off.
		  */
		void pause();
		
		/** Save the complete animation as a movie. \note Just a stub for now */
		void saveAsMovie();
		
		inline bool displayAnimatorValue(){ return _displayNValue; }
		inline void setDisplayAnimatorValue( bool b ){ _displayNValue = b; }

	protected:
		/** Classes that implement this class need to implement this function.
		  * This function is called everytime the next frame is ready and the
		  * subclass should implement an animation with it
		  */
		virtual void nextFrameReady() = 0;

		/** This function should be called after every timer timeout */
		void nextFrame();
		
		/** Call this function before using the animation interface.  It should be given a pointer to
		  * this class's subclass.  This function connects the timeout() signal of the QTimer class used
		  * to the slotNextFrame() slot of the given class.
		  */
		void initializeAnimator( QObject * );
		
		bool isSaving() const{return _save;}
	
	private:
		double min, max, step, current;
		int speed;
		QTimer *_timer;
		bool _displayNValue, _save;
};

#endif

