/***************************************************************************
                          glexpressiongraph.h  -  description
                             -------------------
    begin                : Fri Oct 18 2002
    copyright            : (C) 2002 by Fungmeista
    email                : mizunoami44@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef GLEXPRESSIONGRAPH_H
#define GLEXPRESSIONGRAPH_H

#include <vector>
#include <fstream>

#include "expression.h"
#include "cartesian3dcoord.h"

#include "glbasicgraph.h"
#include "animator.h"

class FungParser;

/**
 * @author Fungmeista
 * @brief Subclass this to create 3D graphs based on expressions.  This handles adding expressions,
 * animating the given expressions, and such things.
 */
class GLExpressionGraph : public GLBasicGraph, public Animator
{
	Q_OBJECT

	public:
		GLExpressionGraph(QWidget *parent=0, const char *name=0);
		~GLExpressionGraph();

		typedef std::vector< std::vector<Cartesian3DCoord> > Cartesian3DCoord2DVector;

		inline void setAngle(int m){angle = m; recompileLists(); updateGL();}
		inline int getAngle() const{return angle;}

		void addExpression(const Expression &);

		void removeExpression(unsigned int);
		inline Expression getExpression(unsigned int i) const{return (functions.size() >= i + 1 && functions[i]) ? *functions[i] : Expression("",true,QColor());}

		inline int getCurrentExpressionIndex() const{return expressionIndex;}
		inline void setCurrentExpressionIndex(int i){ expressionIndex = i;}

		inline void setDrawConnected(bool b){drawConnected = b; recompileLists(); updateGL();}
		inline bool isDrawConnected() const{return drawConnected;}

		inline void setGrid(bool b){showGrid = b; recompileLists(); updateGL();}
		inline bool getGridState() const{return showGrid;}

		inline void setGridColor(const QColor &c){_gridColor = c; recompileLists(); updateGL();}
		inline QColor gridColor() const{return _gridColor;}

		inline bool isTracing() const{return doTrace;}

		inline void setTrace(bool b){doTrace = b; updateGL();}

		/** Returns the number of expressions the graph is holding.  This includes hidden expressions. */
		inline int functionCount() const{return functions.size();}

		/** Sets the resolution in the x direction.  Warning: Any small values for this functions can significantly reduce the speed of the program.
		  * Also, values less than 0.05 could cause errors due to lack of system resources, although it depends on your system.
		  */
		inline void setXRes(GLfloat f){xRes = f; xScale = (getXMax()-getXMin())/f;}

		/** Sets the resolution in the y direction.  Warning: Any small values for this functions can significantly reduce the speed of the program.
		  * Also, values less than 0.05 could cause errors due to lack of system resources, although it depends on your system.
		  */
		inline void setYRes(GLfloat f){yRes = f; yScale = (getYMax()-getYMin())/f;}

		inline GLfloat getXRes() const{return xScale;}
		inline GLfloat getYRes() const{return yScale;}
		
		void exportToDXF( const char * );
		
	public slots:
		/** Sets the resolution of the graph to the specified percent of possible resolution.
		  * This accepts a double between 0 and 100. If less than or greater than 0 or 100,
		  * it sets the resolution to 0 or 100 respectively.
		  */
		void setResolution(int percent);

		int getResolution() const;
	
		/** Makes the next expression active */
		void nextExpression();

		/** Makes the previous expression active */
		void prevExpression();

		void setShowCurrentExpression(bool state);

		void slotNextFrame(){nextFrame();}
		
	signals:
		void animationValueChanged(double);

	protected:
		void paintGL();
		void initializeGL();

		virtual double snapIntervalX() const{return xScale;}
		virtual double snapIntervalY() const{return yScale;}

		virtual void nextFrameReady();

		/* Implement this function to let the ExpressionGraph know the independent variables which the graph takes.
		 * Multiple variables should be seperated by commas.
		 */
		virtual const char * independent_vars() const = 0;

		/* Implement this function to let the ExpressionGraph know the dependent variables which the graph takes. */
		virtual const char * dependent_var() const = 0;

		/* Implement this function which the ExpressionGraph calls when tracing to let it know the x, y, and z
		 * coordinates of the trace box.
		 */
		virtual void trace(GLfloat mouseX, GLfloat mouseY, FungParser &parsed_expression, GLfloat *x, GLfloat *y, GLfloat *z) = 0;

		inline virtual GLfloat traceDepX(GLfloat x) const{return x;}
		inline virtual GLfloat traceDepY(GLfloat y) const{return y;}

		void recompileLists();
		void orthoChanged();

		/** Blends the given color with black according to the elevation
		  * given by the GLfloat
		  */
		void doColorBlend(GLfloat, const QColor &, GLfloat, GLfloat );
		
		/** This class uses the 2-dimensional vector returned from this function to
		  * draw the expression and export it do DXF.  The 2-dimensional vector of
		  * Cartesian3DCoord's should be all of the vertices on the graph.
		  */
		virtual Cartesian3DCoord2DVector getValues( FungParser &, GLfloat * = 0, GLfloat * = 0 ) = 0;

		//events
		void mouseMoveEvent(QMouseEvent *);
		void resizeGL( int w, int h );

		GLfloat xScale, yScale;
		
		

	private:
		void compileList(GLuint, FungParser &, std::string &, const QColor &);
		void exportSingleExprDXF( const Cartesian3DCoord2DVector &, std::ofstream *output, int layer );
		void drawCube(GLfloat x, GLfloat y, GLfloat z, GLfloat percentx, GLfloat percenty);
		std::vector<Expression*> functions;
		unsigned int expressionIndex;
		int angle;
		QColor _gridColor;
		bool drawConnected, showGrid, doTrace;
		GLfloat lastX,lastY,lastZ;
		GLfloat xRes,yRes;
		int x_percent_res, y_percent_res;
};

#endif
