/***************************************************************************
                          mathfunction.h  -  description
                             -------------------
    begin                : Mon Nov 11 2002
    copyright            : (C) 2002-03 by Fungmeista
    email                : mizunoami44@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef MATHFUNCTION_H
#define MATHFUNCTION_H

#include <qpoint.h>
#include <qstring.h>

class QPointArray;
class QPainter;
class QMouseEvent;

class FungParser;
class ExpressionGraph;
class BasicGraph;

/**
 * \author Fungmeista
 * \brief An abstract class which is used by a BasicGraph to execute math functions such as definite integral, arc length, etc.
 *        In order to create your own math function, you must create a subclass of this which reimplements
 *        calculate_and_draw(QPainter *, QString *), and as the name implies, calculates the result of the
 *        function and writes what you wish to display as the result in the given QString pointer.  An example
 *        of this could be "Distance = 1.44" when calculating distance.  Also, a pointer to a QPainter is given
 *        which allows you to draw a visual of the function calculated.  This could shade the area underneath
 *        a curve for definite integral, for example.
 */
class MathFunction
{
	public: 
		MathFunction(BasicGraph * = 0,int expressions = 0);
		virtual ~MathFunction();

		/** When creating a math function subclass, overwrite this function to calculate and then draw
		  * whatever visual you wish using the given QPainter.  You should also call the function setResult()
		  * inside this function and pass to it a string that you wish to be displayed as the result of the
		  * function.
		  */
		virtual void calculate_and_draw(QPainter*){}

		/** When creating a math function subclass, overwrite this function with a function to more quickly
		  * draw the result of the function.  For example, if calculate_and_draw calculates many points,
		  * these can be saved and drawn in draw_stored instead of recalculating them.
		  */
		virtual void draw_stored(QPainter *p){calculate_and_draw(p);}

		/** Call this function to begin determining function.  After calling this function you will be asked
		  * to enter a lower bounds, then an upper bounds, and then the result will be displayed which is
		  * determined by your calculate_and_draw() function.  You may passed a pointer to a parsed expression
		  * which will be stored in the "expression" variable and can be used in determining the value of the
		  * function.
		  */
  
		int expressionsRequired(){ return expression_count;}

		void exec(FungParser *fp = 0, FungParser *fp2 = 0);

		void drawFunction(QPainter *);
		bool active(){return _active;}
		void sendClick(QMouseEvent *e, double arg = 0, double arg2 = 0);
		void cancel();

	protected:
		/** */
		QPoint lowerBounds,upperBounds;
		double arg1, arg2, beta, arg1x, arg2x, arg1y, arg2y, beta2;

		/** Sets the string that will be displayed as the result of the function */
		void setResult(QString &s);

		int expression_count;

		/** This variable is available to subclasses to use for calculating the value of a function with.  It is automatically stored in this variable
		  * for use in the functions draw_and_calculate() and draw_stored().
		  */
		FungParser *expression;
		FungParser *expression2;

		/** \param x_coordinate A y-direction coordinate in the coordinate system of the graph
		  * \return A number that is in the coordinate system of the underlying QWidget in the x direction.
		  * \note When drawing onto the given QPainter it takes actual pixel coordinates where the bottom left is the x-y coordinate
		  * (0,0).
		  */
		int toPixelXCoord(double x_coordinate);

		/** \param y_coordinate A x-direction coordinate in the coordinate system of the graph
		  * \return A number that is in the coordinate system of the underlying QWidget in the x direction.
		  * \note When drawing onto the given QPainter it takes actual pixel coordinates where the bottom left is the x-y coordinate
		  * (0,0).
		  */
		int toPixelYCoord(double y_coordinate);

		/** \param y_coordinate A y-direction coordinate in the coordinate system of the underlying QWidget.
		  * \return A number that is in the coordinate system of the graph in the y direction.
		  * \note When drawing onto the given QPainter it takes actual pixel coordinates where the bottom left is the x-y coordinate
		  * (0,0).
		  */
		double toGraphXCoord(double x_coordinate);

		/** \param x_coordinate A x-direction coordinate in the coordinate system of the underlying QWidget.
		  * \return A number that is in the coordinate system of the graph in the x direction.
		  * \note When drawing onto the given QPainter it takes actual pixel coordinates where the bottom left is the x-y coordinate
		  * (0,0).
		  */
		double toGraphYCoord(double y_coordinate);

		double animatorValue();

		/** Draws a horizontal and vertical line going through the point (x,y) as well as a small box at this point */
		void drawSelectedPoint(double x, double y, QPainter *painter);

		/** This is provided in the case that you will need information from the BasicGraph such as its height/width or something else it provides.*/
		BasicGraph *basicgraph;
	private:
		void getTwoPoints();
		void drawLeftArrow(QPainter *);
		void drawRightArrow(QPainter *);

		QString resultString;              
		bool _active;
		int mouseClicks;
		QPoint lastClick;
		
		static const int LEFTOFFSET = 30;

		static const int CANCEL = -1;
		static const int GETLOWERBOUND = 0;
		static const int GOTLOWERBOUND = 1;
		static const int GETUPPERBOUND = 2;
		static const int GOTUPPERBOUND = 3;
		static const int CALCULATE = 4;
		static const int CLICKTOCONTINUE = 5;

		static const int BOUNDS_TRIANGLE = 4;
};

#endif //MATHFUNCTION_H
