// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the OBJReader that imports Wavefront .obj files
		\author Tim Shead (tshead@k-3d.com)
*/

#include "obj_helpers.h"

#include <k3dsdk/file_helpers.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/imesh_storage.h>
#include <k3dsdk/material.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>

using namespace k3d::xml;

#include <k3dsdk/fstream.h>
#include <boost/filesystem/operations.hpp>

namespace libk3dgeometry
{

/////////////////////////////////////////////////////////////////////////////
// obj_mesh_input_implementation

class obj_mesh_input_implementation :
	public k3d::mesh_source<k3d::persistent<k3d::node> >,
	public k3d::imesh_storage
{
	typedef k3d::mesh_source<k3d::persistent<k3d::node> > base;

public:
	obj_mesh_input_implementation(k3d::idocument& Document) :
		base(Document),
		m_file(init_owner(*this) + init_name("file") + init_label(_("File")) + init_description(_("Input file")) + init_value(boost::filesystem::path()) + init_path_mode(k3d::ipath_property::READ) + init_path_type("obj_files")),
		m_texture_u(init_owner(*this) + init_name("texture_u") + init_label(_("Texture U")) + init_description(_("Texture U")) + init_value(std::string("s"))),
		m_texture_v(init_owner(*this) + init_name("texture_v") + init_label(_("Texture V")) + init_description(_("Texture V")) + init_value(std::string("t"))),
		m_texture_w(init_owner(*this) + init_name("texture_w") + init_label(_("Texture W")) + init_description(_("Texture W")) + init_value(std::string("w")))
	{
		m_file.changed_signal().connect(make_reset_mesh_slot());
		m_texture_u.changed_signal().connect(make_reset_mesh_slot());
		m_texture_v.changed_signal().connect(make_reset_mesh_slot());
		m_texture_w.changed_signal().connect(make_reset_mesh_slot());
	}

	void reset_mesh(k3d::mesh* const Mesh)
	{
		m_output_mesh.reset(Mesh);
	}

	void on_create_mesh(k3d::mesh& Mesh)
	{
		const boost::filesystem::path path = m_file.value();
		const std::string texture_u = m_texture_u.value();
		const std::string texture_v = m_texture_v.value();
		const std::string texture_w = m_texture_w.value();

		detail::load_obj_file(Mesh, document(), path, texture_u, texture_v, texture_w);
	}

	void on_update_mesh(k3d::mesh& Mesh)
	{
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<obj_mesh_input_implementation>,
                k3d::interface_list<k3d::imesh_source,
                k3d::interface_list<k3d::imesh_storage> > > factory(
			k3d::uuid(0x650f96b9, 0xe1c145d5, 0x9eda410d, 0x13f4e98f),
			"OBJMeshInput",
			_("Mesh source that loads external Wavefront (.obj) files into the document by reference"),
			"GeometryReader");

		return factory;
	}

private:
	k3d_data(boost::filesystem::path, immutable_name, change_signal, with_undo, local_storage, no_constraint, path_property, path_serialization) m_file;
	k3d_data(std::string, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_texture_u;
	k3d_data(std::string, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_texture_v;
	k3d_data(std::string, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_texture_w;
};

k3d::iplugin_factory& obj_mesh_input_factory()
{
	return obj_mesh_input_implementation::get_factory();
}

} // namespace libk3dgeometry

