/***************************************************************************
 *   Copyright (C) 2006 by Darth Vader, DARKSTAR Linux Project             *
 *   darkstar@dainet.ro                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef HWPROBE_H
#define HWPROBE_H

/**

  hwprobe.h

  Purpose:      class definition for HwProbe

  Authors:      Darth Vader <darkstar@dainet.ro>
  Maintainer:   Darth Vader <darkstar@dainet.ro>

*/

/*
HwProbe description syntax
--------------------------

Purpose:

The HwProbe is a generalized class to handle access to the libhd hardware autoprobe library. 

Implementation:

The HwProbe probes the hardware when needed. There is no (more) probing at initialization time. It's the sole responsibility of
the underlying libhd library to give a consistent view of the hardware. The libhd data closely resembles the PCI view of hardware,
namely classes and subclasses. The class data is extended to included ISAPnP and other non-PCI hardware. However, this
implementation detail is transparently mapped.

HARDWARE PROBING REPOSITORY PATHS
---------------------------------

The hardware probing repository is accessed with the path prefix .probe followed by sub-paths.

A list of possible sub-paths can be retrieved with Dir (.probe) which currently results in:

"has_pcmcia", "architecture", "boot_arch", "version", "boot_disk", "cdb_isdn", "has_smp", "bios_video",
"framebuffer", "status", "cdrom", "floppy", "disk", "netif", "display", "mouse", "keyboard", "sound", "isdn",
"modem", "storage", "netcard", "monitor", "printer", "tv", "scanner", "system", "camera", "chipcard", "usbctrl",
"hub", "scsi", "ide", "memory", "fbdev", "usb", "pci", "isapnp", "cpu", "braille", "joystick", "bios"

 -----------------------------------------------------------
| Explanation of .probe paths                               |
 -----------------------------------------------------------
path                 type    result                  comment
----                 ----    ------                  -------
.probe.cdrom         List     list of hd-entries     only returns component data for cd-rom like devices (cd-r, cd-rw, dvd) 
.probe.disk          List     list of hd-entries     only returns component data for disk like devices (harddisk, *not* floppy !)
.probe.net           List     list of hd-entries     only returns component data for network like devices (ethernet, loopback (!),
                                                     dummy (!)) 
.probe.has_pcmcia    Boolean                         true if the system has a pcmcia controller, false else  used to decide if pcmcia
                                                     software is to be installed and how
.probe.has_eide      Boolean                         true if the system has a special EIDE controller, false else  used to decide if
                                                     a special kernel (with compiled-in driver) is to be installed
.probe.architecture  String                          reports the system architecture, i.e. "intel" or "alpha"
.probe.boot_arch     String
.probe.system        String                          reports the system type if given, else "" (i.e. "MacRISC", "PReP", or "CHRP" on PPC)
.probe.modem         List     list of modems
.probe.isdn          List     list of isdn cards
.probe.dsl           List     list of DSL cards
.probe.netcard       List     list of network cards

General layout of a hd-entry:

As stated above, most Read(.probe) calls return a list of entries matching a specific feature. All these entries have a common
layout (i.e. vendor-id) and some device specific data (i.e. geometry for harddisks, hsync/vsync values for video monitors, etc.)

This chapter explains the general layout. Each entry is a Map with the following key/value pairs

Important: Always check for the existence of a key/value pair !!!

 ------------------------------------------------------
| GLOBAL                                               |
 ------------------------------------------------------
key             type       value                comment
---             ----       -----                -------
broken          Boolean    true                 hardware seems broken
cardtype        String     AGP                  pci card in agp slot
cardtype        String     PnP                  PnP device
bus             String     (see below)          name of bus
baseclass       String     (see below)          name of baseclass
subclass        String     (see below)          name of subclass
class_id        Integer    (see libhd)          baseclass id
sub_class_id    Integer    (see libhd)          subclass id
device          String     name of device       user readable format
vendor          String     name of vendor       user readable format
sub_device      String     name of subdevice    user readable format
sub_vendor      String     name of subvendor    user readable format
device_id       Integer    value of device      internal format
vendor_id       Integer    value of vendor      internal format
sub_device_id   Integer    value of subdevice   internal format
sub_vendor_id   Integer    value of subvendor   internal format
unique_key      String                          unique key for every hardware item consists of two parts: a position/slot
                                                dependent part and one that is independent of the location of the card/device.
                                                They are separated by a '.' (dot)
rev             String     revision code        revision code is an integer
compat_device   String     name of compatible   device  only used for ISA-PnP
compat_vendor   String     name of compatible   vendor only used for ISA-PnP
attached_to     Map                             recursive  complete data of device
dev_name        String                          /dev/xxx  associated device file
driver info     Map        (see below)          device specific driver information
resource        Map        (see below)          Map of resources used/needed by device

Possible driver info:

As driver info the following keys are defined

 -------------------------------------------------------------
| MODULE                                                      |
 -------------------------------------------------------------
key         type              value                    comment
---         ----              -----                    -------
name      String              module                   name of driver/module
active    Boolean             true                     if module is already loaded
modprobe  Boolean             true/false               if modprobe or insmod is to be used for loading
args      String              module arguments         arguments used for loading module
conf      String              conf.modules entry       data for conf.modules

 -------------------------------------------------------------
| MOUSE                                                       |
 -------------------------------------------------------------
key         type              value                    comment
---         ----              -----                    -------
xf86        String            XF86Config entry
gpm         String            gpm parameter

 -------------------------------------------------------------
| X11                                                         |
 -------------------------------------------------------------
key         type               value                   comment
---         ----               -----                   -------
server      String             X11 server              might not be present in map (voodoo 1/2 special addons don't need a server)
version     String             X11 server              version  "3" ("server" is XFree 3.3.x Server) or "4" ("server" is XFree 4.x module)
has_3d      Boolean                                    3d accel hardware present
c8          Boolean            true/false               8 bpp support
c15         Boolean            true/false              15 bpp support
c16         Boolean            true/false              16 bpp support
c24         Boolean            true/false              24 bpp support
c32         Boolean            true/false              32 bpp support
dacspeed    Integer            (value in MHz)
packages    List (of Strings)                          additional packages to install  if applicable
extensions  List (of Strings)                          additional X extensions to load  usually goes to 'Module' section, i.e. "glx.so"
options     List (of Strings)  special server options  usually goes to 'Device' section, i.e. "no_accel"
raw         String                                     extra info to add to XF86Config to be inserted into 'Device' section

 -------------------------------------------------------------
| DISPLAY                                                     |
 -------------------------------------------------------------
key         type               value                   comment
---         ----               -----                   -------
width       Integer
height      Integer
min_vsync   Integer
max_vsync   Integer
min_hsync   Integer
max_hsync   Integer
bandwidth   Integer

 -------------------------------------------------------------
| DSL                                                         |
 -------------------------------------------------------------
key         type               value                   comment
---         ----               -----                   -------
name        String                                     driver name
mode        String             capiadsl, pppoe

*/

#include <qstring.h>
#include <qstringlist.h>
#include <qvaluelist.h>
#include <qmap.h>
#include <qvariant.h>

/*
typedef QVariant                 HwValue;

typedef QValueList< HwValue >    HwValueList;
typedef QMap< QString, HwValue > HwValueMap;
*/

class HwProbePrivate;

/**
 * @short Class for access to hardware autoprobe
 */
class HwProbe
{
public:
    /**
     * Creates a new HwProbe.
     */
    HwProbe();

    /**
     * Cleans up
     */
    ~HwProbe ();

    /**
     * Reads data. Destroy the result after use.
     * @param path Specifies what part of the subtree should
     * be read. The path is specified _relatively_ to Root()!
     */
    const QVariant read( const QString& path, const QString& arg = QString::null, const QString& optarg = QString::null ) const;

    /**
     * Writes data. Destroy the result after use.
     */
    bool write( const QString& path, const QString& value, const QString& arg = QString::null ) const;

    /**
     * Get a list of all subtrees.
     */
    const QStringList dir( const QString& path ) const;

    // ----------------------------------------
    static const QVariant probe( const QString& path, const QString& arg = QString::null, const QString& optarg = QString::null );

private:
    /**
     * the private HwProbe class...
     */
    HwProbePrivate *d;
};

#endif // HWPROBE_H
