/***************************************************************************
 *   Copyright (C) 2004 by Yogesh Marwaha                                  *
 *   yogeshm02@rediffmail.com                                              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Library General Public License as       *
 *   published by the Free Software Foundation; either version 2 of the    *
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Library General Public     *
 *   License along with this program; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "blurscopeext.h"
#include <kconfig.h>

BlurscopeExt::BlurscopeExt(QWidget *parent, const char *name) : RVisualisation(parent, name), MonoScope(50){
    myBuffer = 0L;
    installEventFilter(parent);
    reloadSettings();
    isOn = myOn = TRUE;
    show();
}

BlurscopeExt::~BlurscopeExt(){
    BlurscopeExt::stop();
    delete myBuffer;
}

void BlurscopeExt::reloadSettings(void){
    KConfig &config = *KGlobal::config();
    config.setGroup("addiKtUI");
    colCycle = config.readBoolEntry("BS_Colour_Cycle", TRUE);
    r = b = 255;
    g = 0;
    cycle = 1;
    if(colCycle == FALSE){
        r = defRed = config.readNumEntry("BS_R", 255);
        g = defGreen = config.readNumEntry("BS_G", 0);
        b = defBlue = config.readNumEntry("BS_B", 0);
    }
    mQualityBest = config.readBoolEntry("BS_QUALITY_BEST", FALSE);
    mQualityUser = config.readBoolEntry("BS_QUALITY_USER", TRUE);
    mUserResolutionWidth = config.readNumEntry("BS_RESOLUTION_WIDTH", 240);
    mUserResolutionHeight = config.readNumEntry("BS_RESOLUTION_HEIGHT", 120);
}

void BlurscopeExt::initializeGL(){
    glClearColor(0.0, 0.0, 0.0, 0.0);
    glShadeModel(GL_FLAT);
    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
}

void BlurscopeExt::paintGL(){
    if(isOn == FALSE){
        glClear(GL_COLOR_BUFFER_BIT);
        glFlush();
        return;
    }
    glRasterPos2i(0, 0);
    glDrawPixels((GLsizei)_width, (GLsizei)_height, GL_RGB, GL_UNSIGNED_BYTE, myBuffer);
    glFlush();
}

void BlurscopeExt::mouseReleaseEvent(QMouseEvent *e){
    QWidget::mouseReleaseEvent(e);
    if (e->button()==LeftButton){//Toggle visualisation state
        myOn = !myOn;
        isOn = !isOn;
        setOn(isOn);
    }
}

void BlurscopeExt::resizeGL( int w, int h ){
    if(isOn == TRUE)
        BlurscopeExt::stop();
    if(mQualityBest){
        _width = width() + 2;
        _height = height() + 2;
        glPixelZoom(1.0, 1.0);
    }else{//mQualityUser
        if(width() <= mUserResolutionWidth && height() <= mUserResolutionHeight){
            _width = width() + 2;
            _height = height() + 2;
            glPixelZoom(1.0, 1.0);
        }else{
            _width = mUserResolutionWidth + 2;
            _height = mUserResolutionHeight + 2;
            glPixelZoom(w/(GLfloat)_width, h/(GLfloat)_height);
        }
    }
    bufSize = _width * _height * 3;
    if(myBuffer) delete myBuffer;
    myBuffer = NULL;
    myBuffer = new GLubyte[bufSize];
    ASSERT(myBuffer);
    for (uint i = 0; i < bufSize; ++i)
        *(myBuffer + i) = 0;
    setSamples(w);

    glViewport(0, 0, w, h);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0.0, (GLdouble)w, 0.0, (GLdouble)h, -1.0, 1.0);//Defines co-ordinate system
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    if(isOn == TRUE)
        BlurscopeExt::start();
}

void BlurscopeExt::setOn(bool state){
    if(state == FALSE){
        if(isOn == TRUE){
            BlurscopeExt::stop();
            isOn = FALSE;
            updateGL();
        }
    }else{
        if(isOn == FALSE && myOn == TRUE){
            BlurscopeExt::start();
            isOn = TRUE;
        }
    }
}

void BlurscopeExt::scopeEvent(float *d, int s){
    if(isOn == FALSE) return;
    int heightHalf = _height/4;
    const int inc = 5;
    if(colCycle == FALSE) cycle = 0;
//Cycle colours, take care to avoid shades of gray
    switch(cycle){
        case 0:
            break;
        case 1:
            b-=inc;
            if(b<0){
                b = 0;
                ++cycle;
            }
            break;
        case 2:
            g+=inc;
            if(g>255){
                g = 255;
                ++cycle;
            }
            break;
        case 3:
            r-=inc;
            if(r<0){
                r = 0;
                ++cycle;
            }
            break;
        case 4:
            b+=inc;
            if(b>255){
                b = 255;
                ++cycle;
            }
            break;
        case 5:
            g-=inc;
            if(g<0){
                g = 0;
                ++cycle;
            }
            break;
        case 6:
            r+=inc;
            if(r>255){
                r = 255;
                cycle=1;
            }
            break;
    }
//Blur the Scope
{
    #define __DIV__ 4.05
    register GLubyte *tempBuf;
    tempBuf = myBuffer;
    register int sizeX = _width;
    for (register int iX = 1; iX < (sizeX - 1); ++iX){
        for (register int iY = 1, offset = 0; iY < (_height - 1); ++iY){
            offset = (sizeX * iY *3) + (iX *3);
            *(tempBuf + offset + 0) = (GLubyte)(( *(tempBuf + (sizeX * iY * 3) + (iX * 3) + 0)
                                        + *(tempBuf + (sizeX * (iY - 1) * 3) + (iX * 3) + 0)
                                        + *(tempBuf + (sizeX * iY * 3) + ((iX - 1) * 3) + 0)
                                        + *(tempBuf + (sizeX * iY * 3) + ((iX + 1) * 3) + 0))/__DIV__);
            *(tempBuf + offset + 1) = (GLubyte)(( *(tempBuf + (sizeX * iY * 3) + (iX * 3) + 1)
                                        + *(tempBuf + (sizeX * (iY - 1) * 3) + (iX * 3) + 1)
                                        + *(tempBuf + (sizeX * iY * 3) + ((iX - 1) * 3) + 1)
                                        + *(tempBuf + (sizeX * iY * 3) + ((iX + 1) * 3) + 1))/__DIV__);
            *(tempBuf + offset + 2) = (GLubyte)(( *(tempBuf + (sizeX * iY * 3) + (iX * 3) + 2)
                                        + *(tempBuf + (sizeX * (iY - 1) * 3) + (iX * 3) + 2)
                                        + *(tempBuf + (sizeX * iY * 3) + ((iX - 1) * 3) + 2)
                                        + *(tempBuf + (sizeX * iY * 3) + ((iX + 1) * 3) + 2))/__DIV__);
        }
    }
    #undef __DIV__
}
//Render new Scope
{
    register int n = 0;
    static int preAmp = heightHalf;
    const int tempHeight = _height - 2;
    int y = tempHeight/2;
    for(int x = 1; x < (_width-2) && x <= s; ++x){
        n = (int) (*(d + x - 1) * heightHalf);
        n += y;
        if (n > tempHeight)
            n = tempHeight;
        else
            if (n < 0)
                n = 0;
        drawVertLine(x, preAmp, n, (GLubyte)r, (GLubyte)g, (GLubyte)b);
        preAmp = n;
    }
}
    updateGL();
}

void BlurscopeExt::drawVertLine(GLsizei xPos, GLsizei y1Pos, GLsizei y2Pos, GLubyte rCol, GLubyte gCol, GLubyte bCol){
    if(y1Pos < y2Pos)
        for(int y = y1Pos; y <= y2Pos; ++y)
            drawPixel(xPos, y, rCol, gCol, bCol);
    else{
        if(y2Pos < y1Pos)
            for(int y = y2Pos; y <= y1Pos; ++y)
                drawPixel(xPos, y, rCol, gCol, bCol);
        else
            drawPixel(xPos, y1Pos, rCol, gCol, bCol);
    }
}

inline void BlurscopeExt::drawPixel(GLsizei xPos, GLsizei yPos, GLubyte rCol, GLubyte gCol, GLubyte bCol){
    if(yPos < 2 || xPos < 0) return;
    if(yPos > (_height - 2) || xPos > (_width - 2)) return;
    const int offset = (_width * yPos *3) + (xPos *3);
    *(myBuffer + offset + 0) = rCol;
    *(myBuffer + offset + 1) = gCol;
    *(myBuffer + offset + 2) = bCol;
}

#include "blurscopeext.moc"
