/***************************************************************************
 *   Copyright (C) 2004 by Yogesh Marwaha                                  *
 *   yogeshm02@rediffmail.com                                              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Library General Public License as       *
 *   published by the Free Software Foundation; either version 2 of the    *
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Library General Public     *
 *   License along with this program; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "plugin_addikt_impl.h"

#include <noatun/playlist.h>
#include <noatun/stdaction.h>
#include <noatun/player.h>
#include <noatun/controls.h>
#include <noatun/effects.h>

#include <qpushbutton.h>
#include <qdragobject.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <qobjectlist.h>
#include <qobjectdict.h>
#include <kconfig.h>
#include <kpopupmenu.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kurldrag.h>
#include <kwin.h>

#include "ronscreendisplay.h"
#include "noatunanalyser.h"
#include "blurscope.h"
#include "spectrumanalyser.h"
#include "rsystemtray.h"
#include "rscrollinglabel.h"
#include "blurscopeext.h"

#include "addiktconfig.h"

addiKt::addiKt() : KMainWindow(0,"addiKt"), UserInterface(){
    mVis = NULL;
    visActive = TRUE;
    osdDisplay = new ROnScreenDisplay();
    curVis = 1;
    totVis = 4;
    visOn = TRUE;
    setAcceptDrops(TRUE);
    mSysTray = new RSystemTray(this, "r_tray");
    mSysTray->setPixmap(BarIcon("noatun"));
    QToolTip::add(mSysTray, i18n("addiKt"));
    connect(mSysTray, SIGNAL(toggleShow()), this, SLOT(toggleShow()));
    connect(mSysTray, SIGNAL(playPause()), napp->player(), SLOT(playpause()));
    connect(mSysTray, SIGNAL(stop()), napp->player(), SLOT(stop()));
    connect(mSysTray, SIGNAL(playPrev()), napp->player(), SLOT(back()));
    connect(mSysTray, SIGNAL(playNext()), napp->player(), SLOT(forward()));
    connect(mSysTray, SIGNAL(loopType(int)), napp->player(), SLOT(loop(int)));
    connect(mSysTray, SIGNAL(toggleListView()), napp->player(), SLOT(toggleListView()));
    mSysTray->show();
    const int buttonSize = 32;

    scrTitle = new RScrollingLabel(this);

    mBack=new QPushButton(this);
    mBack->setFixedSize(buttonSize,buttonSize);
    mBack->setPixmap(BarIcon("noatunback"));
    connect(mBack, SIGNAL(clicked()), napp->player(), SLOT(back()));
    QToolTip::add(mBack,i18n("Previous"));
    mBack->setFocusPolicy(QWidget::NoFocus);

    mStop=new QPushButton(this);
    mStop->setFixedSize(buttonSize,buttonSize);
    mStop->setPixmap(BarIcon("noatunstop"));
    connect(mStop, SIGNAL(clicked()), napp->player(), SLOT(stop()));
    QToolTip::add(mStop, i18n("Stop"));
    mStop->setFocusPolicy(QWidget::NoFocus);

    mPlay=new QPushButton(this);
    mPlay->setFixedSize(buttonSize*2,buttonSize);
    mPlay->setPixmap(BarIcon("noatunplay"));
    connect(mPlay, SIGNAL(clicked()), napp->player(), SLOT(playpause()));
    QToolTip::add(mPlay, i18n("Play"));
    mPlay->setFocusPolicy(QWidget::NoFocus);

    mForward=new QPushButton(this);
    mForward->setFixedSize(buttonSize,buttonSize);
    mForward->setPixmap(BarIcon("noatunforward"));
    connect(mForward, SIGNAL(clicked()), napp->player(), SLOT(forward()));
    QToolTip::add(mForward, i18n("Next"));
    mForward->setFocusPolicy(QWidget::NoFocus);

    mPlaylist=new QPushButton(this);
    mPlaylist->setToggleButton(TRUE);
    mPlaylist->setFixedSize(buttonSize,buttonSize);
    mPlaylist->setPixmap(BarIcon("noatunplaylist"));
    connect(mPlaylist, SIGNAL(clicked()), napp->player(), SLOT(toggleListView()));
    QToolTip::add(mPlaylist, i18n("Playlist"));
    mPlaylist->setFocusPolicy(QWidget::NoFocus);

    mLoop=new QPushButton(this);
    mLoop->setFixedSize(buttonSize,buttonSize);
    mLoop->setPixmap(BarIcon("noatunloopnone"));
    connect(mLoop, SIGNAL(clicked()), napp->player(), SLOT(loop()));
    QToolTip::add(mLoop, i18n("Change loop style"));
    mLoop->setFocusPolicy(QWidget::NoFocus);

    lblTime = new QLabel(this);
    QFont tempFont(font());
    tempFont.setFamily("Courier");
    lblTime->setFont(tempFont);
    lblTime->setText("--:--/--:--");

    mVolume=new L33tSlider(0,100,10,0, Horizontal, this);
    mVolume->setValue(napp->player()->volume());
    mVolume->setFocusPolicy(QWidget::NoFocus);
    mVolume->setMinimumWidth(48);
    QToolTip::add(mVolume, i18n("Volume"));

    mSeeker=new L33tSlider(0,1000,10,0, Horizontal, this);
    mSeeker->setFocusPolicy(QWidget::NoFocus);
    QToolTip::add(mSeeker, i18n("Seek"));

    visMenu = new KPopupMenu(this, "visMenu");
    visMenu->insertTitle(i18n("Select Visualisation"));
    visMenu->insertItem(i18n("Noatun Analyser"), this, 0, 0, 1);
    visMenu->insertItem(i18n("Blurscope"), this, 0, 0, 2);
    visMenu->insertItem(i18n("Spectrum Analyser"), this, 0, 0, 3);
    visMenu->insertItem(i18n("BlurscopeExt"), this, 0, 0, 4);

    mainLayout = new QVBoxLayout(this);
    mainLayout->addWidget(scrTitle);
    thirdLineLayout = new QHBoxLayout();
    mainLayout->addLayout(thirdLineLayout);
    thirdLineLayout->addWidget(mSeeker, 1);
    thirdLineLayout->addWidget(lblTime, 0, Qt::AlignLeft);
    fourthLineLayout = new QHBoxLayout();
    mainLayout->addLayout(fourthLineLayout);
    fourthLineLayout->addWidget(mPlay);
    fourthLineLayout->addWidget(mStop);
    fourthLineLayout->addSpacing(8);
    fourthLineLayout->addWidget(mBack);
    fourthLineLayout->addWidget(mForward);
    fourthLineLayout->addSpacing(8);
    fourthLineLayout->addWidget(mPlaylist, 0, Qt::AlignLeft);
    fourthLineLayout->addSpacing(8);
    fourthLineLayout->addWidget(mLoop);
    fourthLineLayout->addSpacing(8);
    fourthLineLayout->addWidget(mVolume, 10);
    mMinimumSize = minimumSize();
    curVis = -30;//Done explicitly so that the visualisation is not changed when settings are changed through config window
    loadSettings();//Load config settings
    mVis = NULL;
    switch(curVis){
        case 1:
            mVis = new NoatunAnalyser(this);
            mainLayout->insertWidget(1, mVis, 1);
            break;
        case 2:
            mVis = new Blurscope(this);
            mainLayout->insertWidget(1, mVis, 1);
            break;
        case 3:
            mVis = new SpectrumAnalyser(this);
            mainLayout->insertWidget(1, mVis, 1);
            break;
        case 4:
            mVis = new BlurscopeExt(this);
            mainLayout->insertWidget(1, mVis, 1);
            break;
        default:
            curVis = 2;
            mVis = new Blurscope(this);
            mainLayout->insertWidget(1, mVis, 1);
            break;
    }
    mVideo = new VideoFrame((QWidget*)this);
    mainLayout->insertWidget(1, mVideo, 1);

    connect(napp, SIGNAL(hideYourself()), this, SLOT(_hide()) );
    connect(napp, SIGNAL(showYourself()), this, SLOT(_show()) );

    connect(napp->player(), SIGNAL(playing()), this, SLOT(slotPlaying()));
    connect(napp->player(), SIGNAL(stopped()), this, SLOT(slotStopped()));
    connect(napp->player(), SIGNAL(paused()), this, SLOT(slotPaused()));

    connect(napp->player(), SIGNAL(timeout()), this, SLOT(slotTimeout()));
    connect(napp->player(), SIGNAL(loopTypeChange(int)), this, SLOT(changeLoopType(int)));

    /* This skipToWrapper is needed to pass milliseconds to Player() as everybody
     * below the GUI is based on milliseconds instead of some unprecise thingy
     * like seconds or mille */
    connect(seeker(), SIGNAL(userChanged(int)), this, SLOT(skipToWrapper(int)));
    connect(this, SIGNAL(skipTo(int)), napp->player(), SLOT(skipTo(int)));
    connect(seeker(), SIGNAL(sliderMoved(int)), SLOT(sliderMoved(int)));

    /*For OSD*/
    connect(mVolume, SIGNAL(valueChanged(int)), osdDisplay, SLOT(setVolume(int)));

    connect(mVolume, SIGNAL(sliderMoved(int)), napp->player(), SLOT(setVolume(int)));
    connect(mVolume, SIGNAL(userChanged(int)), napp->player(), SLOT(setVolume(int)));

    connect(napp->player(), SIGNAL(playlistShown()), SLOT(playlistShown()));
    connect(napp->player(), SIGNAL(playlistHidden()), SLOT(playlistHidden()));

    // Event Filter for the RMB
    for (QPtrListIterator<QObject> i(*children()); i.current(); ++i)
        (*i)->installEventFilter(this);

    setCaption(i18n("addiKt"));
    setIcon(BarIcon("noatun"));
    if(!(napp->player()->isPlaying() || napp->player()->isPaused()))
        mSeeker->setEnabled(FALSE);

    addiKtConfig *_config = new addiKtConfig(this);

    connect(_config, SIGNAL(saved()), this, SLOT(loadSettings()));//emitted when configuration settings are changed

    QToolTip::add(mLoop, i18n("Change loop style"));
    KConfig &config = *KGlobal::config();
    config.setGroup("addiKtUI");
    napp->player()->handleButtons();
    move(config.readNumEntry("WinX", 0), config.readNumEntry("WinY", 0));
    resize(config.readNumEntry("WinWidth", 360), config.readNumEntry("WinHeight", 240));
    connect(mVideo, SIGNAL(adaptSize(int, int)), this, SLOT(adaptSize(int, int )));
    connect(mVideo, SIGNAL(acquired()), this, SLOT(showVideoWindow()));
    connect(mVideo, SIGNAL(lost()), this, SLOT(hideVideoWindow()));
    if(config.readBoolEntry("Show", TRUE))
        showNormal();
    else
        mVis->setOn(FALSE);
    mVideo->give();
    mVideo->hide();
}

addiKt::~addiKt(){
    KConfig &config = *KGlobal::config();
    config.setGroup("addiKtUI");
    config.writeEntry("Visualisation_Prev", curVis);
    config.writeEntry("WinX", x());
    config.writeEntry("WinY", y());
    config.writeEntry("WinWidth", width());
    config.writeEntry("WinHeight", height());
    bool sShow;
    if(isMinimized())
        sShow = FALSE;
    else
        if(isVisible())
            sShow = TRUE;
        else{
            if(exitOnClose)
                sShow = TRUE;
            else
            sShow = FALSE;
        }
    config.writeEntry("Show", sShow);
    config.sync();
    delete mSysTray;
    delete visMenu;
    delete mVis;
}

void addiKt::showVideoWindow(){
    mVideo->show();
    mVis->hide();
    mVis->setOn(FALSE);
}

void addiKt::hideVideoWindow(){
    mVideo->hide();
    mVis->show();
    mVis->setOn(TRUE);
}

void addiKt::adaptSize(int w, int h){
    if( w <=0 || h <= 0){
        mVideo->hide();
        mVis->show();
        mVis->setOn(TRUE);
    }else{
        resize(mMinimumSize + QSize(w, h));
        mVideo->show();
        mVis->hide();
        mVis->setOn(FALSE);
    }
}

void addiKt::loadSettings(){
    KConfig &config = *KGlobal::config();
    config.setGroup("addiKtUI");
    int temp = config.readNumEntry("Visualisation", 1);
    switch( temp ){
        case -1:
            temp = rand()%2;
            temp += 2;
            break;
        case 0:
            temp = config.readNumEntry("Visualisation_Prev", 1);
        case 1:
        case 2:
        case 3:
        case 4:
            break;
        default:
            temp = 1;
    }
    exitOnClose = config.readBoolEntry("ExitOnClose", TRUE);
    if(curVis == -30){
        curVis = temp;
    }
    osdDisplay->reload();
    if(mVis)
        mVis->reloadSettings();
}

void addiKt::closeEvent(QCloseEvent* e){
    if(!exitOnClose){
        e->ignore();
        if(!isMinimized())
            showMinimized();
        return;
    }
    napp->quit();
}

void addiKt::hideEvent(QHideEvent *e){
    if(mVis)
        mVis->setOn(FALSE);
    KWin::setState( winId(), NET::SkipTaskbar );
    KMainWindow::hideEvent(e);
}

void addiKt::showEvent(QShowEvent *e){
    if(mVis)
        mVis->setOn(TRUE);
    KWin::clearState( winId(), NET::SkipTaskbar );
    KMainWindow::showEvent(e);
}

void addiKt::resizeEvent(QResizeEvent *){
    if(mVideo->isVisible())
        mSizeString = QString("Video Size: %1 x %2").arg(mVideo->width()).arg(mVideo->height());
    else
        mSizeString = QString("Visualisation Size: %1 x %2").arg(mVis->width()).arg(mVis->height());
    QTimer::singleShot(100, this, SLOT(sizeChanged()));
}

void addiKt::sizeChanged(){
    scrTitle->setTextFlash(mSizeString, 2000);
}

void addiKt::dragEnterEvent(QDragEnterEvent *event){
    event->accept(KURLDrag::canDecode(event));
}

void addiKt::dropEvent(QDropEvent *event){
    KURL::List uri;
    if (KURLDrag::decode(event, uri))
        for (KURL::List::Iterator i = uri.begin(); i != uri.end(); ++i)
            napp->player()->openFile(*i, FALSE);
}

void addiKt::mouseReleaseEvent(QMouseEvent *e){
    QWidget::mouseReleaseEvent(e);
    if (e->button() == RightButton){
        NoatunStdAction::ContextMenu::showContextMenu();
    }
}

inline void addiKt::changeStatusbar(const QString& text, const QString &text2){
    if (!text2.isNull())
        lblTime->setText(text2);
    scrTitle->setText(!text.isNull() ? text:napp->player()->current().title());
}

void addiKt::changeCaption(const QString& text){
    setCaption(text);
}

void addiKt::popup(){
    NoatunStdAction::ContextMenu::showContextMenu();
}

void addiKt::slotPlaying(){
    mPlay->setPixmap(BarIcon("noatunpause"));
    mSeeker->setEnabled(TRUE);
    QToolTip::add(mSysTray, QString("Playing: %1").arg(napp->player()->current().title()));
    mSysTray->setState(RSystemTray::Playing);
    osdDisplay->setSong(napp->player()->current().title());
    osdDisplay->setStatus(ROnScreenDisplay::Playing);
    QToolTip::add(mPlay, i18n("Pause"));
}

void addiKt::slotStopped(){
    if (!napp->player()->current()) return;
    seeker()->setValue(0);
    mSeeker->setEnabled(FALSE);
    mPlay->setPixmap(BarIcon("noatunplay"));
    mSysTray->setState(RSystemTray::Stopped);
    QToolTip::add(mSysTray, QString("Stopped: %1").arg(napp->player()->current().title()));
    osdDisplay->setStatus(ROnScreenDisplay::Stopped);
    lblTime->setText("--:--/--:--");
    QToolTip::add(mPlay, i18n("Play"));
}

void addiKt::slotPaused(){
    mPlay->setPixmap(BarIcon("noatunplay"));
    mSysTray->setState(RSystemTray::Paused);
    QToolTip::add(mSysTray, QString("Paused: %1").arg(napp->player()->current().title()));
    osdDisplay->setStatus(ROnScreenDisplay::Paused);
    QToolTip::add(mPlay, i18n("Play"));
}

void addiKt::slotTimeout(){
    mVolume->setValue(napp->player()->volume());
    if (!napp->player()->current()) return;
    if (static_cast<L33tSlider*>(seeker())->currentlyPressed()) return;
    if (seeker()){
        seeker()->setRange ( 0, (int)napp->player()->getLength()/1000 );
        seeker()->setValue ( (int)napp->player()->getTime()/1000 );
    }
    changeStatusbar(0, napp->player()->lengthString());
}

void addiKt::sliderMoved(int seconds){
    if (napp->player()->current())
        changeStatusbar(0, napp->player()->lengthString(seconds*1000));
}

void addiKt::skipToWrapper(int second){
    emit skipTo((long)(second*1000));
}

void addiKt::changeLoopType(int t){
    const int time = 2000;
    switch (t){
        case(Player::None):
            scrTitle->setTextFlash(i18n("No looping"), time);
            mLoop->setPixmap(BarIcon("noatunloopnone"));
            mSysTray->setLoopType(Player::None);
            break;
        case(Player::Song):
            scrTitle->setTextFlash(i18n("Song looping"), time);
            mLoop->setPixmap(BarIcon("noatunloopsong"));
            mSysTray->setLoopType(Player::Song);
            break;
        case(Player::Playlist):
            scrTitle->setTextFlash(i18n("Playlist looping"), time);
            mLoop->setPixmap(BarIcon("noatunloopplaylist"));
            mSysTray->setLoopType(Player::Playlist);
            break;
        case(Player::Random):
            scrTitle->setTextFlash(i18n("Random Play"), time);
            mLoop->setPixmap(BarIcon("noatunlooprandom"));
            mSysTray->setLoopType(Player::Random);
    }
}

bool addiKt::eventFilter(QObject *o, QEvent *e){
    if ((e->type() == QEvent::MouseButtonRelease) && ((static_cast<QMouseEvent*>(e))->button() == RightButton)){
        if( (RVisualisation*)o == mVis){
            switchVisualisation(visMenu->exec(static_cast<QMouseEvent*>(e)->globalPos()));
            return TRUE;
        }
        if( (VideoFrame*)o == mVideo){
            mVideo->popupMenu()->popup(static_cast<QMouseEvent*>(e)->globalPos());
            return TRUE;
        }
        mouseReleaseEvent(static_cast<QMouseEvent*>(e));
        return TRUE;
    }
    if (e->type() == QEvent::Wheel){
        wheelEvent(static_cast<QWheelEvent*>(e));
        return TRUE;
    }
    return QWidget::eventFilter(o, e);
}

void addiKt::playlistShown(){
    mPlaylist->setOn(TRUE);
    mSysTray->playlistVisible = TRUE;
}

void addiKt::playlistHidden(){
    mPlaylist->setOn(FALSE);
    mSysTray->playlistVisible = FALSE;
}

void addiKt::wheelEvent(QWheelEvent *e){
    int delta=e->delta();
    mVolume->setValue(mVolume->value()+(delta/120));
    napp->player()->setVolume(mVolume->value()+(delta/120));
    scrTitle->setTextFlash(QString("Volume: %1%").arg(napp->player()->volume()), 2000);
}

void addiKt::switchVisualisation(int vis_no){
    if(vis_no == -1) return;
    mainLayout->remove(mVis);
    delete mVis;
    mVis = NULL;
    curVis = vis_no;
    switch(curVis){
        case 1:
            mVis = new NoatunAnalyser(this);
            mainLayout->insertWidget(1, mVis, 1);
            break;
        case 2:
            mVis = new Blurscope(this);
            mainLayout->insertWidget(1, mVis, 1);
            break;
        case 3:
            mVis = new SpectrumAnalyser(this);
            mainLayout->insertWidget(1, mVis, 1);
            break;
        case 4:
            mVis = new BlurscopeExt(this);
            mainLayout->insertWidget(1, mVis, 1);
            break;
    }
}

void addiKt::toggleShow(){
    if(isMinimized()){
        showNormal();
        KWin::activateWindow(winId());
    }else{
        if(isVisible())
            showMinimized();
        else
            show();
    }
}

void addiKt::_hide(){
    if(!isMinimized()){
        if(isVisible())
            showMinimized();
    }
}

void addiKt::_show(){
    if(isMinimized()){
        showNormal();
        KWin::activateWindow(winId());
    }else{
        if(!isVisible())
            show();
    }
}

#include "plugin_addikt_impl.moc"
