/***************************************************************************
 *   Copyright (C) 2004 by Yogesh Marwaha                                  *
 *   yogeshm02@rediffmail.com                                              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Library General Public License as       *
 *   published by the Free Software Foundation; either version 2 of the    *
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Library General Public     *
 *   License along with this program; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "spectrumanalyser.h"
#include <math.h>
#include <kconfig.h>

static inline int max(int a, int b){
    return (b > a)?b:a;
}

SpectrumAnalyser::SpectrumAnalyser(QWidget *parent, const char *name) : RVisualisation(parent, name), StereoFFTScope(50){
    installEventFilter(parent);
    mBarWidth = 21;
    mBarHeight = 5;
    mHSpace = 1;
    mVSpace = 1;
    m_pData = m_pPeak = m_pPeakHold = 0L;
    reloadSettings();
    isOn = myOn = TRUE;
    setMinimumSize(60, 20);
    show();
}

SpectrumAnalyser::~SpectrumAnalyser(){
    SpectrumAnalyser::stop();
}

void SpectrumAnalyser::initializeGL(){
    glClearColor(0.0, 0.0, 0.0, 0.0);
    glShadeModel(GL_FLAT);
}

void SpectrumAnalyser::paintGL(){
    if(!isOn){
        glClear(GL_COLOR_BUFFER_BIT);
        glFlush();
        return;
    }
    glClear(GL_COLOR_BUFFER_BIT);
    for(int i = 0; i < mBands; ++i){
        int j = 0;
        glColor3f(colR1, colG1, colB1);
        for(; j < m_pData[i] && j<(mBars/3); ++j)
            glRectf(mHSpace + mPadding + (i * mBarWidth), j * mBarHeight, (mPadding + (i * mBarWidth) + mBarWidth), (j * mBarHeight) + mBarHeight - mVSpace);
        glColor3f(colR2, colG2, colB2);
        for(; j < m_pData[i] && j<(mBars*2/3); ++j)
            glRectf(mHSpace + mPadding + (i * mBarWidth), j * mBarHeight, (mPadding + (i * mBarWidth) + mBarWidth), (j * mBarHeight) + mBarHeight - mVSpace);
        glColor3f(colR3, colG3, colB3);
        for(; j < m_pData[i]; ++j)
            glRectf(mHSpace + mPadding + (i * mBarWidth), j * mBarHeight, (mPadding + (i * mBarWidth) + mBarWidth), (j * mBarHeight) + mBarHeight - mVSpace);
        glColor3f(colR4, colG4, colB4);
        if(m_pPeak[i]!=0 && showPeaks)
            glRectf(mHSpace + mPadding + (i * mBarWidth), (m_pPeak[i] - 1) * mBarHeight, (mPadding + (i * mBarWidth) + mBarWidth ), ((m_pPeak[i] - 1) * mBarHeight) + mBarHeight - mVSpace);
    }
    for(int i=0; i<mBands; ++i){
        int j=0;
        glColor3f(colR1, colG1, colB1);
        for(; j < m_pData[i+mBands] && j<(mBars/3); ++j)
            glRectf(mHSpace + mPadding + ((i + mBands) * mBarWidth), j * mBarHeight, (mPadding + ((i + mBands) * mBarWidth) + mBarWidth), (j * mBarHeight) + mBarHeight - mVSpace);
        glColor3f(colR2, colG2, colB2);
        for(; j < m_pData[i+mBands] && j<(mBars*2/3); ++j)
            glRectf(mHSpace + mPadding + ((i + mBands) * mBarWidth), j * mBarHeight, (mPadding + ((i + mBands) * mBarWidth) + mBarWidth), (j * mBarHeight) + mBarHeight - mVSpace);
        glColor3f(colR3, colG3, colB3);
        for(; j < m_pData[i+mBands]; ++j)
            glRectf(mHSpace + mPadding + ((i + mBands) * mBarWidth), j * mBarHeight, (mPadding + ((i + mBands) * mBarWidth) + mBarWidth), (j * mBarHeight) + mBarHeight - mVSpace);
        glColor3f(colR4, colG4, colB4);
        if(m_pPeak[i + mBands] != 0 && showPeaks)
            glRectf(mHSpace + mPadding + ((i + mBands) * mBarWidth), (m_pPeak[i + mBands] - 1) * mBarHeight, (mPadding + ((i + mBands) * mBarWidth) + mBarWidth), ((m_pPeak[i + mBands] - 1) * mBarHeight) + mBarHeight - mVSpace);
    }
    glFlush();
}

void SpectrumAnalyser::reloadSettings(void){
    //Read configuration settings
    KConfig &config = *KGlobal::config();
    config.setGroup("addiKtUI");
    showPeaks = config.readBoolEntry("SA_Peaks", TRUE);
    colR1 = (config.readNumEntry("SA_R1", 0)/255.0);
    colG1 = (config.readNumEntry("SA_G1", 255)/255.0);
    colB1 = (config.readNumEntry("SA_B1", 0)/255.0);
    colR2 = (config.readNumEntry("SA_R2", 255)/255.0);
    colG2 = (config.readNumEntry("SA_G2", 255)/255.0);
    colB2 = (config.readNumEntry("SA_B2", 0)/255.0);
    colR3 = (config.readNumEntry("SA_R3", 255)/255.0);
    colG3 = (config.readNumEntry("SA_G3", 0)/255.0);
    colB3 = (config.readNumEntry("SA_B3", 0)/255.0);
    colR4 = (config.readNumEntry("SA_R4", 255)/255.0);
    colG4 = (config.readNumEntry("SA_G4", 0)/255.0);
    colB4 = (config.readNumEntry("SA_B4", 0)/255.0);

    mBarSizeIsStatic = config.readBoolEntry("SA_BAR_SIZE_STATIC", TRUE);
    mUserBarWidth = config.readNumEntry("SA_BAR_WIDTH", 20);
    mUserBarHeight = config.readNumEntry("SA_BAR_HEIGHT", 2);
    mUserBarAlongX = config.readNumEntry("SA_BAR_ALONG_X", 10);
    mUserBarAlongY = config.readNumEntry("SA_BAR_ALONG_Y", 15);
    calculate();
}

void SpectrumAnalyser::resizeGL( int w, int h ){
    if(isOn)
        SpectrumAnalyser::stop();
    calculate();
    glViewport(0, 0, w, h);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0.0, (GLdouble)w, 0.0, (GLdouble)h, -1.0, 1.0);//Defines co-ordinate system
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    if(isOn)
        SpectrumAnalyser::start();
}

void SpectrumAnalyser::mouseReleaseEvent(QMouseEvent *e){
    QWidget::mouseReleaseEvent(e);
    if (e->button()==LeftButton){//Toggle visualisation state
        isOn = !isOn;
        myOn = !myOn;
        setOn(isOn);
    }
}

void SpectrumAnalyser::setOn(bool state){
    if(!state){
        if(isOn){
            SpectrumAnalyser::stop();
            isOn = FALSE;
            updateGL();
        }
    }else{
        if(!isOn && myOn){
            SpectrumAnalyser::start();
            isOn = TRUE;
        }
    }
}

void SpectrumAnalyser::scopeEvent(float *left, float *right, int a ){
    if(!isOn) return;
    //generate data
    for(int i = 0; i < mBands; ++i){
        float n;
        //left channel
        if(i >= a)
            n = 0;
        else
            n= *(left + i);
        n += 1.0;
        try{
            n = log10(n)/log(5)*(mBars)*4;
        }
        catch(...){//Is this the right way to do this?
            n = 0;
        }
        int amp = (int)n;
        int prevPeak;
        if(amp > mBars) amp = mBars;
        m_pData[i] = amp;
        prevPeak = m_pPeak[i];
        m_pPeak[i] = max (amp, prevPeak);
        if(prevPeak < m_pPeak[i])
            m_pPeakHold[i] = PEAK_DELAY_IN_FRAMES;
        else{
            if(--m_pPeakHold[i] < 1){
                --m_pPeak[i];
                m_pPeakHold[i] = 1;
            }
        }
        if(m_pPeak[i] < m_pData[i])
            ++m_pPeak[i];
        //right channel
        if(i >= a)
            n = 0;
        else
            n = *(right + i);
        n += 1.0;
        try{
            n = log10(n)/log(5)*(mBars)*4;
        }
        catch(...){
            n = 0;
        }
        amp = (int)n;
        if(amp > mBars) amp = mBars;
        m_pData[i + mBands] = amp;
        prevPeak = m_pPeak[i + mBands];
        m_pPeak[i + mBands] = max (amp, prevPeak);
        if(prevPeak < m_pPeak[i + mBands])
            m_pPeakHold[i + mBands] = PEAK_DELAY_IN_FRAMES;
        else{
            if(--m_pPeakHold[i + mBands] < 1){
                --m_pPeak[i + mBands];
                m_pPeakHold[i + mBands] = 1;
            }
        }
        if(m_pPeak[i + mBands] < m_pData[i + mBands])
            ++m_pPeak[i + mBands];
    }
    updateGL();
}

void SpectrumAnalyser::calculate(){
    if(mBarSizeIsStatic){
        mBarWidth = mUserBarWidth + mHSpace;
        mBarHeight = mUserBarHeight + mVSpace;
        if(mBarWidth < (2 + mHSpace))
            mBarWidth = 2 + mHSpace;
        else
        if(mBarWidth > (width()/2))
            mBarWidth = width()/2;
        if(mBarHeight < (2 + mVSpace))
            mBarHeight = 2 + mVSpace;
        else
        if(mBarHeight > height())
            mBarHeight = height();
        mBands = (int)(((width() - (mHSpace * 2))/mBarWidth) / 2);
        mBars = height() / mBarHeight;
        mPadding = (width() - (mBands * 2 * mBarWidth) - (mHSpace * 2)) / 2;
    }else{
        const int totalHSpace = mUserBarAlongX * mHSpace;
        const int totalVSpace = mUserBarAlongY * mVSpace;
        const int availWidth = width() - totalHSpace;
        const int availHeight = height() - totalVSpace;
        mBarWidth = availWidth/mUserBarAlongX;
        mBarHeight = availHeight/mUserBarAlongY;
        if(mBarWidth < (2 + mHSpace))
            mBarWidth = 2 + mHSpace;
        else
        if(mBarWidth > (width()/2))
            mBarWidth = width()/2;
        if(mBarHeight < (2 + mVSpace))
            mBarHeight = 2 + mVSpace;
        else
        if(mBarHeight > height())
            mBarHeight = height();
        mBands = (int)(((width() - (mHSpace * 2))/mBarWidth) / 2);
        mBars = height() / mBarHeight;
        mPadding = (width() - (mBands * 2 * mBarWidth) - (mHSpace * 2)) / 2;
    }
    if(m_pData)
        delete m_pData;
    if(m_pPeak)
        delete m_pPeak;
    if(m_pPeakHold)
        delete m_pPeakHold;
    m_pData = new int[mBands*2];
    m_pPeak = new int[mBands*2];
    m_pPeakHold = new int[mBands*2];
    ASSERT(m_pData);
    ASSERT(m_pPeak);
    ASSERT(m_pPeakHold);
    for(int i=0; i<(mBands * 2); ++i){
        m_pData[i] = 0;
        m_pPeak[i] = 0;
        m_pPeakHold[i] = PEAK_DELAY_IN_FRAMES;
    }
    setBands(magic(mBands));
}

#include "spectrumanalyser.moc"
