/*
 * CoverFinder - Find and download cover images from Amazon 
 * Copyright (C) 2007 - Sven Salzwedel
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include <gtk/gtk.h>
#include <glib/gstdio.h>

#include "config.h"
#include "ui.h"
#include "coverfinder.h"
#include "control.h"
#include "search.h"
#include "amazon.h"
#include "net.h"

static void slist_free_result (gpointer element, gpointer data)
{
	search_result_free (element);
}

static void get_result (gpointer element, gpointer data)
{
	CFResultData *result_data = (CFResultData *) data;
	SResult *result = (SResult *) element;
	GtkListStore *model = result_data->model;
	CFApp *app = result_data->search_data->app;
	CFGui *gui = app->gui;
	GError *err = NULL;
	GdkPixbuf *pixbuf;
	GtkTreeIter iter;
	gchar *img_filename, *img_basename;

	result_data->results_passed++;
	g_return_if_fail (result != NULL);
	g_return_if_fail (result->title != NULL);
	g_return_if_fail (result->image_url != NULL);

	img_basename = g_path_get_basename (result->image_url);
	img_filename = g_build_filename (app->config->cache_dir, img_basename, NULL);
	g_free (img_basename);

	if (!g_file_test (img_filename, G_FILE_TEST_EXISTS)
		|| g_file_test (img_filename, G_FILE_TEST_IS_DIR)) {
		if (!net_getfile (result->image_url, img_filename, &err)) {
			gdk_threads_enter ();
			ui_error_dialog (gui, err);
			g_error_free (err);
			g_free (img_filename);
			gdk_threads_leave ();
			return;
		}
	}

	gdk_threads_enter ();
	if ((pixbuf = gdk_pixbuf_new_from_file_at_size (img_filename, 100, 100, &err)) == NULL) {
		ui_error_dialog (gui, err);
		g_error_free (err);
		g_free (img_filename);
		gdk_threads_leave ();
		return;
	}

	ui_progress_set (gui, 1.0 / result_data->num_results * result_data->results_passed);
	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, RESULT_COLUMN_THUMBNAIL, pixbuf, RESULT_COLUMN_DESCRIPTION,
						result->markup, RESULT_COLUMN_DATA, result, -1);
	g_object_unref (G_OBJECT (pixbuf));
	result->image_file = img_filename;
	gdk_threads_leave ();
}

gpointer cf_search (gpointer data)
{
	CFSearchData *search_data = (CFSearchData *) data;
	CFResultData *result_data;
	GSList *result;
	CFApp *app = search_data->app;
	CFGui *gui = app->gui;
	GtkListStore *model;
	GtkTreeIter iter;
	void *provider_data;
	gint type;
	GError *err = NULL;

	gdk_threads_enter ();
	ui_progress_reset (gui);
	ui_pop_status (gui, "noresult");
	ui_push_status (gui, "search", "Searching...");
	ui_set_busy_begin (gui);

	gtk_combo_box_get_active_iter (GTK_COMBO_BOX (gui->providers_combo), &iter);
	model = (GtkListStore *) gtk_combo_box_get_model (GTK_COMBO_BOX (gui->providers_combo));
	gtk_tree_model_get (GTK_TREE_MODEL (model), &iter, PROVIDER_COLUMN_TYPE, &type,
						PROVIDER_COLUMN_DATA, &provider_data, -1);
	gdk_threads_leave ();

	switch (type) {
	case SPROVIDER_AMAZON:
		result = amazon_search (provider_data, search_data->token, &err);
		break;
	default:
		// ERROR
		result = NULL;
		break;
	}

	gdk_threads_enter ();
	ui_pop_status (gui, "search");

	model = (GtkListStore *) gtk_tree_view_get_model (GTK_TREE_VIEW (gui->results_view));
	gtk_list_store_clear (model);
	ui_set_save_enabled (gui, FALSE);

	if (result == NULL) {
		ui_set_busy_end (gui);
		if (err != NULL) {
			ui_error_dialog (gui, err);
			g_error_free (err);
		} else {
			/* TODO: put a nice message into treeview */
			ui_push_status (gui, "noresult", "Nothing found!");
		}
		gdk_threads_leave ();
		return NULL;
	}

	if (gui->results_list != NULL) {
		g_slist_foreach (gui->results_list, slist_free_result, NULL);
		g_slist_free (gui->results_list);
	}

	result_data = g_new (CFResultData, 1);
	result_data->model = model;
	result_data->search_data = search_data;
	result_data->num_results = g_slist_length (result);
	result_data->results_passed = 0;
	ui_push_status (gui, "search", "Downloading images...");
	gdk_threads_leave ();

	g_slist_foreach (result, get_result, result_data);
	gui->results_list = result;

	gdk_threads_enter ();
	ui_pop_status (gui, "search");
	ui_progress_reset (gui);
	ui_set_busy_end (gui);
	gdk_threads_leave ();
	g_free (result_data);
	g_free (search_data);

	return NULL;
}

gpointer cf_clear_cache (gpointer data)
{
	CFApp *app = (CFApp *) data;
	GDir *dir;
	GError *err = NULL;
	const gchar *bname;
	gchar *fname;

	gdk_threads_enter ();
	ui_push_status (app->gui, "cache", "Clearing cache...");
	ui_set_busy_begin (app->gui);
	gdk_threads_leave ();


	if ((dir = g_dir_open (app->config->cache_dir, 0, &err)) == NULL) {
		ui_error_dialog (app->gui, err);
		g_error_free (err);
		return NULL;
	}

	while ((bname = g_dir_read_name (dir)) != NULL) {
		fname = g_build_filename (app->config->cache_dir, bname, NULL);
		if (g_file_test (fname, G_FILE_TEST_IS_DIR)) {
			continue;
		}
		g_unlink (fname);
		g_free (fname);
	}
	g_dir_close (dir);

	gdk_threads_enter ();
	ui_pop_status (app->gui, "cache");
	ui_set_busy_end (app->gui);
	gdk_threads_leave ();

	return NULL;
}

void cf_quit (CFApp *app)
{
	gtk_main_quit ();
}

void cf_show_about_dialog (CFApp *app)
{
	gchar *authors[] = { "Sven Salzwedel <sven_salzwedel@web.de>", NULL };
	gchar *artists[] = { "Achim Frase <achim.frase@gmx.de>", NULL };

	gtk_show_about_dialog (GTK_WINDOW (app->gui->window), "authors", authors, "name", "CoverFinder",
						   "website", "http://mookooh.org/coverfinder/", "comments",
						   "Find and download cover images from Amazon", "version", VERSION,
						   "logo-icon-name", "coverfinder", "artists", artists, NULL);
}

/* TODO: add possibility to save multiple files, now every item would get saved to the same filename, d'oh */
static void save_item (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	CFSaveData *save_data = (CFSaveData *) data;
	SResult *result;
	gchar *file, *fname;
	gsize size;

	if (save_data->err != NULL)
		return;

	gtk_tree_model_get (model, iter, RESULT_COLUMN_DATA, &result, -1);
	fname = save_data->filename;
	if (g_file_get_contents (result->image_file, &file, &size, &(save_data->err))) {
		g_file_set_contents (fname, file, size, &(save_data->err));
	}
}

void cf_save_selected (CFApp *app)
{
	CFSaveData *data;
	CFGui *gui = app->gui;
	GtkWidget *chooser;
	GtkTreeSelection *selection;

	chooser = gtk_file_chooser_dialog_new ("Save Cover(s)", GTK_WINDOW (gui->window),
	                                       GTK_FILE_CHOOSER_ACTION_SAVE, GTK_STOCK_CANCEL,
										   GTK_RESPONSE_CANCEL, GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
										   NULL);
	gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER (chooser), TRUE);
	gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (chooser), "cover.jpg");

	if (gtk_dialog_run (GTK_DIALOG (chooser)) == GTK_RESPONSE_ACCEPT) {
		data = g_new (CFSaveData, 1);
		data->items_saved = 0;
		data->filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (chooser));
		data->err = NULL;

		selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (gui->results_view));
		gtk_tree_selection_selected_foreach (selection, save_item, data);
		if (data->err != NULL) {
			ui_error_dialog (app->gui, data->err);
			g_error_free (data->err);
		}
	}

	gtk_widget_destroy (chooser);
}
