/*
 *   PCM time-domain equalizer
 *
 *   Copyright (C) 2002-2006  Felipe Rivera <liebremx at users.sourceforge.net>
 *
 *   19.08.2002 Initial release for XMMS
 *
 *   
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   $Id: configuration.c,v 1.15 2006/01/15 00:13:51 liebremx Exp $
 */

#include "eq.h"

/* Quick hack for xmms-crossfade  */
extern EffectPlugin eq_ep;

/* Global configuration structure */
EQConfig eqcfg;

gboolean eq_has_been_initialized = FALSE;

static GtkWidget *config_window;
static GtkWidget *c_bands;

typedef struct 
{
    gchar *bands_text;
    gint  bands;
    gboolean  original_freq;
}EQBandsChoices;

static const EQBandsChoices bc[] =
{
    { "10 bands, original XMMS frequencies", 10, TRUE },
    { "10 bands, ISO spaced frequencies", 10, FALSE },
    { "15 bands, ISO spaced frequencies", 15, FALSE },
    { "25 bands, ISO spaced frequencies", 25, FALSE },
    { "31 bands, ISO spaced frequencies", 31, FALSE },
    { 0 }
};

static gboolean ouse_independent_channels, 
    ouse_xmms_original_freqs, 
    ogui_visible, 
    oauto_volume_down,
    olock_sliders,
    oextra_filtering,
    conf_dlg_showing = FALSE;

static gint oband_num, oauto_volume_down_ms;

static void apply_changes(void);

static gboolean
on_window_delete_event (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
    gtk_widget_destroy(GTK_WIDGET(config_window));
    conf_dlg_showing = FALSE;
    return FALSE;
}

static void
on_use_independent_toggled (GtkToggleButton *togglebutton, gpointer user_data)
{
    ouse_independent_channels = togglebutton->active;
    gtk_widget_set_sensitive(GTK_WIDGET(GTK_TOGGLE_BUTTON(user_data)), togglebutton->active);
}

static void
on_lock_sliders_toggled (GtkToggleButton *togglebutton, gpointer user_data)
{
    olock_sliders = togglebutton->active;
}

static void
on_extra_filtering_toggled (GtkToggleButton *togglebutton, gpointer user_data)
{
    oextra_filtering = togglebutton->active;
}

static void
on_gui_visible (GtkToggleButton *togglebutton, gpointer user_data)
{
    ogui_visible = togglebutton->active;
}

static void
on_ava (GtkToggleButton *togglebutton, gpointer user_data)
{
    oauto_volume_down = togglebutton->active;
    gtk_widget_set_sensitive(GTK_WIDGET(GTK_SCALE(user_data)), togglebutton->active);
}

static void on_ava_ms (GtkAdjustment *adj)
{
    oauto_volume_down_ms = (gint)adj->value;
}

static void
on_ok_clicked (GtkButton *button, gpointer user_data)
{
    apply_changes();
    gtk_widget_destroy(GTK_WIDGET(config_window));
    conf_dlg_showing = FALSE;
}


static void
on_cancel_clicked (GtkButton *button, gpointer user_data)
{
    gtk_widget_destroy(GTK_WIDGET(config_window));
    conf_dlg_showing = FALSE;
}


static void
on_apply_clicked (GtkButton *button, gpointer user_data)
{
    apply_changes();
}

static void apply_changes()
{
    gchar *bands_string;
    const EQBandsChoices *pbc;

    /* Hide the GUI if asked for */
    if (eqcfg.gui_visible != ogui_visible)
    {
        eqcfg.gui_visible = ogui_visible;
        if (eqcfg.gui_visible)
            EQequalizer_real_show();
        else
            EQequalizer_real_hide();
    }

    eqcfg.auto_volume_down = oauto_volume_down;
    if (eqcfg.auto_volume_down)
        eqcfg.auto_volume_down_ms = oauto_volume_down_ms;
    
    /* Find out if the user selected a different band setting */
    bands_string = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(c_bands)->entry));
    pbc = bc;
    while (pbc->bands)
    {
        if (strcmp(bands_string, pbc->bands_text) == 0)
        {
            oband_num = pbc->bands;
            ouse_xmms_original_freqs = pbc->original_freq;
        }
        pbc++;
    }
    if (ouse_independent_channels != eqcfg.use_independent_channels ||
            oband_num != eqcfg.band_num ||
            ouse_xmms_original_freqs != eqcfg.use_xmms_original_freqs)
    {
        eqcfg.band_num = oband_num;
        eqcfg.use_xmms_original_freqs = ouse_xmms_original_freqs;
        eqcfg.use_independent_channels = ouse_independent_channels;
        init_iir();
        EQequalizer_recreate();
        /* Set the frequency names that will be displayed in XMMS's window */
        EQeqslider_set_names();
    }

    eqcfg.extra_filtering = oextra_filtering;
    eqcfg.lock_sliders = olock_sliders;

    /* Write the configuration */
    eq_write_config();
}

void eq_configuration_window (void)
{
    /* GUI stuff */
    GtkWidget *vbox4;
    GtkWidget *frame4;
    GtkWidget *t_gui_visible;

    /* Bands stuff */
    GtkWidget *vbox1;
    GtkWidget *frame2;
    GList *c_bands_items = NULL;
    GtkWidget *combo_entry1;

    /* Channels stuff */
    GtkWidget *frame1;
    GtkWidget *vbox2;
    GtkWidget *t_use_independent;
    GtkWidget *t_lock_sliders;

    /* Automatic volume adjustment stuff */
    GtkWidget *frame_ava;
    GtkWidget *hbox_ava;
    GtkWidget *t_ava_switch;
    GtkWidget *scale_ava;
    GtkAdjustment *adj_ava;

    /* Other stuff */
    GtkWidget *frame3;
    GtkWidget *vbox3;
    GtkWidget *t_extra_filtering;

    /* Button stuff */
    GtkWidget *hbuttonbox1;
    GtkWidget *b_ok;
    GtkWidget *b_cancel;
    GtkWidget *b_apply;

    GtkTooltips *tooltips;
    const EQBandsChoices *pbc;

    /* Quick hack for xmms-crossfade which doesn't initialize the plugins  */
    if (!eq_has_been_initialized)
        eq_ep.init();

    /* If the configuration dialog is loaded already 
     * return
     */
    if (conf_dlg_showing)
        return;

    conf_dlg_showing = TRUE;

    /* Save original values to avoid recreating the GUI
     * when only the change was in lock_sliders */
    ouse_independent_channels = eqcfg.use_independent_channels;
    oband_num = eqcfg.band_num;
    ouse_xmms_original_freqs = eqcfg.use_xmms_original_freqs;
    ogui_visible = eqcfg.gui_visible;
    oauto_volume_down = eqcfg.auto_volume_down;
    olock_sliders = eqcfg.lock_sliders;
    oextra_filtering = eqcfg.extra_filtering;

    tooltips = gtk_tooltips_new ();

    config_window = gtk_window_new (GTK_WINDOW_DIALOG);
    gtk_widget_set_name (config_window, "window");
    gtk_object_set_data (GTK_OBJECT (config_window), "window", config_window);
    gtk_window_set_title (GTK_WINDOW (config_window), "EQ Plugin Preferences");
    gtk_window_set_wmclass (GTK_WINDOW (config_window), "eq_pref", "xmms");

    /* Vertical box */
    vbox1 = gtk_vbox_new (FALSE, 0);
    gtk_widget_set_name (vbox1, "vbox1");
    gtk_widget_ref (vbox1);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "vbox1", vbox1,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (vbox1);
    gtk_container_add (GTK_CONTAINER (config_window), vbox1);

    /* GUI frame */
    frame4 = gtk_frame_new ("GUI");
    gtk_widget_set_name (frame4, "frame4");
    gtk_widget_ref (frame4);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "frame4", frame4,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (frame4);
    gtk_box_pack_start (GTK_BOX (vbox1), frame4, TRUE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (frame4), 10);

    vbox4 = gtk_vbox_new (TRUE, 0);
    gtk_widget_set_name (vbox4, "vbox4");
    gtk_widget_ref (vbox4);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "vbox4", vbox4,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (vbox4);
    gtk_container_add (GTK_CONTAINER (frame4), vbox4);
    gtk_container_set_border_width (GTK_CONTAINER (vbox4), 10);

    t_gui_visible = gtk_check_button_new_with_label ("Show");
    gtk_widget_set_name (t_gui_visible, "t_gui_visible");
    gtk_widget_ref (t_gui_visible);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "t_gui_visible", t_gui_visible,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(t_gui_visible), ogui_visible);
    gtk_widget_show (t_gui_visible);
    gtk_box_pack_start (GTK_BOX (vbox4), t_gui_visible, FALSE, FALSE, 0);
    gtk_tooltips_set_tip (tooltips, t_gui_visible, "Show/Hide the Equalizer GUI", NULL);

    /* Bands stuff */
    frame2 = gtk_frame_new ("Bands");
    gtk_widget_set_name (frame2, "frame2");
    gtk_widget_ref (frame2);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "frame2", frame2,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (frame2);
    gtk_box_pack_start (GTK_BOX (vbox1), frame2, FALSE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (frame2), 10);

    c_bands = gtk_combo_new ();
    gtk_widget_set_name (c_bands, "c_bands");
    gtk_widget_ref (c_bands);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "c_bands", c_bands,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (c_bands);
    gtk_container_add (GTK_CONTAINER (frame2), c_bands);
    gtk_container_set_border_width (GTK_CONTAINER (c_bands), 10);
    pbc = bc;
    while(pbc->bands)
    {
        if (pbc->bands == eqcfg.band_num && pbc->original_freq == eqcfg.use_xmms_original_freqs)
            c_bands_items = g_list_insert (c_bands_items, (gpointer) pbc->bands_text, 0);
        else
            c_bands_items = g_list_append (c_bands_items, (gpointer) pbc->bands_text);
        pbc++;
    }
    gtk_combo_set_popdown_strings (GTK_COMBO (c_bands), c_bands_items);
    g_list_free (c_bands_items);

    combo_entry1 = GTK_COMBO (c_bands)->entry;
    gtk_widget_set_name (combo_entry1, "combo_entry1");
    gtk_widget_ref (combo_entry1);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "combo_entry1", combo_entry1,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (combo_entry1);
    gtk_tooltips_set_tip (tooltips, combo_entry1, "Select the number of bands to use in the equalizer", NULL);
    gtk_entry_set_editable (GTK_ENTRY (combo_entry1), FALSE);

    /* Independent channel stuff */
    frame1 = gtk_frame_new ("Independent Channels");
    gtk_widget_set_name (frame1, "frame1");
    gtk_widget_ref (frame1);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "frame1", frame1,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (frame1);
    gtk_box_pack_start (GTK_BOX (vbox1), frame1, TRUE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (frame1), 10);

    vbox2 = gtk_vbox_new (TRUE, 0);
    gtk_widget_set_name (vbox2, "vbox2");
    gtk_widget_ref (vbox2);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "vbox2", vbox2,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (vbox2);
    gtk_container_add (GTK_CONTAINER (frame1), vbox2);
    gtk_container_set_border_width (GTK_CONTAINER (vbox2), 10);

    t_use_independent = gtk_check_button_new_with_label ("Use independent channels");
    gtk_widget_set_name (t_use_independent, "t_use_independent");
    gtk_widget_ref (t_use_independent);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "t_use_independent", t_use_independent,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(t_use_independent), eqcfg.use_independent_channels);
    gtk_widget_show (t_use_independent);
    gtk_box_pack_start (GTK_BOX (vbox2), t_use_independent, FALSE, FALSE, 0);
    gtk_tooltips_set_tip (tooltips, t_use_independent, "Have independent gain settings for left and right channels", NULL);

    t_lock_sliders = gtk_check_button_new_with_label ("Lock sliders");
    gtk_widget_set_name (t_lock_sliders, "t_lock_sliders");
    gtk_widget_ref (t_lock_sliders);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "t_lock_sliders", t_lock_sliders,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(t_lock_sliders), eqcfg.lock_sliders);
    gtk_widget_set_sensitive(GTK_WIDGET(t_lock_sliders), eqcfg.use_independent_channels);
    gtk_widget_show (t_lock_sliders);
    gtk_box_pack_start (GTK_BOX (vbox2), t_lock_sliders, FALSE, FALSE, 0);
    gtk_tooltips_set_tip (tooltips, t_lock_sliders, "Lock the sliders in the equalizer so when you change a slider in one channel the equivalent slider in the other channel will be changed in the same way", NULL);

    /* Automatic volume adjustment frame */
    frame_ava = gtk_frame_new ("Automatic volume control");
    gtk_widget_set_name (frame_ava, "frame_ava");
    gtk_widget_ref (frame_ava);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "frame_ava", frame_ava,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_box_pack_start (GTK_BOX (vbox1), frame_ava, FALSE, FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (frame_ava), 10);
    gtk_widget_show (frame_ava);

    hbox_ava = gtk_hbox_new (FALSE, 0);
    gtk_widget_set_name (hbox_ava, "hbox_ava");
    gtk_widget_ref (hbox_ava);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "hbox_ava", hbox_ava,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_container_add (GTK_CONTAINER (frame_ava), hbox_ava);
    gtk_container_set_border_width (GTK_CONTAINER (hbox_ava), 10);
    gtk_widget_show (hbox_ava);

    t_ava_switch = gtk_check_button_new_with_label ("Use (units in ms)");
    gtk_widget_set_name (t_ava_switch, "t_ava_switch");
    gtk_widget_ref (t_ava_switch);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "t_ava_switch", t_ava_switch,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(t_ava_switch), eqcfg.auto_volume_down);
    gtk_box_pack_start (GTK_BOX (hbox_ava), t_ava_switch, FALSE, FALSE, 0);
    gtk_tooltips_set_tip (tooltips, t_ava_switch, "Turn down the volume automatically when the EQ is deactivated to protect your speakers and your ears. When the EQ is reactivated, the volume will be back to its original position gradually using the value on the right", NULL);
    gtk_widget_show (t_ava_switch);

    /* value, lower, upper, step_increment, page_increment, page_size */
    /* Note that the page_size value only makes a difference for
     * scrollbar widgets, and the highest value you'll get is actually
     * (upper - page_size). */
    adj_ava = (GtkAdjustment *)gtk_adjustment_new (eqcfg.auto_volume_down_ms, 10.0, 1001.0, 1.0, 10.0, 1.0);
    gtk_signal_connect (GTK_OBJECT (adj_ava), "value_changed",
            GTK_SIGNAL_FUNC (on_ava_ms), NULL);
    gtk_object_ref (GTK_OBJECT(adj_ava));
    gtk_object_set_data_full (GTK_OBJECT (config_window), "adj_ava", adj_ava,
            (GtkDestroyNotify) gtk_object_unref);

    scale_ava = gtk_hscale_new (adj_ava);
    gtk_widget_set_name (scale_ava, "scale_ava");
    gtk_widget_ref (scale_ava);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "scale_ava", scale_ava,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_set_usize (GTK_WIDGET (scale_ava), 200, 30);
    gtk_range_set_update_policy(GTK_RANGE(scale_ava),
            GTK_UPDATE_DISCONTINUOUS);
    gtk_scale_set_digits(GTK_SCALE(scale_ava), 0);
    gtk_scale_set_draw_value(GTK_SCALE(scale_ava), TRUE);
    gtk_box_pack_start (GTK_BOX (hbox_ava), scale_ava, FALSE, FALSE, 0);
    gtk_tooltips_set_tip (tooltips, scale_ava, "The bigger the number here, the slower the volume will be turned back on to its original level", NULL);
    gtk_widget_set_sensitive(scale_ava, eqcfg.auto_volume_down);
    gtk_widget_show (scale_ava);

    /* Other frame */
    frame3 = gtk_frame_new ("Other");
    gtk_widget_set_name (frame3, "frame3");
    gtk_widget_ref (frame3);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "frame3", frame3,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (frame3);
    gtk_box_pack_start (GTK_BOX (vbox1), frame3, TRUE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (frame3), 10);

    vbox3 = gtk_vbox_new (TRUE, 0);
    gtk_widget_set_name (vbox3, "vbox3");
    gtk_widget_ref (vbox3);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "vbox3", vbox3,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (vbox3);
    gtk_container_add (GTK_CONTAINER (frame3), vbox3);
    gtk_container_set_border_width (GTK_CONTAINER (vbox3), 10);

    t_extra_filtering = gtk_check_button_new_with_label ("Extra filtering");
    gtk_widget_set_name (t_extra_filtering, "t_extra_filtering");
    gtk_widget_ref (t_extra_filtering);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "t_extra_filtering", t_extra_filtering,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(t_extra_filtering), eqcfg.extra_filtering);
    gtk_widget_show (t_extra_filtering);
    gtk_box_pack_start (GTK_BOX (vbox3), t_extra_filtering, FALSE, FALSE, 0);
    gtk_tooltips_set_tip (tooltips, t_extra_filtering, "Provides a second filtering pass to have a better gain ratio at a cost of extra CPU cycles", NULL);

    /* Button stuff */
    hbuttonbox1 = gtk_hbutton_box_new ();
    gtk_widget_set_name (hbuttonbox1, "hbuttonbox1");
    gtk_widget_ref (hbuttonbox1);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "hbuttonbox1", hbuttonbox1,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (hbuttonbox1);
    gtk_box_pack_start (GTK_BOX (vbox1), hbuttonbox1, FALSE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (hbuttonbox1), 5);
    gtk_button_box_set_layout (GTK_BUTTON_BOX (hbuttonbox1), GTK_BUTTONBOX_END);

    /* OK button */
    b_ok = gtk_button_new_with_label ("OK");
    gtk_widget_set_name (b_ok, "b_ok");
    gtk_widget_ref (b_ok);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "b_ok", b_ok,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (b_ok);
    gtk_container_add (GTK_CONTAINER (hbuttonbox1), b_ok);
    GTK_WIDGET_SET_FLAGS (b_ok, GTK_CAN_DEFAULT);

    /* Cancel button */
    b_cancel = gtk_button_new_with_label ("Cancel");
    gtk_widget_set_name (b_cancel, "b_cancel");
    gtk_widget_ref (b_cancel);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "b_cancel", b_cancel,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (b_cancel);
    gtk_container_add (GTK_CONTAINER (hbuttonbox1), b_cancel);
    GTK_WIDGET_SET_FLAGS (b_cancel, GTK_CAN_DEFAULT);
    
    /* Apply button */
    b_apply = gtk_button_new_with_label ("Apply");
    gtk_widget_set_name (b_apply, "b_apply");
    gtk_widget_ref (b_apply);
    gtk_object_set_data_full (GTK_OBJECT (config_window), "b_apply", b_apply,
            (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (b_apply);
    gtk_container_add (GTK_CONTAINER (hbuttonbox1), b_apply);
    GTK_WIDGET_SET_FLAGS (b_apply, GTK_CAN_DEFAULT);

    gtk_signal_connect (GTK_OBJECT (config_window), "delete_event",
            GTK_SIGNAL_FUNC (on_window_delete_event),
            NULL);
    gtk_signal_connect (GTK_OBJECT (t_use_independent), "toggled",
            GTK_SIGNAL_FUNC (on_use_independent_toggled),
            (gpointer)t_lock_sliders);
    gtk_signal_connect (GTK_OBJECT (t_lock_sliders), "toggled",
            GTK_SIGNAL_FUNC (on_lock_sliders_toggled),
            NULL);
    gtk_signal_connect (GTK_OBJECT (t_extra_filtering), "toggled",
            GTK_SIGNAL_FUNC (on_extra_filtering_toggled),
            NULL);
    gtk_signal_connect (GTK_OBJECT (t_gui_visible), "toggled",
            GTK_SIGNAL_FUNC (on_gui_visible),
            NULL);
    gtk_signal_connect (GTK_OBJECT (t_ava_switch), "toggled",
            GTK_SIGNAL_FUNC(on_ava), (gpointer)scale_ava);

    gtk_signal_connect (GTK_OBJECT (b_ok), "clicked",
            GTK_SIGNAL_FUNC (on_ok_clicked),
            NULL);
    gtk_signal_connect (GTK_OBJECT (b_cancel), "clicked",
            GTK_SIGNAL_FUNC (on_cancel_clicked),
            NULL);
    gtk_signal_connect (GTK_OBJECT (b_apply), "clicked",
            GTK_SIGNAL_FUNC (on_apply_clicked),
            NULL);

    gtk_widget_grab_default (b_ok);
    gtk_object_set_data (GTK_OBJECT (config_window), "tooltips", tooltips);
    gtk_window_set_position(GTK_WINDOW(config_window), GTK_WIN_POS_MOUSE);
    gtk_widget_show (config_window);
}

void eq_read_config(void)
{
    ConfigFile *cfgfile;
    const EQBandsChoices *pbc;
    gint i,chn;
    gboolean found;

    /* Defaults */
    eqcfg.x = 10;
    eqcfg.y = 200;
    eqcfg.band_num = 15;
    eqcfg.use_xmms_original_freqs = FALSE;
    eqcfg.use_independent_channels = FALSE;
    eqcfg.lock_sliders = TRUE;
    eqcfg.extra_filtering = TRUE;
    eqcfg.skin = NULL;
    eqcfg.shaded = FALSE;
    eqcfg.gui_visible = TRUE;
    eqcfg.auto_volume_down = TRUE;
    eqcfg.auto_volume_down_ms = 150;
    eqcfg.equalizer_autoload = FALSE;
    eqcfg.equalizer_active = TRUE;
    for (chn = 0; chn<EQ_CHANNELS; chn++)
    {
        eqcfg.equalizer_preamp[chn] = 0.0;
        for (i = 0; i < EQ_MAX_BANDS; i++)
            eqcfg.equalizer_bands[EQ_MAX_BANDS][chn] = 0.0;
    }
    eqcfg.eqpreset_default_file = eqcfg.eqpreset_extension = NULL;

    /* Now read the settings from the file */
    cfgfile = xmms_cfg_open_default_file();

    xmms_cfg_read_int(cfgfile, "eq_plugin", "x", &eqcfg.x);
    xmms_cfg_read_int(cfgfile, "eq_plugin", "y", &eqcfg.y);

    xmms_cfg_read_int(cfgfile, "eq_plugin", "band_num", &eqcfg.band_num);
    /* Verify that the band count is a valid one */
    found = FALSE;
    if (eqcfg.band_num) 
    {
        pbc = bc;
        while (pbc->bands)
        {
            if (eqcfg.band_num == pbc->bands)
            {
                found = TRUE;
                break;
            }
            pbc++;
        }
    }
    if (!found) 
        eqcfg.band_num = bc->bands;

    xmms_cfg_read_boolean(cfgfile, "eq_plugin", "use_xmms_original_freqs", 
            &eqcfg.use_xmms_original_freqs);
    xmms_cfg_read_boolean(cfgfile, "eq_plugin", "use_independent_channels", 
            &eqcfg.use_independent_channels);
    xmms_cfg_read_boolean(cfgfile, "eq_plugin", "lock_sliders",
            &eqcfg.lock_sliders);
    xmms_cfg_read_boolean(cfgfile, "eq_plugin", "extra_filtering",
            &eqcfg.extra_filtering);
    xmms_cfg_read_string(cfgfile, "eq_plugin", "skin", &eqcfg.skin);
    if (!eqcfg.skin)
        eqcfg.skin = "default";
    if (!strcmp(eqcfg.skin, "(null)"))
        eqcfg.skin = "default";

    xmms_cfg_read_boolean(cfgfile, "eq_plugin", "shaded",
            &eqcfg.shaded);
    xmms_cfg_read_boolean(cfgfile, "eq_plugin", "gui_visible",
            &eqcfg.gui_visible);
    xmms_cfg_read_boolean(cfgfile, "eq_plugin", "auto_volume_down",
            &eqcfg.auto_volume_down);
    xmms_cfg_read_int(cfgfile, "eq_plugin", "auto_volume_down_ms", &eqcfg.auto_volume_down_ms);

    xmms_cfg_read_boolean(cfgfile, "eq_plugin", "equalizer_active",
            &eqcfg.equalizer_active);
    xmms_cfg_read_boolean(cfgfile, "eq_plugin", "equalizer_autoload",
            &eqcfg.equalizer_autoload);
    for (chn = 0; chn<EQ_CHANNELS; chn++)
    {
        gchar eqtext[64];
        sprintf(eqtext, "equalizer_preamp%d_%d%s", chn, eqcfg.band_num,
                eqcfg.use_xmms_original_freqs ? "o" : "");
        xmms_cfg_read_float(cfgfile, "eq_plugin", eqtext, 
                &eqcfg.equalizer_preamp[chn]);
        for (i = 0; i < eqcfg.band_num; i++)
        {
            sprintf(eqtext, "equalizer_band%d_%d_%d%s", i, chn, eqcfg.band_num,
                    eqcfg.use_xmms_original_freqs ? "o" : "");
            xmms_cfg_read_float(cfgfile, "eq_plugin", eqtext, 
                    &eqcfg.equalizer_bands[i][chn]);
        }
    }
    xmms_cfg_read_string(cfgfile, "eq_plugin", "eqpreset_default_file", &eqcfg.eqpreset_default_file);
    xmms_cfg_read_string(cfgfile, "eq_plugin", "eqpreset_extension", &eqcfg.eqpreset_extension);
    xmms_cfg_free(cfgfile);
    if (eqcfg.eqpreset_default_file == NULL)
        eqcfg.eqpreset_default_file = g_strdup("eq_dir_default.preset");
    if (eqcfg.eqpreset_extension == NULL)
        eqcfg.eqpreset_extension = g_strdup("eq_preset");
}

void eq_write_config(void)
{
    ConfigFile *cfgfile;
    gint i,chn;
    cfgfile = xmms_cfg_open_default_file();

    if (!eqcfg.skin)
        eqcfg.skin = "default";
    if (!eqcfg.eqpreset_default_file)
        eqcfg.eqpreset_default_file = g_strdup("eq_dir_default.preset");
    if (!eqcfg.eqpreset_extension)
        eqcfg.eqpreset_extension = g_strdup("eq_preset");

    xmms_cfg_write_int(cfgfile, "eq_plugin", "x", eqcfg.x);
    xmms_cfg_write_int(cfgfile, "eq_plugin", "y", eqcfg.y);
    xmms_cfg_write_int(cfgfile, "eq_plugin", "band_num", eqcfg.band_num);
    xmms_cfg_write_boolean(cfgfile, "eq_plugin", "use_xmms_original_freqs", 
            eqcfg.use_xmms_original_freqs);
    xmms_cfg_write_boolean(cfgfile, "eq_plugin", "use_independent_channels", 
            eqcfg.use_independent_channels);
    xmms_cfg_write_boolean(cfgfile, "eq_plugin", "lock_sliders", 
            eqcfg.lock_sliders);
    xmms_cfg_write_boolean(cfgfile, "eq_plugin", "extra_filtering", 
            eqcfg.extra_filtering);
    xmms_cfg_write_string(cfgfile, "eq_plugin", "skin", eqcfg.skin);
    xmms_cfg_write_boolean(cfgfile, "eq_plugin", "shaded",
            eqcfg.shaded);
    xmms_cfg_write_boolean(cfgfile, "eq_plugin", "gui_visible", 
            eqcfg.gui_visible);
    xmms_cfg_write_boolean(cfgfile, "eq_plugin", "auto_volume_down", 
            eqcfg.auto_volume_down);
    xmms_cfg_write_int(cfgfile, "eq_plugin", "auto_volume_down_ms", 
            eqcfg.auto_volume_down_ms);
    xmms_cfg_write_boolean(cfgfile, "eq_plugin", "equalizer_active", 
            eqcfg.equalizer_active);
    xmms_cfg_write_boolean(cfgfile, "eq_plugin", "equalizer_autoload", 
            eqcfg.equalizer_autoload);
    for (chn=0; chn<EQ_CHANNELS;chn++)
    {
        gchar eqtext[64];
        sprintf(eqtext, "equalizer_preamp%d_%d%s", chn, eqcfg.band_num,
                eqcfg.use_xmms_original_freqs ? "o" : "");
        xmms_cfg_write_float(cfgfile, "eq_plugin", eqtext, 
                eqcfg.equalizer_preamp[chn]);
        for (i = 0; i < EQ_MAX_BANDS; i++)
        {
            sprintf(eqtext, "equalizer_band%d_%d_%d%s", i, chn, eqcfg.band_num,
                    eqcfg.use_xmms_original_freqs ? "o" : "");
            xmms_cfg_write_float(cfgfile, "eq_plugin", eqtext, 
                    eqcfg.equalizer_bands[i][chn]);
        }
    }
    xmms_cfg_write_string(cfgfile, "eq_plugin", "eqpreset_default_file", eqcfg.eqpreset_default_file);
    xmms_cfg_write_string(cfgfile, "eq_plugin", "eqpreset_extension", eqcfg.eqpreset_extension);
    xmms_cfg_write_default_file(cfgfile);
    xmms_cfg_free(cfgfile);
}
