/*
 *   PCM time-domain equalizer
 *
 *   Copyright (C) 2002-2006  Felipe Rivera <liebremx at users.sourceforge.net>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   $Id: eq.c,v 1.12 2006/01/15 00:13:49 liebremx Exp $
 */

#include <math.h>
#include "eq.h"

static gboolean on = FALSE;
static void init(void);
static void cleanup(void);
static gint raisevol(gpointer data);
static gint monitor_song_change(gpointer data);
static int volume_diff;
static guint monitor_id = 0;
static gint previous_pos;
static gchar *previous_filename;

EffectPlugin eq_ep =
{
    NULL,
    NULL,
    NULL, /* Description */
    init,
    cleanup,
    about,
    eq_configuration_window,
    mod_samples,
    NULL
};


EffectPlugin *get_eplugin_info(void)
{
    eq_ep.description = g_strdup_printf("EQ Plugin %s", VERSION);
    return &eq_ep;
}

static void init(void)
{
    if (eq_has_been_initialized)
        return;
    /* Init the GUI first to load the configuration values */
    init_gui();
    init_iir();
    on = eqcfg.equalizer_active;
    volume_diff=0;

    /* Monitor a song change to auto adjust the EQ settings */
    previous_pos = -1;
    previous_filename = NULL;
    monitor_id = gtk_timeout_add(100, monitor_song_change, NULL);
    eq_has_been_initialized = TRUE;
}

static void cleanup(void)
{
    clean_gui();
    if (monitor_id)
        gtk_timeout_remove(monitor_id);
    g_free(previous_filename);
}

void onswitch(gboolean onstatus)
{
    on = onstatus;
    if (eqcfg.auto_volume_down)
    {
        if (!onstatus)
            mainwin_set_volume_diff(-35);
        else {
            volume_diff=35;
            (void)gtk_timeout_add(eqcfg.auto_volume_down_ms, raisevol, NULL);
        }
    }
}

void set_eq_value(float value, gint index, gint chn)
{
    /* Map the gain and preamp values */
    if (index >= 0)
        set_gain(index, chn,
             2.5220207857061455181125E-01 * 
            exp(8.0178361802353992349168E-02 * value) 
            - 2.5220207852836562523180E-01);
    else
        /* -12dB .. 12dB mapping */
        set_preamp(chn, 9.9999946497217584440165E-01 *
            exp(6.9314738656671842642609E-02 * value)
            + 3.7119444716771825623636E-07);
}

gint monitor_song_change(gpointer data)
{
    gchar *filename;
    gint pos;

    GDK_THREADS_ENTER();

    pos = get_playlist_position ();

    if (pos != previous_pos)
    {
        filename = playlist_get_filename(pos);
        if (filename != NULL)
        {
            if ( !previous_filename ||
                    g_strcasecmp(filename, previous_filename) != 0 ) 
            {
                EQequalizer_load_auto_preset(g_basename(filename));
                if ( previous_filename )
                    g_free(previous_filename);
                previous_filename = filename;
            }
        }
        previous_pos = pos;
    }

    GDK_THREADS_LEAVE();
    return TRUE;
}

gint raisevol(gpointer data)
{
    GDK_THREADS_ENTER();
    mainwin_set_volume_diff(  1  );
    GDK_THREADS_LEAVE();
    if (--volume_diff == 0) 
        return FALSE;
    return TRUE;
}

/* 
 * The main stuff
 */
int mod_samples(gpointer * d, gint length, AFormat afmt, gint srate, gint nch)
{
    /* Support all this later */
    if (!on || !(afmt == FMT_S16_NE ||
                (afmt == FMT_S16_LE && G_BYTE_ORDER == G_LITTLE_ENDIAN) ||
                (afmt == FMT_S16_BE && G_BYTE_ORDER == G_BIG_ENDIAN)) ||
            !(srate == 48000 || srate == 44100 || 
                srate == 22050 || srate == 11025) || 
            !(nch == 1 || nch == 2))
        return length;

    return iir(d, length, srate, nch);
}
