/*  eqskin.c
 *  simple skin support in EQ plugin
 *  (c) 2003 by Simone Karin Lehmann <simone@lisanet.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "eqskin.h"
#include "eq.h"
#include "bmp.h"

#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <ctype.h>
#include <dirent.h>
#ifdef HAVE_FTS_H
#include <fts.h>
#endif

// include all xpm for the skins
// filename eqboost.xpm only for compatibility with version 0.5, should now be named eqquick.xpm
// file contents is the same, so it doesn't matter
//
#include "defskin/eqmain31.xpm"
#include "defskin/eqmain25.xpm"
#include "defskin/eqmain15.xpm"
#include "defskin/eqmain10.xpm"
#include "defskin/eqmain10_orig.xpm"
#include "defskin/eqcontrols.xpm"
#include "defskin/eqboost.xpm"
#include "defskin/eqoverlay.xpm"
#include "defskin/eq_ex.xpm"

static gint skin_width;
static gint eqskin_offset;

static GdkPixmap *eqskin_controls, *eqskin_quick, *eqskin_overlay, *eqskin_title, *eqskin_shade;

char *tmpdir;

GdkPixmap *eq_load_default_skin(GdkWindow  *window)
{
	char **skin_name = eqmain10_xpm;

	if (eqcfg.band_num == 31)
	{
		skin_name = eqmain31_xpm;
	}
	else if (eqcfg.band_num == 15)
	{
		skin_name = eqmain15_xpm;
	}
	else if (eqcfg.band_num == 25)
	{
		skin_name = eqmain25_xpm;
	}
	else if (eqcfg.band_num == 10)
	{
		if (eqcfg.use_xmms_original_freqs)
			skin_name = eqmain10_orig_xpm;
		else
			skin_name = eqmain10_xpm;
	}

	eqskin_controls = gdk_pixmap_create_from_xpm_d(window, NULL, NULL, eqcontrols_xpm);
	eqskin_quick = gdk_pixmap_create_from_xpm_d(window, NULL, NULL, eqboost_xpm);
	eqskin_overlay = gdk_pixmap_create_from_xpm_d(window, NULL, NULL, eqoverlay_xpm);
	eqskin_shade = gdk_pixmap_create_from_xpm_d(window, NULL, NULL, eq_ex_xpm);

	return gdk_pixmap_create_from_xpm_d(window, NULL, NULL, skin_name);

}

GdkPixmap *eq_load_skin(GdkWindow  *window)
{
	char *filename = NULL;
	gchar *f;
	GdkPixmap *pix;

	if (eqcfg.band_num == 31)
	{
		filename = "eqmain31.bmp";
	}
	else if (eqcfg.band_num == 15)
	{
		filename = "eqmain15.bmp";
	}
	else if (eqcfg.band_num == 25)
	{
		filename = "eqmain25.bmp";
	}
	else if (eqcfg.band_num == 10)
	{
		if (eqcfg.use_xmms_original_freqs)
			filename = "eqmain10_orig.bmp";
		else
			filename = "eqmain10.bmp";
	}

	f = g_strconcat(tmpdir, "/", filename, NULL);
	pix = read_bitmap(window, f);
	g_free(f);

	f = g_strconcat(tmpdir, "/eqcontrols.bmp", NULL);
	eqskin_controls = read_bitmap(window,f);
	g_free(f);

	f = g_strconcat(tmpdir, "/eqboost.bmp", NULL);
	eqskin_quick = read_bitmap(window,f);
	g_free(f);

	f = g_strconcat(tmpdir, "/eqoverlay.bmp", NULL);
	eqskin_overlay = read_bitmap(window,f);
	g_free(f);

	f = g_strconcat(tmpdir, "/eq_ex.bmp", NULL);
	eqskin_shade = read_bitmap(window,f);
	g_free(f);

	return pix;
}


char *eq_decompress_skin(const char* path)
{
	char *tmp = NULL, *tempdir, *unzip, *tar, *ending;

	unzip = getenv("UNZIPCMD");
	if (!unzip)
		unzip = "unzip";
	tar = getenv("TARCMD");
	if (!tar)
		tar = "tar";

	if ((ending = strrchr(path, '.')) == NULL)
		return NULL;

	tempdir = g_strconcat(g_get_tmp_dir(), "/eqskin", NULL);
	if (mkdir(tempdir,0700))
    {
		g_free(tempdir);
		g_message("Failed to create temporary directory: %s.  Unable to load skin.", strerror(errno));
		return NULL;
	}

	if (!strcasecmp(ending, ".zip") || !strcasecmp(ending, ".wsz"))
		tmp = g_strdup_printf("%s >/dev/null -o -j \"%s\" -d %s", unzip, path, tempdir);
	if (!strcasecmp(ending, ".tgz") || !strcasecmp(ending, ".gz"))
		tmp = g_strdup_printf("%s >/dev/null xzf \"%s\" -C %s", tar, path, tempdir);
	if (!strcasecmp(ending, ".bz2"))
		tmp = g_strdup_printf("bzip2 -dc \"%s\" | %s >/dev/null xf - -C %s", path, tar, tempdir);
	if (!strcasecmp(ending, ".tar"))
		tmp = g_strdup_printf("%s >/dev/null xf \"%s\" -C %s", tar, path, tempdir);

	system(tmp);
	g_free(tmp);
	return tempdir;
}




GdkPixmap *eq_create_skin_bg_pixmap(GdkWindow  *window)
{
	GdkPixmap *bg, *pix;
	GdkGC *gc;
	gint w, h, over, in_title;

// load background skin

	if (!strcmp(eqcfg.skin, "default"))
    {
		pix = eq_load_default_skin(window);
	}
	else
	{
		tmpdir = eq_decompress_skin(eqcfg.skin);
		if (tmpdir)
		{
			pix = eq_load_skin(window);
			if (!pix)
			{
				pix = eq_load_default_skin(window);
				eqcfg.skin = "default";
			}
		}
		else
		{
			pix = eq_load_default_skin(window);
			eqcfg.skin = "default";
		}

		del_directory(tmpdir);
		g_free(tmpdir);

	}

	gdk_window_get_size(pix, &skin_width, &h);

	eqskin_offset = (skin_width/2-113)/2;
	if (eqcfg.band_num == 10)
		eqskin_offset += 5;
	if (!eqcfg.use_independent_channels)
		skin_width /= 2;

	// create background
	bg = gdk_pixmap_new(NULL, skin_width, 116, gdk_rgb_get_visual()->depth);
	gc = gdk_gc_new(window);
	// paint skin on background and set it for the window
	gdk_draw_pixmap(bg, gc, pix, 0, 0, 0, 0, skin_width, 116);
	// paint the overlay if we use only hvae no indep. channels
	over = 0;
	if (!eqcfg.use_independent_channels)
	{
		if (eqcfg.band_num == 25)
			over = 37;
		gdk_draw_pixmap(bg, gc, eqskin_overlay, 0, over, skin_width - 60, 0, 60, 36);
	}

	gdk_window_clear(window);
	gdk_window_set_back_pixmap(window, bg, 0);

	// create title bars pixmap
	// use 1 pixel distance between titlebars to have the same starting coords as eq_ex.bmp
	eqskin_title = gdk_pixmap_new(NULL, skin_width, 29, gdk_rgb_get_visual()->depth);
	// paint active title
	gdk_draw_pixmap(eqskin_title, gc, pix, 0, 0, 0, 0, skin_width, 14);
	// paint inactive title, only if it exists, otherwise the same as active title, be compatible
	// with version 0.5 skins
	in_title = 0;
	if (h > 116)
		in_title = 117;
	gdk_draw_pixmap(eqskin_title, gc, pix, 0, in_title, 0, 15, skin_width, 14);
	// paint the overlay if we use only have no indep. channels
	over = 0;
	if (!eqcfg.use_independent_channels)
	{
		if (eqcfg.band_num == 25)
			over = 37;
		// overlay for active title
		gdk_draw_pixmap(eqskin_title, gc, eqskin_overlay, 0, over, skin_width - 60, 0, 60, 14);
		gdk_window_get_size(eqskin_overlay, &w, &h);
		in_title = 0;
		if (h > 74)
		{
			in_title = 74;
			if (eqcfg.band_num == 25)
				over = 15;
		}
		// overlay for inactive title, only if overlay exisits, otherwise the same as active title
		gdk_draw_pixmap(eqskin_title, gc, eqskin_overlay, 0, over + in_title, skin_width - 60, 15, 60, 14);


	}



	return bg;

}

GdkPixmap *eq_skin_controls(void)
{
	return eqskin_controls;
}

GdkPixmap *eq_skin_quick(void)
{
	return eqskin_quick;
}

GdkPixmap *eq_skin_title(void)
{
	return eqskin_title;
}

gint eq_skin_width(void)
{
	return skin_width;
}

gint eq_skin_goffset(void)
{
	return eqskin_offset;
}

GdkPixmap *eq_skin_shade(void)
{
	return eqskin_shade;
}

// ************ taken from xmms *******************

void del_directory(const char *dirname)
{
#ifdef HAVE_FTS_H
	char * const argv[2] = { (char *) dirname, NULL };
	FTS *fts;
	FTSENT *p;

	fts = fts_open(argv, FTS_PHYSICAL, (int (*)())NULL);
	while ((p = fts_read(fts)) != NULL) {
		switch (p->fts_info) {
		case FTS_D:
			break;
		case FTS_DNR:
		case FTS_ERR:
			break;
		case FTS_DP:
			rmdir(p->fts_accpath);
			break;
		default:
			unlink(p->fts_accpath);
			break;
		}
	}
#else /* !HAVE_FTS_H */
	DIR *dir;
	struct dirent *dirent;
	char *file;

	if ((dir = opendir(dirname)) != NULL)
	{
		while ((dirent = readdir(dir)) != NULL)
		{
			if (strcmp(dirent->d_name, ".") && strcmp(dirent->d_name, ".."))
			{
				file = g_strdup_printf("%s/%s", dirname, dirent->d_name);
				if (unlink(file) == -1)
					if (errno == EISDIR)
						del_directory(file);
				g_free(file);
			}
		}
		closedir(dir);
	}
	rmdir(dirname);
#endif /* !HAVE_FTS_H */
}




