/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-2002  Peter Alm, Mikael Alm, Olle Hallnas,
 *                           Thomas Nilsson and 4Front Technologies
 *  Copyright (C) 1999-2002  Haavard Kvaalen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "eq.h"
#include "eqskin.h"
#include "eqskinwin.h"

#include <math.h>

GtkWidget *EQequalizerwin;

static GtkWidget *EQequalizer_load_window = NULL;
static GtkWidget *EQequalizer_load_auto_window = NULL;
static GtkWidget *EQequalizer_save_window = NULL, *EQequalizer_save_entry;
static GtkWidget *EQequalizer_save_auto_window = NULL, *EQequalizer_save_auto_entry;
static GtkWidget *EQequalizer_delete_window = NULL;
static GtkWidget *EQequalizer_delete_auto_window = NULL;
static GtkWidget *EQequalizer_configure_window = NULL;

static GtkWidget *eqconfwin_options_eqdf_entry, *eqconfwin_options_eqef_entry;

GdkPixmap *EQequalizer_bg;
GdkPixmap *EQpixmap, *EQquick, *EQtitle, *EQshade;
GdkGC *EQequalizer_gc;

GList *EQequalizer_wlist = NULL;

GtkAccelGroup *EQequalizer_accel;

static TButton *EQequalizer_on, *EQequalizer_auto;
static PButton *EQequalizer_presets, *EQequalizer_menu;
static PButton *EQequalizer_close, *EQequalizer_minimize;
static PButton *EQequalizer_maximize, *EQequalizer_closeshade, *EQequalizer_menushade;
static HSlider *EQequalizer_volume, *EQequalizer_balance;
static EQEqGraph *EQequalizer_graph[EQ_CHANNELS];
static EQEqSlider *EQequalizer_preamp[EQ_CHANNELS],
                  *EQequalizer_bands[EQ_MAX_BANDS][EQ_CHANNELS];

static GtkItemFactory *EQequalizer_presets_menu, *EQequalizer_system_menu;

gboolean EQequalizer_focus = FALSE;

typedef struct
{
    gchar *name;
    gfloat preamp[EQ_CHANNELS],
    bands[EQ_MAX_BANDS][EQ_CHANNELS];
}
EqualizerPreset;

static GList *equalizer_presets = NULL, *equalizer_auto_presets = NULL;

void EQequalizer_presets_menu_cb(gpointer cb_data, guint action, GtkWidget * w);
void EQequalizer_system_menu_cb(gpointer cb_data, guint action, GtkWidget * w);
GtkWidget * EQequalizer_create_conf_window(void);

enum
{
    EQUALIZER_PRESETS_LOAD_PRESET, 
    EQUALIZER_PRESETS_LOAD_AUTOPRESET,
    EQUALIZER_PRESETS_LOAD_DEFAULT,	
    EQUALIZER_PRESETS_LOAD_ZERO,
    EQUALIZER_PRESETS_LOAD_FROM_FILE,
    EQUALIZER_PRESETS_SAVE_PRESET, 
    EQUALIZER_PRESETS_SAVE_AUTOPRESET,
    EQUALIZER_PRESETS_SAVE_DEFAULT,	
    EQUALIZER_PRESETS_SAVE_TO_FILE,
    EQUALIZER_PRESETS_DELETE_PRESET,
    EQUALIZER_PRESETS_DELETE_AUTOPRESET, 
    EQUALIZER_PRESETS_CONFIGURE,
    EQUALIZER_SKIN_BROWSER,
    EQUALIZER_PRESETS_LOAD_LEFT_ON_RIGHT, 
    EQUALIZER_PRESETS_LOAD_RIGHT_ON_LEFT,
    EQUALIZER_ABOUT, 
    EQUALIZER_PREFERENCES, 
    EQUALIZER_EXIT
};

GtkItemFactoryEntry EQequalizer_presets_menu_entries[] =
{
    {("/Load"),                    NULL, NULL, 0, "<Branch>"},
    {("/Load/Preset"),             NULL, EQequalizer_presets_menu_cb, EQUALIZER_PRESETS_LOAD_PRESET, "<Item>"},
    {("/Load/Auto-load preset"),   NULL, EQequalizer_presets_menu_cb, EQUALIZER_PRESETS_LOAD_AUTOPRESET, "<Item>"},
    {("/Load/Default"),            NULL, EQequalizer_presets_menu_cb, EQUALIZER_PRESETS_LOAD_DEFAULT, "<Item>"},
    {("/Load/-"),                  NULL, NULL, 0, "<Separator>"},
    {("/Load/Zero"),               NULL, EQequalizer_presets_menu_cb, EQUALIZER_PRESETS_LOAD_ZERO, "<Item>"},
    {("/Load/-"),                  NULL, NULL, 0, "<Separator>"},
    {("/Load/From file"),          NULL, EQequalizer_presets_menu_cb, EQUALIZER_PRESETS_LOAD_FROM_FILE, "<Item>"},
    {("/Save"),                    NULL, NULL, 0, "<Branch>"},
    {("/Save/Preset"),             NULL, EQequalizer_presets_menu_cb, EQUALIZER_PRESETS_SAVE_PRESET, "<Item>"},
    {("/Save/Auto-load preset"),   NULL, EQequalizer_presets_menu_cb, EQUALIZER_PRESETS_SAVE_AUTOPRESET, "<Item>"},
    {("/Save/Default"),            NULL, EQequalizer_presets_menu_cb, EQUALIZER_PRESETS_SAVE_DEFAULT, "<Item>"},
    {("/Save/-"),                  NULL, NULL, 0, "<Separator>"},
    {("/Save/To file"),            NULL, EQequalizer_presets_menu_cb, EQUALIZER_PRESETS_SAVE_TO_FILE, "<Item>"},
    {("/Delete"),                  NULL, NULL, 0, "<Branch>"},
    {("/Delete/Preset"),           NULL, EQequalizer_presets_menu_cb, EQUALIZER_PRESETS_DELETE_PRESET, "<Item>"},
    {("/Delete/Auto-load preset"), NULL, EQequalizer_presets_menu_cb, EQUALIZER_PRESETS_DELETE_AUTOPRESET, "<Item>"},
    {("/Configure Presets"),       NULL, EQequalizer_presets_menu_cb, EQUALIZER_PRESETS_CONFIGURE, "<Item>"},
    {("/Left -> Right"),           NULL, EQequalizer_presets_menu_cb, EQUALIZER_PRESETS_LOAD_LEFT_ON_RIGHT, "<Item>"},
    {("/Right -> Left"),           NULL, EQequalizer_presets_menu_cb, EQUALIZER_PRESETS_LOAD_RIGHT_ON_LEFT, "<Item>"},
};

static gint EQequalizer_presets_menu_entries_num =
                sizeof(EQequalizer_presets_menu_entries) /
                sizeof(EQequalizer_presets_menu_entries[0]);

// by simone
GtkItemFactoryEntry EQequalizer_system_menu_entries[] =
{
    {("/About"),          NULL, EQequalizer_system_menu_cb, EQUALIZER_ABOUT, "<Item>"},
    {("/-"),              NULL, NULL, 0, "<Separator>"},
    {("/Preferences"),    NULL, EQequalizer_system_menu_cb, EQUALIZER_PREFERENCES, "<Item>"},
    {("/EQSkin Browser"), NULL, EQequalizer_system_menu_cb, EQUALIZER_SKIN_BROWSER, "<Item>"},
    {("/-"),              NULL, NULL, 0, "<Separator>"},
    {("/Exit EQ"),        NULL, EQequalizer_system_menu_cb, EQUALIZER_EXIT, "<Item>"},
};

static gint EQequalizer_system_menu_entries_num =
                sizeof(EQequalizer_system_menu_entries) /
                sizeof(EQequalizer_system_menu_entries[0]);

static gint xpm_width;

extern void input_get_volume(int *vleft, int *vright);

void EQequalizer_raise(void)
{
    gdk_window_raise(EQequalizerwin->window);
}

void EQequalizer_eq_changed(gint index, gint chn)
{
    gint nchn = (chn+1)%2;
    if (index >= 0)	/* Band slider */
    {
        eqcfg.equalizer_bands[index][chn] = EQeqslider_get_position(EQequalizer_bands[index][chn]);
        set_eq_value(eqcfg.equalizer_bands[index][chn], index, chn);
        if (eqcfg.use_independent_channels)
        {
            if (eqcfg.lock_sliders && eq_has_been_initialized)
            {
                eqcfg.equalizer_bands[index][nchn] =
                    eqcfg.equalizer_bands[index][chn];
                EQeqslider_set_position(EQequalizer_bands[index][nchn],
                        eqcfg.equalizer_bands[index][nchn]);
                set_eq_value(eqcfg.equalizer_bands[index][nchn], index, nchn);
                draw_widget(EQequalizer_graph[nchn]);
            }
        }
        else
        {
            eqcfg.equalizer_bands[index][nchn] =
                eqcfg.equalizer_bands[index][chn];
            set_eq_value(eqcfg.equalizer_bands[index][nchn], index, nchn);
        }
    }
    else 			/* Preamp slider */
    {
        eqcfg.equalizer_preamp[chn] = EQeqslider_get_position(EQequalizer_preamp[chn]);
        set_eq_value(eqcfg.equalizer_preamp[chn], index, chn);
        if (eqcfg.use_independent_channels)
        {
            if (eqcfg.lock_sliders && eq_has_been_initialized)
            {
                eqcfg.equalizer_preamp[nchn] =
                    eqcfg.equalizer_preamp[chn];
                EQeqslider_set_position(EQequalizer_preamp[nchn],
                        eqcfg.equalizer_preamp[nchn]);
                set_eq_value(eqcfg.equalizer_preamp[nchn], index, nchn);
                draw_widget(EQequalizer_graph[nchn]);
            }
        }
        else
        {
            eqcfg.equalizer_preamp[nchn] =
                eqcfg.equalizer_preamp[chn];
            set_eq_value(eqcfg.equalizer_preamp[nchn], index, nchn);
        }

    }
    draw_widget(EQequalizer_graph[chn]);
}

void EQequalizer_copy_bands(gint dst)
{
    int i;

    if (dst == EQ_LEFT)
    {
        for (i = 0; i < eqcfg.band_num; i++)
        {
            eqcfg.equalizer_bands[i][EQ_LEFT] =
                eqcfg.equalizer_bands[i][EQ_RIGHT];
            EQeqslider_set_position(EQequalizer_bands[i][EQ_LEFT],
                    EQeqslider_get_position(EQequalizer_bands[i][EQ_RIGHT])
                    );
            EQequalizer_eq_changed(i, EQ_LEFT);
        }
    }
    else
    {
        for (i = 0; i < eqcfg.band_num; i++)
        {
            eqcfg.equalizer_bands[i][EQ_RIGHT] =
                eqcfg.equalizer_bands[i][EQ_LEFT];
            EQeqslider_set_position(EQequalizer_bands[i][EQ_RIGHT],
                    EQeqslider_get_position(EQequalizer_bands[i][EQ_LEFT])
                    );
            EQequalizer_eq_changed(i, EQ_RIGHT);
        }
    }
    draw_widget(EQequalizer_graph[dst]);
    EQdraw_equalizer_window(TRUE);
}

void EQequalizer_on_pushed(gboolean toggled)
{
    eqcfg.equalizer_active = toggled;
    onswitch(toggled);
}

void EQequalizer_presets_pushed(void)
{
    GdkModifierType modmask;
    gint x, y;

    gdk_window_get_pointer(NULL, &x, &y, &modmask);
    util_item_factory_popup(EQequalizer_presets_menu, x, y, 1, GDK_CURRENT_TIME);
}

void EQequalizer_auto_pushed(gboolean toggled)
{
    eqcfg.equalizer_autoload = toggled;
}

void EQequalizer_set_preamp_value(int chn, float value)
{
    int c;
    if (!eqcfg.use_independent_channels)
    {
        EQeqslider_set_position(EQequalizer_preamp[chn], value);
        EQequalizer_eq_changed(-1, chn);
    }
    else
    {
        if (eqcfg.lock_sliders)
        {
            for (c = 0; c < 2; c++)
            {
                EQeqslider_set_position(EQequalizer_preamp[c], value);
                EQequalizer_eq_changed(-1, c);
            }
        }
        else
        {
            EQeqslider_set_position(EQequalizer_preamp[chn], value);
            EQequalizer_eq_changed(-1, chn);
        }
    }
    EQdraw_equalizer_window(FALSE);
}

void EQequalizer_quickhi_pushed(void)
{
    // reset preamp to +12 db
    EQequalizer_set_preamp_value(0, 20.0);
}

void EQequalizer_quickhi2_pushed(void)
{
    // reset preamp to +12 db
    EQequalizer_set_preamp_value(1, 20.0);
}

void EQequalizer_quickflat_pushed(void)
{
    // reset preamp to zero db
    EQequalizer_set_preamp_value(0, 0.0);
}

void EQequalizer_quickflat2_pushed(void)
{
    // reset preamp to zero db
    EQequalizer_set_preamp_value(1, 0.0);
}

void EQequalizer_quicklo_pushed(void)
{
    // reset preamp to -12 db
    EQequalizer_set_preamp_value(0, -20.0);
}

void EQequalizer_quicklo2_pushed(void)
{
    // reset preamp to -12 db
    EQequalizer_set_preamp_value(1, -20.0);
}

void EQequalizer_set_shade_slider(void)
{
    int left, right, vol;
    input_get_volume(&left, &right);
    vol = right;
    if (left > right)
        vol = left;
    EQequalizer_set_volume_slider(vol);
    EQequalizer_set_balance_slider(right - left);
}

void EQdraw_equalizer_window(gboolean force)
{
    gboolean redraw;
    GdkPixmap *pix;

    lock_widget_list(EQequalizer_wlist);
    if (force)
    {
        if (!eqcfg.shaded)
        {
            pix = EQtitle;
        }
        else
        {
            pix = EQshade;
        }

        if (EQequalizer_focus)
        {
            gdk_draw_pixmap(EQequalizer_bg, EQequalizer_gc, pix, 0, 0, 0, 0, xpm_width, 14);
            EQequalizer_set_shade_slider();
        }
        else
        {
            gdk_draw_pixmap(EQequalizer_bg, EQequalizer_gc, pix, 0, 15, 0, 0, xpm_width, 14);
        }

        draw_widget_list(EQequalizer_wlist, &redraw, TRUE);
    }
    else
        draw_widget_list(EQequalizer_wlist, &redraw, FALSE);

    if (force || redraw)
    {
        clear_widget_list_redraw(EQequalizer_wlist);
        gdk_window_clear(EQequalizerwin->window);
        gdk_flush();
    }
    unlock_widget_list(EQequalizer_wlist);
}

void EQequalizer_close_cb(void)
{
    EQequalizer_real_hide();
    eqcfg.gui_visible = FALSE;
}

// by simone
void EQequalizer_system_menu_pushed(void)
{
    GdkModifierType modmask;
    gint x, y;

    gdk_window_get_pointer(NULL, &x, &y, &modmask);
    util_item_factory_popup(EQequalizer_system_menu, x, y, 1, GDK_CURRENT_TIME);

}

void EQequalizer_set_shade(gboolean shaded)
{
    int visible;

    visible = shaded ? 1 : 0;
    EQequalizer_volume->hs_widget.visible = visible;
    EQequalizer_balance->hs_widget.visible = visible;
    EQequalizer_maximize->pb_widget.visible = visible;
    EQequalizer_closeshade->pb_widget.visible = visible;
    // menu button has differnt pix
    EQequalizer_menushade->pb_widget.visible = visible;
    EQequalizer_menu->pb_widget.visible = !visible;
}

void EQequalizer_minimize_pushed(void)
{
    eqcfg.shaded = !eqcfg.shaded;
    EQequalizer_set_shade(eqcfg.shaded);
    EQequalizer_set_shade_slider();

    // current shade mode is only for standard width 275 pixels
    // other widths don't look good
    // may this should be fixed in the future. simone, 2003-09-16
    gtk_widget_set_usize(EQequalizerwin, (eqcfg.shaded ? 275 : xpm_width), (eqcfg.shaded ? 14 : 116));
    EQdraw_equalizer_window(TRUE);
}

void EQequalizer_maximize_pushed(void)
{
    if (eqcfg.shaded)
        EQequalizer_minimize_pushed();
}

void EQequalizer_closeshade_pushed(void)
{
    if (eqcfg.shaded)
        EQequalizer_close_cb();
}

int EQequalizer_delete(GtkWidget * w, gpointer data)
{
    /*    EQequalizer_show(FALSE);  */
    return TRUE;
}

static gboolean inside_sensitive_widgets(gint x, gint y)
{
    gboolean inside_band_widget = FALSE,
    inside_preamp_widget = FALSE;
    gint i,chn;
    gint channels;

    channels = (eqcfg.use_independent_channels ? 2 : 1);

    for (chn = 0; chn < channels; chn++)
        for (i = 0; i<eqcfg.band_num; i++)
            if (inside_widget(x, y, EQequalizer_bands[i][chn]))
            {
                inside_band_widget = TRUE;
                break;
            }

    if (inside_widget(x, y, EQequalizer_preamp[EQ_LEFT]) ||
            (eqcfg.use_independent_channels && inside_widget(x, y, EQequalizer_preamp[EQ_RIGHT])))
        inside_preamp_widget = TRUE;

    return (inside_widget(x, y, EQequalizer_on) ||
            inside_widget(x, y, EQequalizer_auto) ||
            inside_widget(x, y, EQequalizer_presets) ||
            inside_widget(x, y, EQequalizer_close) ||
            inside_widget(x, y, EQequalizer_menu) ||
            inside_widget(x, y, EQequalizer_minimize) ||
            inside_widget(x, y, EQequalizer_maximize) ||
            inside_widget(x, y, EQequalizer_closeshade) ||
            inside_widget(x, y, EQequalizer_menushade) ||
            inside_widget(x, y, EQequalizer_volume) ||
            inside_widget(x, y, EQequalizer_balance) ||
            inside_preamp_widget || inside_band_widget);
}

void EQequalizer_press(GtkWidget * widget, GdkEventButton * event, gpointer callback_data)
{
    gint mx, my;
    gboolean grab = TRUE;

    mx = event->x;
    my = event->y;

    if (event->button == 1 && event->type == GDK_BUTTON_PRESS &&
            ((cfg.easy_move || event->y < 14) &&
             !inside_sensitive_widgets(event->x, event->y)))
    {

        EQequalizer_raise();
        dock_move_press(dock_window_list, EQequalizerwin, event, FALSE);
    }
    else if (event->button == 1 && event->type == GDK_2BUTTON_PRESS && event->y < 14)
    {
        /* toggle shade mode, hhmmm, EQequalizer_minimize_pushed is ok for that, but this
           name is confusing. It should be named ...toggle_shade or something like that.
           Maybe this will be fixed some time, simone.*/
        EQequalizer_minimize_pushed();
        if(dock_is_moving(EQequalizerwin))
            dock_move_release(EQequalizerwin);
    }
    else if (event->button == 3 &&
            !(inside_widget(event->x, event->y, EQequalizer_on) ||
                inside_widget(event->x, event->y, EQequalizer_auto)))
    {
        grab = FALSE;
    }
    else
    {
        handle_press_cb(EQequalizer_wlist, widget, event);
        EQdraw_equalizer_window(FALSE);
    }
    if (grab)
        gdk_pointer_grab(EQequalizerwin->window, FALSE,
                GDK_BUTTON_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
                GDK_NONE, GDK_NONE, GDK_CURRENT_TIME);
}

void EQequalizer_motion(GtkWidget * widget, GdkEventMotion * event, gpointer callback_data)
{
    XEvent ev;

    if (dock_is_moving(EQequalizerwin))
    {
        dock_move_motion(EQequalizerwin, event);
    }
    else
    {
        handle_motion_cb(EQequalizer_wlist, widget, event);
        EQdraw_equalizer_window(FALSE);
    }
    gdk_flush();
    while (XCheckMaskEvent(GDK_DISPLAY(), ButtonMotionMask, &ev)) ;

}

void EQequalizer_release(GtkWidget * widget, GdkEventButton * event, gpointer callback_data)
{
    gdk_pointer_ungrab(GDK_CURRENT_TIME);
    gdk_flush();
    if (dock_is_moving(EQequalizerwin))
    {
        dock_move_release(EQequalizerwin);
    }
    else
    {
        handle_release_cb(EQequalizer_wlist, widget, event);
        EQdraw_equalizer_window(FALSE);
    }
}

void EQequalizer_focus_in(GtkWidget * widget, GdkEvent * event, gpointer callback_data)
{
    EQequalizer_close->pb_allow_draw = TRUE;
    EQequalizer_menu->pb_allow_draw = TRUE;
    EQequalizer_minimize->pb_allow_draw = TRUE;
    EQequalizer_maximize->pb_allow_draw = TRUE;
    EQequalizer_closeshade->pb_allow_draw = TRUE;
    EQequalizer_menushade->pb_allow_draw = TRUE;
    EQequalizer_focus = TRUE;
    EQdraw_equalizer_window(TRUE);
}

void EQequalizer_focus_out(GtkWidget * widget, GdkEventButton * event, gpointer callback_data)
{
    EQequalizer_close->pb_allow_draw = FALSE;
    EQequalizer_menu->pb_allow_draw = FALSE;
    EQequalizer_minimize->pb_allow_draw = FALSE;
    EQequalizer_maximize->pb_allow_draw = FALSE;
    EQequalizer_closeshade->pb_allow_draw = FALSE;
    EQequalizer_menushade->pb_allow_draw = FALSE;
    EQequalizer_focus = FALSE;
    EQdraw_equalizer_window(TRUE);
}


// shade functions
gint EQequalizer_volume_frame_cb(gint pos)
{
    if(EQequalizer_volume)
    {
        if (pos < 32)
            EQequalizer_volume->hs_knob_nx = EQequalizer_volume->hs_knob_px = 1;
        else if (pos < 63)
            EQequalizer_volume->hs_knob_nx = EQequalizer_volume->hs_knob_px = 4;
        else
            EQequalizer_volume->hs_knob_nx = EQequalizer_volume->hs_knob_px = 7;
    }
    return 1;
}

void EQequalizer_volume_motion_cb(gint pos)
{
    gint v = (gint) rint(pos * 100 / 94.0);
    mainwin_adjust_volume_motion(v);
    mainwin_set_volume_slider(v);
}

void EQequalizer_volume_release_cb(gint pos)
{
    mainwin_adjust_volume_release();
}

gint EQequalizer_balance_frame_cb(gint pos)
{
    if(EQequalizer_balance)
    {
        if(pos < 13)
            EQequalizer_balance->hs_knob_nx = EQequalizer_balance->hs_knob_px = 11;
        else if (pos < 26)
            EQequalizer_balance->hs_knob_nx = EQequalizer_balance->hs_knob_px = 14;
        else
            EQequalizer_balance->hs_knob_nx = EQequalizer_balance->hs_knob_px = 17;
    }

    return 1;
}

void EQequalizer_balance_motion_cb(gint pos)
{
    gint b;
    pos = MIN(pos,38); /* The skin uses a even number of pixels
                          for the balance-slider *sigh* */
    b = (gint) rint((pos - 19) * 100 / 19.0);
    mainwin_adjust_balance_motion(b);
    mainwin_set_balance_slider(b);
}

void EQequalizer_balance_release_cb(gint pos)
{
    mainwin_adjust_balance_release();
}

void EQequalizer_set_balance_slider(gint percent)
{
    EQhslider_set_position(EQequalizer_balance, (gint) rint((percent*19/100.0)+19));
}

void EQequalizer_set_volume_slider(gint percent)
{
    EQhslider_set_position(EQequalizer_volume, (gint) rint(percent*94/100.0));
}
// * shade functions end


gboolean EQequalizer_keypress(GtkWidget * w, GdkEventKey * event, gpointer data)
{
    gtk_widget_event(mainwin, (GdkEvent *) event);
    return TRUE;
}

static gboolean EQequalizer_configure(GtkWidget * window, GdkEventConfigure *event, gpointer data)
{
    if (!GTK_WIDGET_VISIBLE(window))
        return FALSE;

    if (cfg.show_wm_decorations)
        gdk_window_get_root_origin(window->window,
                &eqcfg.x, &eqcfg.y);
    else
        gdk_window_get_deskrelative_origin(window->window,
                &eqcfg.x,
                &eqcfg.y);
    return FALSE;
}

void EQequalizer_set_back_pixmap(void)
{
    gdk_window_set_back_pixmap(EQequalizerwin->window, EQequalizer_bg, 0);
    gdk_window_clear(EQequalizerwin->window);
}

gint EQequalizer_client_event(GtkWidget *w, GdkEventClient *event, gpointer data)
{
    static GdkAtom atom_rcfiles = GDK_NONE;

    if (!atom_rcfiles)
        atom_rcfiles = gdk_atom_intern("_GTK_READ_RCFILES", FALSE);

    if(event->message_type == atom_rcfiles)
    {
        EQequalizer_set_back_pixmap();
        return TRUE;
    }
    return FALSE;
}

static GList *EQequalizer_read_presets(gchar * fname)
{
    gchar *filename, *name;
    ConfigFile *cfgfile;
    GList *list = NULL;
    gint i, chn, p = 0;
    EqualizerPreset *preset;
    gint channels;
    channels = (eqcfg.use_independent_channels ? 2 : 1);

    filename = g_strdup_printf("%s/.xmms/%s", g_get_home_dir(), fname);
    if ((cfgfile = xmms_cfg_open_file(filename)) == NULL)
    {
        g_free(filename);
        return NULL;
    }
    g_free(filename);

    for (;;)
    {
        gchar section[64];

        sprintf(section, "Preset%d", p++);
        if (xmms_cfg_read_string(cfgfile, "Presets", section, &name))
        {
            gchar tmp_name[255];
            preset = g_malloc(sizeof (EqualizerPreset));
            sprintf(tmp_name,"%s_%d_%s",name,eqcfg.band_num,
                    eqcfg.use_xmms_original_freqs ? "xmms" : "iso");
            preset->name = name;
            for (chn = 0; chn < channels; chn++)
            {
                gchar tmp[32];
                sprintf(tmp, "Preamp%d", chn);
                xmms_cfg_read_float(cfgfile, tmp_name, tmp,
                        &preset->preamp[chn]);
                for (i = 0; i < eqcfg.band_num; i++)
                {
                    sprintf(tmp, "Band%d_%d", i, chn);
                    xmms_cfg_read_float(cfgfile, tmp_name, tmp,
                            &preset->bands[i][chn]);
                }
            }
            list = g_list_prepend(list, preset);
        }
        else
            break;
    }
    list = g_list_reverse(list);
    xmms_cfg_free(cfgfile);
    return list;
}

static void EQequalizer_create_widgets(void)
{
    gint i;
    PButton *tpix;

    EQequalizer_on = EQcreate_tbutton(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, 14, 18, 25, 12, 10, 3, 128, 3, 69, 3, 187, 3, EQequalizer_on_pushed);
    tbutton_set_toggled(EQequalizer_on, eqcfg.equalizer_active);
    EQequalizer_auto = EQcreate_tbutton(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, 39, 18, 33, 12, 35, 3, 153, 3, 94, 3, 212, 3, EQequalizer_auto_pushed);
    tbutton_set_toggled(EQequalizer_auto, eqcfg.equalizer_autoload);
    EQequalizer_presets = EQcreate_pbutton(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, xpm_width-58, 18, 44, 12, 224, 48, 224, 60, EQequalizer_presets_pushed, SKIN_EQMAIN);
    EQequalizer_close = EQcreate_pbutton(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, xpm_width-11, 3, 9, 9, 0, 0, 0, 9, EQequalizer_close_cb, SKIN_EQMAIN);
    EQequalizer_close->pb_allow_draw = FALSE;

    EQequalizer_menu = EQcreate_pbutton(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, 6, 3, 9, 9, 224, 73, 224, 82, EQequalizer_system_menu_pushed, SKIN_EQMAIN);
    EQequalizer_menu->pb_allow_draw = FALSE;

    EQequalizer_minimize = EQcreate_pbutton(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, xpm_width-21, 3, 9, 9, 234, 73, 234, 82, EQequalizer_minimize_pushed, SKIN_EQMAIN);
    EQequalizer_minimize->pb_allow_draw = FALSE;

    //shade elements
    EQequalizer_maximize = EQcreate_pbutton(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, 275-21, 3, 9, 9, 1, 38, 1, 47, EQequalizer_maximize_pushed, SKIN_EQSHADE);
    EQequalizer_maximize->pb_allow_draw = FALSE;
    EQequalizer_closeshade = EQcreate_pbutton(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, 275-11, 3, 9, 9, 11, 38, 11, 47, EQequalizer_closeshade_pushed, SKIN_EQSHADE);
    EQequalizer_closeshade->pb_allow_draw = FALSE;
    EQequalizer_menushade = EQcreate_pbutton(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, 6, 3, 9, 9, 21, 38, 21, 47, EQequalizer_system_menu_pushed, SKIN_EQSHADE);
    EQequalizer_menushade->pb_allow_draw = FALSE;
    EQequalizer_volume = EQcreate_hslider(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, 61, 4, 97, 8, 1, 30, 1, 30, 3, 7, 4, 61, 0, 94, EQequalizer_volume_frame_cb, EQequalizer_volume_motion_cb, EQequalizer_volume_release_cb, SKIN_EQSHADE);
    EQequalizer_balance = EQcreate_hslider(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, 164, 4, 42, 8, 11, 30, 11, 30, 3, 7, 4, 164, 0, 39, EQequalizer_balance_frame_cb, EQequalizer_balance_motion_cb, EQequalizer_balance_release_cb, SKIN_EQSHADE);
    EQequalizer_set_shade(eqcfg.shaded);
    // Quick fix for bug #897452
    //EQequalizer_set_shade_slider();
    // end of shade elements

    tpix = EQcreate_pbutton(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, 42, 33, 26, 10, 0, 11, 27, 11, EQequalizer_quickhi_pushed, SKIN_EQQUICK);
    tpix = EQcreate_pbutton(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, 42, 65, 26, 10, 0, 0, 27, 0, EQequalizer_quickflat_pushed, SKIN_EQQUICK);
    tpix = EQcreate_pbutton(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, 42, 92, 26, 10, 0, 22, 27, 22, EQequalizer_quicklo_pushed, SKIN_EQQUICK);

    EQequalizer_graph[EQ_LEFT] = EQcreate_eqgraph(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, eq_skin_goffset(), 17, eqcfg.equalizer_bands, EQ_LEFT);
    EQequalizer_preamp[EQ_LEFT] = EQcreate_eqslider(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, 21, 38, -1, EQ_LEFT);
    EQeqslider_set_position(EQequalizer_preamp[EQ_LEFT],
            eqcfg.equalizer_preamp[EQ_LEFT]);
    /* vars for right preamp and graph need to be initialized */
    EQequalizer_preamp[EQ_RIGHT] = EQequalizer_preamp[EQ_LEFT];
    EQequalizer_graph[EQ_RIGHT] = EQequalizer_graph[EQ_LEFT];
    EQequalizer_eq_changed(-1, EQ_LEFT);

    for (i = 0; i < eqcfg.band_num; i++)
    {
        EQequalizer_bands[i][EQ_LEFT] = EQcreate_eqslider(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, 78 + (i * 18), 38, i, EQ_LEFT);
        /* vars for right bands need to be initialized */
        EQequalizer_bands[i][EQ_RIGHT] = EQequalizer_bands[i][EQ_LEFT];
        EQeqslider_set_position(EQequalizer_bands[i][EQ_LEFT], eqcfg.equalizer_bands[i][EQ_LEFT]);
        EQequalizer_eq_changed(i, EQ_LEFT);
    }

    if (eqcfg.use_independent_channels)
    {
        tpix = EQcreate_pbutton(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, xpm_width-68, 33, 26, 10, 54, 11, 81, 11, EQequalizer_quickhi2_pushed, SKIN_EQQUICK);
        tpix = EQcreate_pbutton(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, xpm_width-68, 65, 26, 10, 54, 0, 81, 0, EQequalizer_quickflat2_pushed, SKIN_EQQUICK);
        tpix = EQcreate_pbutton(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, xpm_width-68, 92, 26, 10, 54, 22, 81, 22, EQequalizer_quicklo2_pushed, SKIN_EQQUICK);

        EQequalizer_graph[EQ_RIGHT] = EQcreate_eqgraph(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, xpm_width - eq_skin_goffset()-113, 17, eqcfg.equalizer_bands, EQ_RIGHT);
        EQequalizer_preamp[EQ_RIGHT] = EQcreate_eqslider(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, xpm_width-35, 38, -1, EQ_RIGHT);

        EQeqslider_set_position(EQequalizer_preamp[EQ_RIGHT],
                eqcfg.equalizer_preamp[EQ_RIGHT]);
        EQequalizer_eq_changed(-1, EQ_RIGHT);
        for (i = 0; i < eqcfg.band_num; i++)
        {
            EQequalizer_bands[i][EQ_RIGHT] = EQcreate_eqslider(&EQequalizer_wlist, EQequalizer_bg, EQequalizer_gc, xpm_width -78 - 14 - ((eqcfg.band_num-i-1) * 18), 38, i, EQ_RIGHT);
            EQeqslider_set_position(EQequalizer_bands[i][EQ_RIGHT], eqcfg.equalizer_bands[i][EQ_RIGHT]);
            EQequalizer_eq_changed(i, EQ_RIGHT);
        }
    }
}

static void EQequalizer_create_gtk(void)
{
    EQequalizerwin = gtk_window_new(GTK_WINDOW_DIALOG);

    dock_add_window(dock_window_list, EQequalizerwin);
    gtk_widget_set_app_paintable(EQequalizerwin, TRUE);
    gtk_window_set_policy(GTK_WINDOW(EQequalizerwin), FALSE, FALSE, TRUE);
    gtk_window_set_title(GTK_WINDOW(EQequalizerwin), "EQEqualizer");

    gtk_window_set_wmclass(GTK_WINDOW(EQequalizerwin), "EQEqualizer", "xmms");
    gtk_window_set_transient_for(GTK_WINDOW(EQequalizerwin), GTK_WINDOW(mainwin));
    if (eqcfg.x != -1 && cfg.save_window_position)
        dock_set_uposition(EQequalizerwin, eqcfg.x, eqcfg.y);

    gtk_widget_set_usize(EQequalizerwin, (eqcfg.shaded ? 275 : xpm_width), (eqcfg.shaded ? 14 : 116));

    gtk_widget_set_events(EQequalizerwin, GDK_FOCUS_CHANGE_MASK | GDK_BUTTON_MOTION_MASK | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);
    gtk_widget_realize(EQequalizerwin);
    /* XXX hint_set_skip_winlist(EQequalizerwin); */
    util_set_cursor(EQequalizerwin);
    if (!cfg.show_wm_decorations)
        gdk_window_set_decorations(EQequalizerwin->window, 0);

    EQequalizer_accel = gtk_accel_group_new();
    gtk_window_add_accel_group(GTK_WINDOW(EQequalizerwin), EQequalizer_accel);

    EQequalizer_bg = eq_create_skin_bg_pixmap(EQequalizerwin->window);
    xpm_width = eq_skin_width();
    EQpixmap = eq_skin_controls();
    EQquick = eq_skin_quick();
    EQtitle = eq_skin_title();
    EQshade = eq_skin_shade();

    gdk_window_clear(EQequalizerwin->window);
    gdk_window_set_back_pixmap(EQequalizerwin->window, EQequalizer_bg, 0);

    gtk_signal_connect(GTK_OBJECT(EQequalizerwin), "delete_event",
            GTK_SIGNAL_FUNC(EQequalizer_delete), NULL);
    gtk_signal_connect(GTK_OBJECT(EQequalizerwin), "button_press_event",
            GTK_SIGNAL_FUNC(EQequalizer_press), NULL);
    gtk_signal_connect(GTK_OBJECT(EQequalizerwin), "button_release_event",
            GTK_SIGNAL_FUNC(EQequalizer_release), NULL);
    gtk_signal_connect(GTK_OBJECT(EQequalizerwin), "motion_notify_event",
            GTK_SIGNAL_FUNC(EQequalizer_motion), NULL);
    gtk_signal_connect(GTK_OBJECT(EQequalizerwin), "focus_in_event",
            GTK_SIGNAL_FUNC(EQequalizer_focus_in), NULL);
    gtk_signal_connect(GTK_OBJECT(EQequalizerwin), "focus_out_event",
            GTK_SIGNAL_FUNC(EQequalizer_focus_out), NULL);
    gtk_signal_connect(GTK_OBJECT(EQequalizerwin), "configure_event",
            GTK_SIGNAL_FUNC(EQequalizer_configure), NULL);
    gtk_signal_connect(GTK_OBJECT(EQequalizerwin), "client_event",
            GTK_SIGNAL_FUNC(EQequalizer_client_event), NULL);
    gtk_signal_connect(GTK_OBJECT(EQequalizerwin), "key-press-event",
            GTK_SIGNAL_FUNC(EQequalizer_keypress), NULL);
}

void EQequalizer_configure_gui()
{
    EQequalizer_presets_menu_entries_num =
        sizeof(EQequalizer_presets_menu_entries) /
        sizeof(EQequalizer_presets_menu_entries[0]);

    EQequalizer_system_menu_entries_num =
        sizeof(EQequalizer_system_menu_entries) /
        sizeof(EQequalizer_system_menu_entries[0]);


    if (!eqcfg.use_independent_channels)
        EQequalizer_presets_menu_entries_num -= 2;

}

void EQequalizer_create(void)
{
    EQequalizer_configure_gui();
    EQequalizer_presets_menu = gtk_item_factory_new(GTK_TYPE_MENU, "<main>", EQequalizer_accel);
    gtk_item_factory_set_translate_func(EQequalizer_presets_menu,
            util_menu_translate, NULL, NULL);
    gtk_item_factory_create_items(EQequalizer_presets_menu, EQequalizer_presets_menu_entries_num, EQequalizer_presets_menu_entries, NULL);

    EQequalizer_system_menu = gtk_item_factory_new(GTK_TYPE_MENU, "<main>", EQequalizer_accel);
    gtk_item_factory_set_translate_func(EQequalizer_system_menu,
            util_menu_translate, NULL, NULL);
    gtk_item_factory_create_items(EQequalizer_system_menu, EQequalizer_system_menu_entries_num, EQequalizer_system_menu_entries, NULL);


    equalizer_presets = EQequalizer_read_presets("eq.preset");
    equalizer_auto_presets = EQequalizer_read_presets("eq.auto_preset");

    EQequalizer_create_gtk();
    EQequalizer_gc = gdk_gc_new(EQequalizerwin->window);
    EQequalizer_create_widgets();
}

void EQequalizer_recreate(void)
{
    dock_window_list = g_list_remove(dock_window_list, EQequalizerwin);

    gtk_widget_destroy(EQequalizerwin);
    gtk_widget_destroy(eqskinwin);

    g_list_free(equalizer_presets);
    g_list_free(equalizer_auto_presets);
    g_list_free(EQequalizer_wlist);
    equalizer_presets = equalizer_auto_presets = EQequalizer_wlist = NULL;
    eq_write_config();
    init_gui();
}

void EQequalizer_real_show(void)
{
    /*
     * This function should only be called from the
     * main menu signal handler
     */
    if (!pposition_broken && eqcfg.x != -1 &&
            cfg.save_window_position && cfg.show_wm_decorations)
        dock_set_uposition(EQequalizerwin, eqcfg.x, eqcfg.y);
    gtk_widget_show(EQequalizerwin);
    if (pposition_broken && eqcfg.x != -1 && cfg.save_window_position)
        dock_set_uposition(EQequalizerwin, eqcfg.x, eqcfg.y);
    gtk_widget_set_usize(EQequalizerwin, (eqcfg.shaded ? 275 : xpm_width), (eqcfg.shaded ? 14 : 116));
    gdk_flush();
    EQdraw_equalizer_window(TRUE);
    /* XXX hint_set_always(cfg.always_on_top);
       hint_set_sticky(cfg.sticky);
       hint_set_skip_winlist(EQequalizerwin);*/
}

void EQequalizer_real_hide(void)
{
    /*
     * This function should only be called from the
     * main menu signal handler
     */
    gtk_widget_hide(EQequalizerwin);
}

static EqualizerPreset *EQequalizer_find_preset(GList * list, gchar * name)
{
    GList *node = list;
    EqualizerPreset *preset;

    while (node)
    {
        preset = node->data;
        if (!strcasecmp(preset->name, name))
            return preset;
        node = g_list_next(node);
    }
    return NULL;
}

static void EQequalizer_write_preset_file(GList * list, gchar * fname)
{
    gchar *filename, *tmp, *tmp2;
    gint i, chn, p;
    EqualizerPreset *preset;
    ConfigFile *cfgfile;
    GList *node;
    gint channels;
    channels = (eqcfg.use_independent_channels ? 2 : 1);

    cfgfile = xmms_cfg_new();
    p = 0;
    node = list;
    while (node)
    {
        preset = node->data;
        tmp = g_strdup_printf("Preset%d", p++);
        tmp2 = g_strdup_printf("%s_%d_%s", preset->name,
                eqcfg.band_num, eqcfg.use_xmms_original_freqs ? "xmms" : "iso");
        xmms_cfg_write_string(cfgfile, "Presets", tmp, preset->name);
        g_free(tmp);
        for (chn = 0; chn<channels; chn++)
        {
            tmp = g_strdup_printf("Preamp%d", chn);
            xmms_cfg_write_float(cfgfile, tmp2, tmp, preset->preamp[chn]);
            g_free(tmp);
            for (i = 0; i < eqcfg.band_num; i++)
            {
                tmp = g_strdup_printf("Band%d_%d\n", i, chn);
                xmms_cfg_write_float(cfgfile, tmp2, tmp, preset->bands[i][chn]);
                g_free(tmp);
            }
        }
        node = g_list_next(node);
    }
    filename = g_strdup_printf("%s/.xmms/%s", g_get_home_dir(), fname);
    xmms_cfg_write_file(cfgfile, filename);
    xmms_cfg_free(cfgfile);
    g_free(filename);
}

static gboolean EQequalizer_load_preset(GList * list, gchar * name)
{
    EqualizerPreset *preset;
    gint i, chn;
    gint channels;
    channels = (eqcfg.use_independent_channels ? 2 : 1);

    if ((preset = EQequalizer_find_preset(list, name)) != NULL)
    {
        for (chn=0; chn<channels; chn++)
        {
            EQeqslider_set_position(EQequalizer_preamp[chn], preset->preamp[chn]);
            EQequalizer_eq_changed(-1, chn);
            for (i = 0; i < eqcfg.band_num; i++)
            {
                EQeqslider_set_position(EQequalizer_bands[i][chn], preset->bands[i][chn]);
                EQequalizer_eq_changed(i, chn);
            }
        }
        EQdraw_equalizer_window(FALSE);
        return TRUE;
    }
    return FALSE;
}

static GList *EQequalizer_save_preset(GList * list, gchar * name, gchar * fname)
{
    gint i,chn;
    EqualizerPreset *preset;
    gint channels;
    channels = (eqcfg.use_independent_channels ? 2 : 1);

    if (!(preset = EQequalizer_find_preset(list, name)))
    {
        preset = g_malloc(sizeof (EqualizerPreset));
        preset->name = g_strdup(name);
        list = g_list_append(list, preset);
    }

    for (chn = 0; chn<channels; chn++)
    {
        preset->preamp[chn] = EQeqslider_get_position(EQequalizer_preamp[chn]);
        for (i = 0; i < eqcfg.band_num; i++)
            preset->bands[i][chn] = EQeqslider_get_position(EQequalizer_bands[i][chn]);
    }

    EQequalizer_write_preset_file(list, fname);

    return list;
}

static GList *EQequalizer_delete_preset(GList * list, gchar * name, gchar * fname)
{
    EqualizerPreset *preset;
    GList *node;

    if ((preset = EQequalizer_find_preset(list, name)) && (node = g_list_find(list, preset)))
    {
        list = g_list_remove_link(list, node);
        g_free(preset->name);
        g_free(preset);
        g_list_free_1(node);

        EQequalizer_write_preset_file(list, fname);
    }

    return list;
}

static void EQequalizer_read_xmms_preset(ConfigFile *cfgfile)
{
    gfloat val;
    gint i,chn;
    gchar tmp[64];
    gint channels;
    channels = (eqcfg.use_independent_channels ? 2 : 1);
    for (chn = 0; chn<channels; chn++)
    {
        sprintf(tmp, "Preamp%d_%d%s", chn,
                eqcfg.band_num, eqcfg.use_xmms_original_freqs ? "o" : "");
        if (xmms_cfg_read_float(cfgfile, "Equalizer preset", tmp, &val))
            EQeqslider_set_position(EQequalizer_preamp[chn], val);
        EQequalizer_eq_changed(-1, chn);
        for (i = 0; i < eqcfg.band_num; i++)
        {
            sprintf(tmp, "Band%d_%d_%d%s", i,chn,
                    eqcfg.band_num, eqcfg.use_xmms_original_freqs ? "o" : "");
            if (xmms_cfg_read_float(cfgfile, "Equalizer preset", tmp, &val))
                EQeqslider_set_position(EQequalizer_bands[i][chn], val);
            EQequalizer_eq_changed(i, chn);
        }
    }
}

static void EQequalizer_save_ok(GtkWidget * widget, gpointer data)
{
    gchar *text;

    text = gtk_entry_get_text(GTK_ENTRY(EQequalizer_save_entry));
    if (strlen(text) != 0)
        equalizer_presets = EQequalizer_save_preset(equalizer_presets, text, "eq.preset");
    gtk_widget_destroy(EQequalizer_save_window);
}

static void EQequalizer_save_select(GtkCList * clist, gint row, gint column, GdkEventButton * event, gpointer data)
{
    gchar *text;

    gtk_clist_get_text(clist, row, 0, &text);

    gtk_entry_set_text(GTK_ENTRY(EQequalizer_save_entry), text);
    if (event->type == GDK_2BUTTON_PRESS)
        EQequalizer_save_ok(NULL, NULL);

}

static void EQequalizer_load_ok(GtkWidget * widget, gpointer data)
{
    gchar *text;
    GtkCList *clist = GTK_CLIST(data);

    if (clist && clist->selection)
    {
        gtk_clist_get_text(clist, GPOINTER_TO_INT(clist->selection->data), 0, &text);
        EQequalizer_load_preset(equalizer_presets, text);
    }
    gtk_widget_destroy(EQequalizer_load_window);
}

static void EQequalizer_load_select(GtkCList * widget, gint row, gint column, GdkEventButton * event, gpointer data)
{
    if (event->type == GDK_2BUTTON_PRESS)
        EQequalizer_load_ok(NULL, widget);
}

static void EQequalizer_delete_delete(GtkWidget * widget, gpointer data)
{
    gchar *text;
    GList *list, *next;
    GtkCList *clist = GTK_CLIST(data);

    g_return_if_fail(clist != NULL);

    list = clist->selection;
    gtk_clist_freeze(clist);
    while (list)
    {
        next = g_list_next(list);
        gtk_clist_get_text(clist, GPOINTER_TO_INT(list->data), 0, &text);
        equalizer_presets = EQequalizer_delete_preset(equalizer_presets,
                text, "eq.preset");
        gtk_clist_remove(clist, GPOINTER_TO_INT(list->data));
        list = next;
    }
    gtk_clist_thaw(clist);
}

static void EQequalizer_save_auto_ok(GtkWidget * widget, gpointer data)
{
    gchar *text;

    text = gtk_entry_get_text(GTK_ENTRY(EQequalizer_save_auto_entry));
    if (strlen(text) != 0)
        equalizer_auto_presets = EQequalizer_save_preset(equalizer_auto_presets, text, "eq.auto_preset");
    gtk_widget_destroy(EQequalizer_save_auto_window);
}

static void EQequalizer_save_auto_select(GtkCList * clist, gint row, gint column, GdkEventButton * event, gpointer data)
{
    gchar *text;

    gtk_clist_get_text(clist, row, 0, &text);

    gtk_entry_set_text(GTK_ENTRY(EQequalizer_save_auto_entry), text);
    if (event->type == GDK_2BUTTON_PRESS)
        EQequalizer_save_auto_ok(NULL, NULL);

}

static void EQequalizer_load_auto_ok(GtkWidget * widget, gpointer data)
{
    gchar *text;
    GtkCList *clist = GTK_CLIST(data);

    if (clist && clist->selection)
    {
        gtk_clist_get_text(clist, GPOINTER_TO_INT(clist->selection->data), 0, &text);
        EQequalizer_load_preset(equalizer_auto_presets, text);
    }
    gtk_widget_destroy(EQequalizer_load_auto_window);
}

static void EQequalizer_load_auto_select(GtkWidget * widget, gint row, gint column, GdkEventButton * event, gpointer data)
{
    if (event->type == GDK_2BUTTON_PRESS)
        EQequalizer_load_auto_ok(NULL, widget);
}

static void EQequalizer_delete_auto_delete(GtkWidget * widget, gpointer data)
{
    gchar *text;
    GList *list, *next;
    GtkCList *clist = GTK_CLIST(data);

    g_return_if_fail(clist != NULL);

    list = clist->selection;
    gtk_clist_freeze(clist);
    while (list)
    {
        next = g_list_next(list);
        gtk_clist_get_text(clist, GPOINTER_TO_INT(list->data), 0, &text);
        equalizer_auto_presets = EQequalizer_delete_preset(equalizer_auto_presets, text, "eq.auto_preset");
        gtk_clist_remove(clist, GPOINTER_TO_INT(list->data));
        list = next;
    }
    gtk_clist_thaw(clist);
}

static void EQequalizer_load_filesel_ok(GtkWidget * w, GtkFileSelection * filesel)
{
    gchar *filename;
    ConfigFile *cfgfile;

    if (util_filebrowser_is_dir(filesel))
        return;

    filename = gtk_file_selection_get_filename(filesel);

    if ((cfgfile = xmms_cfg_open_file(filename)) != NULL)
    {
        EQequalizer_read_xmms_preset(cfgfile);
        xmms_cfg_free(cfgfile);
    }
    gtk_widget_destroy(GTK_WIDGET(filesel));
}

static void EQequalizer_save_filesel_ok(GtkWidget * w, GtkFileSelection * filesel)
{
    gchar *filename;
    ConfigFile *cfgfile;
    gint i,chn;
    gchar tmp[64];
    gint channels;
    channels = (eqcfg.use_independent_channels ? 2 : 1);

    if (util_filebrowser_is_dir(filesel))
        return;

    filename = gtk_file_selection_get_filename(filesel);

    cfgfile = xmms_cfg_new();
    for (chn=0; chn<channels; chn++)
    {
        sprintf(tmp, "Preamp%d_%d%s", chn,
                eqcfg.band_num, eqcfg.use_xmms_original_freqs ? "o" : "");
        xmms_cfg_write_float(cfgfile, "Equalizer preset", tmp, EQeqslider_get_position(EQequalizer_preamp[chn]));
        /* XXX */
        for (i = 0; i < eqcfg.band_num; i++)
        {
            sprintf(tmp, "Band%d_%d_%d%s", i,chn,
                    eqcfg.band_num, eqcfg.use_xmms_original_freqs ? "o" : "");
            xmms_cfg_write_float(cfgfile, "Equalizer preset", tmp,
                    EQeqslider_get_position(EQequalizer_bands[i][chn]));
        }
    }
    xmms_cfg_write_file(cfgfile, filename);
    xmms_cfg_free(cfgfile);
    gtk_widget_destroy(GTK_WIDGET(filesel));
}

static gint EQequalizer_list_sort_func(GtkCList * clist, gconstpointer ptr1, gconstpointer ptr2)
{
    GtkCListRow *row1 = (GtkCListRow *) ptr1;
    GtkCListRow *row2 = (GtkCListRow *) ptr2;

    return strcasecmp(GTK_CELL_TEXT(row1->cell[clist->sort_column])->text, GTK_CELL_TEXT(row2->cell[clist->sort_column])->text);
}

static GtkWidget *EQequalizer_create_list_window(GList * preset_list,
        gchar * title,
        GtkWidget ** window,
        GtkSelectionMode sel_mode,
        GtkWidget ** entry,
        gchar * btn1_caption,
        gchar * btn2_caption,
        GtkSignalFunc btn1_func,
        GtkSignalFunc select_row_func)
{
    GtkWidget *vbox, *scrolled_window, *bbox, *btn1, *btn2, *clist;
    char *preset_text[1];
    GList *node;

    *window = gtk_window_new(GTK_WINDOW_DIALOG);
    gtk_signal_connect(GTK_OBJECT(*window), "destroy",
            GTK_SIGNAL_FUNC(gtk_widget_destroyed), window);
    gtk_window_set_transient_for(GTK_WINDOW(*window), GTK_WINDOW(EQequalizerwin));
    gtk_window_set_position(GTK_WINDOW(*window), GTK_WIN_POS_MOUSE);
    gtk_window_set_title(GTK_WINDOW(*window), title);

    gtk_widget_set_usize(*window, 350, 300);
    gtk_container_set_border_width(GTK_CONTAINER(*window), 10);

    vbox = gtk_vbox_new(FALSE, 10);
    gtk_container_add(GTK_CONTAINER(*window), vbox);

    scrolled_window = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
            GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);

    preset_text[0] = "Presets";
    clist = gtk_clist_new_with_titles(1, preset_text);
    if (select_row_func)
        gtk_signal_connect(GTK_OBJECT(clist), "select_row",
                GTK_SIGNAL_FUNC(select_row_func), NULL);
    gtk_clist_column_titles_passive(GTK_CLIST(clist));
    gtk_clist_set_selection_mode(GTK_CLIST(clist), sel_mode);

    node = preset_list;
    while (node)
    {
        gtk_clist_append(GTK_CLIST(clist),
                &((EqualizerPreset *) node->data)->name);
        node = node->next;
    }
    gtk_clist_set_compare_func(GTK_CLIST(clist), EQequalizer_list_sort_func);
    gtk_clist_sort(GTK_CLIST(clist));

    gtk_container_add(GTK_CONTAINER(scrolled_window), clist);
    gtk_widget_show(clist);
    gtk_widget_show(scrolled_window);

    gtk_box_pack_start(GTK_BOX(vbox), scrolled_window, TRUE, TRUE, 0);

    if (entry)
    {
        *entry = gtk_entry_new();
        gtk_signal_connect(GTK_OBJECT(*entry), "activate",
                GTK_SIGNAL_FUNC(btn1_func), NULL);
        gtk_box_pack_start(GTK_BOX(vbox), *entry, FALSE, FALSE, 0);
        gtk_widget_show(*entry);
    }

    bbox = gtk_hbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(bbox), 5);

    btn1 = gtk_button_new_with_label(btn1_caption);
    gtk_signal_connect(GTK_OBJECT(btn1), "clicked",
            GTK_SIGNAL_FUNC(btn1_func), clist);
    GTK_WIDGET_SET_FLAGS(btn1, GTK_CAN_DEFAULT);
    gtk_box_pack_start(GTK_BOX(bbox), btn1, TRUE, TRUE, 0);
    gtk_widget_show(btn1);

    btn2 = gtk_button_new_with_label(btn2_caption);
    gtk_signal_connect_object(GTK_OBJECT(btn2), "clicked",
            GTK_SIGNAL_FUNC(gtk_widget_destroy),
            GTK_OBJECT(*window));
    GTK_WIDGET_SET_FLAGS(btn2, GTK_CAN_DEFAULT);
    gtk_box_pack_start(GTK_BOX(bbox), btn2, TRUE, TRUE, 0);
    gtk_widget_show(btn2);

    gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);
    gtk_widget_show(bbox);

    gtk_widget_grab_default(btn1);
    gtk_widget_show(vbox);
    gtk_widget_show(*window);
    return *window;
}

void EQequalizer_presets_menu_cb(gpointer cb_data, guint action, GtkWidget * w)
{
    switch (action)
    {
        case EQUALIZER_PRESETS_LOAD_PRESET:
            if (!EQequalizer_load_window)
                EQequalizer_create_list_window(equalizer_presets, "Load preset", &EQequalizer_load_window, GTK_SELECTION_SINGLE, NULL, "Ok", "Cancel", EQequalizer_load_ok, EQequalizer_load_select);
            else
                gdk_window_raise(EQequalizer_load_window->window);
            break;
        case EQUALIZER_PRESETS_LOAD_AUTOPRESET:
            if (!EQequalizer_load_auto_window)
                EQequalizer_create_list_window(equalizer_auto_presets, "Load auto-preset", &EQequalizer_load_auto_window, GTK_SELECTION_SINGLE, NULL, "Ok", "Cancel", EQequalizer_load_auto_ok, EQequalizer_load_auto_select);
            else
                gdk_window_raise(EQequalizer_load_auto_window->window);
            break;
        case EQUALIZER_PRESETS_LOAD_DEFAULT:
            EQequalizer_load_preset(equalizer_presets, "Default");
            break;
        case EQUALIZER_PRESETS_LOAD_ZERO:
            {
                gint i,chn;
                gint channels;
                channels = (eqcfg.use_independent_channels ? 2 : 1);

                for (chn=0; chn<channels;chn++)
                {
                    EQeqslider_set_position(EQequalizer_preamp[chn], 0.0);
                    EQequalizer_eq_changed(-1, chn);
                    for (i = 0; i < eqcfg.band_num; i++)
                    {
                        EQeqslider_set_position(EQequalizer_bands[i][chn], 0.0);
                        EQequalizer_eq_changed(i, chn);
                    }
                }
                break;
            }
        case EQUALIZER_PRESETS_LOAD_FROM_FILE:
            {
                static GtkWidget *load_filesel;
                if (load_filesel != NULL)
                    break;

                load_filesel = gtk_file_selection_new("Load equalizer preset");
                gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(load_filesel)->ok_button), "clicked", GTK_SIGNAL_FUNC(EQequalizer_load_filesel_ok), load_filesel);
                gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(load_filesel)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(load_filesel));
                gtk_signal_connect(GTK_OBJECT(load_filesel), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &load_filesel);
                gtk_widget_show(load_filesel);

                break;
            }

        case EQUALIZER_PRESETS_SAVE_PRESET:
            if (!EQequalizer_save_window)
                EQequalizer_create_list_window(equalizer_presets, "Save preset", &EQequalizer_save_window, GTK_SELECTION_SINGLE, &EQequalizer_save_entry, "Ok", "Cancel", EQequalizer_save_ok, EQequalizer_save_select);
            else
                gdk_window_raise(EQequalizer_save_window->window);
            break;
        case EQUALIZER_PRESETS_SAVE_AUTOPRESET:
            {
                gchar *name;

                if (!EQequalizer_save_auto_window)
                    EQequalizer_create_list_window(equalizer_auto_presets, "Save auto-preset", &EQequalizer_save_auto_window, GTK_SELECTION_SINGLE, &EQequalizer_save_auto_entry, "Ok", "Cancel", EQequalizer_save_auto_ok, EQequalizer_save_auto_select);
                else
                    gdk_window_raise(EQequalizer_save_auto_window->window);
                if ((name = playlist_get_filename(get_playlist_position())) != NULL)
                {
                    gtk_entry_set_text(GTK_ENTRY(EQequalizer_save_auto_entry), g_basename(name));
                    g_free(name);
                }
                break;
            }
        case EQUALIZER_PRESETS_SAVE_DEFAULT:
            equalizer_presets = EQequalizer_save_preset(equalizer_presets, "Default", "eq.preset");
            break;
        case EQUALIZER_PRESETS_SAVE_TO_FILE:
            {
                static GtkWidget *EQequalizer_save_filesel;
                gchar *songname;

                if (EQequalizer_save_filesel != NULL)
                    break;

                EQequalizer_save_filesel = gtk_file_selection_new("Save equalizer preset");

                if ((songname = playlist_get_filename(get_playlist_position())) != NULL)
                {
                    gchar *eqname = g_strdup_printf("%s.%s", songname, eqcfg.eqpreset_extension);
                    g_free(songname);
                    gtk_file_selection_set_filename(GTK_FILE_SELECTION(EQequalizer_save_filesel), eqname);
                    g_free(eqname);
                }

                gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(EQequalizer_save_filesel)->ok_button), "clicked", GTK_SIGNAL_FUNC(EQequalizer_save_filesel_ok), EQequalizer_save_filesel);
                gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(EQequalizer_save_filesel)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(EQequalizer_save_filesel));
                gtk_signal_connect(GTK_OBJECT(EQequalizer_save_filesel), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &EQequalizer_save_filesel);
                gtk_widget_show(EQequalizer_save_filesel);

                break;
            }
        case EQUALIZER_PRESETS_DELETE_PRESET:
            if (!EQequalizer_delete_window)
                EQequalizer_create_list_window(equalizer_presets, "Delete preset", &EQequalizer_delete_window, GTK_SELECTION_EXTENDED, NULL, "Delete", "Close", EQequalizer_delete_delete, NULL);
            break;
        case EQUALIZER_PRESETS_DELETE_AUTOPRESET:
            if (!EQequalizer_delete_auto_window)
                EQequalizer_create_list_window(equalizer_auto_presets, "Delete auto-preset", &EQequalizer_delete_auto_window, GTK_SELECTION_EXTENDED, NULL, "Delete", "Close", EQequalizer_delete_auto_delete, NULL);
            break;
        case EQUALIZER_PRESETS_CONFIGURE:
            if (!EQequalizer_configure_window)
            {
                EQequalizer_configure_window =
                    EQequalizer_create_conf_window();
                gtk_signal_connect(GTK_OBJECT(EQequalizer_configure_window), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &EQequalizer_configure_window);
            }

            break;
        case EQUALIZER_SKIN_BROWSER:
            eq_show_skin_window();
            break;
        case EQUALIZER_PRESETS_LOAD_LEFT_ON_RIGHT:
            EQequalizer_copy_bands(EQ_RIGHT);
            break;
        case EQUALIZER_PRESETS_LOAD_RIGHT_ON_LEFT:
            EQequalizer_copy_bands(EQ_LEFT);
            break;
    }
    EQdraw_equalizer_window(FALSE);
}

void EQequalizer_system_menu_cb(gpointer cb_data, guint action, GtkWidget * w)
{
    switch (action)
    {
        case EQUALIZER_ABOUT:
            about();
            break;
        case EQUALIZER_PREFERENCES:
            eq_configuration_window();
            break;
        case EQUALIZER_SKIN_BROWSER:
            eq_show_skin_window();
            break;
        case EQUALIZER_EXIT:
            EQequalizer_close_cb();
            break;
    }
    EQdraw_equalizer_window(FALSE);
}

void EQequalizer_load_auto_preset(gchar * filename)
{
    gchar *presetfilename, *directory;
    ConfigFile *cfgfile;

    if (!eqcfg.equalizer_autoload)
        return;

    g_return_if_fail(filename != NULL);

    presetfilename = g_strdup_printf("%s.%s", filename, eqcfg.eqpreset_extension);

    /*
     * First try to find a per file preset file
     */
    if (strlen(eqcfg.eqpreset_extension) > 0 &&
            (cfgfile = xmms_cfg_open_file(presetfilename)) != NULL)
    {
        g_free(presetfilename);
        EQequalizer_read_xmms_preset(cfgfile);
        xmms_cfg_free(cfgfile);
        return;
    }

    directory = g_dirname(filename);
    presetfilename = g_strdup_printf("%s/%s", directory, eqcfg.eqpreset_default_file);
    g_free(directory);

    /*
     * Try to find a per directory preset file
     */
    if (strlen(eqcfg.eqpreset_default_file) > 0 &&
            (cfgfile = xmms_cfg_open_file(presetfilename)) != NULL)
    {
        EQequalizer_read_xmms_preset(cfgfile);
        xmms_cfg_free(cfgfile);
    }
    /*
     * Fall back to the oldstyle auto presets
     */
    else if (!EQequalizer_load_preset(equalizer_auto_presets, g_basename(filename)))
        EQequalizer_load_preset(equalizer_presets, "Default");
    g_free(presetfilename);
}

void EQequalizer_set_preamp(gfloat *preamp, gint chn)
{
    EQeqslider_set_position(EQequalizer_preamp[chn], preamp[chn]);
    EQequalizer_eq_changed(-1, chn);
}

void EQequalizer_set_band(gint band, gfloat value, gint chn)
{
    g_return_if_fail(band >= 0 && band < eqcfg.band_num);
    EQeqslider_set_position(EQequalizer_bands[band][chn], value);
}

gfloat EQequalizer_get_preamp(gint chn)
{
    return EQeqslider_get_position(EQequalizer_preamp[chn]);
}

gfloat EQequalizer_get_band(gint band, gint chn)
{
    g_return_val_if_fail(band >= 0 && band < eqcfg.band_num, 0);
    return EQeqslider_get_position(EQequalizer_bands[band][chn]);
}

static void EQequalizer_conf_apply_changes(void)
{
    gchar *start;

    g_free(eqcfg.eqpreset_default_file);
    g_free(eqcfg.eqpreset_extension);

    eqcfg.eqpreset_default_file = gtk_editable_get_chars(GTK_EDITABLE(eqconfwin_options_eqdf_entry), 0, -1);
    eqcfg.eqpreset_extension = gtk_editable_get_chars(GTK_EDITABLE(eqconfwin_options_eqef_entry), 0, -1);

    g_strstrip(eqcfg.eqpreset_default_file);
    start = eqcfg.eqpreset_default_file;
    /* Strip leading '.' */
    while (*start == '.')
        start++;
    if (start != eqcfg.eqpreset_default_file)
        g_memmove(eqcfg.eqpreset_default_file, start, strlen(start) + 1);

    g_strstrip(eqcfg.eqpreset_extension);
    start = eqcfg.eqpreset_extension;
    while (*start == '.')
        start++;
    if (start != eqcfg.eqpreset_extension)
        g_memmove(eqcfg.eqpreset_extension, start, strlen(start) + 1);
}

static void EQequalizer_conf_ok_cb(GtkWidget * w, gpointer data)
{
    EQequalizer_conf_apply_changes();
    gtk_widget_destroy(EQequalizer_configure_window);
}

static void EQequalizer_conf_apply_cb(GtkWidget * w, gpointer data)
{
    EQequalizer_conf_apply_changes();
}

GtkWidget * EQequalizer_create_conf_window(void)
{
    GtkWidget *window, *notebook;
    GtkWidget *options_eqdf_box;
    GtkWidget *options_eqe_box;
    GtkWidget *options_eqdf;
    GtkWidget *options_eqe;
    GtkWidget *options_frame, *options_vbox;
    GtkWidget *options_table;
    GtkWidget *vbox, *hbox, *instructions;
    GtkWidget *ok, *cancel, *apply;

    window = gtk_window_new(GTK_WINDOW_DIALOG);
    gtk_window_set_title(GTK_WINDOW(window), "Configure Equalizer Presets");
    gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);
    gtk_window_set_transient_for(GTK_WINDOW(window), GTK_WINDOW(mainwin));
    gtk_container_border_width(GTK_CONTAINER(window), 10);

    vbox = gtk_vbox_new(FALSE, 10);
    gtk_container_add(GTK_CONTAINER(window), vbox);
    notebook = gtk_notebook_new();
    gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);

    options_vbox = gtk_vbox_new(FALSE, 10);
    gtk_container_border_width(GTK_CONTAINER(options_vbox), 5);
    options_frame = gtk_frame_new("Options");
    gtk_box_pack_start(GTK_BOX(options_vbox), options_frame, FALSE, FALSE, 0);
    gtk_container_border_width(GTK_CONTAINER(options_frame), 0);
    options_table = gtk_table_new(1, 2, FALSE);
    gtk_table_set_col_spacings(GTK_TABLE(options_table), 10);
    gtk_container_add(GTK_CONTAINER(options_frame), options_table);
    gtk_container_border_width(GTK_CONTAINER(options_table), 5);

    options_eqdf_box = gtk_hbox_new(FALSE, 5);
    options_eqdf = gtk_label_new("Directory preset file:");
    gtk_box_pack_start(GTK_BOX(options_eqdf_box), options_eqdf, FALSE, FALSE, 0);
    eqconfwin_options_eqdf_entry = gtk_entry_new_with_max_length(40);
    gtk_widget_set_usize(eqconfwin_options_eqdf_entry, 115, -1);
    gtk_box_pack_start(GTK_BOX(options_eqdf_box), eqconfwin_options_eqdf_entry, FALSE, FALSE, 0);
    gtk_table_attach_defaults(GTK_TABLE(options_table), options_eqdf_box, 0, 1, 0, 1);

    options_eqe_box = gtk_hbox_new(FALSE, 5);
    options_eqe = gtk_label_new("File preset extension:");
    gtk_box_pack_start(GTK_BOX(options_eqe_box), options_eqe, FALSE, FALSE, 0);
    eqconfwin_options_eqef_entry = gtk_entry_new_with_max_length(20);
    gtk_widget_set_usize(eqconfwin_options_eqef_entry, 55, -1);
    gtk_box_pack_start(GTK_BOX(options_eqe_box), eqconfwin_options_eqef_entry, FALSE, FALSE, 0);
    gtk_table_attach_defaults(GTK_TABLE(options_table), options_eqe_box, 1, 2, 0, 1);
    instructions =
        gtk_label_new("If \"Auto\" is enabled on the equalizer, xmms "
                "will try to load equalizer presets like this:\n"
                "1: Look for a preset file in the directory of the "
                "file we are about to play.\n"
                "2: Look for a directory preset file in the same "
                "directory.\n"
                "3: Look for a preset saved with the "
                "\"auto-load\" feature\n"
                "4: Finally, try to load the \"default\" preset");
    gtk_label_set_justify(GTK_LABEL(instructions), GTK_JUSTIFY_LEFT);
    gtk_box_pack_start_defaults(GTK_BOX(options_vbox), instructions);

    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), options_vbox, gtk_label_new("Options"));

    /*
     * Ok, Cancel & Apply
     */

    hbox = gtk_hbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 5);

    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

    ok = gtk_button_new_with_label("Ok");
    gtk_signal_connect(GTK_OBJECT(ok), "clicked", GTK_SIGNAL_FUNC(EQequalizer_conf_ok_cb), NULL);
    GTK_WIDGET_SET_FLAGS(ok, GTK_CAN_DEFAULT);
    gtk_box_pack_start(GTK_BOX(hbox), ok, TRUE, TRUE, 0);
    cancel = gtk_button_new_with_label("Cancel");
    gtk_signal_connect_object(GTK_OBJECT(cancel), "clicked",
            GTK_SIGNAL_FUNC(gtk_widget_destroy),
            GTK_OBJECT(window));
    GTK_WIDGET_SET_FLAGS(cancel, GTK_CAN_DEFAULT);
    gtk_box_pack_start(GTK_BOX(hbox), cancel, TRUE, TRUE, 0);
    apply = gtk_button_new_with_label("Apply");
    gtk_signal_connect(GTK_OBJECT(apply), "clicked", GTK_SIGNAL_FUNC(EQequalizer_conf_apply_cb), NULL);
    GTK_WIDGET_SET_FLAGS(apply, GTK_CAN_DEFAULT);
    gtk_box_pack_start(GTK_BOX(hbox), apply, TRUE, TRUE, 0);

    gtk_entry_set_text(GTK_ENTRY(eqconfwin_options_eqdf_entry), eqcfg.eqpreset_default_file);
    gtk_entry_set_text(GTK_ENTRY(eqconfwin_options_eqef_entry), eqcfg.eqpreset_extension);

    gtk_widget_show_all(window);
    gtk_widget_grab_default(ok);

    return window;
}

void eq_apply_skin(char *path)
{
    eqcfg.skin = path;
    eq_write_config();

    gdk_pixmap_unref(EQequalizer_bg);
    gdk_pixmap_unref(EQpixmap);
    gdk_pixmap_unref(EQquick);
    gdk_pixmap_unref(EQtitle);
    gdk_pixmap_unref(EQshade);

    EQequalizer_bg = eq_create_skin_bg_pixmap(EQequalizerwin->window);
    EQpixmap = eq_skin_controls();
    EQquick = eq_skin_quick();
    EQtitle = eq_skin_title();
    EQshade = eq_skin_shade();

    g_list_free(EQequalizer_wlist);
    EQequalizer_wlist = NULL;

    EQequalizer_create_widgets();

    EQdraw_equalizer_window(TRUE);
}
