//gnaural_dbus_server.c : Gnaural's low-level D-Bus server.
//To use in a GTK project, just call gnaural_dbus_server_main(), then
//gnaural_dbus_server_cleanup() when done. Obviously you will also
//need to include your data sources here.
//
//currently these remote methods are implemented:
//GetVoicecount ()
//GetVoicetype  (unsigned int Voice)
//GetBeatfreq   (unsigned int Voice)
//GetBasefreq   (unsigned int Voice)
//
//Will add things like: Duration, Progress, filename, etc.

#include <dbus/dbus.h>
#include <glib.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include "gnaural-dbus-server.h"
#include "BinauralBeat.h"

//DBUS_SIGNAL_SERVER_STANDALONE only exists so this can be developed
//without being an existing part of a GTK or GLib project:
//#define DBUS_SIGNAL_SERVER_STANDALONE
//Only needed for gds_dbus_server_cleanup():
DBusConnection *gds_remote_connection = NULL;
int gds_thread_alive = 0;

////////////////////////////////
void GetBeatfreq (DBusMessage * msg, DBusConnection * conn)
{
 DBusMessage *reply;
 DBusMessageIter args;
 unsigned int voice = 0;
 double result = 0;

 //read the arguments
 if (!dbus_message_iter_init (msg, &args))
 {
  BB_ERROUT ("Message has no arguments");
 }
 else if (DBUS_TYPE_UINT32 != dbus_message_iter_get_arg_type (&args))
 {
  BB_ERROUT ("Argument is not DBUS_TYPE_UINT32");
 }
 else
 {
  dbus_message_iter_get_basic (&args, &voice);
 }

 BB_DBGOUT_INT ("GetBeatfreq called for voice ", voice);

 //know voice, now can get freq:
 if (voice < BB_VoiceCount)
 {
  result = BB_Voice[voice].cur_beatfreq;
 }

 //create a reply from the message
 reply = dbus_message_new_method_return (msg);

 //add an argument to the reply
 dbus_message_iter_init_append (reply, &args);
 if (!dbus_message_iter_append_basic (&args, DBUS_TYPE_DOUBLE, &result))
 {
  BB_ERROUT ("Out Of Memory");
  return;
 }

 //send the reply && flush the connection
 if (!dbus_connection_send (conn, reply, NULL))
 {
  BB_ERROUT ("Out Of Memory");
  return;
 }
 dbus_connection_flush (conn);

 //free the reply
 dbus_message_unref (reply);
}

////////////////////////////////
void GetBasefreq (DBusMessage * msg, DBusConnection * conn)
{
 DBusMessage *reply;
 DBusMessageIter args;
 unsigned int voice = 0;
 double result = 0;

 //read the arguments
 if (!dbus_message_iter_init (msg, &args))
 {
  BB_ERROUT ("Message has no arguments");
 }
 else if (DBUS_TYPE_UINT32 != dbus_message_iter_get_arg_type (&args))
 {
  BB_ERROUT ("Argument is not DBUS_TYPE_UINT32");
 }
 else
 {
  dbus_message_iter_get_basic (&args, &voice);
 }

 BB_DBGOUT_INT ("GetBasefreq called for voice ", voice);

 //know voice, now can get freq:
 if (voice < BB_VoiceCount)
 {
  result = BB_Voice[voice].cur_basefreq;
 }

 //create a reply from the message
 reply = dbus_message_new_method_return (msg);

 //add an argument to the reply
 dbus_message_iter_init_append (reply, &args);
 if (!dbus_message_iter_append_basic (&args, DBUS_TYPE_DOUBLE, &result))
 {
  BB_ERROUT ("Out Of Memory");
  return;
 }

 //send the reply && flush the connection
 if (!dbus_connection_send (conn, reply, NULL))
 {
  BB_ERROUT ("Out Of Memory");
  return;
 }
 dbus_connection_flush (conn);

 //free the reply
 dbus_message_unref (reply);
}

////////////////////////////////
void GetVoicetype (DBusMessage * msg, DBusConnection * conn)
{
 DBusMessage *reply;
 DBusMessageIter args;
 unsigned int voice = 0;
 unsigned int result = 0;

 //read the arguments
 if (!dbus_message_iter_init (msg, &args))
 {
  BB_ERROUT ("Message has no arguments");
 }
 else if (DBUS_TYPE_UINT32 != dbus_message_iter_get_arg_type (&args))
 {
  BB_ERROUT ("Argument is not DBUS_TYPE_UINT32");
 }
 else
 {
  dbus_message_iter_get_basic (&args, &voice);
 }

 BB_DBGOUT_INT ("GetVoicetype called for voice ", voice);

 //know voice, now can get freq:
 if (voice < BB_VoiceCount)
 {
  result = BB_Voice[voice].type;
 }

 //create a reply from the message
 reply = dbus_message_new_method_return (msg);

 //add an argument to the reply
 dbus_message_iter_init_append (reply, &args);
 if (!dbus_message_iter_append_basic (&args, DBUS_TYPE_UINT32, &result))
 {
  BB_ERROUT ("Out Of Memory");
  return;
 }

 //send the reply && flush the connection
 if (!dbus_connection_send (conn, reply, NULL))
 {
  BB_ERROUT ("Out Of Memory");
  return;
 }
 dbus_connection_flush (conn);

 //free the reply
 dbus_message_unref (reply);
}

////////////////////////////////
//takes no args, returns a unsigned int
void GetVoicecount (DBusMessage * msg, DBusConnection * conn)
{
 DBusMessage *reply;
 DBusMessageIter args;
 unsigned int result = 0;

 BB_DBGOUT ("GetVoicecount called");

 result = BB_VoiceCount;

 //create a reply from the message
 reply = dbus_message_new_method_return (msg);

 //add an argument to the reply
 dbus_message_iter_init_append (reply, &args);
 if (!dbus_message_iter_append_basic (&args, DBUS_TYPE_UINT32, &result))
 {
  BB_ERROUT ("Out Of Memory");
  return;
 }

 //send the reply && flush the connection
 if (!dbus_connection_send (conn, reply, NULL))
 {
  BB_ERROUT ("Out Of Memory");
  return;
 }
 dbus_connection_flush (conn);

 //free the reply
 dbus_message_unref (reply);
}

/////////////////////////////
void gds_show_message_info (DBusMessage * msg)
{
 //spit some info out for illumination:
 BB_DBGOUT_STR ("Interface: ", dbus_message_get_interface (msg));
 BB_DBGOUT_STR ("Object Path: ", dbus_message_get_path (msg));
 BB_DBGOUT_STR ("Member: ", dbus_message_get_member (msg));
 BB_DBGOUT_STR ("Destination: ", dbus_message_get_destination (msg));
 BB_DBGOUT_STR ("Sender: ", dbus_message_get_sender (msg));
 BB_DBGOUT_STR ("Signature: ", dbus_message_get_signature (msg));
 BB_DBGOUT_STR ("Type: ",
		dbus_message_type_to_string (dbus_message_get_type (msg)));
}

////////////////////////////////
//this is the actual "server" part that waits in a loop for incoming calls:
void gds_wait_for_methodcalls (DBusConnection * connection)
{
 DBusMessage *msg;

 if (NULL == connection)
 {
  g_thread_exit (0);
 }

 while (0 != gds_thread_alive)
 {
  //drain queue so dbus_connection_read_write doesn't block forever on backups:
  while (DBUS_DISPATCH_DATA_REMAINS ==
	 dbus_connection_get_dispatch_status (connection))
  {
   msg = dbus_connection_pop_message (connection);
   BB_DBGOUT ("Found undispatched data:");
   gds_show_message_info (msg);
  }
  //dbus_connection_read_write: block until a message arrives, 
  //then drain and process it with dbus_connection_pop_message:
  //BB_DBGOUT ("Waiting at dbus_connection_read_write...");
  dbus_connection_read_write (connection, -1);
  //dbus_connection_pop_message:
  //Returns the first-received message from the incoming
  //message queue, removing it from the queue.
  //If the queue is empty, returns NULL:
  if (NULL != (msg = dbus_connection_pop_message (connection)))
  {
   //see if it is a method call with this interface & method:
   if (dbus_message_is_method_call (msg,	//DBusMessage * msg
				    GNAURAL_DBUS_INTERFACE,	//char * interface
				    "GetBeatfreq"))	//char * method
   {
    GetBeatfreq (msg, connection);
   }
   else
    //see if it is a method call with this interface & method:
    if (dbus_message_is_method_call (msg,	//DBusMessage * msg
				     GNAURAL_DBUS_INTERFACE,	//char * interface
				     "GetBasefreq"))	//char * method
   {
    GetBasefreq (msg, connection);
   }
   else
    //see if it is a method call with this interface & method:
    if (dbus_message_is_method_call (msg,	//DBusMessage * msg
				     GNAURAL_DBUS_INTERFACE,	//char * interface
				     "GetVoicecount"))	//char * method
   {
    GetVoicecount (msg, connection);
   }
   else
    //see if it is a method call with this interface & method:
    if (dbus_message_is_method_call (msg,	//DBusMessage * msg
				     GNAURAL_DBUS_INTERFACE,	//char * interface
				     "GetVoicetype"))	//char * method
   {
    GetVoicetype (msg, connection);
   }
   else
   {
    BB_DBGOUT ("Unhandled message:");
   }
   gds_show_message_info (msg);
   //free the message
   dbus_message_unref (msg);
   msg = NULL;
  }	//else  BB_DBGOUT ("No message");
 }
 g_thread_exit (0);
}

////////////////////////////////
//connects to session bus and exposes a method call
//returns 0 on success, non-0 on error:
DBusConnection *gds_init_connection (void)
{
 DBusConnection *bus;
 DBusError err;
 int ret;

 //initialise the error
 dbus_error_init (&err);

 //connect to the bus and check for errors
 bus = dbus_bus_get (DBUS_BUS_SESSION, &err);
 if (dbus_error_is_set (&err))
 {
  BB_ERROUT ("Connection Error:");
  BB_ERROUT (err.message);
  dbus_error_free (&err);
  return NULL;
 }
 if (NULL == bus)
 {
  BB_ERROUT ("Connection NULL:");
  BB_ERROUT (err.message);
  return NULL;
 }
 //dbus_bus_request_name:
 //Asks the bus to assign the given name to this 
 //connection by invoking the RequestName method on the bus.
 //request our name on the bus and check for errors
 ret =
  dbus_bus_request_name (bus, GNAURAL_DBUS_SERVER,
			 DBUS_NAME_FLAG_REPLACE_EXISTING, &err);
 if (dbus_error_is_set (&err))
 {
  BB_ERROUT ("Name Error:");
  BB_ERROUT (err.message);
  dbus_error_free (&err);
  return NULL;
 }
 if (DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER != ret)
 {
  BB_ERROUT ("Not Primary Owner");
  return NULL;
 }

 return bus;
}

////////////////////////////////
void gnaural_dbus_server_cleanup ()
{
 BB_DBGOUT ("D-Bus cleaning-up for exit");
#ifdef DBUS_SIGNAL_SERVER_STANDALONE
 gdk_threads_leave ();
#endif
 //don't use on shared bus:
 //dbus_connection_close (gds_remote_connection); 
 if (NULL != gds_remote_connection)
 {
  dbus_connection_unref (gds_remote_connection);
 }
}

////////////////////////////////
//Thread starts here:
void gds_start_dbus (gpointer arg)
{
 DBusConnection *bus = gds_init_connection ();

 gds_thread_alive = 1;
 //nasty global needed for cleanup():
 gds_remote_connection = bus;

 if (NULL == bus)
 {
  BB_ERROUT ("Could connect to sesson D-Bus");
  return;
 }
 else
 {
  BB_DBGOUT ("Successfully connected to session D-Bus");
 }
 gds_wait_for_methodcalls (bus);
 BB_DBGOUT ("D-Bus server thread exiting");
 // dbus_connection_unref (bus);
}

////////////////////////////////
////////////////////////////////
#ifdef DBUS_SIGNAL_SERVER_STANDALONE
int main (int argc, char **argv)
#else
int gnaural_dbus_server_main (void)
#endif
{
 //the whole reason to link to the GTK+ libs here: multiplatform threads
 //to write a GTK+ threaded program, you make these three calls, then when 
 //done call gdk_threads_leave ()
#ifdef DBUS_SIGNAL_SERVER_STANDALONE
 BB_DBGOUT ("Initing threads for standalone");
 if (!g_thread_supported ())
  g_thread_init (NULL);
 gdk_threads_init ();
 gdk_threads_enter ();	//if I don't do this, the Win32 version blocks! The documentation is amazingly vague.

 //in most apps, we'd all this too. But here we never call gtk_main():
 // gtk_init (&argc, &argv);
#endif

 GThread *gds_serverthread = NULL;
 GError *thread_err = NULL;

 if (NULL == (gds_serverthread =
	      g_thread_create ((GThreadFunc) gds_start_dbus, (void *) NULL,
			       TRUE, &thread_err)))
 {
  BB_ERROUT ("g_thread_create failed:");
  BB_ERROUT (thread_err->message);
  g_error_free (thread_err);
  gnaural_dbus_server_cleanup ();
  return 1;
 }
 else
 {
  BB_DBGOUT ("D-Bus server thread successfully created");
 }

#ifdef DBUS_SIGNAL_SERVER_STANDALONE
 BB_DBGOUT ("service running");
 // gtk_main ();
 // while (1)
 {
  sleep (10);
 }
 gnaural_dbus_server_cleanup ();
 exit (0);
#endif

 return 0;
}
