/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

////////////////////////////////////////////////////////////
// main.c, by Bret Logan (c) 2007
// Comprises Gnaural2 by uniting ScheduleGUI.c, ScheduleXML.c,
// BinauralBeat.c, and callbacks.c This is where any code should go
// that ties together the different modules (ScheduleGUI, ScheduleXML,
// BinauralBeat) and the GUI; they should not require any reference
// to each other directly.
////////////////////////////////////////////////////////////

//TODO:
// - get D-Bus working (#define'd off now via GNAURAL_DBUS)
// - let user drag progressbar's length to desired timepoint.
// - implement MP3 functionality for files opened at command line; can't currently find file so opens default
// - BUG: when writing WAV to stdout (buggy CMDLINE generally), I pump out so much DBG info that the stream is polluted!
// - Make properties box check for all properties whether all DPs have them or not
// - add balance control to Properties box (no backdoor way to do it now via properties volumes; in fact, they ruin any preexisting balance relationships)
// - Make GUI "Schedule Info" text NOT center justified
//BUGS:
// - Add New Voice sometime checks the Mute box (but isn't even muted)

#undef GNAURAL_DBUS

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>	//needed for memcpy
#include <unistd.h>	//needed for path/file setup: getcwd, getopt
#include <errno.h>	//needed for path/file setup
#include <signal.h>	//needed for SIGINT
#include <sys/stat.h>	//needed for path/file setup: main_testfileforexistance
#include <stdlib.h>
#include <math.h>	//for fabs()
#include <ctype.h>	//for toupper
#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>	//needed for keyboard translations
#include <glade/glade.h>
#include <sndfile.h>

#include "callbacks.h"
#include "support.h"
#include "ScheduleGUI.h"
#include "ScheduleXML.h"
#include "BinauralBeat.h"
#include "src/portaudio.h"
#include "main.h"
#include "main_ArrayConstants.h"
#include "exportAudio.h"
#ifdef GNAURAL_DBUS
#include "gnaural-dbus-server.h"
#endif

#ifdef GNAURAL_WIN32
#include <windows.h>	//for win32
#include <mmsystem.h>	//for sound on win32
#include <io.h>	//for binary-mode stdout on Win32
#include <fcntl.h>	//for binary-mode stdout on Win32
#include <process.h>	//for _spawn on Win32
#endif

#ifndef VERSION
#define VERSION "0.1.20070707"
#endif

//critical assignment for installation stage:
#define GLADE_FILE_NAME "gnaural2.glade"
#define GLADE_FILE PACKAGE_DATA_DIR"/"PACKAGE"/"GLADE_FILE_NAME

//Increments this whenever a change to file format obsoletes old gnaural formats:
//#define main_XMLFILEVERSION "1.20060924"
//#define main_XMLFILEVERSION "1.20070124"
#define main_VERSION_GNAURAL_XMLFILE "1.20070707"
#define main_VERSION_GNAURAL VERSION
#define GNAURAL_GUI_UPDATEINTERVAL 128	//64
enum
{
 MAIN_DND_TARGET_STRING,
 MAIN_DND_TARGET_URL
};

//START Globals for Port Audio handling:
PortAudioStream *mainPA_SoundStream = NULL;	//this will equal NULL if the sound system wasn't detected, so use to check
PaDeviceID mainPA_SoundDevice = GNAURAL_USEDEFAULTSOUNDDEVICE;

//START Need two globals for AudioFileData.
//All Audio files get stored here, and are indexed by their filename
typedef struct main_AFData_type
{
 int *PCM_data;
 int PCM_data_size;		//holds the number of elements in PCM_data;
 char *filename;
} main_AFData;
static main_AFData *main_AudioFileData = NULL;
static int main_AudioFileData_count = 0;	//always holds number of main_AudioFileData elements

//START Globals taken from ScheduleGUI:
GdkPixmap *main_pixmap_Graph;	//main drawing area; must be created by external code (i.e., a main.cpp), then set in SG_Init()
int main_ParserXML_VoiceCount = 0;	//this is a *true* count of Voices (as opposed to listed count in file) taken from  pre-reads of the XML file; has priority
int main_ParserXML_EntryCount = 0;	//this is a *true* count of Entries (as opposed to listed count in file) taken from  pre-reads of the XML file; has priority
char *main_Info_Description = NULL;
char *main_Info_Author = NULL;
char *main_Info_Title = NULL;
GtkWidget *main_window = NULL;
GtkWidget *main_drawing_area = NULL;
GtkWidget *main_ProgressBar = NULL;
GtkButton *main_buttonPlayPause = NULL;
guint GUIUpdateHandle = 0;	//handle to the timer that updates the GUI every GNAURAL_GUI_UPDATEINTERVAL ms.
char *main_AudioWriteFile_name = NULL;	//this gets alloted a default name in main();, NEVER equals NULL
GThread *main_AudioWriteFile_thread = NULL;	//20071201 this will equal NULL if a file is NOT being written to
int main_AudioWriteFile_type = SF_FORMAT_WAV;
int gnaural_pauseflag = (~0);	//pauseflag controls all movement through schedule, also stops and starts sound
GtkLabel *main_labelFileName = NULL;
GtkLabel *main_labelFileDescription = NULL;
GtkLabel *main_labelFileTitle = NULL;
GtkLabel *main_labelFileAuthor = NULL;
GtkLabel *main_labelStatus = NULL;
GtkLabel *main_labelTotalRuntime = NULL;
GtkLabel *main_labelCurrentRuntime = NULL;
GtkEntry *main_entryLoops = NULL;
GtkToggleButton *main_togglebuttonViewFreq = NULL;
GtkToggleButton *main_togglebuttonViewBeat = NULL;
GtkToggleButton *main_togglebuttonViewVol = NULL;
GtkToggleButton *main_togglebuttonViewBal = NULL;
GtkStatusbar *main_Statusbar = NULL;
GtkHScale *main_hscaleVolume = NULL;
GtkHScale *main_hscaleBalance = NULL;
GtkVBox *main_vboxVoices = NULL;
GtkToggleButton *main_checkbuttonSwapStereo = NULL;
int main_Gnaural2File = 0;	//just a way to know internally if a file being opened isn't a valid Gnaurl2 file
gboolean main_KeyDown = FALSE;	//a way to keep track of start of arrow event in order to backup only once per mass event
char main_sTmp[PATH_MAX + PATH_MAX + 1];	//just a place to store tmp strings
const char main_sSec[] = "sec ";
const char main_sMin[] = "min ";
const char main_sInf[] = "inf";

//the only purpose of this is to help load old-style Gnaural files:
struct
{
 float FreqBase;
 float Volume_Tone;
 float Volume_Noiz;
 int loops;
 int loopcount;
 int StereoNoiz;
 int ScheduleEntriesCount;
}
OldGnaualVars;

//path and filename variables:
char main_sPathCurrentWorking[PATH_MAX + 1];	//whatever directory user called gnaural from
char main_sPathHome[PATH_MAX + 1];	// ~/                            -- not meaningfull in Win32
char main_sPathGnaural[PATH_MAX + 1];	// ~/.gnaural/                   -- not meaningfull in Win32
char main_sPathTemp[PATH_MAX + PATH_MAX + 1];	//used to copy the other paths and strcat filenames-to
gchar *main_sPathExecutable = NULL;	//so program can call itself to make MP3
gchar main_sCurrentGnauralFilenameAndPath[PATH_MAX + 1];	// ~/.gnaural/schedule.gnaural

//my getopt() vars/flags:
#define GNAURAL_CMDLINEOPTS     ":w:a:sodpih"
int cmdline_w = 0;		//Output .WAV file directly to file
int cmdline_a = 0;		//tell Gnaural which sound HW to access
int cmdline_s = 0;		//Create fresh Schedule file flag
int cmdline_o = 0;		//Output .WAV directly to stdout
int cmdline_d = 0;		//Show debugging information
int cmdline_p = 0;		//Output to sound system
int cmdline_i = 0;		//Show detailed console information

//  int cmdline_h = 0; //show help [not actually needed, since it is handled during parsing]

int main_gnaural_guiflag = FALSE;	//will be set internall to TRUE if there is a GUI
float main_OverallVolume = 1.f;	//this solely mirrors the OverallVolume slider; don't set
float main_OverallBalance = 0.f;	//this solely mirrors the OverallBalance slider; don't set
gboolean main_vscale_Y_mousebuttondown = FALSE;
float main_vscale_Y = 0.0f;
gboolean main_hscale_X_mousebuttondown = FALSE;
float main_hscale_X = 0.0f;
gboolean main_hscale_X_scaleflag = FALSE;
gboolean main_vscale_Y_scaleflag = FALSE;

/////////////////////////////////////////////////////
int main (int argc, char *argv[])
{
 BB_SeedRand (3676, 2676862);	//anything but 0 on either or twins seems OK
 BB_UserSleep = main_Sleep;
#ifdef ENABLE_NLS
 bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
 bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
 textdomain (GETTEXT_PACKAGE);
#endif

#ifdef GNAURAL_WIN32
 //this ensures that any potential Windows stdout activity is in binary mode:
 _setmode (_fileno (stdout), O_BINARY);
 //  _setmode (_fileno (stdin), O_BINARY);
#endif

 //setup a few globals:
 main_sPathExecutable = argv[0];	//so program can call itself to make MP3
 main_sCurrentGnauralFilenameAndPath[0] = '\0';	//this tells main_SetupPathsAndFiles to use default filename
 SG_StringAllocateAndCopy (&main_AudioWriteFile_name, "Gnaural");

 main_ResetScheduleInfo ();

 //do command line parsing:
 main_ParseCmdLine (argc, argv);

 //Take care of stuff joint to command-line and GUI first:
 //trap Ctrl-C:
 if (signal (SIGINT, SIG_IGN) != SIG_IGN)
 {
  signal (SIGINT, main_InterceptCtrl_C);
 }

 //next need to set up paths, since bb (below) will need to know
 //(among other things) where to access/create gnaural_schedule.txt file:
 main_SetupPathsAndFiles ((main_sCurrentGnauralFilenameAndPath[0] ==
			   '\0') ? TRUE : FALSE);

 //see if file exists; if not, write it:
 if (0 != main_TestPathOrFileExistence (main_sCurrentGnauralFilenameAndPath))
 {
  main_WriteDefaultFile ();
 }

 //Init GDK Threads, needed for file writing:
 //added 20051126 to support GThread cross-compatibility. Yes, it is supposed to be called before gtk_init()
 // call  echo `pkg-config --libs gthread-2.0` to see the libs to link to; I was segfaulting endlessly because I was linking
 //with old libs! This is the proper:
 //g++  -g -O2  -o gnaural  main.o BinauralBeat.o support.o interface.o callbacks.o pa_lib.o pa_unix.o pa_unix_oss.o -lgtk-x11-2.0 -lgdk-x11-2.0 -latk-1.0 -lgdk_pixbuf-2.0 -lm -lpangoxft-1.0 -lpangox-1.0 -lpangoft2-1.0 -lpango-1.0 -lgobject-2.0 -lgmodule-2.0 -ldl -lglib-2.0 -pthread -lgthread-2.0
 //if (!g_thread_supported ()) g_thread_init (NULL);
 g_thread_init (NULL);
 gdk_threads_init ();
 //next line was moved below on 20071126 as per GTK+ FAQ example
 //gdk_threads_enter ();        //if I don't do this, the Win32 version blocks! The documentation is amazingly vague.

 //init GTK:
 gtk_set_locale ();
 gtk_init (&argc, &argv);

 add_pixmap_directory (PACKAGE_DATA_DIR "/" PACKAGE "/pixmaps");

 main_SetIcon ();	//added 20051202; bruteforce method to avoid nasty GTK path/function inconsistencies

 //NOTE: even for command line, this needs to be here:
 //this returns non-0 if it doesn't find the file gnaural2.glade:
 if (0 != main_InitGlade ())
 {
  main_Cleanup ();
  return 0;
 }

 //open user file and put it in BB:
 main_XMLReadFile (main_sCurrentGnauralFilenameAndPath, main_drawing_area,
		   FALSE);

 //sort the args:
 //The main question: Do I need a GUI? Basically boils down to these two categories:
 //THINGS AFFECTING BOTH GUI AND TERM VERSION:
 //- cmdline_a: Tell Gnaural which sound HW to use
 //- cmdline_d: Show debugging information
 //THINGS AFFECTING SOLEY COMMANDLINE VERSION:
 //- cmdline_h: Print "Help" [already did this by this point] DOES NOT REQUIRE SOUND SYSTEM
 //- cmdline_o: dump a .WAV file to sdtout DOES NOT REQUIRE SOUND SYSTEM
 //- cmdline_w: dump a .WAV file to a file DOES NOT REQUIRE SOUND SYSTEM
 //- cmdline_s: create a fresh gnaural_schedule.txt file DOES NOT REQUIRE SOUND SYSTEM
 //- cmdline_i: show terminal-style GUI info
 //- cmdline_p: run the schedule directly through the soundsystem
 if ((cmdline_i + cmdline_p + cmdline_o + cmdline_w + cmdline_s) > 0)
 {	//do the command line version then exit:
  SG_DBGOUT ("Entering console mode");
  main_gnaural_guiflag = FALSE;
  main_RunCmdLineMode ();
  main_Cleanup ();
  exit (EXIT_SUCCESS);
 }

 //got here, so must be using GUI:
 main_gnaural_guiflag = TRUE;

 //set graph view:
 gtk_toggle_button_set_active (main_togglebuttonViewBeat, TRUE);
 main_SetGraphType (main_togglebuttonViewBeat);

 //init the sound:
 mainPA_SoundInit ();

 //start stopped:
 main_OnButton_Stop (NULL);

 //Now we can update some GUI stuff:
 main_UpdateGUI_FileInfo (main_sCurrentGnauralFilenameAndPath);
 main_UpdateGUI_Voices (main_vboxVoices);

 //setup the main GUI timer:
 GUIUpdateHandle =
  g_timeout_add (GNAURAL_GUI_UPDATEINTERVAL, main_UpdateGUI, NULL);

 //this is a hack, but does silence an unsightly gdk_draw error:
 main_configure_event (main_window, NULL);

 //do some elementary drag'n'drop stuff:
 static GtkTargetEntry targetentries[] = {
  {"text/plain", 0, MAIN_DND_TARGET_STRING},	//allows text to be dropped
  {"text/uri-list", 0, MAIN_DND_TARGET_URL},	//allows a file to be dropped
 };

 gtk_drag_dest_set (main_window, GTK_DEST_DEFAULT_ALL, targetentries, 2,
		    GDK_ACTION_COPY | GDK_ACTION_LINK);

 g_signal_connect (main_window, "drag_data_received",
		   G_CALLBACK (main_OnDragDataReceived), NULL);

 //start D-Bus server thread:
#ifdef GNAURAL_DBUS
 gnaural_dbus_server_main ();
#endif

 //enters this until gtk_main_quit() is called
 gdk_threads_enter ();	//essential to do in a threaded GTK+ program
 gtk_main ();

 //flasher_Cleanup (main_FD);//disconnected 20071022
 main_Cleanup ();
 return 0;
}

/////////////////////////////////////////////////////
void main_Cleanup ()
{
 //turn off dbus thread:
#ifdef GNAURAL_DBUS
 gnaural_dbus_server_cleanup ();
#endif

 //cleanup any write threads if active:
 if (NULL != main_AudioWriteFile_thread)
 {
  BB_DBGOUT ("Aborting audio write file thread");
  BB_WriteStopFlag = TRUE;
  while (NULL != main_AudioWriteFile_thread)
  {
   main_Sleep (G_USEC_PER_SEC);
  }
 }
 //clean up sound resources:
 mainPA_SoundCleanup ();

 //cleanup BB:
 BB_CleanupVoices ();

 //cleanup SG. NOTE: this should have been called once already at main_quit(), but to be sure:
 SG_Cleanup ();

 if (main_AudioWriteFile_name != NULL)
 {
  free (main_AudioWriteFile_name);
  main_AudioWriteFile_name = NULL;
 }

 //cleansup the globals:
 if (main_Info_Title != NULL)
 {
  free (main_Info_Title);
  main_Info_Title = NULL;
 }
 if (main_Info_Description != NULL)
 {
  free (main_Info_Description);
  main_Info_Description = NULL;
 }
 if (main_Info_Author != NULL)
 {
  free (main_Info_Author);
  main_Info_Author = NULL;
 }

 if (NULL != main_AudioFileData)
 {
  main_CleanupAudioFileData ();
 }

 //don't forget to do this!:
 BB_DBGOUT ("Calling gdk_threads_leave");
 gdk_threads_leave ();
}

/////////////////////////////////////////
//a20070812
void main_CleanupAudioFileData ()
{
 //SG_DBGOUT_INT("Cleaning up BB:", BB_VoiceCount);
 SG_DBGOUT_INT ("NULL'ing all PCM data in BB, voices:", BB_VoiceCount);
 BB_NullAllPCMData ();	//have to do this, unfortunately
 SG_DBGOUT_INT ("Cleaning up Audio Data, filecount:",
		main_AudioFileData_count);
 int i;

 for (i = 0; i < main_AudioFileData_count; i++)
 {
  main_AudioFileData[i].PCM_data_size = 0;
  if (NULL != main_AudioFileData[i].PCM_data)
  {
   SG_DBGOUT_INT ("Cleaning up Audio Data, array", i);
   free (main_AudioFileData[i].PCM_data);
   main_AudioFileData[i].PCM_data = NULL;
  }
  if (NULL != main_AudioFileData[i].filename)
  {
   free (main_AudioFileData[i].filename);
   main_AudioFileData[i].filename = NULL;
  }
 }
 main_AudioFileData_count = 0;
 SG_DBGOUT ("Cleaning up last bit of Audio Data");
 if (NULL != main_AudioFileData)
 {
  free (main_AudioFileData);
  main_AudioFileData = NULL;
 }
}

/////////////////////////////////////////
//a20070702: to cleanup initialization
//returns 0 on success
int main_InitGlade ()
{
 GladeXML *xml;

 SG_DBGOUT_STR ("Looking for glade file:", GLADE_FILE);
 xml = glade_xml_new (GLADE_FILE, NULL, NULL);
 if (xml == NULL)
 {
  SG_DBGOUT ("File gnaural2.glade not there, trying locally...");
  xml = glade_xml_new (GLADE_FILE_NAME, NULL, NULL);
  if (xml == NULL)
  {
   SG_ERROUT ("Couldn't find glade file");
   return -1;
  }
 }
 SG_DBGOUT ("Found glade file");

 // connect signal handlers:
 glade_xml_signal_autoconnect (xml);

 //connect the GTK main objects:
 //Get the main window:
 main_window = glade_xml_get_widget (xml, "window_main");
 //gtk_widget_set_name (main_window, "Gnaural2 Binaural Beat Generator");
 if (main_window == NULL)
 {
  SG_DBGOUT ("Didn't fine main_window in glade file");
 }

 //Get the progress bar:
 main_ProgressBar = glade_xml_get_widget (xml, "progressbar_main");
 if (main_ProgressBar == NULL)
 {
  SG_DBGOUT ("Didn't fine progressbar_main in glade file");
 }

 // Get the drawing area:
 main_drawing_area = glade_xml_get_widget (xml, "drawingarea_graph");
 gtk_widget_set_size_request (GTK_WIDGET (main_drawing_area), 800, 128);
 gtk_widget_show (main_drawing_area);

 //get the statusbar:
 main_Statusbar =
  (GtkStatusbar *) glade_xml_get_widget (xml, "statusbar_main");

 //get the Play/Pause button:
 main_buttonPlayPause =
  (GtkButton *) glade_xml_get_widget (xml, "buttonPlay");

 //get menuitems that will need updates:
 main_togglebuttonViewFreq =
  (GtkToggleButton *) glade_xml_get_widget (xml,
					    "radiobuttonGraphView_BaseFreq");
 main_togglebuttonViewBeat =
  (GtkToggleButton *) glade_xml_get_widget (xml,
					    "radiobuttonGraphView_BeatFreq");
 main_togglebuttonViewVol =
  (GtkToggleButton *) glade_xml_get_widget (xml,
					    "radiobuttonGraphView_Volume");
 main_togglebuttonViewBal =
  (GtkToggleButton *) glade_xml_get_widget (xml,
					    "radiobuttonGraphView_Balance");

 //get empty frame for Voice info:
 main_vboxVoices = (GtkVBox *) glade_xml_get_widget (xml, "vboxVoices");

 //get all labels:
 main_labelFileName =
  (GtkLabel *) glade_xml_get_widget (xml, "labelFileName");
 main_labelFileAuthor =
  (GtkLabel *) glade_xml_get_widget (xml, "labelFileAuthor");
 main_labelFileDescription =
  (GtkLabel *) glade_xml_get_widget (xml, "labelFileDescription");
 main_labelFileTitle =
  (GtkLabel *) glade_xml_get_widget (xml, "labelFileTitle");
 main_labelStatus = (GtkLabel *) glade_xml_get_widget (xml, "labelStatus");
 main_labelTotalRuntime =
  (GtkLabel *) glade_xml_get_widget (xml, "labelTotalRuntime");
 main_labelCurrentRuntime =
  (GtkLabel *) glade_xml_get_widget (xml, "labelCurrentRuntime");

 //get Entry:
 main_entryLoops = (GtkEntry *) glade_xml_get_widget (xml, "entryLoops");

 //get the Volume hscale:
 main_hscaleVolume = (GtkHScale *) glade_xml_get_widget (xml, "hscaleVolume");

 //get the Volume hscale:
 main_hscaleBalance =
  (GtkHScale *) glade_xml_get_widget (xml, "hscaleBalance");

 //get the checkbuttonSwapStereo:
 main_checkbuttonSwapStereo =
  (GtkToggleButton *) glade_xml_get_widget (xml, "checkbuttonSwapStereo");
 return 0;
}

/////////////////////////////////////////////////////
void main_OnDragDataReceived (GtkWidget * widget,
			      GdkDragContext * context,
			      int x,
			      int y,
			      GtkSelectionData * seldata,
			      guint info, guint time, gpointer userdata)
{
 SG_DBGOUT ("Got some dragged data:");
 SG_DBGOUT_INT ("info:", info);
 SG_DBGOUT_INT ("time:", time);
 SG_DBGOUT_INT ("format:", seldata->format);
 SG_DBGOUT_INT ("length:", seldata->length);
 SG_DBGOUT_STR ("data:", seldata->data);
 /*
    this is GtkSelectionData:
    typedef struct {
    GdkAtom     selection;
    GdkAtom     target;
    GdkAtom     type;
    gint                format;
    guchar       *data;  //for most uses, this is all you need
    gint                length;
    GdkDisplay   *display;
    } GtkSelectionData;
  */
 if (MAIN_DND_TARGET_URL == info)
 {
  //for some reason, g_filename_from_uri returns lots of garbage,
  //so just hunt for any valid filename out of whatever came:
  gchar *filename = g_filename_from_uri ((char *) seldata->data, NULL, NULL);
  int i = strlen (filename);

  while (i > 0)
  {
   if (0 == main_TestPathOrFileExistence (filename))
   {
    break;
   }
   filename[i] = '\0';
   --i;
  }

  if (i > 0)
  {
   SG_DBGOUT_STR (filename, "is a valid file");
   main_XMLReadFile (filename, main_drawing_area, FALSE);
  }
  else
  {
   SG_ERROUT ("Drag and Drop failed: no file found");
  }
  g_free (filename);
 }
}

/////////////////////////////////////////////////////
void main_ResetScheduleInfo ()
{
 SG_StringAllocateAndCopy (&main_Info_Author, NULL);
 SG_StringAllocateAndCopy (&main_Info_Title, NULL);
 SG_StringAllocateAndCopy (&main_Info_Description, NULL);
}

/////////////////////////////////////////////////////
gboolean main_button_release_event (GtkWidget * widget,
				    GdkEventButton * event)
{
 SG_button_release_event (widget, event);
 // if (SG_GraphHasChanged == TRUE)
 if (SG_DataHasChanged == TRUE)
 {
  main_LoadBinauralBeatSoundEngine ();
 }
 return TRUE;
}

/////////////////////////////////////////////////////
gboolean main_motion_notify_event (GtkWidget * widget, GdkEventMotion * event)
{
 SG_motion_notify_event (widget, event);
 //SG_GraphHasChanged = TRUE;
 return FALSE;
}

/////////////////////////////////////////////////////
gboolean main_button_press_event (GtkWidget * widget, GdkEventButton * event)
{
 SG_button_press_event (widget, event);
 if (SG_DataHasChanged == TRUE)
 {
  main_LoadBinauralBeatSoundEngine ();
 }
 return TRUE;
}

/////////////////////////////////////////////////////
gboolean main_key_release_event (GtkWidget * widget, GdkEventKey * event)
{
 main_KeyDown = FALSE;
 return FALSE;
}

/////////////////////////////////////////////////////
gboolean main_key_press_event (GtkWidget * widget, GdkEventKey * event)
{
 //NOTE: MOST GNAURAL KEYBOARD INPUT IS HANDLED BY GTK ACCELLERATORS,
 //therefore almost everything here gets served from callbacks.c:

 // widget = main_drawing_area;
 //====deal with keys pressed with Ctrl:
 if (event->state & GDK_CONTROL_MASK)
 {
  switch (event->keyval)
  {
  case GDK_a:	//select all:
   SG_SelectDataPoints (-8, -8, widget->allocation.width + 8,
			widget->allocation.height + 8, TRUE);
   SG_ConvertDataToXY (widget);
   SG_DrawGraph (widget);
   break;

  case GDK_b:
   {	//Scale Time:
    main_DuplicateSelectedVoice ();
   }
   break;

  case GDK_c:	//copy:
   SG_CopySelectedDataPoints (widget);
   main_UpdateGUI_Progressbar ();
   break;

  case GDK_d:	//Delete currently selected voice:
   main_DeleteSelectedVoice (widget);
   break;

  case GDK_e:	//Select All Points in currently selected voice:
   SG_SelectDataPoints_Voice (SG_SelectVoice (NULL), TRUE);
   SG_DrawGraph (widget);
   break;

  case GDK_g:
   main_ScaleDatPoints_Y (widget);
   break;

   //GDK_h is used by Help

  case GDK_i:	//Inverts selections in all visible voices:
   SG_SelectInvertDataPoints_All ();
   SG_DrawGraph (widget);
   break;

  case GDK_j:	//add new voice:
   main_VoicePropertiesDialog (widget, NULL);
   break;

  case GDK_k:	//Select Interval:
   main_SelectInterval ();
   break;

  case GDK_l:	//line up points:
   SG_AlignDataPoints (widget);
   break;

  case GDK_m:
   main_SelectNeighbor ();
   break;

   //GDK_n is already used

  case GDK_o:
   main_OpenFile (FALSE);
   break;

  case GDK_p:	//Apply Graph to sound engine:
   main_LoadBinauralBeatSoundEngine ();
   break;

  case GDK_q:	//quit:
   main_quit ();
   break;

  case GDK_r:	//reverse voice:
   main_ReverseVoice ();
   break;

  case GDK_s:	//save to file:
   main_XMLWriteFile (main_sCurrentGnauralFilenameAndPath);
   break;

  case GDK_t:
   {	//Edit Voice Properties:
    SG_Voice *tmpVoice = SG_SelectVoice (NULL);

    if (tmpVoice != NULL)
    {
     main_VoicePropertiesDialog (widget, tmpVoice);
    }
   }
   break;

  case GDK_u:	//Select last DPs in visible voices:
   main_SelectLastDPs ();
   break;

  case GDK_v:	//paste:
   SG_BackupDataPoints (main_drawing_area);
   SG_PasteSelectedDataPoints (widget, TRUE);
   break;

  case GDK_w:	//Change Graph View:
   switch (SG_GraphType)
   {
   case SG_GRAPHTYPE_BEATFREQ:
    main_SetGraphType (main_togglebuttonViewFreq);
    break;

   case SG_GRAPHTYPE_BASEFREQ:
    main_SetGraphType (main_togglebuttonViewVol);
    break;

   case SG_GRAPHTYPE_VOLUME:
    main_SetGraphType (main_togglebuttonViewBal);
    break;

   case SG_GRAPHTYPE_VOLUME_BALANCE:
    main_SetGraphType (main_togglebuttonViewBeat);
    break;

   default:
    main_SetGraphType (main_togglebuttonViewFreq);
    break;
   }
   SG_ConvertDataToXY (widget);
   SG_DrawGraph (widget);
   break;

  case GDK_x:	//cut:
   SG_BackupDataPoints (widget);
   SG_CopySelectedDataPoints (widget);
   SG_DeleteDataPoints (widget, FALSE, TRUE);
   break;

  case GDK_y:	//undo-redo:
  case GDK_z:
   //NOTE: undo/redo are dealt with by accellerators because their workings are closely tied with menu states:
   BB_ResetAllVoices ();	//a20070730
   SG_RestoreDataPoints (widget, FALSE);
   break;
  }	//end keyval
  return FALSE;
 }	// end deal with keys pressed with Ctrl

 //====deal with keys pressed with Shift:
 if (event->state & GDK_SHIFT_MASK)
 {
  switch (event->keyval)
  {
  case GDK_A:	//deselect all:
   SG_DeselectDataPoints ();
   SG_DrawGraph (widget);
   break;

  case GDK_E:	//Deselect All Points in currently selected voice:
   SG_SelectDataPoints_Voice (SG_SelectVoice (NULL), FALSE);
   SG_DrawGraph (widget);
   break;

  case GDK_I:	//Inverts selections in all visible voices:
   SG_SelectInvertDataPoints_Voice (SG_SelectVoice (NULL));
   SG_DrawGraph (widget);
   break;

  case GDK_U:	//Select first DPs in visible voices:
   main_SelectFirstDPs ();
   break;

  case GDK_V:	//Pastes whatever's in buffer to very end of schedule
   main_PasteAtEnd ();
   break;

  case GDK_X:	//cut (deleting time also):
   SG_BackupDataPoints (widget);
   SG_CopySelectedDataPoints (widget);
   SG_DeleteDataPoints (widget, TRUE, TRUE);
   break;

  case GDK_Delete:
   //delete points and delete points and time (Shift):
   SG_BackupDataPoints (widget);
   SG_DeleteDataPoints (widget, (event->state & GDK_SHIFT_MASK), TRUE);
   break;

   //a20070620 :
  case GDK_Up:
  case GDK_Down:
   main_key_arrowkeyhandler (widget, (event->keyval == GDK_Up) ? -10 : 10, 0);
   break;

   //a20070620 :
  case GDK_Left:
  case GDK_Right:
   main_key_arrowkeyhandler (widget, 0,
			     (event->keyval == GDK_Left) ? -10 : 10);
   break;
  }	//end keyval
  return FALSE;
 }	// end deal with keys pressed with Shift

 //====Finally, deal with pure keys:
 switch (event->keyval)
 {
 case GDK_Delete:
  SG_BackupDataPoints (widget);
  SG_DeleteDataPoints (widget, (event->state & GDK_SHIFT_MASK), TRUE);
  break;

  //a20070620 :
 case GDK_Up:
 case GDK_Down:
  main_key_arrowkeyhandler (widget, (event->keyval == GDK_Up) ? -1 : 1, 0);
  break;

  //a20070620 :
 case GDK_Left:
 case GDK_Right:
  main_key_arrowkeyhandler (widget, 0, (event->keyval == GDK_Left) ? -1 : 1);
  break;
 }	//end keyval
 return FALSE;
}	//end all keys

/////////////////////////////////////////////////////
//a20070620:
void main_key_arrowkeyhandler (GtkWidget * widget,
			       int vertical, int horizontal)
{
 if (vertical == 0 && horizontal == 0)
 {
  return;
 }
 //20070620 now deal with backup:
 if (main_KeyDown == FALSE)
 {
  SG_BackupDataPoints (widget);	//a20070620
 }
 main_KeyDown = TRUE;
 if (vertical != 0)
 {
  SG_MoveSelectedDataPoints (widget, 0, vertical);
  SG_ConvertYToData_SelectedPoints (widget);
 }

 //a20070620 :
 if (horizontal != 0)
 {
  SG_MoveSelectedDataPoints (widget, horizontal, 0);
  SG_ConvertXToDuration_AllPoints (widget);
 }

 SG_ConvertDataToXY (widget);	// NOTE: I need to call this both to set limits and to bring XY's outside of graph back in
 SG_DrawGraph (widget);
 SG_DataHasChanged = SG_GraphHasChanged = TRUE;	//20070105 tricky way to do main_LoadBinauralBeatSoundEngine in a bulk way
}

/////////////////////////////////////////////////////
// Initializing or resizing, so create a new backing pixmap of the appropriate size
gboolean main_configure_event (GtkWidget * widget, GdkEventConfigure * event)
{
 if (main_pixmap_Graph != NULL)
 {
  SG_DBGOUT ("Destroying old pixmap");
  g_object_unref (main_pixmap_Graph);
 }

 SG_DBGOUT ("Creating new pixmap");
 main_pixmap_Graph =
  gdk_pixmap_new (widget->window, widget->allocation.width,
		  widget->allocation.height, -1);
 // init the graph:
 SG_DBGOUT ("Initing SG_Graph");
 SG_Init (main_pixmap_Graph);

 //try commenting these out:
 SG_ConvertDataToXY (widget);
 SG_DrawGraph (widget);
 return TRUE;
}

/////////////////////////////////////////////////////
// This is the repaint signal, so redraw the screen from the backing pixmap
gboolean main_expose_event (GtkWidget * widget, GdkEventExpose * event)
{
 /*
    The GtkDrawingArea widget is used for creating custom user interface elements.
    It's essentially a blank widget; you can draw on widget->window. After creating a
    drawing area, the application may want to connect to:

    -      Mouse and button press signals to respond to input from the user.
    (Use gtk_widget_add_events() to enable events you wish to receive.)

    -     The "realize" signal to take any necessary actions when the widget is instantiated
    on a particular display. (Create GDK resources in response to this signal.)

    -     The "SG_configure_event" signal to take any necessary actions when the widget changes size.

    -    The "expose_event" signal to handle redrawing the contents of the widget.

    GDK automatically clears the exposed area to the background color before sending the
    expose event, and that drawing is implicitly clipped to the exposed area.
  */
 gdk_draw_drawable (widget->window,
		    widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		    main_pixmap_Graph, event->area.x, event->area.y,
		    event->area.x, event->area.y, event->area.width,
		    event->area.height);

 return FALSE;
}

/////////////////////////////////////////////////////
//NOTE: there is also a main_Cleanup(); not sure if they are in all cases cooperative.
void main_quit ()
{
 //  exit (0);
 g_source_remove (GUIUpdateHandle);
 GUIUpdateHandle = 0;

 SG_Cleanup ();
 gtk_main_quit ();
}

/////////////////////////////////////////////////////
//20070702: This apparently never actually gets called:
//"problem is the common one where libglade has already
//shown the widget before the signal handlers are connected, so you never
//get the signal. The solution is to set the window's 'Visible' property
//to FALSE in glade, and call gtk_widget_show() on it after calling
//glade_xml_signal_autoconnect()."
gboolean main_realize (GtkWidget * widget, GdkEventConfigure * event)
{
 return FALSE;
}

/////////////////////////////////////////////////////
void main_UpdateGUI_entryLoops ()
{
 sprintf (main_sTmp, "%d", BB_Loops);
 gtk_entry_set_text (GTK_ENTRY (main_entryLoops), main_sTmp);
}

/////////////////////////////////////////////////////
void main_SetLoops (unsigned int loops)
{
 int diff = BB_Loops - BB_LoopCount;

 if (loops != 0)
 {	//i.e., we're not in inf. loop mode:
  if (diff >= loops)
   BB_LoopCount = 1;	//this keeps it from running forever
  else
   BB_LoopCount = loops - diff;
 }
 else
 {	// inf. loop mode:
  BB_LoopCount = 0;
 }
 BB_Loops = loops;
 main_UpdateGUI_entryLoops ();
 main_UpdateGUI_ProjectedRuntime ();
 //  main_LoadBinauralBeatSoundEngine();
}

/////////////////////////////////////////////////////
void main_on_entryLoops_activate (GtkEntry * entry, gpointer user_data)
{
 //  const gchar *entry_text;
 //  entry_text = gtk_entry_get_text (GTK_ENTRY (entry));
 //  printf ("Entry contents: %s\n", entry_text);
 //  const gchar *entry_text = gtk_entry_get_text (GTK_ENTRY (entry));
 main_SetLoops (((unsigned
		  int) (g_ascii_strtod (gtk_entry_get_text
					(GTK_ENTRY (entry)), NULL))));
}

/////////////////////////////////////////////////////
//Takes a pointers to hold the Audio Data and a filename, and checks
//if it is valid data, loads it in to a local list so that BB can
//just grab pointers to already loaded audio data here instead of having 
//to reload entire audio files every reload.
void main_ProcessAudioFile (char *filename, int **buffer, unsigned int *size)
{
 int i;

 for (i = 0; i < main_AudioFileData_count; i++)
 {
  if (0 == strcmp (filename, main_AudioFileData[i].filename))
  {	//file was already opened:
   (*buffer) = main_AudioFileData[i].PCM_data;
   (*size) = main_AudioFileData[i].PCM_data_size;
   SG_DBGOUT_STR ("Audio Data already loaded, not loading", filename);
   return;
  }
 }
 SG_DBGOUT_STR ("Audio Data not already loaded, loading", filename);

 //didn't find it, so need to first try to open the file (to be sure it exists before adding it):
 if (0 != main_LoadSoundFile (filename, &(*buffer), &(*size)))
 {	//failed:
  SG_ERROUT ("Did not find valid audio data at:");
  SG_ERROUT (filename);
  return;
 }
 //successful load of PCM data, now need to add it to a list. 
 //first make a copy of the list, one-larger than original:
 main_AFData *tmp_AudioFileData =
  (main_AFData *) malloc (sizeof (main_AFData) *
			  (main_AudioFileData_count + 1));
 if (NULL != main_AudioFileData)
 {
  memcpy (tmp_AudioFileData, main_AudioFileData,
	  sizeof (main_AFData) * main_AudioFileData_count);
 }
 SG_DBGOUT_INT ("Number of audio files open:", main_AudioFileData_count + 1);

 //now add new last element's data:
 tmp_AudioFileData[main_AudioFileData_count].PCM_data = (*buffer);
 tmp_AudioFileData[main_AudioFileData_count].PCM_data_size = (*size);
 tmp_AudioFileData[main_AudioFileData_count].filename = NULL;	//IMPORTANT!
 SG_StringAllocateAndCopy (&
			   (tmp_AudioFileData[main_AudioFileData_count].
			    filename), filename);
 if (NULL != main_AudioFileData)
 {
  free (main_AudioFileData);
 }
 main_AudioFileData = tmp_AudioFileData;
 ++main_AudioFileData_count;
}

/////////////////////////////////////////////////////
//main_LoadBinauralBeatSoundEngine() Loads BB with
//whatever data is currently in SG data.
//Call this whenever ScheduleGUI and BinauralBeat aren't
//data sync'd
void main_LoadBinauralBeatSoundEngine ()
{
 SG_DBGOUT ("Copying Graph to BB Engine");
 SG_Voice *curVoice;
 SG_DataPoint *curDP;
 int count_voice;
 int count_dp;

 // first count the number of voices:
 count_voice = 0;
 curVoice = SG_FirstVoice;
 while (curVoice != NULL)
 {
  ++count_voice;
  curVoice = curVoice->NextVoice;
 }

 SG_DBGOUT_INT ("Total number of Voices:", count_voice);

 //1) prepare the sound engine for that many voices:
 //NOTE: as of 20070731, this also cleans old BB_Voice data:
 BB_InitVoices (count_voice);

 SG_DBGOUT_INT ("Preparing sound engine, number of voices:", count_voice);
 //now go through each voice, count the datapoints,
 //put the datapoints in a raw array, then feed it to
 //the sound engine's data stuctures:
 count_voice = 0;
 curVoice = SG_FirstVoice;

 //CRUCIAL to zero BB_TotalDuration, or else old value might persist (see BB Notes section):
 BB_TotalDuration = 0;
 while (curVoice != NULL)
 {
  //first count how many DPs this voice has:
  count_dp = 0;
  curDP = curVoice->FirstDataPoint;
  do
  {
   ++count_dp;
   //all list loops need this:
   curDP = curDP->NextDataPoint;
  }
  while (curDP != NULL);
  /*
     SG_DBGOUT_INT ("Voice:", count_voice);
     SG_DBGOUT_INT ("\tType:", curVoice->type);
     SG_DBGOUT_INT ("\tEvent Count:", count_dp);
   */
  //2) Allot Entry memory for each voice by running BB_SetupVoice() on each:
  BB_SetupVoice (count_voice, curVoice->type, curVoice->mute, count_dp);

  //3) Load memory for each entry alotted in step 2:
  count_dp = 0;
  curDP = curVoice->FirstDataPoint;
  do
  {
   BB_Voice[count_voice].Entry[count_dp].beatfreq_start_HALF =
    curDP->beatfreq * 0.5;
   /*
      BB_Voice[count_voice].Entry[count_dp].beatfreqL_start =
      curDP->beatfreq * 0.5;
      BB_Voice[count_voice].Entry[count_dp].beatfreqR_start =
      curDP->beatfreq * -0.5;
    */
   BB_Voice[count_voice].Entry[count_dp].duration = curDP->duration;
   BB_Voice[count_voice].Entry[count_dp].basefreq_start = curDP->basefreq;
   BB_Voice[count_voice].Entry[count_dp].volL_start = curDP->volume_left;
   BB_Voice[count_voice].Entry[count_dp].volR_start = curDP->volume_right;
   ++count_dp;
   //all list loops need this:
   curDP = curDP->NextDataPoint;
  }
  while (curDP != NULL);
  /*
     SG_DBGOUT_INT("Voice Type:", BB_Voice[count_voice].type);
     SG_DBGOUT_FLT("Voice Entry HZ:", BB_Voice[count_voice].Entry[2].beatfreqL_start);
     SG_DBGOUT_FLT("Voice Entry Duration:", BB_Voice[count_voice].Entry[2].duration);
     SG_DBGOUT_FLT("Voice Entry basefreq:", BB_Voice[count_voice].Entry[2].basefreq_start);
     SG_DBGOUT_FLT("Voice Entry volume:", BB_Voice[count_voice].Entry[2].volL_start);
   */

  //4) Run BB_CalibrateVoice() on the voice:
  BB_CalibrateVoice (count_voice);

  //5) see if this was a PCM voice:
  if (BB_VOICETYPE_PCM == curVoice->type)
  {
   /*
      //this works well, but I don't want to load directly in to BB anymore (too inefficient)
      main_LoadSoundFile (curVoice->description,
      (int **) &(BB_Voice[count_voice].PCM_samples),
      &(BB_Voice[count_voice].PCM_samples_samples_size));
    */
   //This loads locally, allowing me to check if voice is already and then simply assign pointers to BB:
   main_ProcessAudioFile (curVoice->description,
			  (int **) &(BB_Voice[count_voice].PCM_samples),
			  &(BB_Voice[count_voice].PCM_samples_size));
  }

  //advance to next voice:
  curVoice = curVoice->NextVoice;
  ++count_voice;
 }

 BB_PauseFlag = FALSE;	//CRITICAL, added 20070803
 //next two lines added 20070129 to deal with weird BB_TotalDuration inconsistencies
 //BB_DetermineTotalDuration();
 int fixes = BB_FixVoiceDurations ();

 SG_DBGOUT_INT ("\tVoices fixed:", fixes);

 main_UpdateGUI_ProjectedRuntime ();
}

//################################################
//START PortAudio code:
//NOTE: PortAudio was used for multi-platform compatibility. PortAudio still uses OSS.
//If compiling this for solely Linux, consider replacing all this with some ALSA code;
//OSS is flakey (on some systems, only one sound app can run at a time, etc.)
//NOTE 2: It would eventually be nice to pull this code from this file, since
//it is hardware related and the rest isn't. It is here only out of laziness.
// ======================================
// PortAudio will call this local function:
static int mainPA_MyCallback (void *inputBuffer,
			      void *outputBuffer,
			      unsigned long framesPerBuffer,
			      PaTimestamp outTime, void *userData)
{
 if (mainPA_SoundStream != NULL)
 {
  BB_MainLoop (outputBuffer, (framesPerBuffer << 2));
 }
 // BB_UserSoundProc(outputBuffer, (framesPerBuffer<<2));//IMPORTANT: no matter what units the sndbuffer uses, CBinauralBeat wants sndbufsize in bytes
 return 0;	//return 1 to stop sound server
}

//======================================
void mainPA_SoundStart ()
{
 Pa_StartStream (mainPA_SoundStream);
}

// ======================================
int mainPA_SoundInit ()
{
 char local_tmpstr[1024];

 //Start PortAudio vars:
#define GNAURAL_SAMPLE_RATE (BB_AUDIOSAMPLERATE)
#define GNAURAL_FRAMES_PER_BUFFER (4096)
#define GNAURAL_OUTPUT_DEVICE Pa_GetDefaultOutputDeviceID()

 PaError err;
 int data;			//this is leftover PA tutorial crap

 //end PortAudio vars

 SG_DBGOUT ("Starting PortAudio sound");

 err = Pa_Initialize ();
 if (err != paNoError)
 {
  //let user know status of audio system:
  sprintf (local_tmpstr, "Audio System Init: %s", Pa_GetErrorText (err));
  main_UpdateGUI_Statusbar (local_tmpstr, "");
  BB_ERROUT (local_tmpstr);
  return EXIT_FAILURE;
 }
 if (mainPA_SoundDevice == GNAURAL_USEDEFAULTSOUNDDEVICE)
 {
  mainPA_SoundDevice = Pa_GetDefaultOutputDeviceID ();
 }

 err = Pa_OpenStream (&mainPA_SoundStream, paNoDevice, 0, paInt16,	// short -- use for inteleaved 32-bit stereo
		      NULL, mainPA_SoundDevice, 2, paInt16,	// short -- use for inteleaved 32-bit stereo
		      NULL, GNAURAL_SAMPLE_RATE, GNAURAL_FRAMES_PER_BUFFER, 0,	// number of buffers, if zero then use default minimum
		      paClipOff,	// we won't output out of range samples so don't bother clipping them
		      mainPA_MyCallback, &data);

 //let user know status of audio system:
 sprintf (local_tmpstr, "Init Audio System: %s [device %d]",
	  Pa_GetErrorText (err), mainPA_SoundDevice);

 if (main_gnaural_guiflag == TRUE)
 {
  main_UpdateGUI_Statusbar (local_tmpstr, "");
 }
 else
 {
  SG_DBGOUT (local_tmpstr);
 }

 if (err != paNoError)
 {
  BB_ERROUT (local_tmpstr);
  return EXIT_FAILURE;
 }

 return EXIT_SUCCESS;
}	//end mainPA_SoundInit

// ======================================
void mainPA_SoundCleanup ()
{
 SG_DBGOUT ("Stopping PortAudio sound");
 if (mainPA_SoundStream != NULL)
 {
  Pa_StopStream (mainPA_SoundStream);
  Pa_CloseStream (mainPA_SoundStream);
 }
 Pa_Terminate ();
 mainPA_SoundStream = NULL;
 mainPA_SoundDevice = GNAURAL_USEDEFAULTSOUNDDEVICE;
 //all done with PortAudio
}

//END PortAudio code
//################################################

///////////////////////////////////////////////////////////////
//Parse user's string for patern/name pairs. Titles are bounded
//by "~" symbols, and subsequent filters trailed by commas, like:
//"~All Files~*,~Audio Files~*.wav,*.aiff,*.au,*.flac,~Image Files~*.jpg,*.gif,*.bmp"
void main_DialogAddFileFilters (GtkWidget * dialog, gchar * strFilterString)
{
 GtkFileFilter *filter;
 char *strFilterString_copy = NULL;

 //NOTE: Need to work with our own copy of strUserfilter:
 SG_StringAllocateAndCopy (&strFilterString_copy, strFilterString);
 //Need strFilterString_copy for free'ing later:
 char *sFilterString = strFilterString_copy;
 char *substr;

 BB_DBGOUT_STR ("Filter string: ", sFilterString);
 while (NULL != sFilterString && '\0' != *sFilterString)
 {
  BB_DBGOUT ("Looking for a new title");
  //get a title:
  while ('~' == *sFilterString && '\0' != *sFilterString)
  {
   BB_DBGOUT ("Found start of a new title");
   substr = sFilterString + 1;
   //look for next "~":
   while ('~' != *(++sFilterString));
   //cap it at temporary "end" for substr:
   *sFilterString = '\0';
   BB_DBGOUT_STR (" Filter title  :", substr);
   filter = gtk_file_filter_new ();
   gtk_file_filter_set_name (filter, substr);
   //now start adding filters for that title:
   while ('\0' != *(++sFilterString) && '~' != *sFilterString)
   {
    substr = sFilterString;
    //run to next comma:
    while (',' != *sFilterString && '\0' != *sFilterString)
     ++sFilterString;
    if ('\0' == *sFilterString)
    {
     BB_DBGOUT_STR ("[last one] Filter pattern:", substr);
     gtk_file_filter_add_pattern (filter, substr);
     break;
    }
    //cap for substring
    *sFilterString = '\0';
    BB_DBGOUT_STR (" Filter pattern:", substr);
    gtk_file_filter_add_pattern (filter, substr);
   }
   BB_DBGOUT ("Done collecting patterns for that title");
   gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (dialog), filter);
  }
 }
 BB_DBGOUT ("Done parsing user string");
 free (strFilterString_copy);
}

/////////////////////////////////////////////////////
//returns NULL if failed; otherwise, be sure to free
//what this returns with g_free();
//strUserfilter should be formatted like: 
//"~Audio Files~*.wav,*.aiff,*.au,*.flac,~Image Files~*.jpg,*.gif,*.bmp"
gchar *main_OpenFileDialog (gchar * strUserfilter)
{
 GtkWidget *dialog;
 char *filename = NULL;

 dialog =
  gtk_file_chooser_dialog_new ("Open File", NULL,
			       GTK_FILE_CHOOSER_ACTION_OPEN, GTK_STOCK_CANCEL,
			       GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN,
			       GTK_RESPONSE_ACCEPT, NULL);

 //this isn't very user friendly; it is just easy for me:
 gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog),
				      main_sPathGnaural);
 //these are alternatives:
 // gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), main_sCurrentGnauralFilenameAndPath);
 // gchar * tmpname = gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER (dialog));
 // gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), tmpname);
 // g_free (tmpname);

 main_DialogAddFileFilters (dialog, strUserfilter);

 if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
 {
  filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
 }
 gtk_widget_destroy (dialog);
 return filename;
}

//////////////////////
//Opens an Save As File dialog, fills a user alotted char * array (big enough
//to handle anything) with the filename IF something valid was chosen;
//but be sure to check return value before using it: ==0 means success
//(i.e., use the string), !=0 failure.
int main_AskForSaveAsFilename (char *userfilename)
{
 GtkWidget *dialog;

 dialog =
  gtk_file_chooser_dialog_new ("Save Gnaural Schedule as...", NULL,
			       GTK_FILE_CHOOSER_ACTION_SAVE, GTK_STOCK_CANCEL,
			       GTK_RESPONSE_CANCEL, GTK_STOCK_SAVE,
			       GTK_RESPONSE_ACCEPT, NULL);
 //20070301 This required GTK+2.0 > 2.8, so dropping it for *ugly* stat() hacks.
 //NOTE: next call wasn't even available in Debian unstable a year ago; seems there now
 // gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER (dialog), TRUE);
 gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (dialog),
				main_sCurrentGnauralFilenameAndPath);

 main_DialogAddFileFilters (dialog,
			    "~Gnaural Files~*.gnaural,~Text Files~*.txt,~All Files~,*");

 if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
 {
  char *filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));

  if (strlen (filename) > 0)
  {
   strcpy (userfilename, filename);
  }
  strcpy (userfilename, filename);
  g_free (filename);
  gtk_widget_destroy (dialog);
  return 0;	//user picked a filename
 }
 gtk_widget_destroy (dialog);
 return 1;	//failed to select a file
}

//======================================
//calls a dialog to get a filename, if file exists, asks if
//user is sure about overwriting; if answer is no, repeats,
//otherwise just overwrites.
void main_OnUserSaveAsFile ()
{
 int repeat;

 do
 {
  repeat = 0;
  //first ask for filename:
  if (0 != main_AskForSaveAsFilename (main_sPathTemp))
  {
   return;	//do nothing
  }
  if (0 == main_TestPathOrFileExistence (main_sPathTemp))
  {
   if (0 ==
       main_MessageDialogBox
       ("That file exists. Are you sure you want to overwrite it?",
	GTK_MESSAGE_WARNING, GTK_BUTTONS_YES_NO))
   {
    ++repeat;
   }
  }
 }
 while (0 != repeat);

 if (strlen (main_sPathTemp) > 0)
 {
  main_XMLWriteFile (main_sPathTemp);
  main_UpdateGUI_FileInfo (main_sPathTemp);
 }
}

///////////////////////////////////////////////
//this is only called by main_XMLParser; don't call yerself.
//point of this is to keep Event data separate, since it is the
//most numerous and thus should probably be done with Attributes
int main_XMLEventDataParser (const gchar * DataType,
			     const gchar * Value,
			     const int internal_EntryCount)
{
 //==START OF EVENT PROPERTIES
 if (!strcmp (DataType, "parent"))
 {
  //must add 1, because 0 would look to Restore() like a NULL pointer:
  SG_UndoRedo.DPdata[internal_EntryCount].parent =
   (void *) (atoi (Value) + 1);
  SG_DBGOUT_PNT ("parent:", SG_UndoRedo.DPdata[internal_EntryCount].parent);
  return 0;
 }

 if (!strcmp (DataType, "duration"))
 {
  SG_UndoRedo.DPdata[internal_EntryCount].duration =
   g_ascii_strtod (Value, NULL);
  SG_DBGOUT_FLT ("duration:",
		 SG_UndoRedo.DPdata[internal_EntryCount].duration);
  return 0;
 }

 if (!strcmp (DataType, "volume_left"))
 {
  SG_UndoRedo.DPdata[internal_EntryCount].volume_left =
   g_ascii_strtod (Value, NULL);
  SG_DBGOUT_FLT ("volume_left:",
		 SG_UndoRedo.DPdata[internal_EntryCount].volume_left);
  return 0;
 }

 if (!strcmp (DataType, "volume_right"))
 {
  SG_UndoRedo.DPdata[internal_EntryCount].volume_right =
   g_ascii_strtod (Value, NULL);
  SG_DBGOUT_FLT ("volume_right:",
		 SG_UndoRedo.DPdata[internal_EntryCount].volume_right);
  return 0;
 }

 if (!strcmp (DataType, "beatfreq"))
 {
  SG_UndoRedo.DPdata[internal_EntryCount].beatfreq =
   g_ascii_strtod (Value, NULL);
  SG_DBGOUT_FLT ("beatfreq:",
		 SG_UndoRedo.DPdata[internal_EntryCount].beatfreq);
  return 0;
 }

 if (!strcmp (DataType, "basefreq"))
 {
  SG_UndoRedo.DPdata[internal_EntryCount].basefreq =
   g_ascii_strtod (Value, NULL);
  SG_DBGOUT_FLT ("basefreq:",
		 SG_UndoRedo.DPdata[internal_EntryCount].basefreq);
  return 0;
 }

 if (!strcmp (DataType, "state"))
 {
  SG_UndoRedo.DPdata[internal_EntryCount].state = atoi (Value);
  SG_DBGOUT_INT ("state:", SG_UndoRedo.DPdata[internal_EntryCount].state);
  return 0;
 }
 //==END OF EVENT PROPERTIES
 //=========================
 return 1;
}

///////////////////////////////////////////////
//this is only called by ParserXML; don't call
void main_XMLParser (const gchar * CurrentElement,	//this must always be a valid string
		     const gchar * Attribute,	//beware: this will equal NULL if there are none
		     const gchar * Value)
{	//beware: this will equal NULL to announce end of CurrnetElement
 //Essential to do things in this order:
 // 1) Check if this call represents the end of CurrentElement; if so, increment counter vars then return
 // 2) See if there are Attributes with this CurrentElement; if so, assign them, then return;
 // 3) If I got here, just slog through Elements and Values to assign to vars
 static int internal_EntryCount = 0;
 static int internal_VoiceCount = 0;

 //See if this is the end of CurrentElement:
 if (Value == NULL)
 {
  if (0 == strcmp (CurrentElement, "entry"))
  {
   ++internal_EntryCount;
   SG_DBGOUT_STR ("==END ", CurrentElement);
  }
  else if (0 == strcmp (CurrentElement, "voice"))
  {
   ++internal_VoiceCount;
   SG_DBGOUT_STR ("==END ", CurrentElement);
  }
  return;
 }

 //Now deal with Attributes if there are any:
 if (Attribute != NULL)
 {
  // ParserXML_AbortFlag = TRUE;
  // SG_DBGOUT_STR(Attribute, Value);
  main_XMLEventDataParser (Attribute, Value, internal_EntryCount);
  return;
 }

 //got here, so must slog through assigning vars Element-by-Element (the old way).
 //The first line exists in order to allow old pre-Attribute file versions to
 //still be openable:
 if (0 ==
     main_XMLEventDataParser (CurrentElement, Value, internal_EntryCount))
 {
 }
 else if (!strcmp (CurrentElement, "description"))
 {
  //20070627: fixed bug here if user had no desc. (because Value would equal NULL)
  //  SG_UndoRedo.Voice[internal_VoiceCount].description =
  SG_StringAllocateAndCopy (&
			    (SG_UndoRedo.Voice[internal_VoiceCount].
			     description), Value);
  SG_DBGOUT_STR ("description:",
		 SG_UndoRedo.Voice[internal_VoiceCount].description);
 }
 else if (!strcmp (CurrentElement, "type"))
 {
  SG_UndoRedo.Voice[internal_VoiceCount].type = atoi (Value);
  SG_DBGOUT_INT ("voicetype:", SG_UndoRedo.Voice[internal_VoiceCount].type);
 }
 else if (!strcmp (CurrentElement, "voice_state"))
 {
  SG_UndoRedo.Voice[internal_VoiceCount].state = atoi (Value);
  SG_DBGOUT_INT ("voice_state:",
		 SG_UndoRedo.Voice[internal_VoiceCount].state);
 }
 else if (!strcmp (CurrentElement, "voice_hide"))
 {
  SG_UndoRedo.Voice[internal_VoiceCount].hide = atoi (Value);
  SG_DBGOUT_INT ("voice_hide:", SG_UndoRedo.Voice[internal_VoiceCount].hide);
 }
 else if (!strcmp (CurrentElement, "voice_mute"))
 {
  SG_UndoRedo.Voice[internal_VoiceCount].mute = atoi (Value);
  SG_DBGOUT_INT ("voice_mute:", SG_UndoRedo.Voice[internal_VoiceCount].mute);
 }
 //By the totalentrycount entry, I have all the info needed to make the critical memory allotments:
 else if (!strcmp (CurrentElement, "totalentrycount"))
 {
  int listedcount = atoi (Value);

  if (listedcount > main_ParserXML_EntryCount)
  {
   SG_ERROUT_INT ("Listed entry count larger than actual count by",
		  listedcount - main_ParserXML_EntryCount);
   SG_DBGOUT_INT ("  Listed count:", listedcount);
   SG_DBGOUT_INT ("  Actual count:", main_ParserXML_EntryCount);
  }
  else if (listedcount < main_ParserXML_EntryCount)
  {
   SG_ERROUT_INT ("Listed entry count smaller than actual count by",
		  main_ParserXML_EntryCount - listedcount);
   SG_DBGOUT_INT ("  Listed count:", listedcount);
   SG_DBGOUT_INT ("  Actual count:", main_ParserXML_EntryCount);
  }
  else
  {
   SG_DBGOUT_INT ("Listed entry count matched actual count:",
		  main_ParserXML_EntryCount);
  }

  //  SG_UndoRedo.TotalDataPoints = atoi (Value);   //this is weak, as this number could be wrong (I need to count myself)
  SG_UndoRedo.TotalDataPoints = main_ParserXML_EntryCount;	//I trust the read count more than listed
  SG_AllocateBackupData (&SG_UndoRedo, SG_UndoRedo.TotalVoices,
			 SG_UndoRedo.TotalDataPoints);
  SG_DBGOUT_INT ("TotalEntryCount:", SG_UndoRedo.TotalDataPoints);
  SG_DBGOUT ("==Allocated Loadup Data==");
 }
 else if (!strcmp (CurrentElement, "voicecount"))
 {
  int listedcount = atoi (Value);

  if (listedcount > main_ParserXML_VoiceCount)
  {
   SG_ERROUT_INT ("Listed voice count larger than actual count by",
		  listedcount - main_ParserXML_VoiceCount);
   SG_DBGOUT_INT ("  Listed count:", listedcount);
   SG_DBGOUT_INT ("  Actual count:", main_ParserXML_VoiceCount);
  }
  else if (listedcount < main_ParserXML_VoiceCount)
  {
   SG_ERROUT_INT ("Listed voice count smaller than actual count by",
		  main_ParserXML_VoiceCount - listedcount);
   SG_DBGOUT_INT ("  Listed count:", listedcount);
   SG_DBGOUT_INT ("  Actual count:", main_ParserXML_VoiceCount);
  }
  else
  {
   SG_DBGOUT_INT ("Listed voice count matched actual count:",
		  main_ParserXML_VoiceCount);
  }
  SG_UndoRedo.TotalVoices = main_ParserXML_VoiceCount;	// I trust the real count more than listed
 }

 //finally the rarely called, low-priority global entries:
 else if (!strcmp (CurrentElement, "title"))
 {
  SG_StringAllocateAndCopy (&main_Info_Title, Value);
 }
 else if (!strcmp (CurrentElement, "schedule_description"))
 {
  SG_StringAllocateAndCopy (&main_Info_Description, Value);
 }
 else if (!strcmp (CurrentElement, "author"))
 {
  SG_StringAllocateAndCopy (&main_Info_Author, Value);
 }
 else if (!strcmp (CurrentElement, "loops"))
 {
  main_SetLoops (atoi (Value));
  SG_DBGOUT_INT ("Loops:", BB_Loops);
 }
 else if (!strcmp (CurrentElement, "overallvolume_left"))
 {
  BB_VolumeOverall_left = g_ascii_strtod (Value, NULL);
  SG_DBGOUT_FLT ("BB_VolumeOverall_left", BB_VolumeOverall_left);
 }
 else if (!strcmp (CurrentElement, "overallvolume_right"))
 {
  BB_VolumeOverall_right = g_ascii_strtod (Value, NULL);
  SG_DBGOUT_FLT ("BB_VolumeOverall_right", BB_VolumeOverall_right);
 }
 else if (!strcmp (CurrentElement, "stereoswap"))
 {
  BB_StereoSwap = atoi (Value);
  SG_DBGOUT_INT ("BB_StereoSwap", BB_StereoSwap);
 }
 else if (!strcmp (CurrentElement, "gnauralfile_version"))
 {
  main_Gnaural2File = ~0;
  SG_DBGOUT ("Zeroing running voice and entry counts");
  internal_EntryCount = internal_VoiceCount = 0;
  SG_DBGOUT_STR ("File Version:", Value);
  if (strcmp (main_VERSION_GNAURAL_XMLFILE, Value))
  {
   //SG_ERROUT("File format version too old: Aborting!");
   //ParserXML_AbortFlag = 1;
   SG_ERROUT ("Unknown File Version...");
   SG_ERROUT ("Expected:");
   SG_ERROUT (main_VERSION_GNAURAL_XMLFILE);
   SG_ERROUT ("Got:");
   SG_ERROUT (Value);
   //    if (0 == main_MessageDialogBox ("Unknown Gnaural2 file version or format. Is this a Gnaural 1 file?", GTK_MESSAGE_WARNING, GTK_BUTTONS_YES_NO)) { }
  }
  else
  {
   SG_DBGOUT ("Useable file version, continuing:");
  }
 }
}

/////////////////////////////////////////////////////
//Returns 0 on success
//widget refers to main_drawing_area, BTW
//To maximize reusing code, this procedure actually fills
//ScheduleGUI's variable "SG_UndoRedo" with the incoming
//data, then treats that data like any ordinary restore.
//RETURNS 0 on success;
//20070705: changed a lot of address bugs with console mode
//trying to update a non-existent GUI.
//NOTE: if MergeRestore == TRUE, adds restore to whatever is already
//in SG; if FALSE, erases everything currently in SG
int main_XMLReadFile (char *filename,
		      GtkWidget * widget, gboolean MergeRestore)
{
 if (filename == NULL)
 {
  return -1;
 }

 //to check if maybe it is a Gnaural1 file:
 main_Gnaural2File = 0;

 //Set a few things in case they're skipped later:
 main_SetLoops (1);
 main_Info_Author[0] = '\0';
 main_Info_Title[0] = '\0';
 main_Info_Description[0] = '\0';
 BB_ResetAllVoices ();	//never hurts to do this, and might help a lot
 //  BB_LoopCount = 1;

 //to parse a Gnaural2 file:
 // 1) zero any SG internal variables used to count voice/event progress (ESSENTIAL)
 // 2) Give ParserXML SG's internal callback function
 // 3) call ParserXML_parse_file_xml:
 //Mustn't forget this:
 main_ParserXML_VoiceCount = 0;
 main_ParserXML_EntryCount = 0;
 SetUserParserXMLHandler (main_XMLParser_counter);
 SG_DBGOUT ("Parsing XML file for true voice and entry counts");
 if (0 == ParserXML_parse_file_xml (filename))
 {
  SG_DBGOUT_INT ("Total new Voice count: ", main_ParserXML_VoiceCount);
  SG_DBGOUT_INT ("Total new Entry count: ", main_ParserXML_EntryCount);
 }
 else
 {
  SG_DBGOUT ("Something wrong with XML file, can't count");
 }
 SetUserParserXMLHandler (main_XMLParser);
 SG_DBGOUT ("Starting with real XML file parse:");
 if (0 == ParserXML_parse_file_xml (filename))
 {
  SG_DBGOUT ("XML file parsed, putting data in engine:");
  SG_DBGOUT_INT ("Total new Voice count: ", main_ParserXML_VoiceCount);
  SG_DBGOUT_INT ("Total new Entry count: ", main_ParserXML_EntryCount);
  SG_RestoreDataPoints (widget, MergeRestore);
  SG_DrawGraph (widget);
  SG_DBGOUT_STR ("Title:", main_Info_Title);
  SG_DBGOUT_STR ("Author:", main_Info_Author);
  SG_DBGOUT_STR ("Description:", main_Info_Description);
  main_UpdateGUI_UserDataInfo ();
 }
 else
 {
  if (main_Gnaural2File == 0)
  {
   SG_DBGOUT ("This was not a Gnaural2 format file");
   //try to open a Gnaural1 file:
   if (main_gnaural_guiflag == FALSE ||
       0 != main_MessageDialogBox ("Is this a Gnaural 1 file?",
				   GTK_MESSAGE_WARNING, GTK_BUTTONS_YES_NO))
   {
    SG_DBGOUT ("Processing as a Gnaural 1 format file");
    main_GNAURAL1FILE_SchedFilenameToSchedule (filename);
    //       SG_GetScheduleLimits();
    SG_ConvertDataToXY (widget);
    //    SG_DrawGraph (widget);
    SG_DataHasChanged = SG_GraphHasChanged = TRUE;
   }
   else
   {
    return -1;
   }
  }
  else
  {
   return -1;	//THIS SHOULD BE CLEANED UP; it is confusing
  }
 }

 //Got here, so file is valid, so load BB and set main_sCurrentGnauralFilenameAndPath:
 strcpy (main_sCurrentGnauralFilenameAndPath, filename);
 main_LoadBinauralBeatSoundEngine ();

 //if we have a GUI, might as well do this here:
 if (main_gnaural_guiflag == TRUE)
 {
  main_UpdateGUI_FileInfo (filename);
  main_UpdateGUI_Voices (main_vboxVoices);
  main_OnButton_Stop (NULL);
 }
 return 0;
}

/////////////////////////////////////////////////////
//Writes a Gnaural2 format XML file
void main_XMLWriteFile (char *filename)
{
 //First count how many DPs to copy. One reason to do this first is to be sure
 //there really are selected DPs before throwing away the old buffer:
 SG_DataPoint *curDP;
 int DP_count = 0;
 int voice_count = 0;
 SG_Voice *curVoice = SG_FirstVoice;

 while (curVoice != NULL)
 {
  curDP = curVoice->FirstDataPoint;
  ++voice_count;
  while (curDP != NULL)
  {
   ++DP_count;
   //all list loops need this:
   curDP = curDP->NextDataPoint;
  }
  curVoice = curVoice->NextVoice;
 }

 SG_DBGOUT_INT ("WriteFileXML: DataPoints to copy:", DP_count);
 SG_DBGOUT_INT ("WriteFileXML: Total Voices:", voice_count);

 //if there are no selected DPs, don't do anything:
 if (DP_count < 1)
  return;

 //prepare file access:
 FILE *stream;

 if ((stream = fopen (filename, "w")) == NULL)
 {
  SG_ERROUT ("Failed to open file for writing!");
  return;
 }

 gchar strbuf[G_ASCII_DTOSTR_BUF_SIZE];
 const int strsize = sizeof (strbuf);

 fprintf (stream, "<!-- See http://gnaural.sourceforge.net -->\n");
 fprintf (stream, "<schedule>\n");
 fprintf (stream, "<gnauralfile_version>%s</gnauralfile_version>\n", main_VERSION_GNAURAL_XMLFILE);	//always keep this first
 fprintf (stream, "<gnaural_version>%s</gnaural_version>\n",
	  main_VERSION_GNAURAL);
 time_t curtime;
 struct tm *loctime;

 curtime = time (NULL);
 loctime = (struct tm *) localtime (&curtime);
 fprintf (stream, "<date>%s</date>\n", asctime (loctime));
 fprintf (stream, "<title>%s</title>\n",
	  main_Info_Title == NULL ? "[none]" : main_Info_Title);
 fprintf (stream, "<schedule_description>%s</schedule_description>\n",
	  main_Info_Description == NULL ? "[none]" : main_Info_Description);
 fprintf (stream, "<author>%s</author>\n",
	  main_Info_Author == NULL ? "[none]" : main_Info_Author);
 fprintf (stream, "<totaltime>%s</totaltime>\n",
	  g_ascii_formatd (strbuf, strsize, "%g", SG_TotalScheduleDuration));
 fprintf (stream, "<voicecount>%d</voicecount>\n", voice_count);
 fprintf (stream, "<totalentrycount>%d</totalentrycount>\n", DP_count);
 fprintf (stream, "<loops>%d</loops>\n", BB_Loops);
 fprintf (stream, "<overallvolume_left>%s</overallvolume_left>\n",
	  g_ascii_formatd (strbuf, strsize, "%g", BB_VolumeOverall_left));
 fprintf (stream, "<overallvolume_right>%s</overallvolume_right>\n",
	  g_ascii_formatd (strbuf, strsize, "%g", BB_VolumeOverall_right));
 fprintf (stream, "<stereoswap>%d</stereoswap>\n", BB_StereoSwap);

 // Now put all the selected DPs in to the copy buffer:
 DP_count = 0;
 voice_count = 0;
 curVoice = SG_FirstVoice;
 while (curVoice != NULL)
 {
  //first count the number of DP's in this voice:
  curDP = curVoice->FirstDataPoint;
  int dpcount_local = 0;

  while (curDP != NULL)
  {
   ++dpcount_local;
   //all list loops need this:
   curDP = curDP->NextDataPoint;
  }

  //now do it again with that info:
  curDP = curVoice->FirstDataPoint;
  fprintf (stream, "<voice>\n");
  fprintf (stream, "<description>%s</description>\n",
	   curVoice->description == NULL ? "[none]" : curVoice->description);
  fprintf (stream, "<id>%d</id>\n", voice_count);
  fprintf (stream, "<type>%d</type>\n", curVoice->type);
  fprintf (stream, "<voice_state>%d</voice_state>\n", curVoice->state);
  fprintf (stream, "<voice_hide>%d</voice_hide>\n", curVoice->hide);
  fprintf (stream, "<voice_mute>%d</voice_mute>\n", curVoice->mute);
  fprintf (stream, "<entrycount>%d</entrycount>\n", dpcount_local);
  fprintf (stream, "<entries>\n");

  while (curDP != NULL)
  {
   //NEW STYLE FORMAT
   //START entry writing:
   fprintf (stream, "<entry parent=\"%d\" duration=\"%s\" ", voice_count,
	    g_ascii_formatd (strbuf, strsize, "%g", curDP->duration));
   fprintf (stream, "volume_left=\"%s\" ",
	    g_ascii_formatd (strbuf, strsize, "%g", curDP->volume_left));
   fprintf (stream, "volume_right=\"%s\" ",
	    g_ascii_formatd (strbuf, strsize, "%g", curDP->volume_right));
   fprintf (stream, "beatfreq=\"%s\" ",
	    g_ascii_formatd (strbuf, strsize, "%g", curDP->beatfreq));
   fprintf (stream, "basefreq=\"%s\" state=\"%d\"/>\n",
	    g_ascii_formatd (strbuf, strsize, "%g", curDP->basefreq),
	    curDP->state);
   //END entry writing:
   //END NEW STYLE FORMAT
   ++DP_count;
   //all list loops need this:
   curDP = curDP->NextDataPoint;
  }
  curVoice = curVoice->NextVoice;
  fprintf (stream, "</entries>\n");
  fprintf (stream, "</voice>\n");
  ++voice_count;
 }
 fprintf (stream, "</schedule>\n");
 fclose (stream);
 SG_DBGOUT_INT ("Wrote DataPoints:", DP_count);
}

//========================================
void main_on_edit_activate ()
{	//this is a hack -- no time to write editor or figure out how to access default editor, so let a typical platform editors do it:
 //###############FOR LINUX################
#ifndef GNAURAL_WIN32
 //NOTE: I think it is trying to launch this twice, maybe a mouse-click artifact? Might want to try be sure i can only run once
 //BUG 20060116: if user leaves Gedit open after exiting, it is as if the sound system is still being owned by Gnaual until Gedit is closed
 sprintf (main_sPathTemp, "%s%s%s", "gedit ",
	  main_sCurrentGnauralFilenameAndPath, " &");
 system (main_sPathTemp);
#endif

 //###############FOR WIN32################
#ifdef GNAURAL_WIN32
 ShellExecute (0, "open",	// Operation to perform
	       //"c:\\windows\\notepad.exe", // Application name
	       "notepad.exe",	// Application name
	       main_sCurrentGnauralFilenameAndPath,	// Additional parameters
	       0,	// Default directory
	       SW_SHOW);
#endif
}

/////////////////////////////////////////////////////
void main_EventToKeypress (guint state, guint keyval)
{
 GdkEventKey event;

 //in main_key_press_event, only state and keyval are required:
 event.state = state;
 event.keyval = keyval;
 main_key_press_event (main_drawing_area, &event);
}

/////////////////////////////////////////////////////
//this function only exists to support main_AboutDialogBox below
//#include <gnome.h>
void main_activate_url (GtkAboutDialog * about,
			const gchar * url, gpointer data)
{
#ifndef GNAURAL_WIN32
 //##########FOR LINUX##########
 //don't have anything yet for this
 //gnome_url_show (url, NULL);
 gchar tmpstr[256];

 sprintf (tmpstr, "mozilla %s &", url);
 system (tmpstr);
 // system("mozilla http://gnaural.sourceforge.net/ &");
#endif

#ifdef GNAURAL_WIN32
 //##########FOR WIN32##########
 ShellExecute (NULL, "open", "http://gnaural.sourceforge.net", "", "c:\\",
	       SW_SHOWNORMAL);
#endif
}

/////////////////////////////////////////////////////
void main_AboutDialogBox ()
{
 static GtkWidget *about = NULL;
 const gchar *authors[] = {
  "Bret Logan <gnaural@users.sourceforge.net>",
  NULL
 };
 const gchar *documenters[] = {
  "Bret Logan <gnaural@users.sourceforge.net>",
  NULL
 };
 const gchar *copyright = "Copyright \xc2\xa9 2003-2007 Bret Logan\n";
 const gchar *comments =
  _
  ("A programmable audio generator intended as an aural aid to meditation and relaxation, implementing the binaural beat principle as described in Gerald Oster's Oct. 1973 Scientific American article \"Auditory Beats in the Brain.\"");
 const gchar *license =
  "This program is free software; you can redistribute it and/or modify\n\
    it under the terms of the GNU General Public License as published by\n\
    the Free Software Foundation; either version 2 of the License, or\n\
    (at your option) any later version.\n\n\
    This program is distributed in the hope that it will be useful,\n\
    but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  For more\n\
    details on the GNU General Public License, contact:\n\n\
    \tFree Software Foundation, Inc.\n\
    \t59 Temple Place - Suite 330\n\
    \tBoston, MA 02111-1307, USA.";

 //~ Translators: This is a special message that shouldn't be translated
 //~ literally. It is used in the about box to give credits to
 //~ the translators.
 //~ Thus, you should translate it to your name and email address.
 //~ You can also include other translators who have contributed to
 //~ this translation; in that case, please write them on separate
 //~ lines seperated by newlines (\n).
 const gchar *translator_credits = _("translator-credits");

 if (about != NULL)
 {
  gtk_window_set_transient_for (GTK_WINDOW (about), GTK_WINDOW (main_window));
  gtk_window_present (GTK_WINDOW (about));

  return;
 }
 gtk_about_dialog_set_url_hook (main_activate_url, NULL, NULL);
 about = GTK_WIDGET (g_object_new (GTK_TYPE_ABOUT_DIALOG, "name", _("Gnaural2"), "version", main_VERSION_GNAURAL, "copyright", copyright, "comments", comments, "website", "http://gnaural.sourceforge.net/", "authors", authors, "documenters", documenters, "translator_credits", translator_credits, "logo", NULL,	//this makes it use the default icon set up in main()
				   "license", license,
				   //        "wrap-license", FALSE,
				   NULL));
 gtk_window_set_destroy_with_parent (GTK_WINDOW (about), TRUE);
 g_signal_connect (about, "response", G_CALLBACK (gtk_widget_destroy), NULL);
 g_signal_connect (about, "destroy", G_CALLBACK (gtk_widget_destroyed),
		   &about);
 gtk_window_set_transient_for (GTK_WINDOW (about), GTK_WINDOW (main_window));
 gtk_window_present (GTK_WINDOW (about));
}

/////////////////////////////////////////////////////
//this gets called every GNAURAL_GUI_UPDATEINTERVAL milliseconds
//20070710: changed to only update one GUI feature per call
gboolean main_UpdateGUI (gpointer data)
{
 static int curcount = 0;

 if (++curcount > 3)
 {
  curcount = 0;
 }

 //added 20071126:
 gdk_threads_enter ();

 //deal with Y slider:
 if (TRUE == main_vscale_Y_mousebuttondown)
 {
  main_slider_XY_handler (main_vscale_Y, 0);
 }
 else if (TRUE == main_hscale_X_mousebuttondown)
 {
  main_slider_XY_handler (0, main_hscale_X);
 }

 switch (curcount)
 {
 case 0:
  //added 20070105 to deal with so many inputs not being updated:
  if (TRUE == SG_DataHasChanged)
  {
   main_LoadBinauralBeatSoundEngine ();
   SG_DataHasChanged = FALSE;
  }
  break;

 case 1:
  main_UpdateGUI_PlaceInGraph ();
  break;

 case 2:
  main_UpdateGUI_Progressbar ();
  if (TRUE == SG_GraphHasChanged)
  {
   main_UpdateGUI_Voices (main_vboxVoices);
   SG_GraphHasChanged = FALSE;
  }
  break;

 case 3:
  main_UpdateGUI_Labels ();
  break;
 }

 //added 20071126:
 gdk_threads_leave ();

 return TRUE;
}

/////////////////////////////////////////////////////
void main_UpdateGUI_PlaceInGraph ()
{
 SG_DrawGraph (main_drawing_area);
}

/////////////////////////////////////////////////////
void main_UpdateGUI_Progressbar ()
{
 long total =
  (BB_CurrentSampleCountLooped +
   BB_CurrentSampleCount) / (int) BB_AUDIOSAMPLERATE;
 float i = 0;

 if (BB_TotalDuration > 0)
 {
  if (BB_Loops > 0)
  {
   i = total / (float) (BB_TotalDuration * BB_Loops);
  }
  if (i > 1)
  {
   i = 1;
  }
  else if (i < 0)
  {
   i = 0;
  }
  gtk_progress_bar_set_fraction ((GtkProgressBar *) main_ProgressBar, i);
 }
 else
 {
  gtk_progress_bar_set_fraction ((GtkProgressBar *) main_ProgressBar, 0);
 }
}

//======================================
void main_OnButton_Play (GtkButton * button)
{
 if (main_AudioWriteFile_thread != NULL)
 {
  return;	//disable Play button if we're writing an audio file.
 }

 if (mainPA_SoundStream == NULL)
 {
  mainPA_SoundCleanup ();
  mainPA_SoundInit ();	//try to (re)start sound if it couldn't be initted before
  if (mainPA_SoundStream == NULL)
  {
   return;	//still didn't init, so give up
  }
  gnaural_pauseflag = ~gnaural_pauseflag;	//yes, a little silly, but necessary
 }

 gnaural_pauseflag = ~gnaural_pauseflag;
 if (gnaural_pauseflag == 0)
 {	// i.e., it was already paused:
  //continue playing sound:
  if (mainPA_SoundStream != NULL)
  {
   Pa_StartStream (mainPA_SoundStream);
  }

  //Following is so I don't erase good stuff on screen until full UpdateLoopInforestart:
  if (((BB_InfoFlag) & BB_COMPLETED) != 0)
  {
   BB_InfoFlag &= ~BB_COMPLETED;
   BB_CurrentSampleCountLooped = 0;
  }
  gtk_button_set_label (button, "_Pause");
  GtkWidget *image =
   gtk_image_new_from_stock (GTK_STOCK_MEDIA_PAUSE, GTK_ICON_SIZE_BUTTON);
  gtk_button_set_image (GTK_BUTTON (button), image);
  main_UpdateGUI_Status ("Playing");
 }
 else
 {	//e.g., it was not paused
  //stop playing sound:
  if (mainPA_SoundStream != NULL)
  {
   Pa_StopStream (mainPA_SoundStream);
  }

  gtk_button_set_label (button, "_Play");
  GtkWidget *image =
   gtk_image_new_from_stock (GTK_STOCK_MEDIA_PLAY, GTK_ICON_SIZE_BUTTON);
  gtk_button_set_image (GTK_BUTTON (button), image);
  main_UpdateGUI_Status ("Paused");
 }
 // bbl_UpdateGUI_Info ();
}

/////////////////////////////////////////////////////
//added 20070625, loads main_sTmp
void main_VoiceInfoFormatter (SG_Voice * curVoice)
{
 static char num[64];

 if (curVoice != NULL)
 {
  sprintf (main_sTmp, "%d [", curVoice->ID);

  switch (curVoice->type)
  {
  case BB_VOICETYPE_BINAURALBEAT:
   strcat (main_sTmp, "Binaural Beat] ");
   break;

  case BB_VOICETYPE_PINKNOISE:
   strcat (main_sTmp, "Pink Noise] ");
   break;

  case BB_VOICETYPE_PCM:
   strcat (main_sTmp, "Audio File] ");
   break;
  }
  strcat (main_sTmp, curVoice->description);

  //add count to end:
  int dpcount_all = 0,
   dpcount_selected = 0;

  SG_CountVoiceDPs (curVoice, &dpcount_all, &dpcount_selected);
  sprintf (num, " (%d,%d)", dpcount_all, dpcount_selected);
  strcat (main_sTmp, num);
 }
 else
 {
  main_sTmp[0] = '\0';
 }
}

/////////////////////////////////////////////////////
//negative amount rewinds, positive fast-forwards
void main_OnButton_ForwardRewind (float amount)
{
 if (amount < 0)
 {
  BB_ResetAllVoices ();
 }
 int i =
  BB_CurrentSampleCount + (BB_TotalDuration * amount * BB_AUDIOSAMPLERATE);
 if (i < 0)
  i = 0;
 else if (i > (BB_TotalDuration * BB_AUDIOSAMPLERATE))
  i = BB_TotalDuration * BB_AUDIOSAMPLERATE;
 BB_CurrentSampleCount = i;
}

/////////////////////////////////////////////////////
void main_OnButton_Stop (GtkButton * button)
{
 BB_WriteStopFlag = 1;	//try to kill a write if it is what's happening
 //first step is to pause:
 gnaural_pauseflag = 0;	//to be sure bbl_OnButton_Play() thinks it is was running
 main_OnButton_Play (main_buttonPlayPause);	// Simulate a user button push to enter paused state

 //now make schedule and everything else go back to begining
 BB_Reset ();
 main_UpdateGUI_Status ("Stopped");
 //bbl_UpdateGUI_Info ();
 //gtk_label_set_text (LabelProgramStatus, "Program Stopped");
}

/////////////////////////////////////////////////////
//20070629: returns an SG_Voice for a given index value,
//or NULL if there is none.
SG_Voice *main_VoiceGetIndex (int index)
{
 int voiceindex = 0;
 SG_Voice *curVoice = SG_FirstVoice;

 while (curVoice != NULL)
 {
  if (voiceindex == index)
  {
   break;
  }
  //main loop update
  ++voiceindex;
  curVoice = curVoice->NextVoice;
 }
 return curVoice;
}

/////////////////////////////////////////////////////
void main_callback_VoicesMute (GtkWidget * widget, gpointer data)
{
 //g_print ("Checkbox A%d toggled\n", (int) data);
 int voice = (int) data;

 SG_Voice *curVoice = main_VoiceGetIndex (voice);

 if (curVoice == NULL)
 {
  return;
 }

 if (voice >= BB_VoiceCount)
 {
  // g_print ("voice: %d, BB_VoiceCount: %d\n", (int) data, BB_VoiceCount);
  return;
 }
 if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)) == TRUE)
 {
  BB_Voice[voice].mute = curVoice->mute = TRUE;
 }
 else
 {
  BB_Voice[voice].mute = curVoice->mute = FALSE;
 }
}

/////////////////////////////////////////////////////
//handles the view/hide checkbox in the Voices Box
void main_callback_VoicesView (GtkWidget * widget, gpointer data)
{
 //g_print ("Checkbox B%d toggled\n", (int) data);
 SG_Voice *curVoice = main_VoiceGetIndex ((int) data);

 if (curVoice == NULL)
 {
  return;
 }
 if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)) == TRUE)
 {
  curVoice->hide = FALSE;
 }
 else
 {
  curVoice->hide = TRUE;
  SG_VoiceTestLegalSelection ();
  //SG_SelectDataPoints_Voice (curVoice, FALSE);//it really might be better to call this
  if (curVoice->state == SG_SELECTED)
  {
   curVoice->state = SG_UNSELECTED;
  }
 }
}

/////////////////////////////////////////////////////
//20070629: handles selecting voices and launching
//properties box:
gboolean main_callback_VoicesProperties (GtkWidget * widget,
					 GdkEventButton * event,
					 gpointer data)
{
 // g_print ("Voice Label %d clicked, mouse button %d\n", (int) data, event->button);
 SG_Voice *curVoice = main_VoiceGetIndex ((int) data);

 if (curVoice != NULL)
 {
  SG_SelectVoice (curVoice);
  if (event->button == 3)
  {
   main_VoicePropertiesDialog (main_drawing_area, curVoice);
  }
  main_UpdateGUI_Voices (main_vboxVoices);
 }
 return FALSE;
}

/////////////////////////////////////////////////////
//a20070629 box is simply where you want list put
void main_UpdateGUI_Voices (GtkVBox * mainbox)
{
 GtkWidget *CheckBut;
 GtkWidget *labelVoice;
 GtkWidget *eventbox;
 int voiceindex = 0;
 static GtkWidget *box = NULL;	//create so I can delete something inside box user passed me

 //first create something other than user's box that can be deleted
 if (box != NULL)
 {
  gtk_widget_destroy (box);
 }
 box = gtk_vbox_new (FALSE, 0);
 gtk_widget_show (box);
 gtk_box_pack_start (GTK_BOX (mainbox), box, FALSE, FALSE, 0);

 //create the new hbox for this row:
 GtkWidget *boxh = gtk_hbox_new (FALSE, 0);

 gtk_widget_show (boxh);

 //give a heading for the list:
 //do some chores to get some numbers:
 int dpcount = 0;
 int seldpcount = 0;
 int voicecount = 0;
 int selvoicecount = 0;

 SG_CountAllData (&voicecount, &selvoicecount, &dpcount, &seldpcount);
 SG_DBGOUT ("Updating Voice View");
 sprintf (main_sTmp, "View/Mute/Type/Description/Counts (%d,%d) (%d/%d)",
	  dpcount, seldpcount, selvoicecount, voicecount);
 labelVoice = gtk_label_new (main_sTmp);
 gtk_box_pack_start (GTK_BOX (boxh), labelVoice, FALSE, FALSE, 0);
 gtk_widget_show (labelVoice);
 gtk_box_pack_start (GTK_BOX (box), boxh, FALSE, FALSE, 0);

 //main loop:
 voiceindex = 0;
 SG_Voice *curVoice = SG_FirstVoice;

 while (curVoice != NULL)
 {
  //create the new hbox for this row:
  boxh = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (boxh);

  //make first checkbutton for Show/Hide:
  CheckBut = (GtkWidget *) gtk_check_button_new ();
  gtk_widget_show (CheckBut);

  //Connect the "toggled" signal of the checkbox to our callback
  g_signal_connect (G_OBJECT (CheckBut), "toggled",
		    G_CALLBACK (main_callback_VoicesView),
		    (gpointer) voiceindex);

  //add checkbox to this row:
  gtk_box_pack_start (GTK_BOX (boxh), CheckBut, FALSE, FALSE, 0);

  //check if visible:
  if (curVoice->hide == FALSE)
  {
   gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (CheckBut), TRUE);
  }

  //make second checkbutton for Audible/Mute:
  CheckBut = (GtkWidget *) gtk_check_button_new ();
  gtk_widget_show (CheckBut);

  // Connect the "toggled" signal of the checkbox to our callback
  g_signal_connect (G_OBJECT (CheckBut), "toggled",
		    G_CALLBACK (main_callback_VoicesMute),
		    (gpointer) voiceindex);

  //add checkbox to this row:
  gtk_box_pack_start (GTK_BOX (boxh), CheckBut, FALSE, FALSE, 0);

  //check if muted:
  if (BB_Voice[voiceindex].mute == TRUE)
  {
   gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (CheckBut), TRUE);
  }

  //create an eventbox to recieve mouse events:
  eventbox = gtk_event_box_new ();
  gtk_box_pack_start (GTK_BOX (boxh), eventbox, FALSE, FALSE, 0);
  gtk_widget_show (eventbox);
  // And bind an action to it:
  gtk_widget_set_events (eventbox, GDK_BUTTON_PRESS_MASK);
  g_signal_connect (G_OBJECT (eventbox), "button_press_event",
		    G_CALLBACK (main_callback_VoicesProperties),
		    (gpointer) voiceindex);

  //add label with description to put in the event box:
  main_VoiceInfoFormatter (curVoice);
  labelVoice = gtk_label_new (main_sTmp);
  // gtk_label_set_selectable ((GtkLabel *) labelVoice, TRUE);
  if (curVoice->state == SG_SELECTED)
  {
   //#define MARKUP_STRING "<span size='small'>%s</span>"
#define MARKUP_STRING "<u>%s</u>"
   gchar *markup = g_markup_printf_escaped (MARKUP_STRING, main_sTmp);

   gtk_label_set_markup (GTK_LABEL (labelVoice), markup);
   g_free (markup);
  }
  gtk_container_add (GTK_CONTAINER (eventbox), labelVoice);
  gtk_widget_show (labelVoice);

  //  sprintf (tmpstr, "%d) This is checkbox label", voiceindex);

  //finally, add the hbox and label to the vbox:
  gtk_box_pack_start (GTK_BOX (box), boxh, FALSE, FALSE, 0);

  //main loop update
  ++voiceindex;
  curVoice = curVoice->NextVoice;
 }
}

/////////////////////////////////////////////////////
void main_UpdateGUI_Status (char *msg)
{
 sprintf (main_sTmp, "Status: %s", msg);
 gtk_label_set_text (main_labelStatus, main_sTmp);
}

/////////////////////////////////////////////////////
//20070105 updated to measure looping
void main_UpdateGUI_ProjectedRuntime ()
{
 const char sProjected[] = "Projected Runtime: ";

 if (BB_Loops != 0)
 {
  int total = (int) (BB_TotalDuration * BB_Loops);

  if (1 == BB_Loops)
  {
   sprintf (main_sTmp, "%s%d%s%d%s", sProjected, total / 60, main_sMin,
	    total % 60, main_sSec);
  }
  else
  {
   sprintf (main_sTmp, "%s%d%s%d%s(%d%s%d%sx %d)", sProjected, total / 60,
	    main_sMin, total % 60, main_sSec, ((int) BB_TotalDuration) / 60,
	    main_sMin, ((int) BB_TotalDuration) % 60, main_sSec, BB_Loops);
  }
  gtk_label_set_text (main_labelTotalRuntime, main_sTmp);
 }
 else
 {
  sprintf (main_sTmp, "%sForever (%d%s%d%sx %s)", sProjected,
	   ((int) BB_TotalDuration) / 60, main_sMin,
	   ((int) BB_TotalDuration) % 60, main_sSec, main_sInf);
  gtk_label_set_text (main_labelTotalRuntime, main_sTmp);
 }
}

/////////////////////////////////////////////////////
void main_FormatProgressString ()
{
 const char sProgress[] = "Progress: ";
 const char sLoop[] = "(Loop: ";
 int total =
  (BB_CurrentSampleCountLooped + BB_CurrentSampleCount) / BB_AUDIOSAMPLERATE;
 int remaining = (int) ((BB_TotalDuration * BB_Loops) - total);

 if (BB_Loops != 0)
 {
  sprintf (main_sTmp, "%s%d%s%d%s Remaining: %d%s%d%s%s%d/%d)", sProgress,
	   total / 60, main_sMin, total % 60, main_sSec, remaining / 60,
	   main_sMin, remaining % 60, main_sSec, sLoop,
	   BB_Loops - BB_LoopCount + 1, BB_Loops);
 }
 else
 {
  sprintf (main_sTmp, "%s%d%s%d%s%s%d/%s)", sProgress, (total / 60),
	   main_sMin, total % 60, main_sSec, sLoop, (-BB_LoopCount) + 1,
	   main_sInf);
 }
}

//======================================
//QUESTIONABLE BUG FIXES:
//-BUG: figure out why resetting bb->InfoFlag's caused update labels timer to quit.
//            Wow, this one was a bit as if the garage door opened every time I pulled
//            my hair. Bizarre. One of the wierdest bugs I've ever seen.
// SOLUTION: made BinauralBeat's InfoFlag an unsigned int (from int), made
//            explicit that the timer func returned TRUE, and rephrased bit setting line
//             back to what it had been when the bug first appeared (?!).
//            Hard to believe this really fixed it, but it no longer does the bad behavior. (20051020)
void main_UpdateGUI_Labels ()
{

 //-----------------------------------------------------------
 //START stuff that gets updated EVERY second:
 //
 // current time point within schedule:
 main_FormatProgressString ();
 gtk_label_set_text (main_labelCurrentRuntime, main_sTmp);

 //a20070620 Voice info:
 // main_VoiceInfoFormatter (SG_SelectVoice (NULL));
 // gtk_label_set_text (main_labelVoiceInfo, main_sTmp);
 //
 //END stuff that gets updated EVERY second:
 //-----------------------------------------------------------------------

 //-----------------------------------------------------------------------
 //START check bb->InfoFlag
 //now update things that BinauralBeat says are changed:
 if (BB_InfoFlag != 0)
 {
  //.........................
  //if schedule is done:
  if (((BB_InfoFlag) & BB_COMPLETED) != 0)
  {
   //  bbl_OnButton_Stop ();
   //        gtk_label_set_text (LabelProgramStatus, "Schedule Completed");
   gnaural_pauseflag = 0;	//so bbl_OnButton_Play() thinks I was running
   //main_OnButton_Play (); // I'm simulating user button push to pause
   main_OnButton_Stop (NULL);
   main_UpdateGUI_Status ("Schedule Completed");
   //   bb->loopcount = bb->loops;
   //   gtk_progress_bar_set_fraction (ProgressBar_Overall, 1.0);

  }

  //.........................
  //if starting a new loop:
  if (((BB_InfoFlag) & BB_NEWLOOP) != 0)
  {
   //reset the "new loop" bit of InfoFlag:
   BB_InfoFlag &= ~BB_NEWLOOP;
   //bbl_UpdateGUI_LoopInfo ();
  }
 }	//END check bb->InfoFlag
 //-------------------------------------------------------------
}

/////////////////////////////////////////////////////
void main_UpdateGUI_Statusbar (const char *msg1, const char *msg2)
{
 // g_print ("Trying to update statusbar\n");
 // sprintf(main_sTmp, "Trying to update statusbar\n");
 gint context_id = gtk_statusbar_get_context_id (main_Statusbar, "SC");

 gtk_statusbar_pop (main_Statusbar, context_id);
 sprintf (main_sTmp, "%s %s", msg1, msg2);
 gtk_statusbar_push (main_Statusbar, context_id, main_sTmp);
}

//======================================
//following tries to be sure I don't end with a \ or / (which happens
//only when getcwd returns a root like / or C:\, D:\   :
void main_FixPathTerminator (char *pth)
{
 //  #ifdef GNAURAL_WIN32
 if (pth[(strlen (main_sPathCurrentWorking) - 1)] == '\\')
 {
  pth[(strlen (main_sPathCurrentWorking) - 1)] = '\0';
 }
 // #endif
 else
  // #ifndef GNAURAL_WIN32
 if (main_sPathCurrentWorking[(strlen (main_sPathCurrentWorking) - 1)] == '/')
 {
  main_sPathCurrentWorking[(strlen (main_sPathCurrentWorking) - 1)] = '\0';
 }
 // #endif
}

//======================================
//returns zero if file or path DOES exist
int main_TestPathOrFileExistence (char *PathName)
{
 struct stat buf;

 return stat (PathName, &buf);
}

//======================================
//The most important thing this does is set main_sCurrentGnauralFilenameAndPath
//e20070705: If bDefaultFlag == TRUE sets up $HOME/gnaural.schedule,
//if FALSE, works with whatever is in main_sCurrentGnauralFilenameAndPath
void main_SetupPathsAndFiles (int bDefaultFlag)
{
 //Current Working Directory:
 getcwd (main_sPathCurrentWorking, PATH_MAX);
 main_FixPathTerminator (main_sPathCurrentWorking);

 //DBGOUT(szDir_CurrentWorking);

 //copy CWD to the other vars, just in case I forget to put anything in them:
 strcpy (main_sPathHome, main_sPathCurrentWorking);
 strcpy (main_sPathGnaural, main_sPathCurrentWorking);

 SG_DBGOUT ("This was the command to launch Gnaural:");
 SG_DBGOUT (main_sPathExecutable);

 SG_DBGOUT ("This is the current working directory:");
 SG_DBGOUT (main_sPathCurrentWorking);

#ifdef GNAURAL_WIN32
 //###########START WIN32-ONLY CODE#############
 //In Win32, I try to work from wherever the executable is located:

 //-----
 //START of determine Gnaural's home directory:
 //HKEY_LOCAL_MACHINE\\Software\Microsoft\\Windows\CurrentVersion\\App Paths\\gnaural.exe
 //C:\Program Files\Gnaural;c:\Program Files\Common Files\GTK\2.0\bin
 HKEY hkey;

 if (ERROR_SUCCESS ==
     RegOpenKeyEx (HKEY_LOCAL_MACHINE,
		   "Software\\Microsoft\\Windows\\CurrentVersion\\App Paths\\gnaural2.exe",
		   0, KEY_QUERY_VALUE, &hkey))
 {
  DWORD type,
   count = sizeof (main_sPathTemp);

  if (ERROR_SUCCESS ==
      RegQueryValueExA (hkey, "Path", 0, &type, (LPBYTE) main_sPathTemp,
			&count))
  {
   //Looking for the first path in the list:
   char *tmpch;

   tmpch = strchr (main_sPathTemp, ';');	//search for first ';'
   if (tmpch != NULL)
   {
    (*tmpch) = '\0';
   }
   strcpy (main_sPathGnaural, main_sPathTemp);
   main_FixPathTerminator (main_sPathGnaural);
  }
  RegCloseKey (hkey);
 }
 else
 {
  SG_DBGOUT ("No gnaural.exe registry entry, checking manually for dir");
  //try a bruteforce approach:
  //20070919: I THINK THIS HAS main_TestPathOrFileExistence BACKWARD (not changing until I am sure)
  if (0 != main_TestPathOrFileExistence ("C:\\Program Files\\Gnaural2\\"))
  {
   strcpy (main_sPathGnaural, "C:\\Program Files\\Gnaural2");
  }
 }

 /*
    //NOTE: Original intent (below commented code) was to have all file creation in executable's home directory, but
    //it failed because of "/" and "\" inconsistencies between WINE and Win32, rendering
    //mixed paths, etc. Since getcwd() seems to be mostly consistent on each
    //platform, I eventually stuck with that -- but in the end, a WINE bug causing gcwd to return
    //(in my case) H:\ for both root and my home directory made me give-up and just
    //use the (goddamned) registry.

    strcpy (main_sPathTemp, szFile_ExecutablePath);

    //Now make all slashes consistent with DOS:
    int i=strlen(main_sPathTemp);
    while ((--i)>-1) if (main_sPathTemp[i]=='/') main_sPathTemp[i]='\\';

    //Now throw away executable name if there is any:
    char *tmpch;
    tmpch=strrchr(main_sPathTemp,'\\'); //search for first slash from end
    if (tmpch!=NULL) {
    (*tmpch)='\0';

    GNAURAL_DBGOUT("This is the directory Gnaural is being run from:");
    GNAURAL_DBGOUT(main_sPathTemp);

    //now return all slashes to DOS style:
    i=strlen(main_sPathTemp);
    while ((--i)>-1) if (main_sPathTemp[i]=='/') main_sPathTemp[i]='\\';

    //now change in to that directory [still necessary?!]
    chdir(main_sPathTemp);

    //now ADD starting slash if it isn't there:
    if (main_sPathTemp[0]!='\\')
    {
    for (i=strlen(main_sPathTemp);i>-1;i--)  main_sPathTemp[i+1]=main_sPathTemp[i];
    main_sPathTemp[0]='\\';
    }
    }
    //If I got here, user is already working from Gnaural's directory:
    else main_sPathTemp[0]='\0';

    //now attach this string to the end of CWD:
    strcpy(szDir_Gnaural,szDir_CurrentWorking);
    strcat(szDir_Gnaural,main_sPathTemp);
  */

 //END of determine Gnaural's home directory
 //-----

 //Finally, setup file and path to gnaural_schedule.txt file:
 if (bDefaultFlag == TRUE)
 {
  sprintf (main_sCurrentGnauralFilenameAndPath, "%s\\schedule.gnaural",
	   main_sPathGnaural);
 }
 SG_DBGOUT ("Accessing schedule file: ");
 SG_DBGOUT (main_sCurrentGnauralFilenameAndPath);
 return;
 //###########END WIN32-ONLY CODE#############
#endif

#ifndef GNAURAL_WIN32
 //###########START 'NUX-ONLY CODE#############
 //In Windows, Gnaural just does everything in it's formal installation directory.
 //In 'nix, it works from ~/.gnaural

 //figure out the paths/directories needed for all file access:
 if (NULL != getenv ("HOME"))
 {
  //Home directory:
  strcpy (main_sPathHome, getenv ("HOME"));
  //~/.gnaural directory:
  sprintf (main_sPathGnaural, "%s/.gnaural", main_sPathHome);
 }
 else
 {	//failed, so harmlessly put current directory in there...
  SG_ERROUT ("Couldn't determine home directory");
  strcat (main_sPathHome, main_sPathCurrentWorking);
  strcat (main_sPathGnaural, main_sPathCurrentWorking);
 }

 //create the .gnaural directory if it doesn't exist yet:szDir_CurrentWorking
 if (0 != main_TestPathOrFileExistence (main_sPathGnaural))
 {
  SG_DBGOUT ("Creating ~/.gnaural directory");

  if (0 != mkdir (main_sPathGnaural, 0777))
  {
   fprintf (stderr, "Couldn't make ~/.gnaural directory %s: errno=%i\n",
	    main_sPathGnaural, errno);
  }
 }
 else
 {
  SG_DBGOUT ("Found ~/.gnaural directory; will use.");
 }
 //setup file and path to gnaural_schedule.txt file:
 if (bDefaultFlag == TRUE)
 {
  sprintf (main_sCurrentGnauralFilenameAndPath, "%s/schedule.gnaural",
	   main_sPathGnaural);
 }
 SG_DBGOUT ("Accessing schedule file: ");
 SG_DBGOUT (main_sCurrentGnauralFilenameAndPath);
 return;
#endif
 //###########END 'NUX-ONLY CODE#############
}

/////////////////////////////////////////////////////
//20070824 Highly improved Vol-Bal routine taken from Gnaural
void main_ProcessVolBal ()
{
 BB_VolumeOverall_right = BB_VolumeOverall_left = main_OverallVolume;

 if (main_OverallBalance > 0)
 {	//that means attenuate Left:
  BB_VolumeOverall_left *= (1.f - fabs (main_OverallBalance));
 }
 else
 {	//that means attenuate Right:
  BB_VolumeOverall_right *= (1.f - fabs (main_OverallBalance));
 }
}

/////////////////////////////////////////////////////
void main_on_hscaleBalance (float range)
{
 main_OverallBalance = range *= .01;
 main_ProcessVolBal ();
}

/////////////////////////////////////////////////////
void main_on_hscaleVolume (float range)
{
 main_OverallVolume = range *= .01;
 main_ProcessVolBal ();
}

/////////////////////////////////////////////////////
void main_SetGraphType (GtkToggleButton * togglebutton)
{
 if (togglebutton == main_togglebuttonViewFreq)
 {
  SG_GraphType = SG_GRAPHTYPE_BASEFREQ;
 }
 else if (togglebutton == main_togglebuttonViewBeat)
 {
  SG_GraphType = SG_GRAPHTYPE_BEATFREQ;
 }
 else if (togglebutton == main_togglebuttonViewVol)
 {
  SG_GraphType = SG_GRAPHTYPE_VOLUME;
 }
 else if (togglebutton == main_togglebuttonViewBal)
 {
  SG_GraphType = SG_GRAPHTYPE_VOLUME_BALANCE;
 };
 SG_ConvertDataToXY (main_drawing_area);
 SG_DrawGraph (main_drawing_area);
}

/////////////////////////////////////////////////////
//Basically, a way to get a quick "yes or no" from user, or just present some info.
//Returns 1 for "yes", 0 for "no"
int main_MessageDialogBox (char *msg,
			   GtkMessageType messagetype,
			   GtkButtonsType buttonformation)
{
 GtkWidget *dialog = gtk_message_dialog_new (GTK_WINDOW (main_window),
					     GTK_DIALOG_DESTROY_WITH_PARENT,
					     messagetype,
					     buttonformation,
					     msg);

 //  "Error loading file '%s': %s",  main_sTmp, g_strerror (errno));
 gint res = gtk_dialog_run (GTK_DIALOG (dialog));

 gtk_widget_destroy (dialog);

 if (GTK_RESPONSE_OK == res || GTK_RESPONSE_YES == res ||
     GTK_RESPONSE_APPLY == res || GTK_RESPONSE_ACCEPT == res)
 {
  return 1;
 }
 else
  return 0;
}

//========================================
//NOTE: THIS OPTION WAS DEACTIVATED IN THE GLADE FILE 20071203
void main_on_export_mp3_activate ()
{
 //This is experimental, requires LAME, and runs a second instance of Gnaural in the background;
 //Sorry, Windows users - I don't have a way to hack this for you yet :-(

 main_MessageDialogBox
  ("Sorry, Gnaural2 can't make MP3's yet, but you can make a WAV file then covert it to MP3 with LAME\nhttp://lame.sourceforge.net",
   GTK_MESSAGE_INFO, GTK_BUTTONS_OK);
 return;

#ifdef GNAURAL_WIN32
 main_MessageDialogBox
  ("Currently, the Windows version can only do this at the command line. Be sure LAME is installed\n(see http://lame.sourceforge.net/),\nthen type:\n\n\tgnaural -o | lame - MyMeditation.mp3",
   GTK_MESSAGE_INFO, GTK_BUTTONS_OK);
 return;
#endif

 if (BB_Loops < 1)
 {
  if (0 ==
      main_MessageDialogBox
      ("You are in endless-loop mode. Do you really want to fill all the space on your hard drive?",
       GTK_MESSAGE_WARNING, GTK_BUTTONS_YES_NO))
  {
   return;	//don't start write an infinitely large file
  }
 }

 //TODO: check automatically for LAME
 if (0 ==
     main_MessageDialogBox
     ("Do you have LAME installed? If not, this step will fail.\nSee http://lame.sourceforge.net/",
      GTK_MESSAGE_WARNING, GTK_BUTTONS_YES_NO))
 {
  return;
 }
 //TODO check disk space for user, update GUI status window to tell that writing is going on

 //Ask for a name for the MP3 file:
 GtkWidget *dialog;

 dialog =
  gtk_file_chooser_dialog_new ("Save MP3 Audio File",
			       GTK_WINDOW (main_window),
			       GTK_FILE_CHOOSER_ACTION_SAVE, GTK_STOCK_CANCEL,
			       GTK_RESPONSE_CANCEL, GTK_STOCK_SAVE,
			       GTK_RESPONSE_ACCEPT, NULL);
 //gtk_entry_set_text (GTK_ENTRY (entry_Input), "Gnaural.wav");
 gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog),
				      main_sPathCurrentWorking);
 gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (dialog), "Gnaural.mp3");

 //gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER (dialog), TRUE);//doesn't exist yet in Debian unstable

 if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
 {
  char *filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));

  if (strlen (filename) > 0)
  {
   //this works great, just doesn't give user any way to know when it is done, and (worse) doesn't always reliably find it's own executable.
   //###############FOR LINUX################
#ifndef GNAURAL_WIN32
   sprintf (main_sPathTemp, "%s -o | lame - %s &", main_sPathExecutable,
	    filename);
   system (main_sPathTemp);
#endif
   //###############FOR WIN32################
#ifdef GNAURAL_WIN32
   sprintf (main_sPathTemp, "%s -o | lame - %s", main_sPathExecutable,
	    filename);
   system (main_sPathTemp);
   //  _spawn(main_sPathTemp);
   //_spawnlp (_P_DETACH, szFile_ExecutablePath, "-o", "|", "lame","-",  filename, NULL);
   //_execlp (szFile_ExecutablePath, "-o", "|", "lame","-",  filename, NULL);
   /*
      ShellExecute(0,
      "open", // Operation to perform
      main_sPathTemp, // Application name
      NULL, // Additional parameters
      0, // Default directory
      SW_SHOW);
    */
#endif

   main_MessageDialogBox
    ("A second copy of Gnaural has be spawned, and will run in the background until finished. In case of a problem, you can kill the processes with:\nkillall gnaural2",
     GTK_MESSAGE_INFO, GTK_BUTTONS_OK);
  }
  g_free (filename);
 }
 gtk_widget_destroy (dialog);
}

//START writethreadstuff======================================
/////////////////////////////////////////////////////
void main_on_export_audio_to_file_activate ()
{
 char *filename = NULL;
 char wav[] = ".wav";
 char flac[] = ".flac";
 char aiff[] = ".aiff";
 char au[] = ".au";
 char raw[] = ".raw";
 char *typestring = wav;

 if (BB_Loops < 1)
 {
  if (0 ==
      main_MessageDialogBox
      ("You are in endless-loop mode. Do you really want to fill all the space on your hard drive?",
       GTK_MESSAGE_WARNING, GTK_BUTTONS_YES_NO))
  {
   return;	//can't write an infinitely large file! (COULD USE A DIALOG)
  }
 }
 if (main_AudioWriteFile_thread != NULL)
 {
  main_MessageDialogBox ("You are already writing a file.",
			 GTK_MESSAGE_ERROR, GTK_BUTTONS_OK);
  return;	//this means someone is already writing (COULD USE A DIALOG)
 }
 //TODO check disk space for user, update GUI status window to tell that writing is going on
 GtkWidget *dialog;

 dialog =
  gtk_file_chooser_dialog_new ("Save WAV Audio File",
			       GTK_WINDOW (main_window),
			       GTK_FILE_CHOOSER_ACTION_SAVE, GTK_STOCK_CANCEL,
			       GTK_RESPONSE_CANCEL, GTK_STOCK_SAVE,
			       GTK_RESPONSE_ACCEPT, NULL);

 main_DialogAddFileFilters (dialog,
			    "~Gnaural Audio Files~*.wav,*.aiff,*.au,*.flac,~All Files~*");

 //create a Box to put some extra stuff in:
 GtkWidget *box = gtk_vbox_new (FALSE, 0);

 gtk_widget_show (box);
 gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER (dialog), box);

 //add some information for user
 GtkWidget *label_Info =
  gtk_label_new
  ("Filename extension will be appended automatically according to format chosen below:");
 gtk_widget_show (label_Info);
 gtk_box_pack_start (GTK_BOX (box), label_Info, FALSE, FALSE, 0);

 GtkWidget *comboboxentry1 = gtk_combo_box_entry_new_text ();

 gtk_widget_show (comboboxentry1);

 //DON'T change order on the enum without changing in the following 
 //gtk_combo_box_append_text's:
 enum
 { WAV, FLAC, AIFF, AU, RAW };

 gtk_combo_box_append_text (GTK_COMBO_BOX (comboboxentry1),
			    "WAV - Microsoft WAV format (little endian)");
 gtk_combo_box_append_text (GTK_COMBO_BOX (comboboxentry1),
			    "FLAC - Free Lossless Audio Codec format");
 gtk_combo_box_append_text (GTK_COMBO_BOX (comboboxentry1),
			    "AIFF - Apple/SGI AIFF format (big endian)");
 gtk_combo_box_append_text (GTK_COMBO_BOX (comboboxentry1),
			    "AU - Sun/NeXT AU format (big endian)");
 gtk_combo_box_append_text (GTK_COMBO_BOX (comboboxentry1),
			    "RAW - RAW PCM data");

 //Translate last main_AudioWriteFile_type to a comboboxentry:
 int filetype = WAV;

 switch (main_AudioWriteFile_type)
 {
 case SF_FORMAT_WAV:
  filetype = WAV;
  break;

 case SF_FORMAT_AIFF:
  filetype = AIFF;
  break;

 case SF_FORMAT_AU:
  filetype = AU;
  break;

 case SF_FORMAT_RAW:
  filetype = RAW;
  break;

 case SF_FORMAT_FLAC:
  filetype = FLAC;
  break;

 default:
  filetype = WAV;
  break;
 }

 //make filetype hightlight last chosen comboboxentry for format:
 gtk_combo_box_set_active ((GtkComboBox *) comboboxentry1, filetype);
 gtk_box_pack_end (GTK_BOX (box), comboboxentry1, FALSE, FALSE, 0);

 //extract base from filename for convenience:
 filename = g_path_get_basename (main_AudioWriteFile_name);
 gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (dialog), filename);
 g_free (filename);

 if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
 {
  filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
  if (strlen (filename) > 0)
  {
   filetype = gtk_combo_box_get_active ((GtkComboBox *) comboboxentry1);
   switch (filetype)
   {
   case WAV:
    main_AudioWriteFile_type = SF_FORMAT_WAV;
    typestring = wav;
    break;

   case AIFF:
    main_AudioWriteFile_type = SF_FORMAT_AIFF;
    typestring = aiff;
    break;

   case AU:
    main_AudioWriteFile_type = SF_FORMAT_AU;
    typestring = au;
    break;

   case RAW:
    main_AudioWriteFile_type = SF_FORMAT_RAW;
    typestring = raw;
    break;

   case FLAC:
    main_AudioWriteFile_type = SF_FORMAT_FLAC;
    typestring = flac;
    break;

   default:
    main_AudioWriteFile_type = SF_FORMAT_WAV;
    typestring = wav;
    break;
   }

   //now, check for extension, if there is one, see if it is same
   //as filetype; if so do nothing, otherwise add correct one here:
   SG_StringAllocateAndCopy (&main_AudioWriteFile_name, filename);
   char *ext;

   for (ext = &(main_AudioWriteFile_name[strlen (main_AudioWriteFile_name)]);
	main_AudioWriteFile_name != ext && ext[0] != '.'; ext--);
   if (ext == main_AudioWriteFile_name)
   {
    BB_DBGOUT ("Filename has no extension");
   }
   else
   {
    //make sure really removing a Gnaural-understood extension:
    BB_DBGOUT_STR ("Previous file extension:", ext);
    if (0 == strncasecmp (ext, flac, 5) ||
	0 == strncasecmp (ext, wav, 4) ||
	0 == strncasecmp (ext, aiff, 5) ||
	0 == strncasecmp (ext, au, 3) || 0 == strncasecmp (ext, raw, 4))
    {
     //TODO: it would be nice if I alerted user IF their extension
     //is different from the SF_AUDIOFORMAT they "chose", because
     //I just replace it anyway:
     BB_DBGOUT ("Extension known - removing");
     *ext = '\0';	//"remove" extension
    }
    else
    {
     BB_DBGOUT ("Extension unknown - not removing");
    }
   }

   //put an extension on:
   strcpy (main_sPathTemp, main_AudioWriteFile_name);
   strcat (main_sPathTemp, typestring);

   //see if file already exists:
   if (0 == main_TestPathOrFileExistence (main_sPathTemp) &&
       0 == main_MessageDialogBox ("File exists. Overwrite it?",
				   GTK_MESSAGE_WARNING, GTK_BUTTONS_YES_NO))
   {
    BB_DBGOUT ("Not starting Audio File Write");
   }
   else
   {
    SG_StringAllocateAndCopy (&main_AudioWriteFile_name, main_sPathTemp);
    BB_DBGOUT_STR ("New Filename:", main_AudioWriteFile_name);
    main_OnButton_Stop (NULL);
    main_AudioWriteFile_start ();
   }
  }
  g_free (filename);
 }
 gtk_widget_destroy (dialog);
}

//========================================
void main_AudioWriteFile (void *ptr)
{
 BB_DBGOUT ("Starting Audio Write File thread");
 BB_DBGOUT_INT ("Audio SF_FORMAT type:", main_AudioWriteFile_type);
 //BB_WriteWAVFile (main_AudioWriteFile_name); //OLD WAY, replaced 20071130
 exportAudioToFile (main_AudioWriteFile_name, main_AudioWriteFile_type);
 main_AudioWriteFile_thread = NULL;	//so rest of program knows writing is done
 BB_DBGOUT ("Audio Write File thread exiting");
 g_thread_exit (0);	//this should be updated to g_thread_join(GThread) when I get more time
}

//======================================
void main_AudioWriteFile_start ()
{
 if (main_AudioWriteFile_thread != NULL)
 {
  main_MessageDialogBox ("You are already writing one", GTK_MESSAGE_ERROR,
			 GTK_BUTTONS_OK);
  return;	//this means someone is already writing
 }
 //Note: you must be in paused state to do writing!
 //bbl_UpdateGUI_Info ();

 //user can set this to non-0 at any time to stop write:
 BB_WriteStopFlag = 0;
 //new gthread code (new 20051126):
 GError *thread_err = NULL;

 if (NULL ==
     (main_AudioWriteFile_thread
      =
      g_thread_create ((GThreadFunc) main_AudioWriteFile, (void *) NULL, TRUE,
		       &thread_err)))
 {
  fprintf (stderr, "g_thread_create failed: %s!!\n", thread_err->message);
  g_error_free (thread_err);
 }
}

//END writethreadstuff======================================

//====START OF IMPORT OLD GNAURAL FILE SECTION====
//======================================
//this just reads the values found between brackets []
void main_GNAURAL1FILE_ParseCmd (FILE * stream)
{
 char strNum[128];
 char strCmd[128];
 char tmpchar;
 int inum = 0,
  icmd = 0;

 strNum[0] = strCmd[0] = '\0';
 while ((tmpchar = fgetc (stream)) != ']' && feof (stream) == 0)
 {
  //eat up any extra characters
  //while (tmpchar=='=' || ' ') tmpchar=fgetc( stream );
  //if it is a number, put it in the number string:
  if ((tmpchar >= '0' && tmpchar <= '9') || tmpchar == '.' || tmpchar == '-')
  {
   strNum[inum] = tmpchar;
   inum++;
  }

  //if it is a string, put it in Cmd string:
  else
   if ((tmpchar >= 'A'
	&& tmpchar <= 'Z') || (tmpchar >= 'a' && tmpchar <= 'z'))
  {
   strCmd[icmd] = toupper (tmpchar);
   icmd++;
  }
  else if (tmpchar == '#')
  {	//user put a comment after a command!
   while (feof (stream) == 0 && tmpchar != '\n')
    tmpchar = fgetc (stream);
   return;
  }
 }	//end of parsing line:

 strNum[inum] = '\0';
 strCmd[icmd] = '\0';
 if (!strcmp (strCmd, "BASEFREQ"))
 {
  OldGnaualVars.FreqBase = (float) g_ascii_strtod (strNum, NULL);
  if (OldGnaualVars.FreqBase < 40)
   OldGnaualVars.FreqBase = 40;
  if (OldGnaualVars.FreqBase > 1000)
   OldGnaualVars.FreqBase = 1000;
 }

 if (!strcmp (strCmd, "TONEVOL"))
 {
  OldGnaualVars.Volume_Tone = (float) g_ascii_strtod (strNum, NULL);
  if (OldGnaualVars.Volume_Tone < 0)
   OldGnaualVars.Volume_Tone = 0;
  if (OldGnaualVars.Volume_Tone > 100)
   OldGnaualVars.Volume_Tone = 100;
  //    OldGnaualVars.Volume_Tone = (int) (163.84 * ftemp + 0.5);
  OldGnaualVars.Volume_Tone = (float) (OldGnaualVars.Volume_Tone * 0.01);
 }

 if (!strcmp (strCmd, "NOISEVOL"))
 {
  OldGnaualVars.Volume_Noiz = (float) g_ascii_strtod (strNum, NULL);
  if (OldGnaualVars.Volume_Noiz < 0)
   OldGnaualVars.Volume_Noiz = 0;
  if (OldGnaualVars.Volume_Noiz > 100)
   OldGnaualVars.Volume_Noiz = 100;
  OldGnaualVars.Volume_Noiz = (float) (OldGnaualVars.Volume_Noiz * 0.01);
 }

 if (!strcmp (strCmd, "LOOPS"))
 {
  OldGnaualVars.loops = atoi (strNum);
  if (OldGnaualVars.loops < 0)
   OldGnaualVars.loops = 0;
  OldGnaualVars.loopcount = OldGnaualVars.loops;
 }

 if (!strcmp (strCmd, "STEREONOISE"))
 {
  if (atoi (strNum) > 0)
   OldGnaualVars.StereoNoiz = 1;
  else
   OldGnaualVars.StereoNoiz = 0;
 }
}

//======================================
void main_GNAURAL1FILE_SchedBuffToSchedule (char *str)
{
 //the philosophy here: take a string with commas separating the
 //floats, hunt for commas. When one is found, start a new string
 //beginning with next character. Take every following character up
 //to (but not including) the next comma. Then store that float, dispose
 //of the tempstring, and hunt some more until length of string is
 //covered.
 char tmpstr[256];
 float *tmpfloat = NULL;
 char substr[256];

 //first count how many floats are in the string:
 int floatcount = 1;
 unsigned long i;

 for (i = 0; i < strlen (str); i++)
 {
  if (str[i] == ',')
  {
   floatcount++;
  }	// end if comma
 }	//end for i
 //do it all again, now that I know how many floats:
 tmpfloat = (float *) malloc (floatcount * sizeof (float));
 floatcount = 0;
 int j = 0;

 for (i = 0; i < strlen (str); i++)
 {
  //first extract a number to a string:
  if ((str[i] >= '0' && str[i] <= '9') || str[i] == '.' || str[i] == '-')
   tmpstr[j++] = str[i];
  //if I found a comma, end the string:
  else if (str[i] == ',')
  {
   tmpstr[j] = '\0';
   strcpy (substr, tmpstr);
   tmpfloat[floatcount] = (float) g_ascii_strtod (substr, NULL);
   j = 0;
   floatcount++;
  }	// end if comma
 }	//end for i
 if (j != 0)
 {	//there should be one more float to get:
  tmpstr[j] = '\0';
  strcpy (substr, tmpstr);
  tmpfloat[floatcount] = (float) g_ascii_strtod (substr, NULL);
  ++floatcount;
 }	// end if j!=0
 //==START LOADING THE FLOATS IN TO THE GRAPH:
 OldGnaualVars.ScheduleEntriesCount = floatcount / 3;
 //Now init SG so that it can start to receive the schedule.
 //Since this is an old Gnaural file, there must be only two voices, first being
 //the explicitly directed binaural beat, the second simple unmodulated noise. The
 //philosophy of Gnaural file opening is to let the Graph (SG) always lead the BB
 //engine, so load the graph by creating 2 voices with SG_SetupDefaultDataPoints(),
 //then delete all DPs and start loading adding real ones to the end (the noise one only
 //has one DP, with the volume set).
 SG_SetupDefaultDataPoints (2);	//first voice is BB data, second will be noise
 //important: cleanup all datapoints (since new data will be instated), and NULL first DPs:
 SG_CleanupDataPoints (SG_FirstVoice->FirstDataPoint);
 SG_FirstVoice->FirstDataPoint = NULL;
 SG_CleanupDataPoints (SG_FirstVoice->NextVoice->FirstDataPoint);
 SG_FirstVoice->NextVoice->FirstDataPoint = NULL;
 //for reference:
 /*
    struct  {
    float FreqBase;
    float Volume_Tone;
    float Volume_Noiz;
    int loops;
    int loopcount;
    int StereoNoiz;
    int ScheduleEntriesCount;
    } OldGnaualVars;
  */
 //==START load BinauralBeat voice:
 int buffcount = 0;
 float FreqLStart,
  FreqRStart,
  Duration,
  Duration_total = 0;
 int entry;
 SG_DataPoint *curDP = NULL;

 for (entry = 0; entry < OldGnaualVars.ScheduleEntriesCount; entry++)
 {
  //get the buffer values, put them in temporary storage:
  FreqLStart = (float) tmpfloat[buffcount++];
  FreqRStart = (float) tmpfloat[buffcount++];
  Duration = (float) tmpfloat[buffcount++];
  Duration_total += Duration;
  //apply temporary storage to a real: DP
  curDP = SG_AddNewDataPointToEnd (SG_FirstVoice->FirstDataPoint, SG_FirstVoice, Duration,	//duration
				   fabs (FreqLStart - FreqRStart),	//beatfreq
				   OldGnaualVars.Volume_Tone,	//volume_left
				   OldGnaualVars.Volume_Tone,	//volume_right
				   OldGnaualVars.FreqBase,	//basefreq
				   SG_UNSELECTED);	//state
  if (SG_FirstVoice->FirstDataPoint == NULL)
   SG_FirstVoice->FirstDataPoint = curDP;
 }	//end for
 //==END load BinauralBeat voice

 //==START load Noise voice:
 curDP = SG_AddNewDataPointToEnd (SG_FirstVoice->NextVoice->FirstDataPoint, SG_FirstVoice->NextVoice, Duration_total,	//duration
				  0,	//beatfreq
				  OldGnaualVars.Volume_Noiz,	//volume_left
				  OldGnaualVars.Volume_Noiz,	//volume_right
				  0,	//basefreq
				  SG_UNSELECTED);	//state
 SG_FirstVoice->NextVoice->FirstDataPoint = curDP;
 //==END load Noise voice:
 //==END LOADING THE FLOATS IN TO THE GRAPH:
 main_SetLoops (OldGnaualVars.loops);
 //  SG_DBGOUT("Got to line 1934");
 free (tmpfloat);
}	//end of extractfloats()

//======================================
//THIS IS INCLUDED SOLEY TO OPEN OLD STYLE GNAURAL SCHEDULES
//opens whatever is passed as filename, dumps it in to str,
//then uses SchedBuffToSchedule to fill Schedule. has to go like this
//because it has to segregate all "command strings" [] from schedule entries
int main_GNAURAL1FILE_SchedFilenameToSchedule (char *filename)
{
 FILE *stream;
 char *str;

 if ((stream = fopen (filename, "r")) == NULL)
 {	//didn't find a schedule file, so return (could try to create one at this point with WriteScheduleToFile()
  return 0;
 }
 //FileFlag = true;

 //START count file elements to know how big to make the char buffer:
 char tmpchar;
 unsigned int i = 0;

 while (feof (stream) == 0)
 {
  tmpchar = fgetc (stream);
  //comments: eat entire lines starting with #:
  if (tmpchar == '#')
  {
   while (feof (stream) == 0 && tmpchar != '\n')
    tmpchar = fgetc (stream);
  }

  //  if (tmpchar>='0' && tmpchar <='9') i++;
  //  else if (tmpchar==',' || tmpchar=='.') i++;

  if ((tmpchar >= '0'
       && tmpchar <= '9')
      || tmpchar == ',' || tmpchar == '.' || tmpchar == '-')
  {
   i++;
  }
 }	//end while
 //END count file elements

 /*
    //20070116 DECIDED NOT TO USE THIS METHOD:
    //Now init SG so that it can start to receive the schedule.
    //Since this is an old Gnaural file, there must be only two voices, first being
    //the assumed noise, the second being the explicitly directed binaural beat. The
    //philosophy of Gnaural file opening is to let the Graph (SG) always lead the BB
    //engine; and in loading the graph, it is customary to load the SG_UndoRedo buffer
    //with the file data, because when using SG_RestoreDataPoints() you only need:
    // 1) SG_UndoRedo.count field filled (total count of ALL datapoints)
    // 2) SG_UndoRedo.VoiceType allocated (one for each voice) and filled
    // 3) SG_UndoRedo.DPdata allotted and these DPdata struct fields filled:
    //  -  duration
    //  -  beatfreq
    //  -  basefreq
    //  -  volume_left
    //  -  volume_right
    //  -  state
    //  -  parent
  */

 //preparing the holding vars for command line parsing:
 OldGnaualVars.FreqBase = 220;
 OldGnaualVars.Volume_Tone = .5;
 OldGnaualVars.Volume_Noiz = .5;
 OldGnaualVars.loops = 1;
 OldGnaualVars.loopcount = 1;
 OldGnaualVars.StereoNoiz = 1;	//not implemented right now; needs to set BB_Voice[voice].StereoNoiseFlag, but SG doesn't have such a voice parameter in the struct
 //START allocate char buffer, rewind file, and do it again this time putting elements in buffer:
 str = (char *) malloc ((++i));
 rewind (stream);
 i = 0;
 while (feof (stream) == 0)
 {
  tmpchar = fgetc (stream);
  //deal with comments:
  if (tmpchar == '#')
  {	//throw whole line away:
   while (feof (stream) == 0 && tmpchar != '\n')
    tmpchar = fgetc (stream);
  }

  //deal with command strings:
  else if (tmpchar == '[')
  {
   main_GNAURAL1FILE_ParseCmd (stream);
  }

  //if it is a number, add it to the monster string:
  else
   if ((tmpchar >= '0'
	&& tmpchar <= '9')
       || tmpchar == ',' || tmpchar == '.' || tmpchar == '-')
  {
   str[i] = tmpchar;
   i++;
  }
 }
 //END allocate char buffer, rewind file, and do it again this time putting elements in buffer:

 fclose (stream);
 //put end on the string!!
 str[i] = '\0';
 main_GNAURAL1FILE_SchedBuffToSchedule (str);
 free (str);
 return 1;
}

//====END OF IMPORT OLD GNAURAL FILE SECTION====

//======================================
//I bet I could do this without actually writing to disk, but
//have wrestled with so many bizarrely non-functional GTK
//themed/builtin/window/named icon functions that
//after a full day of failing, I must simply do something
//I understand (see icon_problems.txt):
void main_SetIcon ()
{
#ifdef GNAURAL_WIN32
 // Get the temp path.
 if (0 != GetTempPath (sizeof (main_sPathTemp), main_sPathTemp))	//NOTE: returned string ends in backslash
 {	//success
  strcat (main_sPathTemp, "gnaural2-icon.png");
 }
 else
 {	//fail
  sprintf (main_sPathTemp, "gnaural2-icon.png");
 }
#endif

#ifndef GNAURAL_WIN32
 sprintf (main_sPathTemp, "%s/%s", main_sPathGnaural, "gnaural2-icon.png");
#endif
 FILE *stream;

 if ((stream = fopen (main_sPathTemp, "wb")) == NULL)
 {
  return;
 }
 fwrite (&main_GnauralIcon, sizeof (main_GnauralIcon), 1, stream);
 fclose (stream);
 gtk_window_set_default_icon_from_file (main_sPathTemp, NULL);
}

//======================================
void main_InterceptCtrl_C (int sig)
{
 SG_DBGOUT ("Caught Ctrl-C");
 /*
    if (gnaural_guiflag == GNAURAL_GUIMODE) {
    //20060328: I used to just call gtk_main_quit() here, but it
    //is smarter to let the callback functions associated with window1
    //do their own cleanup:
    gtk_widget_destroy( window1 );
    //   ScheduleGUI_delete_event
    DBGOUT ("Quit GUI mode");
    }
  */
 main_Cleanup ();
 exit (EXIT_SUCCESS);
}

//======================================
void main_ParseCmdLine (int argc, char *argv[])
{
 opterr = 0;	//this means I want to check the ? character myself for errors
 int c;

 //NOTE: options with ":" after them require arguments
 while ((c = getopt (argc, argv, GNAURAL_CMDLINEOPTS)) != -1)
  switch (c)
  {
  case 'h':	//print help
   fprintf (stdout, "Gnaural (ver. %s) - ", main_VERSION_GNAURAL);
   fputs (main_GnauralHelp, stdout);
   exit (EXIT_SUCCESS);	//completely quit if the user asked for help.
   break;
  case 'i':	//show terminal-style gui info
   ++cmdline_i;
   break;
  case 'p':	//output directly to sound system
   ++cmdline_p;
   break;
  case 'd':	//output debugging info to stderr
   ++cmdline_d;
   SG_DebugFlag = TRUE;
   BB_DebugFlag = TRUE;
   //may go back in to GUI mode, can't know from this
   break;
  case 'o':	//output a .WAV stream to stdout
   ++cmdline_o;
   break;
  case 's':	//create default gnaural_schedule.txt file
   ++cmdline_s;
   if (0 < cmdline_w)
   {
    SG_ERROUT ("Ignoring -w; -s has priority");
    cmdline_w = 0;	//just in case I missed something...
   }
   break;
  case 'a':
   ++cmdline_a;
   mainPA_SoundDevice = atoi (optarg);
   //may go back in to GUI mode, can't know from this
   break;
  case 'w':	//output a .WAV stream to a user specified .wav file
   if (cmdline_s > 0)
   {	//CHANGE 051110: you don't want -s and -w in the same line
    SG_ERROUT ("Ignoring -w; -s has priority");
    cmdline_w = 0;	//just in case I missed something...
    break;
   }
   ++cmdline_w;
   //20071201 Following could probably be done better now:
   SG_StringAllocateAndCopy (&main_AudioWriteFile_name, optarg);
   break;
  case ':':	//some option was missing its argument
   fprintf (stderr, "Missing argument for `-%c'.\n", optopt);
   fputs (main_GnauralHelpShort, stderr);
   exit (EXIT_FAILURE);	//completely quit on error
  case '?':
   fprintf (stderr, "Unknown option `-%c'.\n", optopt);
   fputs (main_GnauralHelpShort, stderr);
   exit (EXIT_FAILURE);	//completely quit on error
  default:
   fputs (main_GnauralHelpShort, stderr);
   exit (EXIT_FAILURE);	//completely quit on error
   break;
  }

 //now see if there was any command-line stuff user inputted that didn't
 //get caught above.
 //a20070705: The last thing to get here is considered a filename
 int index;

 for (index = optind; index < argc; index++)
 {
  sprintf (main_sCurrentGnauralFilenameAndPath, argv[index]);
  //  fprintf (stderr, "Error: non-option argument \"%s\"\n", argv[index]);
 }
 //END command line parsing
 return;
}

//======================================
//this is basically just the command line version dropped in as a function-call in
//to the GUI version.
void main_RunCmdLineMode ()
{
 //don't need GUI, so turn pause off:
 gnaural_pauseflag = 0;
 //first do the things that don't require sound at all:
 //- cmdline_s: create a fresh gnaural_schedule.txt file then exits DOES NOT REQUIRE SOUND SYSTEM
 //- cmdline_o: dump a .WAV file to sdtout DOES NOT REQUIRE SOUND SYSTEM
 //- cmdline_w: dump a .WAV file to a file DOES NOT REQUIRE SOUND SYSTEM
 // user asked to create a new default Schedule file;  do that then exit:
 if (cmdline_s > 0)
 {
  //  bb->WriteDefaultScheduleToFile ();
  SG_SetupDefaultDataPoints (2);
  main_XMLWriteFile ("schedule.gnaural");
  SG_DBGOUT ("Wrote default gnaural_schedule.txt file");
  return;
 }

 ////////////////
 //NOTE:
 //from this point forward, I work on the premise that user may be piping
 //output to stdout to another program expecting a WAV file
 ////////////////

 //these next three are mutually exclusive, and only cmdline_p needs sound system:
 if (cmdline_w != 0)
 {
  //old direct way of doing this:
  //      fprintf (stderr, "Writing schedule %s to file %s... ",
  //        bb->SchedFilename, w_str);
  //      bb->WriteWAVFile (w_str);
  //      fprintf (stderr, "Done.\n");
  //      return (EXIT_SUCCESS);

  //threaded way of doing the above:
  if (cmdline_i != 0)
  {
   fputs ("\033[2J", stderr);	//clear entire screen
   fputs ("\033[14;1H", stderr);	//place cursor
  }

  fprintf (stderr,
	   "Writing WAV data to file %s... \n", main_AudioWriteFile_name);
  //new gthread code (new 20051126):
  GError *thread_err = NULL;

  if ((main_AudioWriteFile_thread =
       g_thread_create ((GThreadFunc) main_AudioWriteFile, (void *) NULL,
			TRUE, &thread_err)) == NULL)
  {
   fprintf (stderr, "g_thread_create failed: %s!!\n", thread_err->message);
   g_error_free (thread_err);
  }

  while
   (main_AudioWriteFile_thread
    != NULL && (((BB_InfoFlag) & BB_COMPLETED) == 0))
  {
   if (cmdline_i != 0)
   {
    main_UpdateTerminalGUI (stdout);
    fflush (stdout);	//must flush often when using stdout
   }
   main_Sleep (G_USEC_PER_SEC);
  }
  if (cmdline_i != 0)
   fputs ("\033[14;1H", stderr);	//place cursor
  return;
 }

 //----------

 else if (cmdline_o != 0)
 {
  BB_WriteWAVToStream (stdout);
  return;
 }

 //----------

 else if (cmdline_p != 0)
 {	//write to sound system:
  //First, set up the sound system, or exit if not possible:
  //first set-up sound, since program may not return from parsing command line
  if (EXIT_FAILURE == mainPA_SoundInit ())
  {
   SG_DBGOUT ("Couldn't intialize the sound system, exiting");
   Pa_Terminate ();
   mainPA_SoundStream = NULL;	//this is how the rest of the program knows if we're using sound or not.
   return;
  }

  SG_DBGOUT ("Writing to sound system");
  if (cmdline_i != 0)
  {
   fputs ("\033[2J", stderr);	//clear entire screen
   fputs ("\033[14;1H", stderr);	//place cursor
  }

  mainPA_SoundStart ();
  while (mainPA_SoundStream != NULL && (((BB_InfoFlag) & BB_COMPLETED) == 0))
  {
   if (cmdline_i != 0)
   {
    main_UpdateTerminalGUI (stdout);
    fflush (stdout);	//must flush often when using stdout
   }
   Pa_Sleep (1000);
  };
  if (cmdline_i != 0)
  {
   fputs ("\033[14;1H", stderr);	//place cursor
  }
  return;
 }
}

//======================================
//updates a terminal window via either stdout or stderr streams
void main_UpdateTerminalGUI (FILE * gstream)
{
 //NOTE THIS IS CURRENTLY MOSTLY UNIMPLEMENTED; a lot of decisions
 //will need to be made how to present multi voiced data without a GUI
 // current time point within schedule:

 main_FormatProgressString ();
 fputs ("\033[2;1H", gstream);	//place cursor
 fprintf (gstream, main_sTmp);
 fputs ("\033[K", gstream);	//clear to end of line
 /*
    //-----------------------------------------------------------
    //START stuff that gets updated EVERY second:
    //
    // current time point within schedule:
    fputs("\033[2;1H", gstream); //place cursor
    fprintf (gstream, "Current Progress: \t%d min. \t%d sec.",
    (BB_CurrentSampleCount + BB_CurrentSampleCountLooped) / 6000,
    ((BB_CurrentSampleCount + BB_CurrentSampleCountLooped) / 100) % 60);
    fputs("\033[K", gstream); //clear to end of line

    //remaining time in entry:
    fputs("\033[5;1H", gstream); //place cursor
    fprintf (gstream, "Time left: \t\t%d sec.",
    ((bb->Schedule[bb->ScheduleCount].AbsoluteEndTime_100 - BB_CurrentSampleCount) / 100));
    fputs("\033[K", gstream); //clear to end of line

    //Left frequency:
    fputs("\033[9;1H", gstream); //place cursor
    fprintf (gstream, "Current Freq. Left: \t%g\n",
    bb->FreqL);
    fputs("\033[K", gstream); //clear to end of line

    //Right frequency:
    // fputs("\033[15;1H",gstream);//place cursor
    fprintf (gstream, "Current Freq. Right: \t%g\n",
    bb->FreqR);
    fputs("\033[K", gstream); //clear to end of line

    //Beat frequency:
    // fputs("\033[16;1H",gstream);//place cursor
    fprintf (gstream, "Current Beat Freq.: \t%g\n",
    bb->FreqL - bb->FreqR);
    fputs("\033[K", gstream); //clear to end of line

    //
    //END stuff that gets updated EVERY second:
    //-----------------------------------------------------------------------

    //-----------------------------------------------------------------------
    //START check bb->InfoFlag
    //now update things that BinauralBeat says are changed:
    if (bb->InfoFlag != 0) {

    //.........................
    //if schedule is done:
    if (((bb->InfoFlag) & BB_COMPLETED) != 0) {
    fputs("\033[7;1H", gstream); //place cursor
    fprintf (gstream, "Schedule Completed");
    fputs("\033[K", gstream); //clear to end of line
    }

    //.........................
    //if starting a new loop:
    if (((bb->InfoFlag) & BB_NEWLOOP) != 0) {
    //reset the "new loop" bit of InfoFlag:
    bb->InfoFlag &= ~BB_NEWLOOP;
    //bbl_UpdateGUI_LoopInfo ();
    }

    //.........................
    //START things that get updated every new entry:
    //if new entry
    if (((bb->InfoFlag) & BB_NEWENTRY) != 0) {
    //reset the "new entry" bit of InfoFlag:
    (bb->InfoFlag) &= (~BB_NEWENTRY);

    //current entry number:
    fputs("\033[3;1H", gstream); //place cursor
    fprintf (gstream, "Current Entry: \t\t%d of %d",
    bb->ScheduleCount + 1, bb->ScheduleEntriesCount);
    fputs("\033[K", gstream); //clear to end of line

    //total time in entry:
    fputs("\033[4;1H", gstream); //place cursor
    fprintf (gstream, "Duration: \t\t%g sec.",
    bb->Schedule[bb->ScheduleCount].Duration);
    fputs("\033[K", gstream); //clear to end of line

    //fprintf(gstream,"%d",bb->Schedule[bb->ScheduleCount].AbsoluteEndTime_100/100);
    //SetDlgItemText(lEndCount, gstream);

    fputs("\033[6;1H", gstream); //place cursor
    fprintf (gstream, "Beat Range: \t\t%g to %g Hz\n",
    bb->Schedule[bb->ScheduleCount].FreqLStart -
    bb->Schedule[bb->ScheduleCount].FreqRStart,
    bb->Schedule[bb->ScheduleCount].FreqLEnd -
    bb->Schedule[bb->ScheduleCount].FreqREnd);
    fputs("\033[K", gstream); //clear to end of line

    //left
    // fputs("\033[7;1H",gstream);//place cursor
    fprintf (gstream, "Left ear: \tStart:\t%g",
    bb->Schedule[bb->ScheduleCount].FreqLStart);
    fputs("\033[K", gstream); //clear to end of line
    fprintf (gstream, "\tEnd:\t%g\n",
    bb->Schedule[bb->ScheduleCount].FreqLEnd);
    fputs("\033[K", gstream); //clear to end of line

    //right
    // fputs("\033[8;1H",gstream);//place cursor
    fprintf (gstream, "Right ear: \tStart:\t%g",
    bb->Schedule[bb->ScheduleCount].FreqRStart);
    fputs("\033[K", gstream); //clear to end of line
    fprintf (gstream, "\tEnd:\t%g",
    bb->Schedule[bb->ScheduleCount].FreqREnd);
    fputs("\033[K", gstream); //clear to end of line

    //this is overdoing it, but need it updated sometime:

    if (bb->loops > 0) {
    fputs("\033[1;1H", gstream); //place cursor
    fprintf (gstream, "Projected Runtime: \t%d min. \t%d sec.",
    (int) (bb->loops * bb->ScheduleTime) / 60,
    (int) (bb->loops * bb->ScheduleTime) % 60);
    fputs("\033[K", gstream); //clear to end of line
    } else {
    fputs("\033[1;1H", gstream); //place cursor
    fprintf (gstream, "Projected Runtime:  FOREVER (Inf. Loop Mode)");
    fputs("\033[K", gstream); //clear to end of line
    }

    }
    //END things that get updated every new entry
    }
  */
}

//======================================
void main_WriteDefaultFile ()
{
 //20070705: Used to write to whatever was in main_sCurrentOpenGnauralFile,
 //but now sets-up default names and paths first:
 main_sCurrentGnauralFilenameAndPath[0] = '\0';	//not necessary, I guess
 main_SetupPathsAndFiles (TRUE);
 //prepare file access:
 FILE *stream;

 if ((stream = fopen (main_sCurrentGnauralFilenameAndPath, "w")) == NULL)
 {
  SG_ERROUT ("Failed to open file for writing!");
  return;
 }
 fprintf (stream, "%s", main_DefaultSchedule);
 fclose (stream);
}

/////////////////////////////////////////////////////
//20070627: This edits properties of an existing
//voices, OR creates a new voice if curVoice = NULL
gboolean main_VoicePropertiesDialog (GtkWidget * widget, SG_Voice * curVoice)
{
 GtkWidget *dialog = gtk_dialog_new_with_buttons ("Voice Properties", NULL,
						  (GtkDialogFlags)
						  (GTK_DIALOG_MODAL |
						   GTK_DIALOG_DESTROY_WITH_PARENT),
						  GTK_STOCK_OK,
						  GTK_RESPONSE_OK,
						  GTK_STOCK_CANCEL,
						  GTK_RESPONSE_CANCEL,
						  "Choose Audio File",
						  GTK_RESPONSE_APPLY,
						  "Delete Voice",
						  GTK_RESPONSE_NO, NULL);

 //add some stuff:
 GtkWidget *label_VoiceType = gtk_label_new ("Voice Type:");
 GtkWidget *comboboxentry1 = gtk_combo_box_entry_new_text ();

 gtk_widget_show (comboboxentry1);
 //e20070621: List order: don't change; depends on #define values in BinauralBeat.h:
 gtk_combo_box_append_text
  (GTK_COMBO_BOX (comboboxentry1), "Binaural Beat Freqency");
 gtk_combo_box_append_text (GTK_COMBO_BOX (comboboxentry1), "Pink Noise");
 gtk_combo_box_append_text (GTK_COMBO_BOX (comboboxentry1), "Audio File");
 GtkWidget *label_VoiceDescrip =
  gtk_label_new ("Voice Description [or Filename if Audio File]:");
 GtkWidget *entry_Input_VoiceDesrip = gtk_entry_new ();

 if (curVoice != NULL)
 {
  gtk_entry_set_text ((GtkEntry
		       *) entry_Input_VoiceDesrip, curVoice->description);
 }

 gtk_combo_box_set_active ((GtkComboBox *) comboboxentry1,
			   (curVoice != NULL) ? curVoice->type : 0);
 // Add the label, and show everything we've added to the dialog.
 gtk_container_add
  (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), label_VoiceType);
 gtk_container_add
  (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), comboboxentry1);
 gtk_container_add
  (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), label_VoiceDescrip);
 gtk_container_add
  (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), entry_Input_VoiceDesrip);
 gtk_widget_show_all (dialog);
 gint res = FALSE;

 do
 {
  //block until I get a response:
  res = gtk_dialog_run (GTK_DIALOG (dialog));
  switch (res)
  {
  case GTK_RESPONSE_OK:
   {
    if (curVoice != NULL)
    {
     curVoice->type =
      gtk_combo_box_get_active ((GtkComboBox *) comboboxentry1);
    }
    else
    {
     curVoice =
      SG_AddNewVoiceToEnd
      (SG_FirstVoice,
       gtk_combo_box_get_active ((GtkComboBox *) comboboxentry1));
     //always must manually add a first data point to the end of a new voice:
     SG_AddNewDataPointToEnd (curVoice->FirstDataPoint, curVoice, SG_TotalScheduleDuration,	//duration
			      0,	//beatfreq
			      .1,	//volume
			      .1,	//volume
			      110,	//basefreq
			      SG_SELECTED);
    }
    SG_StringAllocateAndCopy (&
			      (curVoice->
			       description),
			      (char
			       *)
			      gtk_entry_get_text
			      (GTK_ENTRY (entry_Input_VoiceDesrip)));
    SG_SelectVoice (curVoice);
    //   SG_DeselectDataPoints ();
    //   SG_SelectDataPoints_Voice (curVoice, TRUE);
    //   SG_ConvertDataToXY (widget);
    SG_DrawGraph (widget);
    main_UpdateGUI_Voices (main_vboxVoices);
    SG_GraphHasChanged = SG_DataHasChanged = TRUE;	//datahaschanged because voicetype may have changed locally
   }
   break;
   //pick audio file:
  case GTK_RESPONSE_APPLY:
   {
    gchar *tmpfilename =
     main_OpenFileDialog
     ("~Gnaural Audio Files~*.wav,*.aiff,*.au,*.flac,~All Files~*");

    if (NULL != tmpfilename)
    {
     gtk_entry_set_text ((GtkEntry *) entry_Input_VoiceDesrip, tmpfilename);
    }
    g_free (tmpfilename);
   }
   break;
   //delete voice:
  case GTK_RESPONSE_NO:
   main_DeleteSelectedVoice (widget);
   break;
  default:
   res = FALSE;
   break;
  }
 }
 while (res == GTK_RESPONSE_APPLY);
 gtk_widget_destroy (dialog);
 return res;
}

/////////////////////////////////////////////////////
void main_on_revert_activate (GtkMenuItem * menuitem, gpointer user_data)
{
 if (0 != main_TestPathOrFileExistence (main_sCurrentGnauralFilenameAndPath))
 {
  SG_ERROUT ("Can't revert to a file that doesn't exist.");
  return;
 }
 main_XMLReadFile
  (main_sCurrentGnauralFilenameAndPath, main_drawing_area, FALSE);
}

/////////////////////////////////////////////////////
//erases all voices and leaves user with one voice with one DP
void main_NewGraph (int voices)
{
 if (FALSE == main_SetScheduleInfo (FALSE))
 {
  return;
 }
 SG_SetupDefaultDataPoints (voices);
 main_SetLoops (1);
 //SG_DrawGraph (main_drawing_area);
 SG_GraphHasChanged = SG_DataHasChanged = TRUE;	//20070105 tricky way to do main_LoadBinauralBeatSoundEngine in a bulk way
 sprintf (main_sTmp, "%s/untitled.gnaural", main_sPathGnaural);
 main_UpdateGUI_FileInfo (main_sTmp);
 main_UpdateGUI_Voices (main_vboxVoices);
 //this is really just a way for user to free up big chunks of memory if they had lots of AFs open:
 main_CleanupAudioFileData ();
}

/////////////////////////////////////////////////////
void main_UpdateGUI_FileInfo (char *filename)
{
 strcpy (main_sCurrentGnauralFilenameAndPath, filename);
 sprintf (main_sTmp, "File: %s", main_sCurrentGnauralFilenameAndPath);
 gtk_label_set_text (main_labelFileName, main_sTmp);
 sprintf (main_sTmp, "Author: %s", main_Info_Author);
 gtk_label_set_text (main_labelFileAuthor, main_sTmp);
 sprintf (main_sTmp, "Description: %s", main_Info_Description);
 gtk_label_set_text (main_labelFileDescription, main_sTmp);
 sprintf (main_sTmp, "Title: %s", main_Info_Title);
 gtk_label_set_text (main_labelFileTitle, main_sTmp);
}

/////////////////////////////////////////////////////
//clears all but first DP:
void main_ClearDataPointsInVoice (SG_Voice * curVoice)
{
 if (curVoice == NULL)
 {
  return;
 }
 SG_DeselectDataPoints ();
 SG_SelectDataPoints_Voice (curVoice, TRUE);
 curVoice->FirstDataPoint->state = SG_UNSELECTED;
 SG_DeleteDataPoints (main_drawing_area, TRUE, TRUE);
}

/////////////////////////////////////////////////////
//clears all but first DP in ALL voices:
void main_ClearDataPointsInVoices ()
{
 SG_BackupDataPoints (main_drawing_area);
 SG_Voice *curVoice = SG_FirstVoice;

 while (curVoice != NULL)
 {
  main_ClearDataPointsInVoice (curVoice);
  curVoice = curVoice->NextVoice;
 }
}

///////////////////////////////////////////
//returns TRUE if user clicked OK, filling val with result
double main_AskUserForNumberDialog
 (char *title, char *question, double *startingval)
{
 GtkWidget *dialog = gtk_dialog_new_with_buttons (title,
						  (GtkWindow *) main_window,
						  (GtkDialogFlags)
						  (GTK_DIALOG_MODAL |
						   GTK_DIALOG_DESTROY_WITH_PARENT),
						  GTK_STOCK_OK,
						  GTK_RESPONSE_ACCEPT,
						  GTK_STOCK_CANCEL,
						  GTK_RESPONSE_REJECT, NULL);
 gboolean res = FALSE;

 //add some stuff:
 GtkWidget *label = gtk_label_new (question);
 GtkWidget *entry_Input = gtk_entry_new ();

 sprintf (main_sTmp, "%g", *startingval);
 gtk_entry_set_text (GTK_ENTRY (entry_Input), main_sTmp);
 // Add the label, and show everything we've added to the dialog.
 gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), label);
 gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), entry_Input);
 gtk_widget_show_all (dialog);
 //block until I get a response:
 gint result = gtk_dialog_run (GTK_DIALOG (dialog));

 switch (result)
 {
 case GTK_RESPONSE_ACCEPT:
  {
   *startingval = atof (gtk_entry_get_text (GTK_ENTRY (entry_Input)));
   res = TRUE;
  }
  break;
 default:
  break;
 }
 gtk_widget_destroy (dialog);
 return res;
}

///////////////////////////////////////////
gboolean main_SetScheduleInfo (gboolean FillEntriesFlag)
{
 gboolean ok = FALSE;
 GtkWidget *dialog = gtk_dialog_new_with_buttons ("Edit Schedule Info",
						  (GtkWindow *) main_window,
						  (GtkDialogFlags)
						  (GTK_DIALOG_MODAL |
						   GTK_DIALOG_DESTROY_WITH_PARENT),
						  GTK_STOCK_OK,
						  GTK_RESPONSE_ACCEPT,
						  GTK_STOCK_CANCEL,
						  GTK_RESPONSE_REJECT, NULL);

 //add some stuff:
 GtkWidget *label_title = gtk_label_new ("Title:");
 GtkWidget *entry_Input_title = gtk_entry_new ();

 if (TRUE == FillEntriesFlag)
 {
  gtk_entry_set_text (GTK_ENTRY (entry_Input_title), main_Info_Title);
 }
 GtkWidget *label_description = gtk_label_new ("Description");
 GtkWidget *entry_Input_description = gtk_entry_new ();

 if (TRUE == FillEntriesFlag)
 {
  gtk_entry_set_text (GTK_ENTRY
		      (entry_Input_description), main_Info_Description);
 }
 GtkWidget *label_author = gtk_label_new ("Author:");
 GtkWidget *entry_Input_author = gtk_entry_new ();

 if (TRUE == FillEntriesFlag)
 {
  gtk_entry_set_text (GTK_ENTRY (entry_Input_author), main_Info_Author);
 }
 // Add the label, and show everything we've added to the dialog.
 gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), label_title);
 gtk_container_add
  (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), entry_Input_title);
 gtk_container_add
  (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), label_description);
 gtk_container_add
  (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), entry_Input_description);
 gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), label_author);
 gtk_container_add
  (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), entry_Input_author);
 gtk_widget_show_all (dialog);
 //block until I get a response:
 gint result = gtk_dialog_run (GTK_DIALOG (dialog));

 switch (result)
 {
 case GTK_RESPONSE_ACCEPT:
  {
   ok = TRUE;
   SG_StringAllocateAndCopy
    (&main_Info_Title, gtk_entry_get_text (GTK_ENTRY (entry_Input_title)));
   SG_StringAllocateAndCopy
    (&main_Info_Description,
     gtk_entry_get_text (GTK_ENTRY (entry_Input_description)));
   SG_StringAllocateAndCopy
    (&main_Info_Author, gtk_entry_get_text (GTK_ENTRY (entry_Input_author)));
   main_UpdateGUI_FileInfo (main_sCurrentGnauralFilenameAndPath);
  }
  break;
 default:
  break;
 }
 gtk_widget_destroy (dialog);
 return ok;
}

///////////////////////////////////
void main_ReverseVoice ()
{
 SG_BackupDataPoints (main_drawing_area);
 SG_ReverseVoice (main_drawing_area);
}

///////////////////////////////////
void main_DuplicateSelectedVoice ()
{
 SG_BackupDataPoints (main_drawing_area);
 SG_DuplicateSelectedVoice ();
 main_UpdateGUI_Voices (main_vboxVoices);
}

///////////////////////////////////
void main_ScaleDataPoints_Time (GtkWidget * widget)
{
 static double scalar = 1.0;

 if (TRUE ==
     main_AskUserForNumberDialog
     ("Time Scale",
      "Number to scale duration of selected points by:", &scalar))
 {
  SG_BackupDataPoints (widget);
  SG_ScaleDataPoints_Time (widget, scalar);
 }
}

///////////////////////////////////
void main_DeleteSelectedVoice (GtkWidget * widget)
{
 //first check if there was a legally selected voice:  
 SG_Voice *curVoice = SG_SelectVoice (NULL);

 if (curVoice != NULL)
 {
  SG_BackupDataPoints (widget);
  SG_DeleteVoice (curVoice);	//NOTE: Might want to base deletion on SG_Voice ID instead...
  SG_SelectVoice (SG_FirstVoice);
  SG_DrawGraph (widget);
  SG_GraphHasChanged = TRUE;
  main_UpdateGUI_Voices (main_vboxVoices);
 }
}

///////////////////////////////////
void main_ScaleDatPoints_Y (GtkWidget * widget)
{
 static double scalar = 1.0;

 if (TRUE ==
     main_AskUserForNumberDialog
     ("Scale Y", "Number to scale Y values of selected points by:", &scalar))
 {
  SG_BackupDataPoints (widget);
  SG_ScaleDataPoints_Y (widget, scalar);
 }
}

///////////////////////////////////
void main_AddRandomToDataPoints_Y (GtkWidget * widget)
{
 static double val = 0;

 if (TRUE ==
     main_AskUserForNumberDialog
     ("Add Randomness to Y",
      "+/- range of random number to add to selected points:", &val))
 {
  SG_BackupDataPoints (widget);
  SG_AddRandomToDataPoints (widget, val, FALSE, TRUE);
 }
}

///////////////////////////////////
void main_AddRandomToDataPoints_time (GtkWidget * widget)
{
 static double val = 0;

 if (TRUE ==
     main_AskUserForNumberDialog
     ("Add Randomness to Durations",
      "+/- range of seconds to add to selected points:", &val))
 {
  SG_BackupDataPoints (widget);
  SG_AddRandomToDataPoints (widget, val, TRUE, FALSE);
 }
}

///////////////////////////////////
void main_SelectInterval ()
{
 static double val = 2;

 if (TRUE ==
     main_AskUserForNumberDialog
     ("Select Interval", "Select at this interval:", &val))
 {
  SG_BackupDataPoints (main_drawing_area);
  SG_SelectIntervalDataPoints_All ((int) val, TRUE, FALSE);
 }
}

///////////////////////////////////
void main_SelectNeighbor ()
{	//Select neighbor to right:
 static double val = 0;

 if (TRUE ==
     main_AskUserForNumberDialog
     ("Select Next Datapoints",
      "Deselect currently selected? (1=yes, 0=no):", &val))
 {
  SG_BackupDataPoints (main_drawing_area);
  SG_SelectNeighboringDataPoints (TRUE, ((val == 1) ? TRUE : FALSE));
 }
}

///////////////////////////////////
void main_OpenFile (gboolean OpenMerge)
{	//Open an XML data file:
 gchar *tmpfilename =
  main_OpenFileDialog ("~Gnaural Files~*.gnaural,*.txt,~All Files~*");

 if (tmpfilename == NULL)
 {
  SG_DBGOUT ("Not opening a file.");
  return;
 }
 main_XMLReadFile (tmpfilename, main_drawing_area, OpenMerge);
 g_free (tmpfilename);
}

///////////////////////////////////
void main_SelectLastDPs ()
{
 SG_BackupDataPoints (main_drawing_area);
 SG_SelectLastDP_All ();
}

///////////////////////////////////
void main_SelectFirstDPs ()
{
 SG_BackupDataPoints (main_drawing_area);
 SG_SelectFirstDP_All ();
}

///////////////////////////////////
void main_TruncateSchedule ()
{
 static double val = 60;

 if (TRUE ==
     main_AskUserForNumberDialog
     ("Truncate (or Grow) Schedule",
      "Schedule End Time (sec.):", &val) && val >= 0)
 {
  SG_BackupDataPoints (main_drawing_area);
  BB_ResetAllVoices ();
  SG_TruncateSchedule (main_drawing_area, val);
 }
}

///////////////////////////////////
void main_PasteAtEnd ()
{
 SG_BackupDataPoints (main_drawing_area);
 SG_PasteDataPointsAtEnd (main_drawing_area);
}

///////////////////////////////////
void main_InvertY ()
{
 SG_BackupDataPoints (main_drawing_area);
 SG_InvertY (main_drawing_area);
}

///////////////////////////////////
void main_SelectDuration ()
{
 static double val = 1;

 if (TRUE ==
     main_AskUserForNumberDialog
     ("Select by Duration (sec.)",
      "Selection Threshold (positive for above, negative for below):", &val))
 {
  SG_BackupDataPoints (main_drawing_area);
  SG_SelectDuration (main_drawing_area, val);
 }
}

///////////////////////////////////
void main_SelectProximity_All ()
{
 static double val = 1;

 if (TRUE ==
     main_AskUserForNumberDialog
     ("Select by X,Y Proximity (pixels)",
      "Proximity Threshold:", &val) && val >= 0)
 {
  SG_BackupDataPoints (main_drawing_area);
  SG_SelectProximity_All (val);
  //  SG_SelectProximity_SingleDP(SG_FirstVoice->FirstDataPoint, val);
 }
}

///////////////////////////////////
void main_SelectProximity_SinglePoint ()
{
 static double val = 1;

 if (TRUE ==
     main_AskUserForNumberDialog
     ("Select by X,Y Proximity (pixels)",
      "Proximity Threshold:", &val) && val >= 0)
 {
  SG_DataPoint *curDP = NULL;
  SG_Voice *curVoice = SG_FirstVoice;

  while (curVoice != NULL)
  {
   if (curVoice->hide == FALSE)
   {
    curDP = curVoice->FirstDataPoint;
    do
    {
     if (SG_SELECTED == curDP->state)
     {
      break;
     }
     curDP = curDP->NextDataPoint;
    }
    while (curDP != NULL);
   }
   if (curDP != NULL)
   {
    break;
   }
   curVoice = curVoice->NextVoice;
  }

  if (curDP != NULL && SG_SELECTED == curDP->state)
  {
   SG_BackupDataPoints (main_drawing_area);
   SG_SelectProximity_SingleDP (curDP, val);
  }
 }
}

///////////////////////////////////
void main_Preferences ()
{
 double val = M_E;

 sprintf (main_sTmp, "Preferences should be reducible to one number");
 do
 {
  main_AskUserForNumberDialog (main_sTmp, "Preferences?", &val);
  sprintf (main_sTmp, "Your Pi is no good around here");
 }
 while ((int) (val * 10000) == (int) (M_PI * 10000));
}

///////////////////////////////////
//just a short multiplatform nap; but Sleep is a wonderfully convluted
//occupation. Look what you might do: clock, delay, ftime, gettimeofday, 
//msleep, nap, napms, nanosleep, setitimer, sleep, Sleep, times, usleep
//Note the handy G_USEC_PER_SEC
void main_Sleep (int microseconds)
{
 g_usleep (microseconds);
 /*
    #ifdef GNAURAL_WIN32
    SG_DBGOUT_INT ("[Win32] Sleep(ms):", ms);
    Sleep (ms);
    #endif

    #ifndef GNAURAL_WIN32
    SG_DBGOUT_INT ("[POSIX] usleep(ms):", ms);
    //Obsolete, but using nanosleep feels like using a bazooka for a fly:
    usleep (ms);
    #endif
  */
}

///////////////////////////////////
void main_DuplicateAllVoices ()
{
 SG_BackupDataPoints (main_drawing_area);
 SG_RestoreDataPoints (main_drawing_area, TRUE);
}

///////////////////////////////////
//pass this an empty buffer, and it fills it with sound from filename
//and fills size with number of elements in the array of shorts.
//returns 0 for success.
int main_LoadSoundFile_shorts
 (char *filename, short **buffer, unsigned int *size)
{
 SNDFILE *sndfile;
 SF_INFO sfinfo;

 (*buffer) = NULL;
 memset (&sfinfo, 0, sizeof (sfinfo));
 SG_DBGOUT_STR ("Loading PCM File:", filename);
 if (!(sndfile = sf_open (filename, SFM_READ, &sfinfo)))
 {
  puts (sf_strerror (NULL));
  return 1;
 };
 if (sfinfo.channels < 1 || sfinfo.channels > 2)
 {
  SG_DBGOUT_INT ("Error : channels", sfinfo.channels);
  return 1;
 };
 //http://www.mega-nerd.com/libsndfile/FAQ.html
 //Q12 : I'm looking at sf_read*. What are items? What are frames?
 //For a sound file with only one channel, a frame is the same as a item.
 //For multi channel sound files, a single frame contains a single item for each channel.
 (*buffer) = (short *) malloc (sfinfo.frames * sizeof (short) * 2);
 (*size) = sfinfo.frames * 2;
 SG_DBGOUT_INT ("Channels:", sfinfo.channels);
 if (NULL == (*buffer))
 {
  return 1;
 }
 unsigned int readcount = 0;

 if (2 == sfinfo.channels)
 {
  SG_DBGOUT ("Reading stereo data");
  readcount = sf_readf_short (sndfile, (*buffer), (sfinfo.frames));
 }
 else
 {
  SG_DBGOUT ("Reading mono data; converting to stereo");
  while (0 != sf_readf_short (sndfile, &((*buffer)[readcount]), 1))
  {
   ++readcount;
   (*buffer)[readcount] = (*buffer)[(readcount - 1)];
   ++readcount;
  }
 }
 sf_close (sndfile);
 SG_DBGOUT_INT ("readcount: ", readcount);
 SG_DBGOUT_INT ("framecount: ", (int) sfinfo.frames);
 SG_DBGOUT_INT ("array sizeof (bytes):",
		((int) sfinfo.frames) * sizeof (short) * 2);
 SG_DBGOUT_INT ("array elements: (shorts)", (*size));
 return 0;	//success
 //  Gnaural.wav
}

///////////////////////////////////
//pass this an empty buffer, and it fills it with sound from filename
//and fills size with number of elements in the array of shorts.
//returns 0 for success, filling (*buffer) and (*size) with data
//return non-zero on failure, with  (*buffer) = NULL (*size) = 0
int main_LoadSoundFile (char *filename, int **buffer, unsigned int *size)
{
 SNDFILE *sndfile;
 SF_INFO sfinfo;

 (*buffer) = NULL;
 (*size) = 0;
 memset (&sfinfo, 0, sizeof (sfinfo));
 SG_DBGOUT_STR ("Loading PCM File:", filename);
 if (!(sndfile = sf_open (filename, SFM_READ, &sfinfo)))
 {
  puts (sf_strerror (NULL));
  return 1;
 };
 if (sfinfo.channels < 1 || sfinfo.channels > 2)
 {
  SG_DBGOUT_INT ("Error : channels", sfinfo.channels);
  return 1;
 };
 //http://www.mega-nerd.com/libsndfile/FAQ.html
 //Q12 : I'm looking at sf_read*. What are items? What are frames?
 //For a sound file with only one channel, a frame is the same as a item.
 //For multi channel sound files, a single frame contains a single item for each channel.
 (*buffer) = (int *) malloc (sfinfo.frames * sizeof (int));
 (*size) = sfinfo.frames;
 SG_DBGOUT_INT ("Channels:", sfinfo.channels);
 if (NULL == (*buffer))
 {
  return 1;
 }
 unsigned int readcount = 0;

 if (2 == sfinfo.channels)
 {
  SG_DBGOUT ("Reading stereo data");
  readcount = sf_readf_short (sndfile, (short *) (*buffer), (sfinfo.frames));
 }
 else
 {
  SG_DBGOUT ("Reading mono data; converting to stereo");
  while (0 != sf_readf_short (sndfile, (short *) &((*buffer)[readcount]), 1))
  {
   (*buffer)[readcount] &= 0x0000FFFF;
   (*buffer)[readcount] |= (((*buffer)[readcount]) << 16);
   ++readcount;
  }
 }
 sf_close (sndfile);
 SG_DBGOUT_INT ("readcount: ", readcount);
 SG_DBGOUT_INT ("framecount: ", (int) sfinfo.frames);
 SG_DBGOUT_INT ("array sizeof (bytes):",
		((int) sfinfo.frames) * sizeof (int));
 SG_DBGOUT_INT ("array elements: (shorts)", (*size));
 return 0;	//success
}

///////////////////////////////////////////////
//this is only called by ParserXML; don't call it has one job: 
//to count the number of voices and number of datapoints
void main_XMLParser_counter (const gchar * CurrentElement,	//this must always be a valid string
			     const gchar * Attribute,	//beware: this will equal NULL if there are none
			     const gchar * Value)	//beware: this will equal NULL to announce end of CurrnetElement
{
 //See if this is the end of CurrentElement:
 if (Value == NULL)
 {
  if (0 == strcmp (CurrentElement, "entry"))
  {
   ++main_ParserXML_EntryCount;
  }
  else if (0 == strcmp (CurrentElement, "voice"))
  {
   ++main_ParserXML_VoiceCount;
  }
  return;
 }
}

///////////////////////////////////////////////
//called whenever opening/re-initing a user file for data
void main_UpdateGUI_UserDataInfo ()
{
 //start checkboxes init
 gtk_toggle_button_set_active
  (main_checkbuttonSwapStereo, (BB_StereoSwap == 0) ? FALSE : TRUE);
 //end checkboxes init
 //start sliders init
 //now for the tricky overall vol and bal:
 if (BB_VolumeOverall_left > BB_VolumeOverall_right)
 {
  main_OverallVolume = BB_VolumeOverall_left;
  main_OverallBalance =
   -1.f + (BB_VolumeOverall_right / BB_VolumeOverall_left);
 }
 else if (BB_VolumeOverall_left < BB_VolumeOverall_right)
 {
  main_OverallVolume = BB_VolumeOverall_right;
  main_OverallBalance =
   1.f - (BB_VolumeOverall_left / BB_VolumeOverall_right);
 }
 else if (BB_VolumeOverall_left == BB_VolumeOverall_right)
 {
  main_OverallVolume = BB_VolumeOverall_right;
  main_OverallBalance = 0;
 }
 gtk_range_set_value ((GtkRange *) main_hscaleVolume,
		      100 * main_OverallVolume);
 gtk_range_set_value ((GtkRange *) main_hscaleBalance,
		      100 * main_OverallBalance);
 //end sliders init
}

///////////////////////////////////////////////
void main_vscale_Y_button_event (GtkWidget * widget, gboolean pressed)
{
 main_vscale_Y_mousebuttondown = pressed;
 if (FALSE == main_vscale_Y_mousebuttondown)
 {
  gtk_range_set_value ((GtkRange *) widget, 0);
 }
 else
 {
  SG_BackupDataPoints (main_drawing_area);	//a20070620
 }
}

///////////////////////////////////////////////
void main_vscale_Y_value_change (GtkRange * range)
{
 if (TRUE == main_vscale_Y_mousebuttondown)
 {
  main_vscale_Y = -gtk_range_get_value (range);
 }
 else
 {
  main_vscale_Y = 0.0f;
  gtk_range_set_value ((GtkRange *) range, 0);
 }
}

///////////////////////////////////////////////
void main_hscale_X_button_event (GtkWidget * widget, gboolean pressed)
{
 main_hscale_X_mousebuttondown = pressed;
 if (FALSE == main_hscale_X_mousebuttondown)
 {
  gtk_range_set_value ((GtkRange *) widget, 0);
 }
 else
 {
  SG_BackupDataPoints (main_drawing_area);	//a20070620
 }
}

///////////////////////////////////////////////
void main_hscale_X_value_change (GtkRange * range)
{
 if (TRUE == main_hscale_X_mousebuttondown)
 {
  main_hscale_X = gtk_range_get_value (range);
 }
 else
 {
  main_hscale_X = 0.0f;
  gtk_range_set_value ((GtkRange *) range, 0);
 }
}

/////////////////////////////////////////////////////
//a20070620:
void main_slider_XY_handler (float vertical, float horizontal)
{
 if (0.0f == vertical && 0.0f == horizontal)
 {
  return;
 }

 if (vertical != 0.0f)
 {
  if (FALSE == main_vscale_Y_scaleflag)
  {
   SG_MoveSelectedDataPoints (main_drawing_area, 0, vertical);
  }
  else
  {
   SG_ScaleDataPoints_Y (main_drawing_area, 1 + (-.1 * vertical));
  }
  SG_ConvertYToData_SelectedPoints (main_drawing_area);
 }

 //a20070620 :
 if (horizontal != 0.0f)
 {
  if (FALSE == main_hscale_X_scaleflag)
  {
   SG_MoveSelectedDataPoints (main_drawing_area, horizontal, 0);
  }
  else
  {
   SG_ScaleDataPoints_Time (main_drawing_area, 1 + (.1 * horizontal));
  }
  SG_ConvertXToDuration_AllPoints (main_drawing_area);
 }

 SG_ConvertDataToXY (main_drawing_area);	// NOTE: I need to call this both to set limits and to bring XY's outside of graph back in
 SG_DrawGraph (main_drawing_area);
 SG_DataHasChanged = SG_GraphHasChanged = TRUE;	//20070105 tricky way to do main_LoadBinauralBeatSoundEngine in a bulk way
}

///////////////////////////////////
void main_RoundValues ()
{
 static double val = 100;
 static double param = 0;
 char namestr[] = "Round Selected DPs";

 if (TRUE ==
     main_AskUserForNumberDialog
     (namestr,
      "Round which parameter?\n0: duration\n1: volume_left\n2: volume_right\n3: basefreq\n4: beatfreq",
      &param)
     && TRUE == main_AskUserForNumberDialog (namestr,
					     "Rounding value:\n1=1th\n10=10th\n100=100th",
					     &val))
 {
  SG_BackupDataPoints (main_drawing_area);
  SG_RoundValues_All (main_drawing_area, val, param);
 }
}
