/*
#   idjcmixer.c: central core of IDJC's mixer.
#   Copyright (C) 2005-2007 Stephen Fairchild (s-fairchild@users.sourceforge.net)
#
#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program in the file entitled COPYING.
#   If not, see <http://www.gnu.org/licenses/>.
*/

#define _GNU_SOURCE
#include <../config.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <math.h>
#include <jack/jack.h>
#include <jack/transport.h>
#include <jack/ringbuffer.h>
#include <getopt.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <locale.h>

#include "kvpparse.h"
#include "dbconvert.h"
#include "compressor.h"
#include "noisegate.h"
#include "xlplayer.h"
#include "ialloc.h"
#include "vorbinfo.h"
#include "mp4tag.h"
#include "sndfileinfo.h"
#include "avcodecdecode.h"

#define TRUE 1
#define FALSE 0

/* size of the ringbuffer (roughly one second) in bytes */
#define RB_SIZE 262144
/* number of samples in the fade (ring) buffer */
#define FB_SIZE (RB_SIZE / sizeof (float))

/* the different VOIP modes */
#define NO_PHONE 0
#define PHONE_PUBLIC 1
#define PHONE_PRIVATE 2

typedef jack_default_audio_sample_t sample_t;

/* values of the volume sliders in the GUI */
int volume, crossfade, jinglesvolume, interludevol, mixbackvol;
/* back and forth status indicators re. jingles */
int jingles_playing, jingles_audio_f;
/* used for gapless playback to indicate an almost empty buffer */
int left_audio_runout = 0, right_audio_runout = 0;
/* the main-player unmute buttons */
int left_stream = 1, left_audio = 1, right_stream = 1, right_audio = 1;
/* status variables for the button cluster in lower right of main window */
int lmic_on, rmic_on, aux_on, mixermode = NO_PHONE;
/* these are TRUE when the mic and aux are transitioning between on and off */
int lmic_flux, rmic_flux, aux_flux;
/* simple mixer mode: uses less space on the screen and less cpu as well */
int simple_mixer;
/* currentvolumes are used to implement volume smoothing */
int current_volume, current_jingles_volume, current_interlude_volume,
    current_crossfade, currentmixbackvol; 
/* the id. number microphone filter we are using */
int mic_filter;
/* value of the stream mon. button */
int stream_monitor = 0;
/* when this is set the end of track alarm is started */
int eot_alarm_f = 0;
/* threshold values for a premature indicator that a player is about to finish */
int jingles_samples_cutoff;
int player_samples_cutoff;
/* the number of samples processed this song - used to calculate the player progress bars */
int left_samples_total, right_samples_total;
/* used to implement interlude player fade in/out: true when playing a track */
int main_play;
/* if the main app segfaults or similar this counter will see that the mixer terminates */
int timeout;
int shutdown;
/* flag set when jack closes the client thread for example when the jackd is closed */
int jack_closed_f;
/* flag to indicate whether to use the player reading function which supports speed variance */
int speed_variance;
/* flag to indicate if microphone audio should be present in the dj stream */
int mic_to_dj;
/* buffers that process_audio uses when reading media player data */
sample_t *lp_lc, *lp_rc, *rp_lc, *rp_rc, *jp_lc, *jp_rc, *ip_lc, *ip_rc;
sample_t *lp_lcf, *lp_rcf, *rp_lcf, *rp_rcf, *jp_lcf, *jp_rcf, *ip_lcf, *ip_rcf;
/* used for signal level silence threshold tracking */
sample_t left_peak = -1.0F, right_peak = -1.0F;
/* handle for beat processing */
/*struct beatproc *beat_lp, *beat_rp;*/

/* part of CTRL+C handling */
sigset_t mixersigset;

/* the number of samples worth of data in the fadeout buffer */
jack_nframes_t alarm_size;

float headroom_db;			/* player muting level when mic is open */
float mic_lpeak, mic_rpeak;		/* used for vu meters in the main app */
float str_lpeak, str_rpeak;		/* used to store the peak levels */
float mic_l_tally, mic_r_tally;		/* used to calculate the rms value */
float str_l_tally, str_r_tally;
int rms_tally_count;
float mic_l_meansqrd, mic_r_meansqrd;
float str_l_meansqrd, str_r_meansqrd;
int reset_vu_stats_f;			/* when set the mixer will reset the above */
float *dblookup, *antidblookup;		/* a table for speeding up log / antilog operations */
float *fade_table;			/* a table of values used for autofade */
float dfmod;				/* used to reduce the ducking factor */
float dj_audio_level;			/* used to reduce the level of dj audio */
float dj_audio_gain = 1.0;		/* same as above but not in dB */
float current_dj_audio_level = 0.0;

struct compressor left_mic_compressor =
   {
   0.0, -15.0, -20.0, 4.0, 1.0F/10.0F, 1.0F/10000.0F, 15.0F, 1.0F, 10000, 0, 0.0, 0.0
   }, right_mic_compressor =
   {
   0.0, -15.0, -20.0, 4.0, 1.0F/10.0F, 1.0F/10000.0F, 15.0, 1.0, 10000, 0, 0.0, 0.0
   }, stream_limiter =
   {
   0.0, -0.05, -0.2, INFINITY, 1, 1.0F/4000.0F, 0.0, 0.0, 1, 1, 0.0, 0.0
   }, audio_limiter =
   {
   0.0, -0.05, -0.2, INFINITY, 1, 1.0F/4000.0F, 0.0, 0.0, 1, 1, 0.0, 0.0
   }, phone_limiter =
   {
   0.0, -0.05, -0.2, INFINITY, 1, 1.0F/4000.0F, 0.0, 0.0, 1, 1, 0.0, 0.0
   }, new_mic_compressor;

int new_compressor_stats=FALSE;
 
struct noisegate left_mic_noisegate =
   {
   -43, 3, 50000, -40, 35, 40000, 0, 0.0, 0
   }, right_mic_noisegate =
   {
   -43, 3, 50000, -40, 35, 40000, 0, 0.0, 0
   }, new_ng;
   
int new_noisegate_stats=FALSE;

struct normalizer str_normalizer =
   {
   0, 0.0F, -12.0F, 1.0F/120000.0F, 1.0F/90000.0F, 12.0
   }, new_normalizer;

int new_normalizer_stats = FALSE;

/* the different player's gain factors */
/* lp=left player, rp=right player, jp=jingles player, ip=interlude player */ 
/* lc=left channel, rc=right channel */
/* aud = the DJs audio, str = the listeners (stream) audio */
/* the initial settings are 'very' temporary */
sample_t lp_lc_aud = 1.0, lp_rc_aud = 1.0, rp_lc_aud = 1.0, rp_rc_aud = 1.0;
sample_t lp_lc_str = 1.0, lp_rc_str = 1.0, rp_lc_str = 0.0, rp_rc_str = 0.0;
sample_t jp_lc_str = 0.0, jp_rc_str = 0.0, jp_lc_aud = 0.0, jp_rc_aud = 0.0;
sample_t ip_lc_str = 0.0, ip_rc_str = 0.0, ip_lc_aud = 0.0, ip_rc_aud = 0.0;
				/* like above but for fade */
sample_t lp_lc_audf = 1.0, lp_rc_audf = 1.0, rp_lc_audf = 1.0, rp_rc_audf = 1.0;
sample_t lp_lc_strf = 1.0, lp_rc_strf = 1.0, rp_lc_strf = 1.0, rp_rc_strf = 1.0;
sample_t jp_lc_strf = 1.0, jp_rc_strf = 1.0, jp_lc_audf = 1.0, jp_rc_audf = 1.0;
sample_t ip_lc_strf = 1.0, ip_rc_strf = 1.0, ip_lc_audf = 0.0, ip_rc_audf = 0.0;
sample_t mic_lc = 0.0, mic_rc = 0.0;	/* microphone gain */
sample_t dj_mic_aud;		/* additional gain applied to mic in dj audio */
	 
sample_t mic_linv = 1.0, mic_rinv = 1.0; 	/* For phase inversion when set to -1.0 */
sample_t mic_lgain = 1.0, mic_rgain = 1.0;	/* User specified mic attenuation */
/* used to apply the stereo mix of the microphones */
sample_t mic_l_lc = 1.0, mic_l_rc = 0.0, mic_r_lc = 0.0, mic_r_rc = 1.0;
/* aux input gain factor - typically 1.0=on or 0.0=off */
sample_t aux_lc = 0.0, aux_rc = 0.0;
/* media player mixback level for when in RedPhone mode */
sample_t mb_lc_aud = 1.0, mb_rc_aud = 1.0;
sample_t current_headroom;	/* the amount of mic headroom being applied */
sample_t *eot_alarm_table;	/* the wave table for the DJ alarm */
	 
jack_client_t *client;		/* client handle to JACK */
jack_port_t *audio_left_port;	/* handles for the various jack ports */
jack_port_t *audio_right_port;
jack_port_t *stream_left_port;
jack_port_t *stream_right_port;
jack_port_t *mic_left_channel;
jack_port_t *mic_right_channel;
jack_port_t *aux_left_channel;
jack_port_t *aux_right_channel;
jack_port_t *phone_left_send;	/* used for VOIP */
jack_port_t *phone_right_send;
jack_port_t *phone_left_recv;
jack_port_t *phone_right_recv;

unsigned long sr;		/* the sample rate reported by JACK */

int transport_aware = 0;	/* arcane jack stuff */
jack_transport_state_t transport_state;

struct xlplayer *plr_l, *plr_r, *plr_j, *plr_i; /* pipe reader instance stuctures */

/* these are set in the parse routine - the contents coming from the GUI */
char *mixer_string, *compressor_string, *gate_string, *microphone_string;
char *normalizer_string;
char *micl, *micr, *auxl, *auxr, *audl, *audr, *strl, *strr, *action;
char *oggpathname, *sndfilepathname, *mp4pathname, *mp4taglist, *avformatpathname;
char *playerpathname, *seek_s, *size, *playerplaylist, *loop, *resamplequality;

/* dictionary look-up type thing used by the parse routine */
struct kvpdict kvpdict[] = {
	 { "PLRP", &playerpathname },		/* The media-file pathname for playback */
         { "SEEK", &seek_s },			/* Playback initial seek time in seconds */
         { "SIZE", &size },			/* Size of the file in seconds */
         { "PLPL", &playerplaylist },		/* A playlist for the media players */
         { "LOOP", &loop },			/* play in a loop */
         { "MIXR", &mixer_string },		/* Control strings */
         { "COMP", &compressor_string },	/* packed full of data */
         { "GATE", &gate_string },
         { "MICS", &microphone_string },
         { "NORM", &normalizer_string },
         { "MICL", &micl },			/* JACK port names to bind to */
         { "MICR", &micr },
         { "AUXL", &auxl },
         { "AUXR", &auxr },
         { "AUDL", &audl },
         { "AUDR", &audr },
         { "STRL", &strl },
         { "STRR", &strr },
         { "OGGP", &oggpathname },
         { "MP4P", &mp4pathname },
         { "SNDP", &sndfilepathname },
         { "AVFP", &avformatpathname },
         { "MP4T", &mp4taglist },
         { "RSQT", &resamplequality },
         { "ACTN", &action },			/* Action to take */
         { "", NULL }};

/* These two point to either sharp, mild, or NULL - optional mic hi-pass filter */   
struct digital_filter *left_mic_filter;
struct digital_filter *right_mic_filter;   

/* variables for the various microphone filters */
struct digital_filter *sharp_left_mic_filter;
struct digital_filter *sharp_right_mic_filter;
dfilter_t a_cheby100[] = { 1 , -4 , 6, -4, 1 };
size_t a_cheby100_count = 5;
dfilter_t b_cheby100[] = { 3.9418017792,-5.8274608161, 3.8294852689 ,-0.9438267667};
size_t b_cheby100_count = 4;
dfilter_t cheby100_filter_gain =1.029430476;

struct digital_filter *mild_left_mic_filter;
struct digital_filter *mild_right_mic_filter;
dfilter_t a_butterworth90[] = { 1, -1 };
size_t a_butterworth90_count = 2;
dfilter_t b_butterworth90[] = { 0.9872586881 };
size_t b_butterworth90_count = 1;
dfilter_t butterworth90_filter_gain = 1.006411501;

/* used for de-essing */
struct digital_filter *ess_left_filter;
struct digital_filter *ess_right_filter;
dfilter_t a_ess8k12k[] = { 1, 0, -4, 0, 6, 0, -4, 0, 1 };
size_t a_ess8k12k_count = 9;
dfilter_t b_ess8k12k[] = { 1.0672882610,-3.4723263985,2.5984638141,-4.4589555851, 2.1983120838,-2.4940727230, 0.6389926185,-0.5067329547};
size_t b_ess8k12k_count = 8;
dfilter_t ess8k12k_gain = 5.472512765e+02;
 
/* used to produce a clean feed for the noisegate */
struct digital_filter *ng_left_filter;
struct digital_filter *ng_right_filter;
dfilter_t a_ngbandpass[] = { 1, 0, -4, 0, 6, 0, -4, 0, 1 };
size_t a_ngbandpasscount = 9;
dfilter_t b_ngbandpass[] = { 7.0389284721, -21.9220056594, 39.5069302814, -45.1037888240, 33.4221059835, -15.7003351003, 4.2743904241, -0.5162255821 };
size_t b_ngbandpasscount = 8;
dfilter_t ngbandpassgain = 6.381624743e+02;

/* the rms filter currently in use */
struct rms_calc *lm_rms_filter, *rm_rms_filter;

/* Does this ever get run, or is it cruft? */
void process_silence(jack_nframes_t nframes)
   {
   sample_t *la_buffer = (sample_t *) jack_port_get_buffer(audio_left_port, nframes);
   sample_t *ra_buffer = (sample_t *) jack_port_get_buffer(audio_right_port, nframes);
   sample_t *ls_buffer = (sample_t *) jack_port_get_buffer(audio_left_port, nframes);
   sample_t *rs_buffer = (sample_t *) jack_port_get_buffer(audio_right_port, nframes);
   
   memset(la_buffer, 0, sizeof (jack_default_audio_sample_t) * nframes);
   memset(ra_buffer, 0, sizeof (jack_default_audio_sample_t) * nframes);
   memset(ls_buffer, 0, sizeof (jack_default_audio_sample_t) * nframes);
   memset(rs_buffer, 0, sizeof (jack_default_audio_sample_t) * nframes);
   }
   
/* handle_mute_button: soft on/off for the mute buttons */
void handle_mute_button(sample_t *gainlevel, int switchlevel)
   {
   if (switchlevel)
      {
      if (*gainlevel < 0.99F)		/* switching on */
         {
         *gainlevel += (1.0F - *gainlevel) * 0.09F;
         if (*gainlevel >= 0.99F)
            *gainlevel = 1.0F;
         }
      }
   else 
      {
      if (*gainlevel > 0.0F)		/* switching off */
         {
         *gainlevel -= *gainlevel * 0.075F * (2.0F - *gainlevel) * (2.0F - *gainlevel);
         if (*gainlevel < 0.00002)
            *gainlevel = 0.0F;
         }
      }
   }

/* update_smoothed_volumes: stuff that gets run once every 32 samples */
void update_smoothed_volumes()
   {
   static sample_t vol_rescale = 1.0F, jingles_vol_rescale = 1.0F, interlude_vol_rescale = 1.0F, cross_left = 1.0F, cross_right = 0.0F, mixback_rescale = 1.0F;
   static sample_t lp_listen_mute = 1.0F, rp_listen_mute = 1.0F, lp_stream_mute = 1.0F, rp_stream_mute = 1.0F;
   sample_t mic_target, diff;
   static float interlude_autovol = -128.0F, old_autovol = -128.0F;
   float vol;
   float xprop, yprop;
   const float bias = 0.35386F;

   timeout++;
   
   if (dj_audio_level != current_dj_audio_level)
      {
      current_dj_audio_level = dj_audio_level;
      dj_audio_gain = db2level(dj_audio_level);
      }
   
   if (crossfade != current_crossfade)
      {
      if (crossfade > current_crossfade)
         current_crossfade++;
      else
         current_crossfade--;
         
      /* This crossfader is based on a linear potentiometer with a pull-up resistor (bias) */
      xprop = current_crossfade * 0.01F;
      yprop = -xprop + 1.0F;
      cross_left = yprop / ((xprop * bias) / (xprop + bias) + yprop);
      cross_right = xprop / ((yprop * bias) / (yprop + bias) + xprop); 
      
      /* Okay, but now for stage 2.  The result should be an S curve. */
      if (xprop >= 0.5F)
         cross_left /= 1 + (xprop - 0.5) * 8.0F;
      else
         cross_right /= 1 + (yprop - 0.5) * 8.0F; 
      }

   if (volume != current_volume)
      {
      if (volume > current_volume)
         current_volume++;
      else
         current_volume--;
      vol_rescale = 1.0F/powf(10.0F,current_volume/55.0F);	/* a nice logarithmic volume scale */
      }
      
   if (jinglesvolume != current_jingles_volume)
      {
      if (jinglesvolume > current_jingles_volume)
         current_jingles_volume++;
      else
         current_jingles_volume--;
      jingles_vol_rescale = 1.0F/powf(10.0F,current_jingles_volume/55.0F);
      }
      
   /* interlude_autovol rises and falls as and when no media players are playing */
   /* it indicates the playback volume in dB in addition to the one specified by the user */
   
   old_autovol = interlude_autovol;
   if (main_play == TRUE)
      {
      if (interlude_autovol > -128.0F)
         interlude_autovol -= 0.05F;
      }
   else
      {
      if (interlude_autovol < 0.0F)
         interlude_autovol += 0.05F;
      if (interlude_autovol < -30.0F)
         interlude_autovol += 0.1F;
      if (interludevol > 40.0F && interlude_autovol < -10.0F)
         interlude_autovol += 0.02;
      }   
   
   if (interludevol != current_interlude_volume || interlude_autovol != old_autovol )
      {
      if (interludevol > current_interlude_volume)
         current_interlude_volume++;
      else
         current_interlude_volume--;
      interlude_vol_rescale = powf(10.0F, -(current_interlude_volume * 0.025 + interlude_autovol * -0.05F));
      }
      
   if (mixbackvol != currentmixbackvol)
      {
      if (mixbackvol > currentmixbackvol)
         currentmixbackvol++;
      else
         currentmixbackvol--;
      mixback_rescale = powf(10.0F, -(currentmixbackvol * 0.018181818F));
      }
   
   handle_mute_button(&lp_listen_mute, left_audio);
   handle_mute_button(&lp_stream_mute, left_stream);
   handle_mute_button(&rp_listen_mute, right_audio);
   handle_mute_button(&rp_stream_mute, right_stream);
   
   /* the factors that will be applied in the mix on the media players */
   lp_lc_aud = lp_rc_aud = vol_rescale * lp_listen_mute;
   rp_lc_aud = rp_rc_aud = vol_rescale * rp_listen_mute;
   lp_lc_str = lp_rc_str = vol_rescale * cross_left * lp_stream_mute;
   rp_lc_str = rp_rc_str = vol_rescale * cross_right * rp_stream_mute;
   jp_lc_str = jp_rc_str = jp_lc_aud = jp_rc_aud = jingles_vol_rescale;
   mb_lc_aud = mb_rc_aud = mixback_rescale;
   ip_lc_aud = ip_rc_aud = 0.0F;
   ip_lc_str = ip_rc_str = interlude_vol_rescale;
      
   mic_target = ((lmic_on || rmic_on) ? -1.0F : 0.0F) * headroom_db;
   if ((diff = mic_target - current_headroom))
      {
      current_headroom += diff * 0.0666666;
      if (mic_target)
         {
         if (diff > -0.00001F)
            {
            current_headroom = mic_target;
            }
         }
      else
         if (diff < 0.0000004F)
            {
            current_headroom = 0.0F; 
            }
      }
      
   /* calculate the ducking factor reduction based on gain controls */
   if (jingles_playing)
      vol = current_jingles_volume * 0.06666666F;
   else
      vol = current_volume * 0.06666666F;
   dfmod = 1.0F / (vol * vol + 1.0F);

   dj_mic_aud = (float)mic_to_dj;
   }

/* update_mic_and_aux: provide gradual mute/unmute for mic and aux buttons */
void update_mic_and_aux()	/* mic and aux mute/unmute smoothing */
   {
   const sample_t onfactor = 0.0006F;
   const sample_t offfactor = 0.00028F;
   const sample_t upper = 0.999999;	/* these values are to prevent cpu usage going ^^^^ */
   const sample_t lower = 0.0000004;	/* -120 dB */
   
   if (lmic_flux)
      {
      if (lmic_on)
         {
         if (mic_lc < upper)
            mic_lc += (1.0F - mic_lc) * onfactor;
         else
            {
            mic_lc = 1.0F;
            lmic_flux = !lmic_on;
            }
         }   
      else
         {
         if (mic_lc > lower)
            mic_lc -= mic_lc * offfactor;
         else
            {
            mic_lc = 0.0F;
            lmic_flux = lmic_on;
            }
         }
      }
   
   if (rmic_flux)
      {
      if (rmic_on)
         {
         if (mic_rc < upper)
            mic_rc += (1.0F - mic_rc) * onfactor;
         else
            {
            mic_rc = 1.0F;
            rmic_flux = !rmic_on;
            }
         }   
      else
         {
         if (mic_rc > lower)
            mic_rc -= mic_rc * offfactor;
         else
            {
            mic_rc = 0.0F;
            rmic_flux = rmic_on;
            }
         }
      }
   
   if (aux_flux)
      {
      if (aux_on)
         {
         if (aux_lc < upper)
            aux_lc = aux_rc += (1.0F - aux_lc) * onfactor;
         else
            {
            aux_lc = aux_rc = 1.0F;
            aux_flux = !aux_on;
            }
         }   
      else
         {
         if (aux_lc > lower)
            aux_lc = aux_rc -= aux_lc * offfactor;
         else
            {
            aux_lc = aux_rc = 0.0F;
            aux_flux = rmic_on;
            }
         }
      }
   }

/* process_audio: the JACK callback routine */
void process_audio(jack_nframes_t nframes)
   {
   int samples_todo;		/* The samples remaining counter in the main loop */
   static float df = 1.0;	/* the ducking factor - generated by the compressor */
   static int mic_proc_counter; /* reduce cpu usage when mic is off */
   /* the following are used to turn the individual microphone inputs into a stereo mix */
   sample_t lmic, rmic, lstr, rstr, lc_micmix, rc_micmix;
   /* the following are used to apply the output of the compressor code to the audio levels */
   static float lmic_gain = 1.0, rmic_gain = 1.0;
   sample_t compressor_gain = 1.0, noisegate_gain = 1.0, dfdb, str_normalizer_gain;
   /* a counter variable used to trigger the volume smoothing on a regular basis */
   static unsigned vol_smooth_count = 0;
   /* index values for reading from a table of fade gain values */
   static jack_nframes_t alarm_index = 0;
   /* pointers to buffers provided by JACK */
   sample_t *lap, *rap, *lsp, *rsp, *lmp, *rmp, *lxp, *rxp, *lpsp, *rpsp, *lprp, *rprp;
   sample_t *la_buffer, *ra_buffer, *ls_buffer, *rs_buffer, *lps_buffer, *rps_buffer;
   /* ponters to buffers for reading the media players */
   sample_t *lplcp, *lprcp, *rplcp, *rprcp, *jplcp, *jprcp, *iplcp, *iprcp;
   /* pointers to buffers for fading */
   sample_t *lplcpf, *lprcpf, *rplcpf, *rprcpf, *jplcpf, *jprcpf, *iplcpf, *iprcpf;
   /* temporary storage for processed fade values */
   sample_t lp_lc_fade, lp_rc_fade, rp_lc_fade, rp_rc_fade;
   sample_t jp_lc_fade, jp_rc_fade, ip_lc_fade, ip_rc_fade;

   if (timeout > 8000)
      {
      if (!shutdown)
         {
         fprintf(stderr, "timeout exceeded - unblocking xlplayer threads, shutdown flag set\n");
         shutdown = TRUE;
         }
      plr_l->command = CMD_COMPLETE;
      plr_r->command = CMD_COMPLETE;
      plr_j->command = CMD_COMPLETE;
      plr_i->command = CMD_COMPLETE;
      }

   /* get the data for the jack ports */
   la_buffer = lap = (sample_t *) jack_port_get_buffer (audio_left_port, nframes);
   ra_buffer = rap = (sample_t *) jack_port_get_buffer (audio_right_port, nframes);
   ls_buffer = lsp = (sample_t *) jack_port_get_buffer (stream_left_port, nframes);
   rs_buffer = rsp = (sample_t *) jack_port_get_buffer (stream_right_port, nframes);
   lps_buffer = lpsp = (sample_t *) jack_port_get_buffer (phone_left_send, nframes);
   rps_buffer = rpsp = (sample_t *) jack_port_get_buffer (phone_right_send, nframes);
   lmp = (sample_t *) jack_port_get_buffer (mic_left_channel, nframes);
   rmp = (sample_t *) jack_port_get_buffer (mic_right_channel, nframes);
   lxp = (sample_t *) jack_port_get_buffer (aux_left_channel, nframes);
   rxp = (sample_t *) jack_port_get_buffer (aux_right_channel, nframes);
   lprp = (sample_t *) jack_port_get_buffer (phone_left_recv, nframes);
   rprp = (sample_t *) jack_port_get_buffer (phone_right_recv, nframes);
   
   /* recreate buffers for data read from the pipes via the jack ringbuffer */
   lp_lc = lplcp = irealloc(lp_lc, nframes);
   lp_rc = lprcp = irealloc(lp_rc, nframes);
   rp_lc = rplcp = irealloc(rp_lc, nframes);
   rp_rc = rprcp = irealloc(rp_rc, nframes);
   jp_lc = jplcp = irealloc(jp_lc, nframes);
   jp_rc = jprcp = irealloc(jp_rc, nframes);
   ip_lc = iplcp = irealloc(ip_lc, nframes);
   ip_rc = iprcp = irealloc(ip_rc, nframes);
   
   /* recreate buffers and pointers for fade */
   lp_lcf = lplcpf = irealloc(lp_lcf, nframes);
   lp_rcf = lprcpf = irealloc(lp_rcf, nframes);
   rp_lcf = rplcpf = irealloc(rp_lcf, nframes);
   rp_rcf = rprcpf = irealloc(rp_rcf, nframes);
   jp_lcf = jplcpf = irealloc(jp_lcf, nframes);
   jp_rcf = jprcpf = irealloc(jp_rcf, nframes);
   ip_lcf = iplcpf = irealloc(ip_lcf, nframes);
   ip_rcf = iprcpf = irealloc(ip_rcf, nframes);

   if (!(lp_lc && lp_rc && rp_lc && rp_rc && jp_lc && jp_rc && ip_lc && ip_rc &&
         lp_lcf && lp_rcf && rp_lcf && rp_rcf && jp_lcf && jp_rcf && ip_lcf && ip_rcf))
      {
      if (!shutdown)
         {
         printf("Malloc failure in process audio\n");
         shutdown = TRUE;
         }
      plr_l->command = CMD_COMPLETE;
      plr_r->command = CMD_COMPLETE;
      plr_j->command = CMD_COMPLETE;
      plr_i->command = CMD_COMPLETE;
      }  

   if (speed_variance)
      read_from_player_sv(plr_l, lp_lc, lp_rc, lp_lcf, lp_rcf, nframes);
   else
      read_from_player(plr_l, lp_lc, lp_rc, lp_lcf, lp_rcf, nframes);
   if (plr_l->have_swapped_buffers_f)
      {
      lp_lc_audf = lp_lc_aud * df;	/* volume levels of player at stoppage time */
      lp_lc_strf = lp_lc_str * df;	/* these are used to modify the fade volume level */
      lp_rc_audf = lp_rc_aud * df;
      lp_rc_strf = lp_rc_str * df; 
      }
   left_audio_runout = (plr_l->avail < player_samples_cutoff);

   if (speed_variance)
      read_from_player_sv(plr_r, rp_lc, rp_rc, rp_lcf, rp_rcf, nframes);
   else
      read_from_player(plr_r, rp_lc, rp_rc, rp_lcf, rp_rcf, nframes);
   if (plr_r->have_swapped_buffers_f)
      {
      rp_lc_audf = rp_lc_aud * df;
      rp_lc_strf = rp_lc_str * df;
      rp_rc_audf = rp_rc_aud * df;
      rp_rc_strf = rp_rc_str * df;
      }
   right_audio_runout = (plr_r->avail < player_samples_cutoff);
      
   read_from_player(plr_j, jp_lc, jp_rc, jp_lcf, jp_rcf, nframes);
   if (plr_j->have_swapped_buffers_f)
      {
      jp_lc_audf = jp_lc_aud * df;
      jp_lc_strf = jp_lc_str * df;
      jp_rc_audf = jp_rc_aud * df;
      jp_rc_strf = jp_rc_str * df;
      } 
   jingles_audio_f = (plr_j->avail > jingles_samples_cutoff);

   read_from_player(plr_i, ip_lc, ip_rc, ip_lcf, ip_rcf, nframes);
   if (plr_i->have_swapped_buffers_f)
      {
      ip_lc_audf = ip_lc_aud * df;
      ip_lc_strf = ip_lc_str * df;
      ip_rc_audf = ip_rc_aud * df;
      ip_rc_strf = ip_rc_str * df;
      } 
      
   /* resets the running totals for the vu meter stats */      
   if (reset_vu_stats_f)
      {
      mic_lpeak = mic_rpeak = str_lpeak = str_rpeak = 0.0;
      mic_l_tally = mic_r_tally = str_l_tally = str_r_tally = 0.0;
      rms_tally_count = 0;
      left_peak = right_peak = -1.0;
      reset_vu_stats_f = FALSE;
      }

   /* Update mic compression and noisegate control values as needed */
   if (new_compressor_stats)
      {
      new_mic_compressor.gain_db = left_mic_compressor.gain_db;
      new_mic_compressor.ducking_hold_count = left_mic_compressor.gain_db;
      new_mic_compressor.ducking_db = left_mic_compressor.ducking_db;
      left_mic_compressor = new_mic_compressor;
      left_mic_compressor.filter = ess_left_filter;
      left_mic_compressor.rms_filter = lm_rms_filter;
      new_mic_compressor.gain_db = right_mic_compressor.gain_db;
      new_mic_compressor.ducking_hold_count = right_mic_compressor.gain_db;
      new_mic_compressor.ducking_db = right_mic_compressor.ducking_db;
      right_mic_compressor = new_mic_compressor;
      right_mic_compressor.filter = ess_right_filter;
      right_mic_compressor.rms_filter = rm_rms_filter;
      new_compressor_stats = FALSE;
      }
   if (new_noisegate_stats)
      {
      new_ng.delaycount = left_mic_noisegate.delaycount;
      new_ng.gain = left_mic_noisegate.gain;
      new_ng.mute_state = left_mic_noisegate.mute_state;
      left_mic_noisegate = new_ng;
      new_ng.delaycount = right_mic_noisegate.delaycount;
      new_ng.gain = right_mic_noisegate.gain;
      new_ng.mute_state = right_mic_noisegate.mute_state;
      right_mic_noisegate = new_ng;
      new_noisegate_stats = FALSE;
      }
   if (new_normalizer_stats)
      {
      new_normalizer.level = str_normalizer.level;
      str_normalizer = new_normalizer;
      new_normalizer_stats = FALSE;
      }

   /* apply the user selected microphone filter */
   if (mic_filter == 1)
      {
      left_mic_filter = mild_left_mic_filter;
      right_mic_filter = mild_right_mic_filter;
      }
   if (mic_filter == 2)
      {
      left_mic_filter = sharp_right_mic_filter;
      right_mic_filter = sharp_right_mic_filter;
      }
   /* there are four mixer modes and the only seemingly efficient way to do them is */
   /* to basically copy a lot of code four times over hence the huge size */
   if (simple_mixer == FALSE && mixermode == NO_PHONE)	/* Fully featured mixer code */
      {
      memset(lps_buffer, 0, nframes * sizeof (sample_t)); /* send silence to VOIP */
      memset(rps_buffer, 0, nframes * sizeof (sample_t));
      for(samples_todo = nframes; samples_todo--; lap++, rap++, lsp++, rsp++, lxp++, rxp++, lmp++, rmp++,
      		lplcp++, lprcp++, rplcp++, rprcp++, jplcp++, jprcp++, iplcp++, iprcp++)
         {	 
	 /* make separate varibles lmic and rmic for the buffer values */
	 /* so that we can safely mess with the values */
	 if (isunordered(*lmp, *rmp))
	    lmic = rmic = 0.0;
	 else
	    {
	    lmic = *lmp * mic_linv * mic_lgain;
	    rmic = *rmp * mic_rinv * mic_rgain;
	    }
	    
	 /* update the microphone compression settings */
	 if (mic_lc || !(mic_proc_counter & 0x3))
	    {
            if (mic_filter != 0)
               lmic = digital_filter(left_mic_filter, lmic);
            compressor_gain = compressor(&left_mic_compressor, lmic, 0);
            noisegate_gain = noisegate(&left_mic_noisegate, digital_filter(ng_left_filter, lmic));
            lmic_gain = db2level(compressor_gain + noisegate_gain);
	    }
         if (mic_rc || !(mic_proc_counter & 0x3))
            {
            if (mic_filter != 0)
               rmic = digital_filter(right_mic_filter, rmic);
            compressor_gain = compressor(&right_mic_compressor, rmic, 0);
            noisegate_gain = noisegate(&right_mic_noisegate, digital_filter(ng_right_filter, rmic));
            rmic_gain = db2level(compressor_gain + noisegate_gain);
            }
         mic_proc_counter++;
         lmic *= lmic_gain;
         rmic *= rmic_gain;
         
	 if (vol_smooth_count++ % 100 == 0) /* Can change volume level every so many samples */
            update_smoothed_volumes();

	 /* do the stereo mix for the microphones */
	 lc_micmix = (mic_lc * lmic * mic_l_lc) + (mic_rc * rmic * mic_r_lc);
	 rc_micmix = (mic_lc * lmic * mic_l_rc) + (mic_rc * rmic * mic_r_rc); 
	 
	 /* compute the gain factor used to apply ducking - later in the main mix */
	 /* this is done from the stronger of the two microphones ducking factors */
	 /* also neatly incorporated is the headroom for when the mic is open */
         if (mic_lc || mic_rc)
            {
            if (left_mic_compressor.ducking_db * mic_lc < right_mic_compressor.ducking_db * mic_rc)
               dfdb = left_mic_compressor.ducking_db * mic_lc;
            else
               dfdb = right_mic_compressor.ducking_db * mic_rc;
            df = db2level((dfdb + current_headroom) * dfmod);
            }
         
         if (plr_l->fadeindex != FB_SIZE)
            {
            lp_lc_fade = fade_table[plr_l->fadeindex] * *lplcpf++; 
            lp_rc_fade = fade_table[plr_l->fadeindex] * *lprcpf++;
            plr_l->fadeindex++;
            }
         else
            lp_lc_fade = lp_rc_fade = 0.0;
            
         if (plr_r->fadeindex != FB_SIZE)
            {
            rp_lc_fade = fade_table[plr_r->fadeindex] * *rplcpf++; 
            rp_rc_fade = fade_table[plr_r->fadeindex] * *rprcpf++;
            plr_r->fadeindex++;
            }
         else
            rp_lc_fade = rp_rc_fade = 0.0;
         
         if (plr_j->fadeindex != FB_SIZE)
            {
            jp_lc_fade = fade_table[plr_j->fadeindex] * *jplcpf++; 
            jp_rc_fade = fade_table[plr_j->fadeindex] * *jprcpf++;
            plr_j->fadeindex++;
            }
         else
            jp_lc_fade = jp_rc_fade = 0.0;
            
         if (plr_i->fadeindex != FB_SIZE)
            {
            ip_lc_fade = fade_table[plr_i->fadeindex] * *iplcpf++; 
            ip_rc_fade = fade_table[plr_i->fadeindex] * *iprcpf++;
            plr_i->fadeindex++;
            }
         else
            ip_lc_fade = ip_rc_fade = 0.0;
            
         if (lmic_flux || rmic_flux || aux_flux)
            update_mic_and_aux();		/* mic fade in/out */
            
         if (fabs(*lplcp) > left_peak)		/* peak levels used for song cut-off */
            left_peak = fabs(*lplcp);
         if (fabs(*lprcp) > left_peak)
            left_peak = fabs(*lprcp);
         if (fabs(*rplcp) > right_peak)
            right_peak = fabs(*rplcp);
         if (fabs(*rprcp) > right_peak)
            right_peak = fabs(*rprcp);
         
	 /* This is it folks, the main mix */
         *lsp = ((*lplcp * lp_lc_str) + (*rplcp * rp_lc_str) + (*lxp * aux_lc) + (*jplcp * jp_lc_str)) * df + lc_micmix + (*iplcp * ip_lc_str) + (ip_lc_fade * ip_lc_strf) + 
         (lp_lc_fade * lp_lc_strf) + (rp_lc_fade * rp_lc_strf) + (jp_lc_fade * jp_lc_strf);
         *rsp = ((*lprcp * lp_rc_str) + (*rprcp * rp_rc_str) + (*rxp * aux_rc) + (*jprcp * jp_rc_str)) * df + rc_micmix + (*iprcp * ip_rc_str) + (ip_rc_fade * ip_rc_strf) +
         (lp_rc_fade * lp_rc_strf) + (rp_rc_fade * rp_rc_strf) + (jp_rc_fade * jp_rc_strf);
	 
         /* apply normalization at the stream level */
         str_normalizer_gain = db2level(normalizer(&str_normalizer, *lsp, *rsp));
         
         *lsp *= str_normalizer_gain;
         *rsp *= str_normalizer_gain;
         
         /* hard limit the levels if they go outside permitted limits */
	 /* note this is not the same as clipping */
	 compressor_gain = db2level(limiter(&stream_limiter, *lsp, *rsp));

	 *lsp *= compressor_gain;
	 *rsp *= compressor_gain;
	 
         if (stream_monitor == FALSE)
	    {
	    *lap = ((*lplcp * lp_lc_aud) + (*rplcp * rp_lc_aud) + (*jplcp * jp_lc_aud)) * df + (lc_micmix * dj_mic_aud) + (*iplcp * ip_lc_aud) + (ip_lc_fade * ip_lc_aud) +
            (lp_lc_fade * lp_lc_audf) + (rp_lc_fade * rp_lc_audf) + (jp_lc_fade * jp_lc_audf);
	    *rap = ((*lprcp * lp_rc_aud) + (*rprcp * rp_rc_aud) + (*jprcp * jp_rc_aud)) * df + (rc_micmix * dj_mic_aud) + (*iprcp * ip_rc_aud) + (ip_rc_fade * ip_rc_aud) +
            (lp_rc_fade * lp_rc_audf) + (rp_rc_fade * rp_rc_audf) + (jp_rc_fade * jp_rc_audf);
	    compressor_gain = db2level(limiter(&audio_limiter, *lap, *rap));
	    *lap *= compressor_gain;
	    *rap *= compressor_gain;
	    }
	 else
	    {
	    *lap = *lsp;  /* allow the DJ to hear the mix that the listeners are hearing */
	    *rap = *rsp;
	    }
            
         if (eot_alarm_f)	/* mix in the end-of-track alarm tone */
            {
            if (alarm_index >= alarm_size)
               {
               alarm_index = 0;
               eot_alarm_f = 0;
               }
            else
               {
               *lap += eot_alarm_table[alarm_index];
               *lap *= 0.5;
               *rap += eot_alarm_table[alarm_index];
               *rap *= 0.5;
               alarm_index++;
               }
            }
	 	 
	 *lap *= dj_audio_gain;
	 *rap *= dj_audio_gain;	 
	 	 
	 /* make note of the peak volume levels */
	 lmic = fabsf(lmic);
	 if (lmic > 1.0F)
	    lmic = 1.0F;
	 rmic = fabsf(rmic);
	 if (rmic > 1.0F)
	    rmic = 1.0F;
	 lstr = fabsf(*lsp);
	 rstr = fabsf(*rsp);
	 if (lmic > mic_lpeak)
	    mic_lpeak = lmic;
	 if (rmic > mic_rpeak)
	    mic_rpeak = rmic;
	 if (lstr > str_lpeak)
	    str_lpeak = lstr;
	 if (rstr > str_rpeak)
	    str_rpeak = rstr;

	 /* a running total of sound pressure levels used for rms calculation */
	 mic_l_tally += lmic * lmic;
	 mic_r_tally += rmic * rmic;
	 str_l_tally += lstr * lstr;
	 str_r_tally += rstr * rstr;
	 rms_tally_count++;	/* the divisor for the above running counts */
         /* beat analysis */
         /*beat_add(beat_lp, *lplcp * *lplcp, *lprcp * *lprcp);
         beat_add(beat_rp, *rplcp * *rplcp, *rprcp * *rprcp);*/
	 }
      mic_l_meansqrd = mic_l_tally/rms_tally_count;	/* calculations done here for */
      mic_r_meansqrd = mic_r_tally/rms_tally_count;	/* reasons of thread safety */
      str_l_meansqrd = str_l_tally/rms_tally_count;
      str_r_meansqrd = str_r_tally/rms_tally_count;
      }
   else
      if (simple_mixer == FALSE && mixermode == PHONE_PUBLIC)
         {
         for(samples_todo = nframes; samples_todo--; lap++, rap++, lsp++, rsp++, lxp++, rxp++,
         	lmp++, rmp++, lplcp++, lprcp++, rplcp++, rprcp++, jplcp++, jprcp++,
         	lpsp++, rpsp++, lprp++, rprp++, iplcp++, iprcp++)
            {	 
            /* make separate varibles lmic and rmic for the buffer values */
            /* so that we can safely mess with the values */
            if (isunordered(*lmp, *rmp))
               lmic = rmic = 0.0;
            else
               {
               lmic = *lmp * mic_linv * mic_lgain;
               rmic = *rmp * mic_rinv * mic_rgain;
               }
            
            /* update the microphone compression settings */
            if (mic_lc || !(mic_proc_counter & 0x3))
               {
               if (mic_filter != 0)
                  lmic = digital_filter(left_mic_filter, lmic);
               compressor_gain = compressor(&left_mic_compressor, lmic, 0);
               noisegate_gain = noisegate(&left_mic_noisegate, digital_filter(ng_left_filter, lmic));
               lmic_gain = db2level(compressor_gain + noisegate_gain);
               }
            if (mic_rc || !(mic_proc_counter & 0x3))
               {
               if (mic_filter != 0)
                  rmic = digital_filter(right_mic_filter, rmic);
               compressor_gain = compressor(&right_mic_compressor, rmic, 0);
               noisegate_gain = noisegate(&right_mic_noisegate, digital_filter(ng_right_filter, rmic));
               rmic_gain = db2level(compressor_gain + noisegate_gain);
               }
            mic_proc_counter++;
            lmic *= lmic_gain;
            rmic *= rmic_gain;
            
            if (vol_smooth_count++ % 100 == 0) /* Can change volume level every so many samples */
               update_smoothed_volumes();
   
            /* do the stereo mix for the microphones */
            lc_micmix = (lmic * mic_l_lc) + (rmic * mic_r_lc);
            rc_micmix = (lmic * mic_l_rc) + (rmic * mic_r_rc); 
            
            /* No ducking */
            df = 1.0;
            
            if (plr_l->fadeindex != FB_SIZE)
               {
               lp_lc_fade = fade_table[plr_l->fadeindex] * *lplcpf++; 
               lp_rc_fade = fade_table[plr_l->fadeindex] * *lprcpf++;
               plr_l->fadeindex++;
               }
            else
               lp_lc_fade = lp_rc_fade = 0.0;
               
            if (plr_r->fadeindex != FB_SIZE)
               {
               rp_lc_fade = fade_table[plr_r->fadeindex] * *rplcpf++; 
               rp_rc_fade = fade_table[plr_r->fadeindex] * *rprcpf++;
               plr_r->fadeindex++;
               }
            else
               rp_lc_fade = rp_rc_fade = 0.0;
            
            if (plr_j->fadeindex != FB_SIZE)
               {
               jp_lc_fade = fade_table[plr_j->fadeindex] * *jplcpf++; 
               jp_rc_fade = fade_table[plr_j->fadeindex] * *jprcpf++;
               plr_j->fadeindex++;
               }
            else
               jp_lc_fade = jp_rc_fade = 0.0;
            
            if (plr_i->fadeindex != FB_SIZE)
               {
               ip_lc_fade = fade_table[plr_i->fadeindex] * *iplcpf++; 
               ip_rc_fade = fade_table[plr_i->fadeindex] * *iprcpf++;
               plr_i->fadeindex++;
               }
            else
               ip_lc_fade = ip_rc_fade = 0.0;
            
            /* do the phone send mix */
            *lpsp = lc_micmix + (*jplcp * jp_lc_str) + (jp_lc_fade * jp_lc_strf);
            *rpsp = rc_micmix + (*jprcp * jp_rc_str) + (jp_rc_fade * jp_rc_strf);
            
            if (lmic_flux || rmic_flux || aux_flux)
               update_mic_and_aux();		/* smooth mic rise and fall mute/unmute */

            if (fabs(*lplcp) > left_peak)		/* peak levels used for song cut-off */
               left_peak = fabs(*lplcp);
            if (fabs(*lprcp) > left_peak)
               left_peak = fabs(*lprcp);
            if (fabs(*rplcp) > right_peak)
               right_peak = fabs(*rplcp);
            if (fabs(*rprcp) > right_peak)
               right_peak = fabs(*rprcp);

            /* The main mix */
            *lsp = ((*lplcp * lp_lc_str) + (*rplcp * rp_lc_str) + (*lxp * aux_lc)) + *lprp + *lpsp +
            (lp_lc_fade * lp_rc_strf) + (rp_lc_fade * rp_lc_strf) + (*iplcp * ip_lc_str) + (ip_lc_fade * ip_lc_strf);
            *rsp = ((*lprcp * lp_rc_str) + (*rprcp * rp_rc_str) + (*rxp * aux_rc)) + *rprp + *rpsp +
            (lp_rc_fade * lp_rc_strf) + (rp_rc_fade * rp_rc_strf) + (*iprcp * ip_rc_str) + (ip_rc_fade * ip_rc_strf);
            
            compressor_gain = db2level(limiter(&phone_limiter, *lpsp, *rpsp));
            *lpsp *= compressor_gain;
            *rpsp *= compressor_gain;

            /* apply normalization at the stream level */
            str_normalizer_gain = db2level(normalizer(&str_normalizer, *lsp, *rsp));
            
            *lsp *= str_normalizer_gain;
            *rsp *= str_normalizer_gain;
            
            /* hard limit the levels if they go outside permitted limits */
            /* note this is not the same as clipping */
            compressor_gain = db2level(limiter(&stream_limiter, *lsp, *rsp));
   
            *lsp *= compressor_gain;
            *rsp *= compressor_gain;
            if (stream_monitor == FALSE)
               {
               *lap = ((*lplcp * lp_lc_aud) + (*rplcp * rp_lc_aud)) + *lprp +
               (lp_lc_fade * lp_rc_audf) + (rp_lc_fade * rp_lc_audf) + (*iplcp * ip_lc_aud) + (ip_lc_fade * ip_lc_audf) + (lc_micmix * dj_mic_aud) + (*jplcp * jp_lc_str) + (jp_lc_fade * jp_lc_strf);
               *rap = ((*lprcp * lp_rc_aud) + (*rprcp * rp_rc_aud)) + *rprp +
               (lp_rc_fade * lp_rc_audf) + (rp_rc_fade * rp_rc_audf) + (*iprcp * ip_rc_aud) + (ip_rc_fade * ip_rc_audf) + (rc_micmix * dj_mic_aud) + (*jprcp * jp_rc_str) + (jp_rc_fade * jp_rc_strf);
               compressor_gain = db2level(limiter(&audio_limiter, *lap, *rap));
               *lap *= compressor_gain;
               *rap *= compressor_gain;
               }
            else
               {
               *lap = *lsp;  /* allow the DJ to hear the mix that the listeners are hearing */
               *rap = *rsp;
               }
               
            if (eot_alarm_f)	/* mix in the end-of-track alarm tone */
               {
               if (alarm_index >= alarm_size)
                  {
                  alarm_index = 0;
                  eot_alarm_f = 0;
                  }
               else
                  {
                  *lap += eot_alarm_table[alarm_index];
                  *lap *= 0.5;
                  *rap += eot_alarm_table[alarm_index];
                  *rap *= 0.5;
                  alarm_index++;
                  }
               }
                  
            *lap *= dj_audio_gain;
            *rap *= dj_audio_gain;	 
                  
            /* make note of the peak volume levels */
            lmic = fabsf(lmic);
            if (lmic > 1.0F)
               lmic = 1.0F;
            rmic = fabsf(rmic);
            if (rmic > 1.0F)
               rmic = 1.0F;
            lstr = fabsf(*lsp);
            rstr = fabsf(*rsp);
            if (lmic > mic_lpeak)
               mic_lpeak = lmic;
            if (rmic > mic_rpeak)
               mic_rpeak = rmic;
            if (lstr > str_lpeak)
               str_lpeak = lstr;
            if (rstr > str_rpeak)
               str_rpeak = rstr;
            
            /* a running total of sound pressure levels used for rms calculation */
            mic_l_tally += lmic * lmic;
            mic_r_tally += rmic * rmic;
            str_l_tally += lstr * lstr;
            str_r_tally += rstr * rstr;
            rms_tally_count++;	/* the divisor for the above running counts */
            }
         mic_l_meansqrd = mic_l_tally/rms_tally_count;	/* calculations done here for */
         mic_r_meansqrd = mic_r_tally/rms_tally_count;	/* reasons of thread safety */
         str_l_meansqrd = str_l_tally/rms_tally_count;
         str_r_meansqrd = str_r_tally/rms_tally_count;
         }
      else
         if (simple_mixer == FALSE && mixermode == PHONE_PRIVATE && lmic_on == 0 && rmic_on == 0)
            {
            for(samples_todo = nframes; samples_todo--; lap++, rap++, lsp++, rsp++, lxp++, rxp++, lmp++, rmp++,
            lplcp++, lprcp++, rplcp++, rprcp++, jplcp++, jprcp++, lpsp++, rpsp++, 
            lprp++, rprp++, iplcp++, iprcp++)
               {	 
               /* make separate varibles lmic and rmic for the buffer values */
               /* so that we can safely mess with the values */
               if (isunordered(*lmp, *rmp))
                  lmic = rmic = 0.0;
               else
                  {
                  lmic = *lmp * mic_linv * mic_lgain;
                  rmic = *rmp * mic_rinv * mic_rgain;
                  }
               
               /* update the microphone compression settings */
               if (mic_lc || !(mic_proc_counter & 0x3))
                  {
                  if (mic_filter != 0)
                     lmic = digital_filter(left_mic_filter, lmic);
                  compressor_gain = compressor(&left_mic_compressor, lmic, 0);
                  noisegate_gain = noisegate(&left_mic_noisegate, digital_filter(ng_left_filter, lmic));
                  lmic_gain = db2level(compressor_gain + noisegate_gain);
                  }
               if (mic_rc || !(mic_proc_counter & 0x3))
                  {
                  if (mic_filter != 0)
                     rmic = digital_filter(right_mic_filter, rmic);
                  compressor_gain = compressor(&right_mic_compressor, rmic, 0);
                  noisegate_gain = noisegate(&right_mic_noisegate, digital_filter(ng_right_filter, rmic));
                  rmic_gain = db2level(compressor_gain + noisegate_gain);
                  }
               mic_proc_counter++;
               lmic *= lmic_gain;
               rmic *= rmic_gain;
               
               if (vol_smooth_count++ % 100 == 0) /* Can change volume level every so many samples */
                  update_smoothed_volumes();
      
               /* do the stereo mix for the microphones */
               lc_micmix = (lmic * mic_l_lc) + (rmic * mic_r_lc);
               rc_micmix = (lmic * mic_l_rc) + (rmic * mic_r_rc); 
               
               /* No ducking */
               df = 1.0;
               
               if (plr_l->fadeindex != FB_SIZE)
                  {
                  lp_lc_fade = fade_table[plr_l->fadeindex] * *lplcpf++; 
                  lp_rc_fade = fade_table[plr_l->fadeindex] * *lprcpf++;
                  plr_l->fadeindex++;
                  }
               else
                  lp_lc_fade = lp_rc_fade = 0.0;
                  
               if (plr_r->fadeindex != FB_SIZE)
                  {
                  rp_lc_fade = fade_table[plr_r->fadeindex] * *rplcpf++; 
                  rp_rc_fade = fade_table[plr_r->fadeindex] * *rprcpf++;
                  plr_r->fadeindex++;
                  }
               else
                  rp_lc_fade = rp_rc_fade = 0.0;
               
               if (plr_j->fadeindex!= FB_SIZE)
                  {
                  jp_lc_fade = fade_table[plr_j->fadeindex] * *jplcpf++; 
                  jp_rc_fade = fade_table[plr_j->fadeindex] * *jprcpf++;
                  plr_j->fadeindex++;
                  }
               else
                  jp_lc_fade = jp_rc_fade = 0.0;
               
               if (plr_i->fadeindex != FB_SIZE)
                  {
                  ip_lc_fade = fade_table[plr_i->fadeindex] * *iplcpf++; 
                  ip_rc_fade = fade_table[plr_i->fadeindex] * *iprcpf++;
                  plr_i->fadeindex++;
                  }
               else
                  ip_lc_fade = ip_rc_fade = 0.0;
               
               if (lmic_flux || rmic_flux || aux_flux)
                  update_mic_and_aux();		/* smooth mic rise and fall mute/unmute */
               
               if (fabs(*lplcp) > left_peak)		/* peak levels used for song cut-off */
                  left_peak = fabs(*lplcp);
               if (fabs(*lprcp) > left_peak)
                  left_peak = fabs(*lprcp);
               if (fabs(*rplcp) > right_peak)
                  right_peak = fabs(*rplcp);
               if (fabs(*rprcp) > right_peak)
                  right_peak = fabs(*rprcp);
               
               /* This is it folks, the main mix */
               *lsp = (*lplcp * lp_lc_str) + (*rplcp * rp_lc_str) + (*lxp * aux_lc) +
               (lp_lc_fade * lp_rc_strf) + (rp_lc_fade * rp_lc_strf) + (*iplcp * ip_lc_str) + (ip_lc_fade * ip_lc_strf);
               *rsp = (*lprcp * lp_rc_str) + (*rprcp * rp_rc_str) + (*rxp * aux_rc) +
               (lp_rc_fade * lp_rc_strf) + (rp_rc_fade * rp_rc_strf) + (*iprcp * ip_rc_str) + (ip_rc_fade * ip_rc_strf);
               
               /* apply normalization at the stream level */
               str_normalizer_gain = db2level(normalizer(&str_normalizer, *lsp, *rsp));
               
               *lsp *= str_normalizer_gain;
               *rsp *= str_normalizer_gain;
               
               /* hard limit the levels if they go outside permitted limits */
               /* note this is not the same as clipping */
               compressor_gain = db2level(limiter(&stream_limiter, *lsp, *rsp));
      
               *lsp *= compressor_gain;
               *rsp *= compressor_gain;
               
               /* The mix the voip listeners receive */
               *lpsp = (*lsp * mb_lc_aud) + (*jplcp * jp_lc_aud) + lc_micmix + 
               (jp_lc_fade * jp_lc_strf);
               *rpsp = (*rsp * mb_lc_aud) + (*jprcp * jp_rc_aud) + rc_micmix +
               (jp_rc_fade * jp_rc_strf);
               compressor_gain = db2level(limiter(&phone_limiter, *lpsp, *rpsp));
               *lpsp *= compressor_gain;
               *rpsp *= compressor_gain;
               
               if (stream_monitor == FALSE) /* the DJ can hear the VOIP phone call */
                  {
                  *lap = (*lsp * mb_lc_aud) + (*jplcp * jp_lc_aud) + (lc_micmix * dj_mic_aud) + (jp_lc_fade * jp_lc_strf) + *lprp;
                  *rap = (*rsp * mb_lc_aud) + (*jprcp * jp_rc_aud) + (rc_micmix * dj_mic_aud) + (jp_rc_fade * jp_rc_strf) + *rprp;
                  compressor_gain = db2level(limiter(&audio_limiter, *lap, *rap));
                  *lap *= compressor_gain;
                  *rap *= compressor_gain;
                  }
               else
                  {
                  *lap = *lsp;  /* allow the DJ to hear the mix that the listeners are hearing */
                  *rap = *rsp;
                  }
                  
               if (eot_alarm_f)	/* mix in the end-of-track alarm tone */
                  {
                  if (alarm_index >= alarm_size)
                     {
                     alarm_index = 0;
                     eot_alarm_f = 0;
                     }
                  else
                     {
                     *lap += eot_alarm_table[alarm_index];
                     *lap *= 0.5;
                     *rap += eot_alarm_table[alarm_index];
                     *rap *= 0.5;
                     alarm_index++;
                     }
                  }
                     
               *lap *= dj_audio_gain;
               *rap *= dj_audio_gain;	 
                     
               /* make note of the peak volume levels */
               lmic = fabsf(lmic);
               if (lmic > 1.0F)
                  lmic = 1.0F;
               rmic = fabsf(rmic);
               if (rmic > 1.0F)
                  rmic = 1.0F;
               lstr = fabsf(*lsp);
               rstr = fabsf(*rsp);
               if (lmic > mic_lpeak)
                  mic_lpeak = lmic;
               if (rmic > mic_rpeak)
                  mic_rpeak = rmic;
               if (lstr > str_lpeak)
                  str_lpeak = lstr;
               if (rstr > str_rpeak)
                  str_rpeak = rstr;
               
               /* a running total of sound pressure levels used for rms calculation */
               mic_l_tally += lmic * lmic;
               mic_r_tally += rmic * rmic;
               str_l_tally += lstr * lstr;
               str_r_tally += rstr * rstr;
               rms_tally_count++;	/* the divisor for the above running counts */
               }
            mic_l_meansqrd = mic_l_tally/rms_tally_count;	/* calculations done here for */
            mic_r_meansqrd = mic_r_tally/rms_tally_count;	/* reasons of thread safety */
            str_l_meansqrd = str_l_tally/rms_tally_count;
            str_r_meansqrd = str_r_tally/rms_tally_count;
            }
         else
            if (simple_mixer == FALSE && mixermode == PHONE_PRIVATE) /* note: mic is on */
               {
               for(samples_todo = nframes; samples_todo--; lap++, rap++, lsp++, rsp++, lxp++, rxp++, lmp++, rmp++,
                        lplcp++, lprcp++, rplcp++, rprcp++, jplcp++, jprcp++, lpsp++, rpsp++,
                  	iplcp++, iprcp++)
                  {	 
                  /* make separate varibles lmic and rmic for the buffer values */
                  /* so that we can safely mess with the values */
                  if (isunordered(*lmp, *rmp))
                     lmic = rmic = 0.0;
                  else
                     {
                     lmic = *lmp * mic_linv * mic_lgain;
                     rmic = *rmp * mic_rinv * mic_rgain;
                     }
                  
                  /* update the microphone compression settings */
                  if (mic_lc || !(mic_proc_counter & 0x3))
                     {
                     if (mic_filter != 0)
                        lmic = digital_filter(left_mic_filter, lmic);
                     compressor_gain = compressor(&left_mic_compressor, lmic, 0);
                     noisegate_gain = noisegate(&left_mic_noisegate, digital_filter(ng_left_filter, lmic));
                     lmic_gain = db2level(compressor_gain + noisegate_gain);
                     }
                  if (mic_rc || !(mic_proc_counter & 0x3))
                     {
                     if (mic_filter != 0)
                        rmic = digital_filter(right_mic_filter, rmic);
                     compressor_gain = compressor(&right_mic_compressor, rmic, 0);
                     noisegate_gain = noisegate(&right_mic_noisegate, digital_filter(ng_right_filter, rmic));
                     rmic_gain = db2level(compressor_gain + noisegate_gain);
                     }
                  mic_proc_counter++;
                  lmic *= lmic_gain;
                  rmic *= rmic_gain;
                  
                  if (vol_smooth_count++ % 100 == 0) /* Can change volume level every so many samples */
                     update_smoothed_volumes();
         
                  /* do the stereo mix for the microphones */
                  lc_micmix = (mic_lc * lmic * mic_l_lc) + (mic_rc * rmic * mic_r_lc);
                  rc_micmix = (mic_lc * lmic * mic_l_rc) + (mic_rc * rmic * mic_r_rc); 
                  
                  /* compute the gain factor used to apply ducking - later in the main mix */
                  /* this is done from the stronger of the two microphones ducking factors */
                  /* also neatly incorporated is the headroom for when the mic is open */
                  if (mic_lc || mic_rc)
                     {
                     if (left_mic_compressor.ducking_db * mic_lc < right_mic_compressor.ducking_db * mic_rc)
                        dfdb = left_mic_compressor.ducking_db * mic_lc;
                     else
                        dfdb = right_mic_compressor.ducking_db * mic_rc;
                     df = db2level((dfdb + current_headroom) * dfmod);
                     }
                     
                  if (plr_l->fadeindex != FB_SIZE)
                     {
                     lp_lc_fade = fade_table[plr_l->fadeindex] * *lplcpf++; 
                     lp_rc_fade = fade_table[plr_l->fadeindex] * *lprcpf++;
                     plr_l->fadeindex++;
                     }
                  else
                     lp_lc_fade = lp_rc_fade = 0.0;
                     
                  if (plr_r->fadeindex != FB_SIZE)
                     {
                     rp_lc_fade = fade_table[plr_r->fadeindex] * *rplcpf++; 
                     rp_rc_fade = fade_table[plr_r->fadeindex] * *rprcpf++;
                     plr_r->fadeindex++;
                     }
                  else
                     rp_lc_fade = rp_rc_fade = 0.0;
                  
                  if (plr_j->fadeindex!= FB_SIZE)
                     {
                     jp_lc_fade = fade_table[plr_j->fadeindex] * *jplcpf++; 
                     jp_rc_fade = fade_table[plr_j->fadeindex] * *jprcpf++;
                     plr_j->fadeindex++;
                     }
                  else
                     jp_lc_fade = jp_rc_fade = 0.0;

                  if (plr_i->fadeindex != FB_SIZE)
                     {
                     ip_lc_fade = fade_table[plr_i->fadeindex] * *iplcpf++; 
                     ip_rc_fade = fade_table[plr_i->fadeindex] * *iprcpf++;
                     plr_i->fadeindex++;
                     }
                  else
                     ip_lc_fade = ip_rc_fade = 0.0;
                  
                  if (lmic_flux || rmic_flux || aux_flux)
  	             update_mic_and_aux();	/* smooth mic rise and fall mute/unmute */

                  if (fabs(*lplcp) > left_peak)		/* peak levels used for song cut-off */
                     left_peak = fabs(*lplcp);
                  if (fabs(*lprcp) > left_peak)
                     left_peak = fabs(*lprcp);
                  if (fabs(*rplcp) > right_peak)
                     right_peak = fabs(*rplcp);
                  if (fabs(*rprcp) > right_peak)
                     right_peak = fabs(*rprcp);

                  /* This is it folks, the main mix */
                  *lsp = ((*lplcp * lp_lc_str) + (*rplcp * rp_lc_str) + (*lxp * aux_lc) + (*jplcp * jp_lc_str)) * df + lc_micmix + (*iplcp * ip_lc_str) + (ip_lc_fade * ip_lc_strf) +
                  (lp_lc_fade * lp_lc_strf) + (rp_lc_fade * rp_lc_strf) + (jp_lc_fade * jp_lc_strf);
                  *rsp = ((*lprcp * lp_rc_str) + (*rprcp * rp_rc_str) + (*rxp * aux_rc) + (*jprcp * jp_rc_str)) * df + rc_micmix + (*iprcp * ip_rc_str) + (ip_rc_fade * ip_rc_strf) +
                  (lp_rc_fade * lp_rc_strf) + (rp_rc_fade * rp_rc_strf) + (jp_rc_fade * jp_rc_strf);
                  
                  /* apply normalization at the stream level */
                  str_normalizer_gain = db2level(normalizer(&str_normalizer, *lsp, *rsp));
                  
                  *lsp *= str_normalizer_gain;
                  *rsp *= str_normalizer_gain;
                  
                  /* hard limit the levels if they go outside permitted limits */
                  /* note this is not the same as clipping */
                  compressor_gain = db2level(limiter(&stream_limiter, *lsp, *rsp));
         
                  *lsp *= compressor_gain;
                  *rsp *= compressor_gain;
                  
                  *lpsp = *lsp * mb_lc_aud;	/* voip callers get stream mix at a certain volume */ 
                  *rpsp = *rsp * mb_rc_aud;
                  
                  if (stream_monitor == FALSE)
                     {
                     *lap = ((*lplcp * lp_lc_aud) + (*rplcp * rp_lc_aud) + (*jplcp * jp_lc_aud)) * df + (lc_micmix * dj_mic_aud) + (*iplcp * ip_lc_aud) + (ip_lc_fade * ip_lc_audf) +
                     (lp_lc_fade * lp_lc_audf) + (rp_lc_fade * rp_lc_audf) + (jp_lc_fade * jp_lc_audf);
                     *rap = ((*lprcp * lp_rc_aud) + (*rprcp * rp_rc_aud) + (*jprcp * jp_rc_aud)) * df + (rc_micmix * dj_mic_aud) + (*iprcp * ip_rc_aud) + (ip_rc_fade * ip_rc_audf) +
                     (lp_rc_fade * lp_rc_audf) + (rp_rc_fade * rp_rc_audf) + (jp_rc_fade * jp_rc_audf);
                     compressor_gain = db2level(limiter(&audio_limiter, *lap, *rap));
                     *lap *= compressor_gain;
                     *rap *= compressor_gain;
                     }
                  else
                     {
                     *lap = *lsp;  /* allow the DJ to hear the mix that the listeners are hearing */
                     *rap = *rsp;
                     }
                     
                  if (eot_alarm_f)	/* mix in the end-of-track alarm tone */
                     {
                     if (alarm_index >= alarm_size)
                        {
                        alarm_index = 0;
                        eot_alarm_f = 0;
                        }
                     else
                        {
                        *lap += eot_alarm_table[alarm_index];
                        *lap *= 0.5;
                        *rap += eot_alarm_table[alarm_index];
                        *rap *= 0.5;
                        alarm_index++;
                        }
                     }
                        
                  *lap *= dj_audio_gain;
                  *rap *= dj_audio_gain;	 
                        
                  /* make note of the peak volume levels */
                  lmic = fabsf(lmic);
                  if (lmic > 1.0F)
                     lmic = 1.0F;
                  rmic = fabsf(rmic);
                  if (rmic > 1.0F)
                     rmic = 1.0F;
                  lstr = fabsf(*lsp);
                  rstr = fabsf(*rsp);
                  if (lmic > mic_lpeak)
                     mic_lpeak = lmic;
                  if (rmic > mic_rpeak)
                     mic_rpeak = rmic;
                  if (lstr > str_lpeak)
                     str_lpeak = lstr;
                  if (rstr > str_rpeak)
                     str_rpeak = rstr;
                  
                  /* a running total of sound pressure levels used for rms calculation */
                  mic_l_tally += lmic * lmic;
                  mic_r_tally += rmic * rmic;
                  str_l_tally += lstr * lstr;
                  str_r_tally += rstr * rstr;
                  rms_tally_count++;	/* the divisor for the above running counts */
                  }
               mic_l_meansqrd = mic_l_tally/rms_tally_count;	/* calculations done here for */
               mic_r_meansqrd = mic_r_tally/rms_tally_count;	/* reasons of thread safety */
               str_l_meansqrd = str_l_tally/rms_tally_count;
               str_r_meansqrd = str_r_tally/rms_tally_count;
               }
            else
               if (simple_mixer == TRUE)
                  {
                  if (left_audio)
                     {
                     if (dj_audio_level != current_dj_audio_level)
                        {
                        current_dj_audio_level = dj_audio_level;
                        dj_audio_gain = db2level(dj_audio_level);
                        }
                     samples_todo = nframes;
                     while(samples_todo--)
                        {
                        *lap++ = *lplcp++ * dj_audio_gain;
                        *rap++ = *lprcp++ * dj_audio_gain;
                        }
                     }
                  else
                     {
                     memset(la_buffer, 0, nframes * sizeof (sample_t));
                     memset(ra_buffer, 0, nframes * sizeof (sample_t));
                     }
                  if (left_stream)
                     {
                     memcpy(ls_buffer, lp_lc, nframes * sizeof (sample_t));
                     memcpy(rs_buffer, lp_rc, nframes * sizeof (sample_t));
                     }
                  else
                     {
                     memset(ls_buffer, 0, nframes * sizeof (sample_t));
                     memset(rs_buffer, 0, nframes * sizeof (sample_t));
                     }
                  }
               else
                  fprintf(stderr,"Error: no mixer mode was chosen\n");
   }
 
int process (jack_nframes_t nframes, void *arg)
   {
   if(transport_aware)
      {
      jack_position_t pos;
      
      if(jack_transport_query(client, &pos) != JackTransportRolling)
         {
	 process_silence(nframes);
	 return 0;
	 }
      }
   process_audio(nframes);
   return 0;
   }
   
int peak_to_log(float peak)
   {
   if (peak <= 0.0)
      return 128;
   if (peak >= 1.0)
      return 0;
   return (int) (fabsf(level2db(peak)));
   }
   
void my_jack_error_callback(const char *message)
   {
   printf("JACK: %s\n", message);
   fflush(stdout);
   }
   
void interrupt_handler(int data)	/* eats the first ^C */
   {
   static int count = 0;
   
   /* each thread running will call this handler so the number */
   /* needs to be high if we mean to exit in an orderly fashion */
   if (count++ > 7)
      {
      fprintf(stderr, "Mixer exiting due to ^C interrupt.\n");      
      exit(130);
      }
   else
      {
      if (count == 1)
         fprintf(stderr, "Mixer got ^C interrupt.\n");
      /* ignore this interrupt and wait instead for idjc to clean exit */
      sigdelset(&mixersigset, SIGINT);
      }
   }

void segfault_handler(int data)
   {
   printf("Segmentation Fault\n");
   fflush(stdout);
   exit(5);
   }

void jack_shutdown_handler(void *data)
   {
   jack_closed_f = TRUE;
   }

void alarm_handler(int data)
   {
   struct xlplayer *cause;
   
   if ((cause = plr_l)->watchdog_timer++ == 4 ||
       (cause = plr_r)->watchdog_timer++ == 4 ||
       (cause = plr_i)->watchdog_timer++ == 4 ||
       (cause = plr_j)->watchdog_timer++ == 4) 
      {
      if (cause->playmode == PM_INITIATE)
         {
         cause->initial_audio_context = cause->current_audio_context;
         cause->playmode = PM_STOPPED;
         cause->command = CMD_COMPLETE;
         }
      fprintf(stderr, "watchdog timer frozen for one of the media players -- possible bad media file\nshutting down the mixer in one second\n");
      signal(SIGALRM, segfault_handler);
      }
   alarm(1);
   }

void atexit_handler()
   {
   if (client)
      jack_client_close(client);
   }

void send_metadata_update(struct xlp_dynamic_metadata *dm)
   {
   pthread_mutex_lock(&(dm->meta_mutex));
   fprintf(stderr, "dm->data_type = %d\n", dm->data_type);
   if (dm->data_type < DM_JOINED_UC)
      {
      fprintf(stdout, "new_metadata=d%d:%sd%d:%sd%d:%sd9:%09dd9:%09dx\n", strlen(dm->artist), dm->artist, strlen(dm->title), dm->title, strlen(dm->artist_title), dm->artist_title, dm->current_audio_context, dm->rbdelay);
      }
   else
      {
      fprintf(stderr, "send_metadata_update: utf16 chapter info not supported\n");
      }
   dm->data_type = DM_NONE_NEW;
   pthread_mutex_unlock(&(dm->meta_mutex));
   }

int main(int argc, char **argv)
   {
   FILE *fp =stdin;
   char *client_name = 0;
   const char **inport;
   const char **outport;
   int mic_l_peak_db, mic_r_peak_db, str_l_peak_db, str_r_peak_db;
   int mic_l_rms_db, mic_r_rms_db, str_l_rms_db, str_r_rms_db;
   float normrise, normfall;
   float comp_ratio;
   int comp_knee, comp_depth, comp_attack, comp_release, comp_gain, comp_relative_f, comp_ducking_hold, fadeout_f;
   int filterslow;
   int flush_left, flush_right, flush_jingles, flush_interlude;
   float comp_ducking, de_ess;
   int mic_left_invert, mic_right_invert, mic_left_level, mic_right_level,
       mic_left_balance, mic_right_balance;
   int i, new_left_pause, new_right_pause;
   int ngthresh, nghyster, ngatten, ngontime, ngofftime;
   static int old_lmic_on = 0, old_rmic_on = 0, old_aux_on = 0;
   jack_nframes_t nframes;
   char *artist = NULL, *title = NULL;
   double length;
   int sync = FALSE;
   struct rms_calc *lm_fast_rms, *lm_slow_rms, *rm_fast_rms, *rm_slow_rms;

   setenv("LC_ALL", "C", 1);		/* ensure proper sscanf operation */

   if (sigfillset(&mixersigset))
      {
      fprintf(stderr, "sigfillset failed\n");
      return 1;
      }
   signal(SIGINT, interrupt_handler);
   signal(SIGALRM, alarm_handler); 
   signal(SIGSEGV, segfault_handler);

   jack_set_error_function(my_jack_error_callback);
   client_name = (char *) malloc(9 * sizeof (char));
   strcpy(client_name, "idjc-mx");

   atexit(atexit_handler);

   if((client = jack_client_open(client_name, JackUseExactName | JackServerName, NULL, getenv("IDJC_JACK_SERVER"))) == 0)
      {
      printf("IDJC: Error\n");
      fflush(stdout);
      return 1;
      }
      
   jack_set_process_callback (client, process, NULL);
      
   /* create the jack ports by which sound is communicated through the jack system */
   /* this one normally connects to the sound card output */
   audio_left_port = jack_port_register(client, "aud_lt", JACK_DEFAULT_AUDIO_TYPE, 
   					JackPortIsOutput, 0);
   audio_right_port = jack_port_register(client, "aud_rt", JACK_DEFAULT_AUDIO_TYPE, 
   					JackPortIsOutput, 0);
   /* this one connects to the server module which in turn streams to the internet */
   stream_left_port = jack_port_register(client, "str_lt", JACK_DEFAULT_AUDIO_TYPE, 
   					JackPortIsOutput, 0);
   stream_right_port = jack_port_register(client, "str_rt", JACK_DEFAULT_AUDIO_TYPE, 
   					JackPortIsOutput, 0);
   /* connects to the capture port normally, which can be set to things other than just mic */
   mic_left_channel = jack_port_register(client, "mic_lt", JACK_DEFAULT_AUDIO_TYPE,
   					JackPortIsInput, 0);
   mic_right_channel = jack_port_register(client, "mic_rt", JACK_DEFAULT_AUDIO_TYPE,
   					JackPortIsInput, 0);
   /* intended as a spare for connection to almost any jack app you like */
   aux_left_channel = jack_port_register(client, "aux_lt", JACK_DEFAULT_AUDIO_TYPE,
   					JackPortIsInput, 0);
   aux_right_channel = jack_port_register(client, "aux_rt", JACK_DEFAULT_AUDIO_TYPE,
   					JackPortIsInput, 0);
   /* inteneded for connection to a VOIP application */
   phone_left_send = jack_port_register(client, "voip_send_lt", JACK_DEFAULT_AUDIO_TYPE,
      					JackPortIsOutput, 0);
   phone_right_send = jack_port_register(client, "voip_send_rt", JACK_DEFAULT_AUDIO_TYPE,
      					JackPortIsOutput, 0);
   phone_left_recv = jack_port_register(client, "voip_recv_lt", JACK_DEFAULT_AUDIO_TYPE,
      					JackPortIsInput, 0);
   phone_right_recv = jack_port_register(client, "voip_recv_rt", JACK_DEFAULT_AUDIO_TYPE,
      					JackPortIsInput, 0);

   sr = jack_get_sample_rate(client);
   jingles_samples_cutoff = sr / 12;		/* A twelfth of a second early */
   player_samples_cutoff = sr * 0.25;		/* for gapless playback */
      
   if(!(xlplayer_create_fadein_lookup(0.07F, sr)))
      {
      printf("failed to create the fadein lookup table for xlplayer\n");
      return 1;
      }

   if(! ((plr_l = xlplayer_create(sr, RB_SIZE, "leftplayer", &jack_closed_f)) &&
         (plr_r = xlplayer_create(sr, RB_SIZE, "rightplayer", &jack_closed_f))))
      {
      printf("failed to create main player modules\n");
      return 1;
      }
   
   if (!(plr_j = xlplayer_create(sr, RB_SIZE, "jinglesplayer", &jack_closed_f)))
      {
      printf("failed to create jingles player module\n");
      return 1;
      }

   if (!(plr_i = xlplayer_create(sr, RB_SIZE, "interludeplayer", &jack_closed_f)))
      {
      printf("failed to create interlude player module\n");
      return 1;
      }

   /* microphone audio filters */
   sharp_left_mic_filter = new_digital_filter(a_cheby100_count, a_cheby100, b_cheby100_count, b_cheby100, cheby100_filter_gain);
   sharp_right_mic_filter = new_digital_filter(a_cheby100_count, a_cheby100, b_cheby100_count, b_cheby100, cheby100_filter_gain);
   
   mild_left_mic_filter = new_digital_filter(a_butterworth90_count, a_butterworth90, b_butterworth90_count, b_butterworth90, butterworth90_filter_gain);
   mild_right_mic_filter = new_digital_filter(a_butterworth90_count, a_butterworth90, b_butterworth90_count, b_butterworth90, butterworth90_filter_gain);
   
   lm_fast_rms = rms_calc_create(sr, 3.0F   / 1000.0F);	/* used for RMS calculation in compressor */
   rm_fast_rms = rms_calc_create(sr, 3.0F   / 1000.0F);
   lm_slow_rms = rms_calc_create(sr, 30.0F  / 1000.0F);
   rm_slow_rms = rms_calc_create(sr, 30.0F  / 1000.0F);
   left_mic_compressor.rms_filter = lm_rms_filter = lm_fast_rms;
   right_mic_compressor.rms_filter = rm_rms_filter = rm_fast_rms;

   left_mic_filter = mild_left_mic_filter;
   right_mic_filter = mild_right_mic_filter;
   
   /* filter for the compressor de-ess mechanism */
   ess_left_filter = new_digital_filter(a_ess8k12k_count, a_ess8k12k, b_ess8k12k_count, b_ess8k12k, ess8k12k_gain);
   ess_right_filter = new_digital_filter(a_ess8k12k_count, a_ess8k12k, b_ess8k12k_count, b_ess8k12k, ess8k12k_gain);
   
   /* filter for the noisegate favouring the vocal audio range */
   ng_left_filter = new_digital_filter(a_ngbandpasscount, a_ngbandpass, b_ngbandpasscount, b_ngbandpass, ngbandpassgain);
   ng_right_filter = new_digital_filter(a_ngbandpasscount, a_ngbandpass, b_ngbandpasscount, b_ngbandpass, ngbandpassgain);
   
   if (mild_left_mic_filter == NULL || mild_right_mic_filter == NULL || 				sharp_left_mic_filter == NULL || sharp_right_mic_filter == NULL || ess_left_filter == NULL || ess_right_filter == NULL)
      {
      fprintf(stderr, "Failed to allocate filter\n");
      exit(5);
      }

   if (!init_dblookup_table())
      {
      fprintf(stderr, "Failed to allocate space for signal to db lookup table\n");
      exit(5);
      }
      
   if (!init_signallookup_table())
      {
      fprintf(stderr, "Failed to allocate space for db to signal lookup table\n");
      exit(5);
      } 
      
   /* generate the wave table for the DJ alarm */
   if (!(eot_alarm_table = calloc(sizeof (sample_t), sr)))
      {
      fprintf(stderr, "Failed to allocate space for end of track alarm wave table\n");
      exit(5);
      }
   else
      {
      alarm_size = (sr / 900) * 900;	/* builds the alarm tone wave table */
      for (i = 0; i < alarm_size ; i++)	/* note it has a nice 2nd harmonic added */
         {
         eot_alarm_table[i] = 0.83F * sinf((i % (sr/900)) * 6.283185307F / (sr/900));
         eot_alarm_table[i] += 0.024F * sinf((i % (sr/900)) * 12.56637061F / (sr/900) + 3.141592654F / 4.0F);
         }
      }
         
   /* computes the decay curve for track fadeouts */
   if (!(fade_table = malloc(FB_SIZE * sizeof(float))))
      {
      fprintf(stderr, "Failed to allocate space for fade table\n");
      exit(5);
      }
   else
      {
      plr_l->fadeindex = FB_SIZE;
      plr_r->fadeindex = FB_SIZE;
      plr_j->fadeindex = FB_SIZE;
      plr_i->fadeindex = FB_SIZE;
      for (i = 0; i < FB_SIZE; i ++)
         fade_table[i] = pow10f(i / -20000.0);
      }
         
   nframes = jack_get_sample_rate (client);
   lp_lc = ialloc(nframes);
   lp_rc = ialloc(nframes);
   rp_lc = ialloc(nframes);
   rp_rc = ialloc(nframes);
   jp_lc = ialloc(nframes);
   jp_rc = ialloc(nframes);
   ip_lc = ialloc(nframes);
   ip_rc = ialloc(nframes);
   lp_lcf = ialloc(nframes);
   lp_rcf = ialloc(nframes);
   rp_lcf = ialloc(nframes);
   rp_rcf = ialloc(nframes);
   jp_lcf = ialloc(nframes);
   jp_rcf = ialloc(nframes);
   ip_lcf = ialloc(nframes);
   ip_rcf = ialloc(nframes);   
         
   if (!(lp_lc && lp_rc && rp_lc && rp_rc && jp_lc && jp_rc && ip_lc && ip_rc &&         
       lp_lcf && lp_rcf && rp_lcf && rp_rcf && jp_lcf && jp_rcf && ip_lcf && ip_rcf))
      {
      fprintf(stderr, "Failed to allocate read-buffers for player_reader reading\n");
      exit(5);
      }
         
#if 0
   beat_lp = beat_init(sr, 2.4, 48);	/* initialise beat analysis engine */
   beat_rp = beat_init(sr, 2.4, 48);
#endif

   jack_on_shutdown(client, jack_shutdown_handler, NULL);

   if (jack_activate(client))
      {
      fprintf(stderr, "Failed to activate client\n");
      return 1;
      }
   
   /* report the sample rate back to the main app where it is used to calibrate mplayer */
   /* the main app waits on this signal in order to prevent a race with the server code */
   fprintf(stdout, "IDJC: Sample rate %d\n", (int)sr);
   fflush(stdout);
 
   		/* Scan for physical audio IO ports to use as defaults */
   inport = jack_get_ports(client, NULL, NULL, JackPortIsPhysical | JackPortIsOutput);
   outport = jack_get_ports(client, NULL, NULL, JackPortIsPhysical | JackPortIsInput);
 
   /* Make voip input audio available on the voip output so you can hear your own voice */
   /* and hear jingles and mixed back audio too */
   /* If the voip app provides this you can break the connection automatically */
   /* using the events feature in idjc prefs and the jack_disconnect program */
   /*jack_connect(client, "idjc-mx:voip_send_lt", "idjc-mx:voip_recv_lt");
   jack_connect(client, "idjc-mx:voip_send_rt", "idjc-mx:voip_recv_rt");*/
      
   alarm(3);		/* handles timeouts on media player worker threads */
      
   while (kvp_parse(kvpdict, fp))
      {
      if (jack_closed_f == TRUE || shutdown == TRUE)
         break;
      if (!strcmp(action, "sync"))
         {
         fprintf(stdout, "IDJC: sync reply\n");
         fflush(stdout);
         sync = TRUE;
         }
      if (sync == FALSE)
         continue;

      if (!strcmp(action, "playleft"))
         {
         fprintf(stdout, "context_id=%d\n", xlplayer_play(plr_l, playerpathname, atoi(seek_s), atoi(size)));
         fflush(stdout);
         }
      if (!strcmp(action, "playright"))
         {
         fprintf(stdout, "context_id=%d\n", xlplayer_play(plr_r, playerpathname, atoi(seek_s), atoi(size)));
         fflush(stdout);
         }
      if (!strcmp(action, "playnoflushleft"))
         {
         fprintf(stdout, "context_id=%d\n", xlplayer_play_noflush(plr_l, playerpathname, atoi(seek_s), atoi(size)));
         fflush(stdout);
         }
      if (!strcmp(action, "playnoflushright"))
         {
         fprintf(stdout, "context_id=%d\n", xlplayer_play_noflush(plr_r, playerpathname, atoi(seek_s), atoi(size)));
         fflush(stdout);
         }
      
      if (!strcmp(action, "playmanyjingles"))
         {
         fprintf(stdout, "context_id=%d\n", xlplayer_playmany(plr_j, playerplaylist, loop[0]=='1'));
         fflush(stdout);
         }
      if (!strcmp(action, "playmanyinterlude"))
         {
         fprintf(stdout, "context_id=%d\n", xlplayer_playmany(plr_i, playerplaylist, loop[0]=='1'));
         fflush(stdout);
         }

      if (!strcmp(action, "stopleft"))
         xlplayer_eject(plr_l);
      if (!strcmp(action, "stopright"))
         xlplayer_eject(plr_r);
      if (!strcmp(action, "stopjingles"))
         xlplayer_eject(plr_j);
      if (!strcmp(action, "stopinterlude"))
         xlplayer_eject(plr_i);
 
      if (!strcmp(action, "dither"))
         {
         xlplayer_dither(plr_l, TRUE);
         xlplayer_dither(plr_r, TRUE);
         xlplayer_dither(plr_j, TRUE);
         xlplayer_dither(plr_i, TRUE);
         }

      if (!strcmp(action, "dontdither"))
         {
         xlplayer_dither(plr_l, FALSE);
         xlplayer_dither(plr_r, FALSE);
         xlplayer_dither(plr_j, FALSE);
         xlplayer_dither(plr_i, FALSE);
         }
      
      if (!strcmp(action, "resamplequality"))
         {
         plr_l->rsqual = plr_r->rsqual = plr_j->rsqual = plr_i->rsqual = resamplequality[0] - '0';
         }
      
      if (!strcmp(action, "ogginforequest"))
         {
         if (vorbinfo(oggpathname, &artist, &title, &length))
            {
            fprintf(stdout, "OIR:ARTIST=%s\nOIR:TITLE=%s\nOIR:LENGTH=%f\nOIR:end\n", artist, title, length);
            fflush(stdout);
            }
         else
            {
            fprintf(stdout, "OIR:NOT VALID\n");
            fflush(stdout);
            }
         }
      
      if (!strcmp(action, "sndfileinforequest"))
         sndfileinfo(sndfilepathname);

#ifdef HAVE_MP4FF
      if (!strcmp(action, "mp4inforequest"))
         mp4_file_info(mp4pathname);
      if (!strcmp(action, "mp4readtagrequest"))
         mp4_tag_read(mp4pathname);
      if (!strcmp(action, "mp4writetagrequest"))
         mp4_tag_write(mp4pathname, mp4taglist);
#endif

#ifdef HAVE_AVCODEC
#ifdef HAVE_AVFORMAT
      if (!strcmp(action, "avformatinforequest"))
         avformatinfo(avformatpathname);
#endif
#endif

      if (!strcmp(action, "micmixstats"))
         {
	 if (sscanf(microphone_string, ":%d:%d:%d:%d:%d:%d:%d:%f:%d:", &mic_filter,
	 	 &mic_left_invert, &mic_right_invert, &mic_left_level, &mic_right_level,
		 &mic_left_balance, &mic_right_balance,
		 &headroom_db,
		 &mic_to_dj) != 9)
	    {
	    fprintf(stderr, "mixer got bad microphone string\n");
	    break;
	    }
	 mic_linv = mic_left_invert ? -1.0 : 1.0;
	 mic_rinv = mic_right_invert ? -1.0 : 1.0;
	 mic_lgain = pow(10.0, mic_left_level / 20.0);
	 mic_rgain = pow(10.0, mic_right_level / 20.0);
	 mic_l_lc = cos(mic_left_balance / 63.66197724);
	 mic_l_rc = sin(mic_left_balance / 63.66197724);
	 mic_r_lc = cos(mic_right_balance / 63.66197724); 
	 mic_r_rc = sin(mic_right_balance / 63.66197724);
	 }
      
      if (!strcmp(action, "normalizerstats"))
         {
         if (sscanf(normalizer_string, ":%f:%f:%f:%f:%d:", &new_normalizer.maxlevel,
         	&new_normalizer.ceiling, &normrise, &normfall, &new_normalizer.active) != 5)
            {
            fprintf(stderr, "mixer got bad normalizer string\n");
            break;
            }
         new_normalizer.rise = 1.0F / normrise;
         new_normalizer.fall = 1.0F / normfall;
         new_normalizer_stats = TRUE;
         }
               
      if (!strcmp(action, "compstats"))
         {
	 if(sscanf(compressor_string,":%f:%d:%d:%d:%d:%d:%d:%f:%d:%f:%d:", &comp_ratio, &comp_knee, &comp_depth, &comp_attack, &comp_release, &comp_gain, &comp_relative_f, &comp_ducking, &comp_ducking_hold, &de_ess, &filterslow) != 11)
	    {
	    fprintf(stderr, "mixer got bad compressor string\n");
	    break;
	    }
         if (filterslow)
            {
            lm_rms_filter = lm_slow_rms;
            rm_rms_filter = rm_slow_rms;
            }
         else
            {
            lm_rms_filter = lm_fast_rms;
            rm_rms_filter = rm_fast_rms;
            }
	 new_mic_compressor.gain_db = 0.0;
	 new_mic_compressor.k1 = comp_knee;
	 new_mic_compressor.k2 = comp_knee - comp_depth;
	 new_mic_compressor.ratio = comp_ratio;
	 new_mic_compressor.attack = 1.0F;
	 new_mic_compressor.release = 1.0F / (float)comp_release;
	 new_mic_compressor.ducking = comp_ducking;
	 new_mic_compressor.ducking_hold = comp_ducking_hold;
	 new_mic_compressor.de_ess_db = 0.0;
	 if (comp_relative_f)
	    {
	    new_mic_compressor.opgain = 0.0;
	    new_mic_compressor.opgain = (-compressor(&new_mic_compressor, 1.0, 1)) + comp_gain;
	    }
	 else
	    new_mic_compressor.opgain = comp_gain;
	 
	 new_mic_compressor.attack = 1.0F / (float)comp_attack;
	 new_mic_compressor.de_ess_db = de_ess;
	 new_compressor_stats = TRUE;
	 }

      if (!strcmp(action, "gatestats"))
         {
         if (sscanf(gate_string, ":%d:%d:%d:%d:%d:%d:", &ngthresh, &nghyster, &new_ng.offdelay, &ngatten, &ngontime, &ngofftime) != 6)
	    {
	    fprintf(stderr, "got bad gatestats string\n");
	    break;
	    }
         new_ng.thresh = (float)ngthresh;
         new_ng.hyster = (float)nghyster;
         new_ng.attenuation = (float)ngatten;
         new_ng.ontime = 1.0F / (float)ngontime;
         new_ng.offtime = 1.0F / (float)ngofftime;
	 new_noisegate_stats = TRUE;
	 }	 
       
      if (!strcmp(action, "mixstats"))
         {
         if(sscanf(mixer_string,
                ":%03d:%03d:%03d:%03d:%03d:%d:%1d%1d%1d%1d%1d:%1d%1d:%1d%1d%1d:%1d%1d%1d%1d:%1d:%1d:%1d:%1d:%1d:%f:%f:%1d:%f:",
      		&volume, &crossfade, &jinglesvolume, &interludevol, &mixbackvol, &jingles_playing,
                &left_stream, &left_audio, &right_stream, &right_audio, &stream_monitor,
		&new_left_pause, &new_right_pause, &lmic_on, &rmic_on, &aux_on, &flush_left, &flush_right, &flush_jingles, &flush_interlude, &simple_mixer, &eot_alarm_f, &mixermode, &fadeout_f, &main_play, &(plr_l->newpbspeed), &(plr_r->newpbspeed), &speed_variance, &dj_audio_level) !=29)
	    {
	    fprintf(stderr, "mixer got bad mixer string\n");
	    break;
	    }

         plr_l->fadeout_f = plr_r->fadeout_f = plr_j->fadeout_f = plr_i->fadeout_f = fadeout_f;

         if (new_left_pause != plr_l->pause)
            {
            if (new_left_pause)
               xlplayer_pause(plr_l);
            else
               xlplayer_unpause(plr_l);
            }
            
         if (new_right_pause != plr_r->pause)
            {
            if (new_right_pause)
               xlplayer_pause(plr_r);
            else
               xlplayer_unpause(plr_r);
	    }

	 if (lmic_on != old_lmic_on)
            lmic_flux = TRUE;
         old_lmic_on = lmic_on;
         
         if (rmic_on != old_rmic_on)
            rmic_flux = TRUE;
         old_rmic_on = rmic_on;
         
         if (aux_on != old_aux_on)
            aux_flux = TRUE;
         old_aux_on = aux_on;
         }
      /* the means by which the jack ports are connected to the soundcard */
      /* notably the main app requests the connections */
      if (!strcmp(action, "remakemicl"))
         {
         jack_port_disconnect(client, mic_left_channel);
	 if (strcmp(micl, "default"))
            {
            if (micl[0] != '\0')
	       jack_connect(client, micl, "idjc-mx:mic_lt");
            }
	 else
	    if (inport)
	       jack_connect(client, inport[0], "idjc-mx:mic_lt");
	 }
      if (!strcmp(action, "remakemicr"))
         {
	 jack_port_disconnect(client, mic_right_channel);
	 if (strcmp(micr, "default"))
            {
            if (micr[0] != '\0')
	       jack_connect(client, micr, "idjc-mx:mic_rt");
            }
	 else
	    if (inport && inport[1])
	       jack_connect(client, inport[1], "idjc-mx:mic_rt");
	 }
      if (!strcmp(action, "remakeaudl"))
         {
	 jack_port_disconnect(client, audio_left_port);
	 if (strcmp(audl, "default"))
            {
            if (audl[0] != '\0')
	       jack_connect(client, "idjc-mx:aud_lt", audl);
            }
	 else
	    if (outport)
	       jack_connect(client, "idjc-mx:aud_lt", outport[0]);
	 }
      if (!strcmp(action, "remakeaudr"))
         {
	 jack_port_disconnect(client, audio_right_port);
	 if (strcmp(audr, "default"))
            {
            if (audr[0] != '\0')
	       jack_connect(client, "idjc-mx:aud_rt", audr);
            }
	 else
	    if (outport && outport[1])
	       jack_connect(client, "idjc-mx:aud_rt", outport[1]);
	 }
      if (!strcmp(action, "remakestrl"))
         {
	 jack_port_disconnect(client, stream_left_port);
	 jack_connect(client, "idjc-mx:str_lt", "idjc-sc:lt_in");	/* this is kludgy */
	 if (strcmp(strl, "default"))
            {
            if (strl[0] != '\0')
	       jack_connect(client, "idjc-mx:str_lt", strl);
            }
	 else
	    if (outport && outport[1] && outport[2] && outport[3] && outport[4])
	       jack_connect(client, "idjc-mx:str_lt", outport[4]);
	    else
	       if(outport && outport[1] && outport[2])
	          jack_connect(client, "idjc-mx:str_lt", outport[2]);
	 }
      if (!strcmp(action, "remakestrr"))
         {
	 jack_port_disconnect(client, stream_right_port);
	 jack_connect(client, "idjc-mx:str_rt", "idjc-sc:rt_in");
	 if (strcmp(strr, "default"))
            {
            if (strr[0] != '\0')
	       jack_connect(client, "idjc-mx:str_rt", strr);
            }
	 else
	    if (outport && outport[1] && outport[2] && outport[3] && outport[4] && outport[5])
	       jack_connect(client, "idjc-mx:str_rt", outport[5]);
	    else
	       if (outport && outport[1] && outport[2] && outport[3])
	          jack_connect(client, "idjc-mx:str_rt", outport[3]);
	 }
      if (!strcmp(action, "remakeauxl"))
         {
	 jack_port_disconnect(client, aux_left_channel);
         if (auxl[0] != '\0')
	    jack_connect(client, auxl, "idjc-mx:aux_lt");
	 }
      if (!strcmp(action, "remakeauxr"))
         {
	 jack_port_disconnect(client, aux_right_channel);
         if (auxr[0] != '\0')
	    jack_connect(client, auxr, "idjc-mx:aux_rt");
	 }
      if (!strcmp(action, "serverbind"))
         {
         fprintf(stderr, "remaking connection to server\n");
         jack_connect(client, "idjc-mx:str_lt", "idjc-sc:lt_in"); 
         jack_connect(client, "idjc-mx:str_rt", "idjc-sc:rt_in");
         }
      if (!strcmp(action, "requestlevels"))
         {
         timeout = 0;		/* the main app has proven it is alive */
	 /* make logarithmic values for the peak levels */
	 mic_l_peak_db = peak_to_log(mic_lpeak);
	 mic_r_peak_db = peak_to_log(mic_rpeak);
	 str_l_peak_db = peak_to_log(str_lpeak);
	 str_r_peak_db = peak_to_log(str_rpeak);
	 /* set reply values for a totally blank signal */
	 mic_l_rms_db = mic_r_rms_db = str_l_rms_db = str_r_rms_db = 120;
	 /* compute the rms values */
	 if (mic_l_meansqrd)
	    mic_l_rms_db = (int) fabs(level2db(sqrt(mic_l_meansqrd)));
	 if (mic_r_meansqrd)
	    mic_r_rms_db = (int) fabs(level2db(sqrt(mic_r_meansqrd)));
	 if (str_l_meansqrd)
	    str_l_rms_db = (int) fabs(level2db(sqrt(str_l_meansqrd)));
	 if (str_r_meansqrd)
	    str_r_rms_db = (int) fabs(level2db(sqrt(str_r_meansqrd)));
	    
         /* send the meter and other stats to the main app */
	 fprintf(stdout, "mic_l_peak=%d\nmic_r_peak=%d\n"
	 		 "str_l_peak=%d\nstr_r_peak=%d\n"
			 "mic_l_rms=%d\nmic_r_rms=%d\n"
			 "str_l_rms=%d\nstr_r_rms=%d\n"
			 "left_compressor_gain=%d\n"
			 "right_compressor_gain=%d\n"
			 "jingles_playing=%d\n"
			 "left_elapsed=%d\n"
			 "right_elapsed=%d\n"
            		 "left_playing=%d\n"
            		 "right_playing=%d\n"
            		 "interlude_playing=%d\n"
            		 "left_signal=%d\n"
            		 "right_signal=%d\n"
            		 "left_cid=%d\n"
            		 "right_cid=%d\n"
            		 "jingles_cid=%d\n"
            		 "interlude_cid=%d\n"
            		 "left_audio_runout=%d\n"
            		 "right_audio_runout=%d\n"
            		 "left_additional_metadata=%d\n"
            		 "right_additional_metadata=%d\n"
			 "end\n",
			 mic_l_peak_db, mic_r_peak_db, str_l_peak_db, str_r_peak_db,
			 mic_l_rms_db, mic_r_rms_db, str_l_rms_db, str_r_rms_db,
			 (int)fabs(left_mic_compressor.gain_db),
			 (int)fabs(right_mic_compressor.gain_db),
            		 jingles_audio_f | (plr_j->current_audio_context & 0x1),
            		 plr_l->play_progress_ms / 1000,
            		 plr_r->play_progress_ms / 1000,
            		 plr_l->have_data_f | (plr_l->current_audio_context & 0x1),
            		 plr_r->have_data_f | (plr_r->current_audio_context & 0x1),
            		 plr_i->have_data_f | (plr_i->current_audio_context & 0x1),
            		 left_peak > 0.0008F || left_peak < 0.0F || plr_l->pause,
            		 right_peak > 0.0008F || right_peak < 0.0F || plr_r->pause,
                         plr_l->current_audio_context,
                         plr_r->current_audio_context,
                         plr_j->current_audio_context,
                         plr_i->current_audio_context,
            		 left_audio_runout && (!(plr_l->current_audio_context & 0x1)),
            		 right_audio_runout && (!(plr_r->current_audio_context & 0x1)),
            		 plr_l->dynamic_metadata.data_type,
            		 plr_r->dynamic_metadata.data_type);
	 /* tell the jack mixer it can reset its vu stats now */
	 reset_vu_stats_f = TRUE;
	 if (plr_l->dynamic_metadata.data_type)
	    send_metadata_update(&(plr_l->dynamic_metadata));
	 if (plr_r->dynamic_metadata.data_type)
	    send_metadata_update(&(plr_r->dynamic_metadata));
	 fflush(stdout);
         }
      }
   jack_client_close(client);
   client = NULL;
   rms_calc_destroy(lm_fast_rms);
   rms_calc_destroy(lm_slow_rms);
   rms_calc_destroy(rm_fast_rms);
   rms_calc_destroy(rm_slow_rms);
   /*beat_free(beat_lp);
   beat_free(beat_rp);*/
   ifree(lp_lc);
   ifree(lp_rc);
   ifree(rp_lc);
   ifree(rp_rc);
   ifree(jp_lc);
   ifree(jp_rc);
   ifree(ip_lc);
   ifree(ip_rc);
   ifree(lp_lcf);
   ifree(lp_rcf);
   ifree(rp_lcf);
   ifree(rp_rcf);
   ifree(jp_lcf);
   ifree(jp_rcf);
   ifree(ip_lcf);
   ifree(ip_rcf);
   xlplayer_destroy(plr_l);
   xlplayer_destroy(plr_r);
   xlplayer_destroy(plr_j);
   xlplayer_destroy(plr_i);
   xlplayer_destroy_fadein_lookup();
   fprintf(stderr, "Mixer module has closed\n");
   return 0;
   }
