/*
**	$Id: aspectcorrection.c 937 2006-08-10 09:37:18Z gromeck $
**
**	Copyright (c) 2004 by Christian Lorenz
**
**	do the aspect correction stuff
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <config.h>
#include <math.h>
#include "gip.h"

/*
**	init the API
*/
int GIP_aspectcorrection_init(void)
{
	return 1;
}

/*
**	shutdown the API
*/
int GIP_aspectcorrection_exit(void)
{
	return 1;
}

/*
**	convert a aspectcorrection name into the corresponding type code
*/
int GIP_aspectcorrection_str2type(const char *s)
{
#define GIP_ASPECTCORRECTION_STR2TYPE(type)	if (!strcasecmp(s,# type) || !strcasecmp(s,"GIP_ASPECTCORRECTION_TYPE_" # type)) \
										return GIP_ASPECTCORRECTION_TYPE_ ## type;
	GIP_ASPECTCORRECTION_STR2TYPE(AUTO);
	GIP_ASPECTCORRECTION_STR2TYPE(NONE);
	GIP_ASPECTCORRECTION_STR2TYPE(FITIN);
	GIP_ASPECTCORRECTION_STR2TYPE(FILLIN);
	return GIP_ASPECTCORRECTION_TYPE_NONE;
#undef GIP_ASPECTCORRECTION_STR2TYPE
}

/*
**	convert a aspectcorrection type code into the corresponding name
*/
const char *GIP_aspectcorrection_type2str(int type)
{
#define GIP_ASPECTCORRECTION_TYPE2STR(type)	case GIP_ASPECTCORRECTION_TYPE_ ## type:	return # type;
	switch (type) {
		GIP_ASPECTCORRECTION_TYPE2STR(AUTO);
		GIP_ASPECTCORRECTION_TYPE2STR(NONE);
		GIP_ASPECTCORRECTION_TYPE2STR(FITIN);
		GIP_ASPECTCORRECTION_TYPE2STR(FILLIN);
	}
	return NULL;
#undef GIP_ASPECTCORRECTION_TYPE2STR
}

/*
**	do the aspect correction for the given image
*/
GIP_IMAGE_T *GIP_aspectcorrection_process(GIP_IMAGE_T *img,int width,int height,double aspectratio,int aspectcorrection)
{
	double img_aspectratio;
	GIP_IMAGE_T *newimg = NULL;

	img_aspectratio = (double) img->width / (double) img->height;

	if (aspectcorrection == GIP_ASPECTCORRECTION_TYPE_AUTO) {
		/*
		**	find out, if we should use fit-in, fill-in, or none
		*/
#if 1
		if (img_aspectratio > 1.0)
			aspectcorrection = GIP_ASPECTCORRECTION_TYPE_FILLIN;
		else
			aspectcorrection = GIP_ASPECTCORRECTION_TYPE_FITIN;
#else
#define ASPECT_DELTA	0.7
		if (fabs(img_aspectratio - aspectratio) < ASPECT_DELTA) {
			/*
			**	the input image is more or less
			**	in the same aspect ration like
			**	the expected aspect ratio:
			*/
			aspectcorrection = GIP_ASPECTCORRECTION_TYPE_FILLIN;
		}
		else if (fabs(1 / img_aspectratio - aspectratio) < ASPECT_DELTA) {
			/*
			**	input seems to be portrait orientated,
			**	but the rotated input image is more or
			**	less in the same aspect ration like
			**	the expected aspect ratio:
			*/
			aspectcorrection = GIP_ASPECTCORRECTION_TYPE_FITIN;
		}
		else
			aspectcorrection = GIP_ASPECTCORRECTION_TYPE_NONE;
#endif
	}

	switch (aspectcorrection) {
		case GIP_ASPECTCORRECTION_TYPE_FITIN:
			/*
			**	do the fit-in:
			**
			**	black bars, but the whole image is seen
			*/
			if (img_aspectratio < aspectratio) {
				/*
				**	the resulting image will have black
				**	bars on the left and right side
				*/
				if (!(newimg = GIP_image_create(GIP_COLOR_STR_BLACK,
						img->height * aspectratio,img->height)))
					CRITICAL("couldn't create image\n");
				GIP_image_overlay(newimg,(newimg->width - img->width) / 2,0,img);
				DEBUG("FIT-IN(bars on the left+right side): %dx%dpx to %dx%dpx\n",
					img->width,img->height,
					newimg->width,newimg->height);
			}
			else {
				/*
				**	the resulting image will have black
				**	bars on the top and bottom side
				*/
				if (!(newimg = GIP_image_create(GIP_COLOR_STR_BLACK,
						img->width,img->width / aspectratio)))
					CRITICAL("couldn't create image\n");
				GIP_image_overlay(newimg,0,(newimg->height - img->height) / 2,img);
				DEBUG("FIT-IN(bars on the top+bottom side): %dx%dpx to %dx%dpx\n",
					img->width,img->height,
					newimg->width,newimg->height);
			}
			break;
		case GIP_ASPECTCORRECTION_TYPE_FILLIN:
			/*
			**	do the fill-in:
			**
			**	no black bars but part of the image is cut off
			*/
			if (img_aspectratio < aspectratio) {
				/*
				**	the resulting image will be cut-off
				**	on the top and bottom side
				*/
				if (!(newimg = GIP_image_create(GIP_COLOR_STR_BLACK,
						img->width,img->width / aspectratio)))
					CRITICAL("couldn't create image\n");
				GIP_image_bitblit(newimg,0,0,img,0,(img->height - newimg->height) / 2,newimg->width,newimg->height);
				DEBUG("FILL-IN(cut-off on the top+bottom side): %dx%dpx to %dx%dpx\n",
					img->height,img->width,
					newimg->height,newimg->width);
			}
			else {
				/*
				**	the resulting image will be cut-off
				**	on the left and right side
				*/
				if (!(newimg = GIP_image_create(GIP_COLOR_STR_BLACK,
						img->height * aspectratio,img->height)))
					CRITICAL("couldn't create image\n");
				GIP_image_bitblit(newimg,0,0,img,(img->width - newimg->width) / 2,0,newimg->width,newimg->height);
				DEBUG("FILL-IN(cut-off on the left+right side): %dx%dpx to %dx%dpx\n",
					img->height,img->width,
					newimg->height,newimg->width);
			}
			break;
		case GIP_ASPECTCORRECTION_TYPE_NONE:
		default:
			/*
			**	no correction needed?
			*/
			break;
	}

	if (newimg) {
		/*
		**	swap the images if we got a new one
		*/
		GIP_image_free(img);
		img = newimg;
	}
	return img;
}/**/
