/*
**	$Id: geometry.c 1085 2007-07-31 08:48:10Z gromeck $
**
**	Copyright (c) 2004 by Christian Lorenz
**
**	handle geometries
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <string.h>
#include <config.h>
#include <math.h>
#include "gip.h"

#define MIN(a,b)		(((a) < (b)) ? (a) : (b))
#define MAX(a,b)		(((a) > (b)) ? (a) : (b))

/*
**	init the API
*/
int GIP_geometry_init(void)
{
	return 1;
}

/*
**	shutdown the API
*/
int GIP_geometry_exit(void)
{
	return 1;
}

/*
**	find the first char out of sep
*/
static char *strfindsep(const char *s,char *sep)
{
	while (*s) {
		if (strchr(sep,*s))
			return (char *) s;
		s++;
	}
	return NULL;
}

/*
**	parse a geometry string and return the resulting
**	absolute dimensions
**
**	geometry strings are of the form
**
**		<width>[x<height>][{+-}<x>[{+-}<y>]][%]
*/
static int geometry_parse(const char *geo,
		double image_width,double image_height,
		double frame_width,double frame_height,
		double *x,double *y,double *w,double *h)
{
	char *s;
	int no_x = 1,no_y = 1;
	int neg_x = 0,neg_y = 0;

	DEBUG("geo:%s  image:%fx%fpx  frame:%fx%fpx  x:%p  y:%p  w:%p  h:%p\n",geo,image_width,image_height,frame_width,frame_height,w,h,x,y);

	/*
	**	no geometry => use the default
	*/
	if (!geo)
		geo = GIP_GEOMETRY_PERCENT_100;

	/*
	**	init
	*/
	*x = *y = *w = *h = 0;

	/*
	**	find and parse w and h
	*/
	*w = abs(atoi(geo));
	*h = ((s = strchr(geo,'x'))) ? abs(atoi(s + 1)) : *w;

	/*
	**	find and parse x and y
	**	if not x or y is used, we will
	**	set them to center the selected area
	*/
	if ((s = strfindsep(geo,"+-"))) {
		no_x = 0;
		neg_x = (*s == '-');
		*x = abs(atoi(s));
		if ((s = strfindsep(s + 1,"+-"))) {
			no_y = 0;
			neg_y = (*s == '-');
			*y = abs(atoi(s));
		}
	}

	/*
	**	solve relative values
	*/
	if (strchr(geo,'%')) {
		*x = image_width * *x / 100.0;
		*y = image_height * *y / 100.0;
		*w = image_width * *w / 100.0;
		*h = image_height * *h / 100.0;
	}

	/*
	**	solve missing w or h
	*/
	if (*w <= 0) {
		*w = *h * frame_width / frame_height;
	}
	else if (*h <= 0) {
		*h = *w * frame_height / frame_width;
	}

	/*
	**	solve negative x and y values
	*/
	if (neg_x)
		*x = image_width - *x - *w;
	if (neg_y)
		*y = image_height - *y - *h;

	/*
	**	solve unset x and y values
	*/
	if (no_x)
		*x = (image_width - *w) / 2.0;
	if (no_y)
		*y = (image_height - *h) / 2.0;
	
	DEBUG("geo:%s  image:%fx%fpx  frame:%fx%fpx => %fx%f+%f+%f\n",geo,image_width,image_height,frame_width,frame_height,*w,*h,*x,*y);
	return 1;
}

/*
**	parse a geometry string and return the resulting
**	absolute dimensions
**
**	geometry strings are of the form
**
**		<width>[x<height>][{+-}<x>[{+-}<y>]][%]
*/
int GIP_geometry_parse(const char *geo,
		int image_width,int image_height,
		int frame_width,int frame_height,
		int *x,int *y,int *w,int *h)
{
	double dx,dy,dw,dh;

	DEBUG("geo:%s  image:%dx%dpx  frame:%dx%dpx  x:%p  y:%p  w:%p  h:%p\n",geo,image_width,image_height,frame_width,frame_height,w,h,x,y);

	geometry_parse(geo,image_width,image_height,frame_width,frame_height,&dx,&dy,&dw,&dh);
	*x = lround(dx);
	*y = lround(dy);
	*w = lround(dw);
	*h = lround(dh);
	
	DEBUG("geo:%s  image:%dx%dpx  frame:%dx%dpx => %dx%d+%d+%d\n",geo,image_width,image_height,frame_width,frame_height,*w,*h,*x,*y);
	return 1;
}

/*
**	process the geometries and return the absolute
**	dimensions in dependance of the current step
*/
int GIP_geometry_process(const char *from_geo,const char *to_geo,
				int image_width,int image_height,
				int frame_width,int frame_height,
				int step,int steps,
				int *x,int *y,int *w,int *h)
{
	double from_x,from_y,from_w,from_h;
	double to_x,to_y,to_w,to_h;

	DEBUG("from_geo:%p  to_geo:%p\n",
		from_geo,
		to_geo);
	DEBUG("from_geo:%s  to_geo:%s  image:%dx%dpx  frame:%dx%dpx  step:%d/%d\n",
		from_geo,
		to_geo,
		image_width,image_height,
		frame_width,frame_height,
		step,steps);

	/*
	**	parse the geometries
	*/
	if (!geometry_parse(from_geo,image_width,image_height,frame_width,frame_height,&from_x,&from_y,&from_w,&from_h))
		return 0;
	if (!geometry_parse(to_geo,image_width,image_height,frame_width,frame_height,&to_x,&to_y,&to_w,&to_h))
		return 0;

	/*
	**	compute the dimensions in dependance of the current step
	*/
	steps--;
	if (step <= 0) {
		*x = lround(from_x);
		*y = lround(from_y);
		*w = lround(from_w);
		*h = lround(from_h);
	}
	else if (step >= steps) {
		*x = lround(to_x);
		*y = lround(to_y);
		*w = lround(to_w);
		*h = lround(to_h);
	}
	else {
		*x = lround(from_x + (to_x - from_x) * (double) step / (double) steps);
		*y = lround(from_y + (to_y - from_y) * (double) step / (double) steps);
		*w = lround(from_w + (to_w - from_w) * (double) step / (double) steps);
		*h = lround(from_h + (to_h - from_h) * (double) step / (double) steps);
	}

	*x = MIN(MAX(*x,0),image_width - 1);
	*y = MIN(MAX(*y,0),image_height - 1);
	*w = MIN(MAX(*w,0),image_width - 1 - *x);
	*h = MIN(MAX(*h,0),image_height - 1 - *y);
	DEBUG("from_geo:%s  to_geo:%s  image:%dx%dpx  frame:%dx%dpx  step:%d/%d => %dx%d+%d+%d\n",
		from_geo,
		to_geo,
		image_width,image_height,
		frame_width,frame_height,
		step,steps,
		*w,*h,*x,*y);
	return 1;
}/**/
